<?php

/**
 * Functionality to create, check and update the plugins custom database tables.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- This file contains the functionality required to create or update the custom database tables required by the plugin. Direct database queries and schema changes are required to support this. Caching would not be appropriate.

class Logtastic_DB_Utilities {

	/**
	 * An array of options/settings for the plugin
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array    $settings    Plugin options/settings
	 */
	protected $settings;


	/**
	 * The default database structure
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array    $db_structure    Default database structure
	 */
	protected $db_structure;


	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $settings = null ) {
		
		// If settings passed
		if ( is_array( $settings ) ) {

			// Add settings passed to __construct to $settings variable
			$this->settings = $settings;

		} else {

			// Load settings
			$this->settings = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME );

		}
		
		// Load default settigns (if not already loaded)
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/default-settings/class-logtastic-default-settings_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';

		// Set $db_structure from default settings
		$this->db_structure = Logtastic_Default_Settings::$db_structure;

	}


    /**
	 * Function to check and update the database structure, for all active logs, to match the default structure stored in $db_structure
	 *
	 * @since      	1.0.0
	 * @return bool True if all active logs successfully checked and updated, false if any failed.
	 */
	public function check_and_update_db_structure_all_active_logs() {

		global $wpdb;

		// Assume success unless a failure occurs
		$success = true;

		// Get the active logs
		if ( !empty( $this->settings['enabled_logs'] ) ) {
			$active_logs = array_filter( $this->settings['enabled_logs'] );
		}

		// Loop over each enabled log
		if ( isset( $active_logs ) ) {
			
			foreach ( $active_logs as $log_id => $active ) {

				// Check and update table structure
				$result = $this->check_and_update_db_structure_single_log( $log_id );
				if ( false === $result ) {
					$success = false;
				}

			}

		}

		return $success;

	}
	
	
	/**
	 * Check and update the database structure for a specific log.
	 *
	 * Iterates over each table in the defined database structure for the given log,
	 * creating, updating, or deleting tables as needed.
	 *
	 * @since 1.0.0
	 *
	 * @param string $log_id The ID of the log whose database structure should be verified and updated.
	 * @return bool True if all database operations succeeded, false if any failed.
	 */
	public function check_and_update_db_structure_single_log( $log_id ) {

		global $wpdb;

		// Assume success unless a failure occurs
		$success = true;

		// Get the table structure for the specified log
		$log_table_structure = $this->db_structure[ $log_id ] ?? false;

		// Loop over each defined table in the db structure
		if ( is_array( $log_table_structure ) ) {

			foreach ( $log_table_structure as $table_name => $columns ) {

				$full_table_name = $wpdb->prefix . $table_name;

				// Delete table if explicitly set to false
				if ( false === $columns ) {

					// Delete table
					$deleted = $this->delete_table( $full_table_name );

					if ( false === $deleted ) {
						$success = false;
					}

				// Handle tables defined as arrays of columns
				} else if ( is_array( $columns ) ) {

					// Does the table already exist?
					if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $full_table_name ) ) != $full_table_name ) {
						
						// Table doesn't exist, create it
						$created = $this->create_table( $full_table_name, $columns );

						if ( false === $created ) {
							$success = false;
						}

					} else {

						// Table exists, check columns
						$updated = $this->update_table( $full_table_name, $columns );

						if ( false === $updated ) {
							$success = false;
						}

					}

				}

			}

		} else {

			// Invalid or missing structure definition
			$success = false;

		}

		return $success;

	}


	/**
	 * Create a new DB table.
	 *
	 * @since 1.0.0
	 * @param string $table_name Full table name (with prefix).
	 * @param array  $columns    Column definitions (your schema array).
	 * @return bool True on success, false on any failure.
	 */
	private function create_table( $table_name, $columns ) {

		global $wpdb;

		// Assume success unless a failure occurs
		$success = true;

		// Reset $wpdb->last_error
		$wpdb->last_error = ''; 
	
		// Retrieve the database character collate
		$charset_collate = $wpdb->get_charset_collate();

		// Begin sql query definition
		$sql = "CREATE TABLE $table_name (";
	
		// Create empty array to store column definitions
		$column_definitions = [];

		// Loop over each column in $columns array and add column definition to array
		foreach ($columns as $column_name => $options) {
			$column_definitions[] = $this->get_column_definition_prepared( $column_name, $options );
		}
	
		// Implode column definitions array and append to sql query
		$sql .= implode(', ', $column_definitions);

		// Get unique columns (excluding primary)
		$unique_cols = array_keys(
			array_filter($columns, function ( $column ) {
				return
					( ! empty( $column['unique'] ) ) &&
					( empty( $column['primary_key'] ) );
			})
		);

		// If unique column(s) found, define unique keys and append to sql query
		if (!empty($unique_cols)) {
			foreach ( $unique_cols as $col ) {
				$sql .= ", UNIQUE KEY " . $col . "_unique (" . $col . ")";
			}
		}
	
		// Get primary key column(s)
		$primary_keys = array_keys(array_filter($columns, function($column) {
			return $column['primary_key'] ?? false;
		}));

		// If primary key column(s) found, define primary key and append to sql query
		if (!empty($primary_keys)) {
			$sql .= ", PRIMARY KEY (" . implode(', ', $primary_keys) . ")";
		}
	
		// Complete sql query and append charset collate
		$sql .= ") $charset_collate;";
	
		// Create table
		require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
		dbDelta($sql);

		// Check for errors
		if ( ! empty( $wpdb->last_error ) ) {
			$success = false;
		}

		return $success;

	}
	
	
	/**
	 * Update an existing DB table to match schema.
	 *
	 * @since 1.0.0
	 * @param string $table_name Full table name (with prefix).
	 * @param array  $columns    Column definitions (your schema array).
	 * @return bool True if all operations succeed, false on any failure.
	 */
	private function update_table( $table_name, $columns ) {
		
		global $wpdb;

		// Assume success unless a failure occurs
		$success = true;

		// Reset $wpdb->last_error
		$wpdb->last_error = ''; 
	
		// Get existing columns
		$existing_columns = $wpdb->get_results( $wpdb->prepare( 'SHOW COLUMNS FROM %i', $table_name ), ARRAY_A );
		if ( false === $existing_columns || ! empty( $wpdb->last_error ) ) {
			return false;
		}
		// Get existing column names
		$existing_column_names = array_column( $existing_columns, 'Field' );

		// Get table indexes
		$indexes = $wpdb->get_results( $wpdb->prepare( 'SHOW INDEXES FROM %i', $table_name ), ARRAY_A );
		if ( false === $indexes || ! empty( $wpdb->last_error ) ) {
			return false;
		}

		// Create variables to store primary column and unique keys
		$primary_cols_db = [];
		$unique_cols_db = [];

		// Get primary column and unique keys from indexes
		foreach ( $indexes as $index ) {
			// Primary Key
			if ( $index['Key_name'] === 'PRIMARY' ) {
				$primary_cols_db[] = $index['Column_name'];
			// Unique keys
			} else if ( !empty( $index['Key_name'] ) && $index['Non_unique'] == '0' ) {
				$unique_cols_db[] = $index['Column_name'];
			}
		}

		// Loop over each column in column definition array	
		foreach ( $columns as $column_name => $options ) {

			// Check if column exists
			if ( !in_array( $column_name, $existing_column_names ) ) {

				// No, add missing column
				
				// phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- $this->get_column_definition_prepared() is prepared/validated within function
				$result = $wpdb->query(
					$wpdb->prepare(
						// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $this->get_column_definition_prepared() is prepared/validated within function
						'ALTER TABLE %i ADD ' . $this->get_column_definition_prepared($column_name, $options),
						$table_name
					)
				);

				if ( false === $result || ! empty( $wpdb->last_error ) ) {
					$success = false;
				}

				
			} else {
				
				// Yes, get existing column definition
				$existing_column = current(array_filter($existing_columns, function($col) use ($column_name) {
					return $col['Field'] == $column_name;
				}));
				
				// Check if column needs to be modified
				$needs_update = $this->does_column_need_update($existing_column, $options);
				
				// If needs to be modified, modify
				if ($needs_update) {
					// phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- $this->get_column_definition_prepared() is prepared/validated within function
					$result = $wpdb->query(
						$wpdb->prepare(
							// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- $this->get_column_definition_prepared() is prepared/validated within function
							'ALTER TABLE %i MODIFY ' . $this->get_column_definition_prepared($column_name, $options),
							$table_name
						)
					);
					if ( false === $result || ! empty( $wpdb->last_error ) ) {
						$success = false;
					}
				}
			}
		}
	
		// Drop extra columns
		foreach ( $existing_column_names as $existing_column_name ) {
			if ( !isset( $columns[$existing_column_name] ) ) {
				$result = $wpdb->query(
					$wpdb->prepare(
						'ALTER TABLE %i DROP COLUMN %i',
						$table_name,
						$existing_column_name
					)
				);
				if ( false === $result || ! empty( $wpdb->last_error ) ) {
					$success = false;
				}
			}
		}

		// Check table primary key - extract primary cols from column definintions
		$primary_cols_definitions = array_keys(array_filter($columns, function($column) {
			return $column['primary_key'] ?? false;
		}));

		// If primary cols defined in db and/or primary cols defined in col definitions, check 
		if ( !empty( $primary_cols_db ) || !empty( $primary_cols_definitions ) ) {

			sort($primary_cols_db);
			sort($primary_cols_definitions);

			// If primary cols from db are not the same as primary cols from definitions
			if ( $primary_cols_db != $primary_cols_definitions ) {

				// If existing primary key in database, drop
				if ( !empty( $primary_cols_db ) ) {
					$result = $wpdb->query(
						$wpdb->prepare(
							'ALTER TABLE %i DROP PRIMARY KEY',
							$table_name
						)
					);
					if ( false === $result || ! empty( $wpdb->last_error ) ) {
						$success = false;
					}
				}

				// If new key defined in definitions, add to database
				if ( !empty( $primary_cols_definitions ) ) {
					$primary_key_placeholders = implode( ', ', array_fill( 0, count( $primary_cols_definitions ), '%i' ) );
					$result = $wpdb->query(
						$wpdb->prepare(
							// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $primary_key_placeholders - defined above, outputs %i, %i, etc. depending on number of primary columns
							"ALTER TABLE %i ADD PRIMARY KEY ({$primary_key_placeholders})",
							array_merge( [ $table_name ], $primary_cols_definitions )
						)
					);
					if ( false === $result || ! empty( $wpdb->last_error ) ) {
						$success = false;
					}
				}

			}

		}

		// Check table unique keys - extract unique cols from column definintions
		$unique_cols_definitions = array_keys(
			array_filter($columns, function ( $column ) {
				return
					( ! empty( $column['unique'] ) ) &&
					( empty( $column['primary_key'] ) );
			})
		);

		// If unique cols defined in db and/or unique cols defined in col definitions, check 
		if ( !empty( $unique_cols_db ) || !empty( $unique_cols_definitions ) ) {

			sort( $unique_cols_db );
			sort( $unique_cols_definitions );

			// If unique cols from db are not the same as unique cols from definitions
			if ( $unique_cols_db != $unique_cols_definitions ) {

				// Loop over all unique cols defined in DB
				foreach( $unique_cols_db as $col ) {

					// If not present in table definitions, drop
					if ( !in_array( $col, $unique_cols_definitions ) ) {
						$result = $wpdb->query( 
							$wpdb->prepare( 
								"ALTER TABLE %i DROP INDEX %i",
								$table_name,
								'unique_' . $col
							)
						);
						if ( false === $result || ! empty( $wpdb->last_error ) ) {
							$success = false;
						}
					}

				}

				// Loop over all unique cols from column definitions
				foreach( $unique_cols_definitions as $col ) {

					// If not present in db array, add index to table
					if ( !in_array( $col, $unique_cols_db ) ) {
						$result = $wpdb->query( 
							$wpdb->prepare( 
								"ALTER TABLE %i ADD UNIQUE INDEX %i (%i)",
								$table_name,
								'unique_' . $col,
								$col
							)
						);
						if ( false === $result || ! empty( $wpdb->last_error ) ) {
							$success = false;
						}
					}

				}

			}

		}

		return $success;

	}


	/**
	 * Delete a table from the database
	 *
	 * @since      	 1.0.0
	 * @param string $table_name	Full table name (with prefix).
	 * @return bool True if table successfully delted, false on failure.
	 */
	private function delete_table( $table_name ) {

		global $wpdb;

		// Assume success unless a failure occurs
		$success = true;

		// Reset $wpdb->last_error
		$wpdb->last_error = ''; 

		$result = $wpdb->query( $wpdb->prepare( "DROP TABLE IF EXISTS %i", $table_name ) );

		if ( false === $result || ! empty( $wpdb->last_error ) ) {
			$success = false;
		}

		return $success;

	}
	

	/**
	 * Function to prepare column definition SQL syntax based on passed array of options
	 *
	 * @since      	 1.0.0
	 * @param string $column_name	The name of the database column to be created/updated 
	 * @param array  $options		An array of options for the database column
	 */
		private function get_column_definition_prepared( $column_name, $options ) {

		global $wpdb;

		// Column Name
		$definition = "%i";
		$variables = array (
			$column_name
		);

		// Data Type
		$valid_types = [
			'TINYINT', 'SMALLINT', 'MEDIUMINT', 'INT', 'INTEGER', 'BIGINT',
			'DECIMAL', 'DEC', 'NUMERIC', 'FLOAT', 'DOUBLE', 'REAL',
			'DATE', 'DATETIME', 'TIMESTAMP', 'TIME', 'YEAR',
			'CHAR', 'VARCHAR', 'BINARY', 'VARBINARY',
			'TINYTEXT', 'TEXT', 'MEDIUMTEXT', 'LONGTEXT',
			'TINYBLOB', 'BLOB', 'MEDIUMBLOB', 'LONGBLOB',
			'ENUM', 'SET',
			'JSON', 'BIT',
			'GEOMETRY', 'POINT', 'LINESTRING', 'POLYGON',
			'MULTIPOINT', 'MULTILINESTRING', 'MULTIPOLYGON', 'GEOMETRYCOLLECTION'
		];
		if ( in_array( $options['data_type'], $valid_types ) ) {
			$definition .= " {$options['data_type']}";
		} else {
			return false;
		}
		
		// Length	
		if (!empty($options['length'])) {
			$definition .= "(%d)";
			$variables[] = $options['length'];
		}
	
		// Unsigned
		if (!empty($options['unsigned'])) {
			$definition .= " UNSIGNED";
		}
	
		// Not Null
		if (empty($options['allow_null'])) {
			$definition .= " NOT NULL";
		}
	
		// Auto Increment
		if (!empty($options['auto_increment'])) {
			$definition .= " AUTO_INCREMENT";
		}
	
		// Default value
		if (!empty($options['default'])) {
			$definition .= " DEFAULT %s";
			$variables[] = $options['default'];
		}
	
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $definition built dynamically above including placeholders, $variables passed as array
		return $wpdb->prepare( $definition, $variables );

	}
	

	/**
	 * Function to compare column definitions between an existing db column and the default options and check if an update is required. 
	 * Returns true if any of the checks fail and the column needs updating, otherwise returns false if column does not need updating
	 *
	 * @since      	 1.0.0
	 * @param array $column_name	The name of the database column to be created/updated 
	 * @param array  $options		An array of options for the database column
	 * 
	 */
	private function does_column_need_update($existing_column, $options) {

		// Check column type definition (including data type, length and signed/unsigned)
		if ( $existing_column['Type'] != $this->get_column_type($options) ) {
			return true;
		}
	
		// Check allow null
		if ( !empty( $options['allow_null'] ) && $existing_column['Null'] != 'YES') {
			return true;
		}

		// Check auto increment
		if ( !empty( $options['auto_increment'] ) && ( false === strpos( strtolower( $existing_column['Extra'] ), 'auto_increment' ) ) ) {
			return true;
		}
	
		// Check default value
		if ( !empty( $options['default'] ) && $existing_column['Default'] != $options['default']) {
			return true;
		}
	
		return false;

	}
	

	/**
	 * Function to define a column type defination (including data type, length and unsigned)
	 *
	 * @since      	 1.0.0
	 * @param array  $options		An array of options for the database column
	 * 
	 */
	private function get_column_type( $options ) {
		
		$type = $options['data_type'];
	
		if (!empty($options['length'])) {
			$type .= "({$options['length']})";
		}
	
		if (!empty($options['unsigned'])) {
			$type .= " UNSIGNED";
		}
	
		return $type;

	}

	
	/**
	 * Gemerate an array of the plugin's custom database table names
	 * This is used during deactivation to delete the custom tables
	 *
	 * @since      	 1.0.0
	 * 
	 */
	public function list_all_db_tables() {
		
		$table_names = [];

		foreach ( $this->db_structure as $log_id => $log_db_structure ) {
		
			foreach ( $log_db_structure as $table_name => $table_columns ) {
				
				$table_names[] = $table_name;
				
			}

		}
		
		return $table_names;
		
	}

}