<?php

/**
 * The scheduled tasks functionality for the JavaScript Error Log.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}


class Logtastic_JS_Error_Log_Scheduled_Tasks {

	/**
	 * The JS error table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_TABLE_NAME = 'logtastic_js_errors';


	/**
	 * The JS error occurrences table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_OCCURRENCES_TABLE_NAME = 'logtastic_js_error_occurrences';


	/**
	 * The JS error stack traces table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_STACK_TRACES_TABLE_NAME = 'logtastic_js_error_stack_traces';
	
	
	/**
	 * Delete expired error occurrences from the database (where the occurrence is older than the max age passed in days)
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$max_age_days 	The maximum age in days
	 * @global 	 wpdb 		$wpdb 			WordPress database abstraction object.
	 */
	private function delete_expired_occurrences( int $max_age_days ) {
		
		global $wpdb;

		// Prepare and execute the query
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->query(
			$wpdb->prepare(
				"
					DELETE FROM %i
					WHERE timestamp < NOW() - INTERVAL %d DAY
				",
				array( 
					$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
					$max_age_days
				)
				
			)
		);
		
	}
	
	
	/**
	 * Delete errors with no recorded error occurrences from the database (where the error is older than the max age passed in days and is not set to ignore)
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$max_age_days 	The maximum age in days
	 * @global 	 wpdb 		$wpdb 			WordPress database abstraction object.
	 */
	private function delete_errors_with_no_occurrences( int $max_age_days ) {
		
		global $wpdb;

		// Prepare and execute the query
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->query(
			$wpdb->prepare(
				"
					DELETE e
					FROM %i e
					LEFT JOIN %i o 
						ON e.error_id = o.error_id
					WHERE e.ignore_error = 0
						AND e.timestamp < NOW() - INTERVAL %d DAY
						AND o.error_id IS NULL
				",
				array( 
					$wpdb->prefix . self::ERROR_TABLE_NAME,
					$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
					$max_age_days
				)	
			)
		);
		
	}

	
	/**
	 * Delete error stack traces from the database where there is no corresponding error occurrence (eg. the error occurrence has been deleted)
	 *
	 * @since 1.0.0
	 *
	 * @global 	 wpdb 		$wpdb 			WordPress database abstraction object.
	 */
	private function delete_orphaned_stack_traces() {
		
		global $wpdb;

		// Prepare and execute the query
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->query(
			$wpdb->prepare(
				"
					DELETE s
					FROM %i s 
					LEFT JOIN %i oa 
						ON s.stack_trace_id = oa.stack_trace_array_id
					LEFT JOIN %i os 
						ON s.stack_trace_id = os.stack_trace_string_id	
					WHERE oa.stack_trace_array_id IS NULL
						AND os.stack_trace_string_id IS NULL
				",
				array( 
					$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
					$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
					$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME
				)	
			)
		);
		
	}
	

	/**
	 * Delete orphaned error occurrneces from the database where there is no corresponding error (eg. the error has been deleted)
	 *
	 * @since 1.0.0
	 *
	 * @global 	 wpdb 		$wpdb 			WordPress database abstraction object.
	 */
	private function delete_orphaned_error_occurrences() {
		
		global $wpdb;

		// Prepare and execute the query
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->query(
			$wpdb->prepare(
				"
					DELETE o
					FROM %i o
					LEFT JOIN %i e 
						ON o.error_id = e.error_id
					WHERE e.error_id IS NULL
				",
				array( 
					$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
					$wpdb->prefix . self::ERROR_TABLE_NAME
				)	
			)
		);
		
	}
	

	/**
	 * Process the actions for the JavaScript error log retention schedule (called via scheduled task)
	 *
	 * @since 1.0.0
	 *
	 * @global 	 wpdb 		$wpdb 			WordPress database abstraction object.
	 */
	public function process_js_error_log_retention_schedule() {
		
		// Get php error log settings
		$log_settings = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_js_error_log');
		
		// Get max age to keep records for from settings
		if ( isset( $log_settings['retention_schedule'] ) && is_numeric( $log_settings['retention_schedule'] ) ) {
			$max_age_days = intval( $log_settings['retention_schedule'] ) ;
		}
		
		// If max age greater than 0, process cleanup
		if ( isset( $max_age_days ) && 0 != $max_age_days ) {
		
			// Delete expired occurrences
			$this->delete_expired_occurrences( $max_age_days );
			
			// Delete errors older than one hour with no occurrences, where ignore is set to 0
			$this->delete_errors_with_no_occurrences( $max_age_days );
			
			// Delete orphaned occurrences (there shouldn't really be any)
			$this->delete_orphaned_error_occurrences();
			
			// Delete orphaned stack traces
			$this->delete_orphaned_stack_traces();
			
		}
		
	}
	
}

