<?php

/**
 * The error logging functionality for the PHP Error Log.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Base class for logging PHP errors.
 *
 * @since 1.0.0
 */
class Logtastic_PHP_Error_Logger {
	

	/**
	 * The current plugin settings.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $settings;
	

	/**
	 * PHP error level names.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $error_levels = [
			E_ERROR => "e_error",
			E_WARNING => "e_warning",
			E_PARSE => "e_parse",
			E_NOTICE => "e_notice",
			E_CORE_ERROR => "e_core_error",
			E_CORE_WARNING => "e_core_warning",
			E_COMPILE_ERROR => "e_compile_error",
			E_COMPILE_WARNING => "e_compile_warning",
			E_USER_ERROR => "e_user_error",
			E_USER_WARNING => "e_user_warning",
			E_USER_NOTICE => "e_user_notice",
			E_RECOVERABLE_ERROR => "e_recoverable_error",
			E_DEPRECATED => "e_deprecated",
			E_USER_DEPRECATED => "e_user_deprecated",
			E_ALL => "e_all"
	];


	/**
	 * The php error table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_TABLE_NAME = 'logtastic_php_errors';


	/**
	 * The php error occurrences table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_OCCURRENCES_TABLE_NAME = 'logtastic_php_error_occurrences';


	/**
	 * The php error stack traces table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_STACK_TRACES_TABLE_NAME = 'logtastic_php_error_stack_traces';


	/**
	 * Stores the previously registered PHP eerror handler.
	 *
	 * Used so the plugin can call the original handler after logging
	 * an uncaught exception, preserving default or other plugin behavior.
	 *
	 * @since 1.0.0
	 * @var callable|null
	*/
	public $previous_error_handler;


	/**
	 * Stores the previously registered PHP exception handler.
	 *
	 * Used so the plugin can call the original handler after logging
	 * an uncaught exception, preserving default or other plugin behavior.
	 *
	 * @since 1.0.0
	 * @var callable|null
	*/
	public $previous_exception_handler;
	
	
	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	function __construct() {
		
		$this->settings = $this->get_settings();
		
	}
	
	/**
	 * Attempt to find a matching error in the database, based on error level, file, line and message.
	 * If a match is found, return error_id, source_type, source_slug and ignore_error
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 * @param array 	$error 		The current error array
	 */
	public function get_php_error_details($error) { 
		
		global $wpdb;
		
		// Prepare and execute the query
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table, Caching would not be appropriate in this context.
		$result = $wpdb->get_row(
			$wpdb->prepare(
				"
					SELECT error_id, source, source_slug, ignore_error FROM %i
					WHERE error_level = %s
					AND file = %s
					AND line = %d
					AND message = %s
					LIMIT 1
				",
				array (
					$wpdb->prefix . self::ERROR_TABLE_NAME,
					$error['type'],
					$error['file'],
					$error['line'],
					$error['message']
				)
			)
		);
		
		// Update the $error array based on the results
		if ($result) {
			// Record found, mark as existing = false and add additional details to error array 
			$error['existing'] = true;
			$error['error_id'] = $result->error_id;
			if ( isset( $result->source ) ) {
				$error['source'] = $result->source;
			}
			if ( isset( $result->source_slug ) ) {
				$error['source_slug'] = $result->source_slug;
			}
			if ( isset( $result->ignore_error ) ) {
				$error['ignore_error'] = $result->ignore_error;
			}
		} else {
			// No record found, mark as existing = false 
			$error['existing'] = false;
		}
		
		// Return the updated error array
		return $error;
	}
	

	/**
	 * Create a new php error record in the database
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 * @param array 	$error 		The current error array
	 */
	public function add_new_php_error($error) { 
		
		global $wpdb;
		
		// Insert new error
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct query as custom table
		$wpdb->insert(
			$wpdb->prefix . self::ERROR_TABLE_NAME,
			array(
				'error_level' => $error['type'],
				'file' => $error['file'],
				'line' => $error['line'],
				'source' => $error['source'],
				'source_slug' => $error['source_slug'],
				'message' => $error['message'],
				'timestamp' => current_time( 'mysql' ),
			),
			array(
				'%d',
				'%s',
				'%d',
				'%s',
				'%s',
				'%s',
				'%s'
			)
		);
		
		// Add the error ID of the inserted error to the error array
		$error['error_id'] = $wpdb->insert_id;
		
		// Return the updated error array
		return $error;
		
	}
	

	/**
	 * Map a stack-trace argument to an array containing its type and value.
	 *
	 * Accepts any variable and returns a structured array with:
	 *  - 'type'    => the PHP data type of the variable (gettype()).
	 *                 The special case "double" is normalized to "float".
	 *  - 'content' => the original variable value itself.
	 *
	 * Used when generating a stack trace so that each argument
	 * can be logged along with its PHP type.
	 *
	 * @since 1.0.0
	 *
	 * @param mixed $arg The argument value to inspect.
	 * @return array{type:string,content:mixed} Array with the variable type and its content.
	 */
	public function add_arg_types_to_stacktrace_array_map($arg) {
		
		// Get the variable type for the passed variable
		$type = gettype( $arg );
		
		// If the variable is of type 'double', change to 'float'
		if ( $type == 'double' ) {
			$type = 'float';
		}
		
		// Return as an array containing the 'type' and the originally passed $arg as 'content'
		return [
			'type' => $type,
			'content' => $arg
		];
		
	}
	
	
	/**
	 * Add type information to each argument inside a stack trace.
	 *
	 * Iterates through a PHP debug backtrace array and, for every stack trace
	 * frame that contains an 'args' key, replaces the raw argument list with
	 * an array of structures describing both the argument value and its PHP type.
	 * Uses add_arg_types_to_stacktrace_array_map() for the actual mapping.
	 *
	 * @since 1.0.0
	 *
	 * @param array $stacktrace A PHP stack trace (e.g. from debug_backtrace()).
	 * @return array The same stack trace with each 'args' entry transformed into
	 *               an array of ['type' => string, 'content' => mixed] pairs.
	 */
	public function add_arg_types_to_stacktrace( array $stacktrace ) {

		// Loop through each stack frame in the trace.
		foreach ( $stacktrace as &$stacktraceitem ) {

			// If this frame contains an 'args' key and it is an array of arguments…
			if ( isset( $stacktraceitem['args'] ) && is_array( $stacktraceitem['args'] ) ) {

				// Replace each raw argument with a structured array describing
				// the argument's PHP type and value.
				$stacktraceitem['args'] = array_map(
					[ $this, 'add_arg_types_to_stacktrace_array_map' ],
					$stacktraceitem['args']
				);
			}
		}

		// Return the enriched stack trace.
		return $stacktrace;
	}


	/**
	 * Recursively replace PHP resources within a data structure with descriptive strings.
	 *
	 * Walks through an array or object and converts any PHP resource values
	 * (open or closed) into a human-readable description such as:
	 *   resource(5) of type (stream)
	 *
	 * @since 1.0.0
	 *
	 * @param mixed 	$data Array, object, or scalar value that may contain resources.
	 * @return mixed    The same data structure with any resources replaced
	 *                  by descriptive strings.
	 */
	public function convert_resources_to_description($data) {

		// If $data is an array, iterate its elements.
		if ( is_array( $data ) ) {
			foreach ( $data as $key => $value ) {

				if ( is_resource( $value ) ) {

					// Replace open resource with an id/type description.
					$data[ $key ] = 'resource(' . get_resource_id( $value ) . ') of type (' . get_resource_type( $value ) . ')';

				} elseif ( gettype( $value ) === 'resource (closed)' ) {

					// Replace closed resource with a similar description.
					$data[ $key ] = 'resource(' . get_resource_id( $value ) . ') of type (' . get_resource_type( $value ) . ')';

				} elseif ( is_array( $value ) || is_object( $value ) ) {

					// Recurse into nested arrays/objects.
					$data[ $key ] = $this->convert_resources_to_description( $value );
				}
			}

		// If $data is an object, iterate its properties.
		} elseif ( is_object( $data ) ) {
			foreach ( $data as $key => $value ) {

				if ( is_resource( $value ) ) {

					// Replace open resource with an id/type description.
					$data->$key = 'resource(' . get_resource_id( $value ) . ') of type (' . get_resource_type( $value ) . ')';

				} elseif ( gettype( $value ) === 'resource (closed)' ) {

					// Replace closed resource with a similar description.
					$data->$key = 'resource(' . get_resource_id( $value ) . ') of type (' . get_resource_type( $value ) . ')';

				} elseif ( is_array( $value ) || is_object( $value ) ) {

					// Recurse into nested arrays/objects.
					$data->$key = $this->convert_resources_to_description( $value );

				}
			}
		}

		return $data;
	}
	

	/**
	 * Recursively sanitize a value so it can be safely serialized or JSON-encoded.
	 *
	 * Handles arrays, objects, closures, and circular references to produce a
	 * structure of only scalar values and arrays that can be stored or logged.
	 * Specific behavior:
	 *  - Arrays: Each element is processed recursively.
	 *  - Closures: Returned as the string "[closure]".
	 *  - Objects: Converted to an array containing:
	 *      - '__class__'      => the object's class name
	 *      - '__properties__' => an array of sanitized property values
	 *    Circular references are detected via spl_object_hash() and replaced
	 *    with the string "[circular_reference]" to avoid infinite recursion.
	 *  - Scalars: Returned as-is.
	 *
	 * @since 1.0.0
	 *
	 * @param mixed $data Data to sanitize (array, object, scalar, etc.).
	 * @param array $seen Optional. Internal reference map of processed objects
	 *                    used to detect circular references. Passed by reference.
	 * @return mixed Sanitized value suitable for safe serialization or JSON output.
	 */
	protected function sanitize_for_serialization($data, &$seen = []) {
		
		if (is_array($data)) {
			$clean = [];
			foreach ($data as $key => $value) {
				$clean[$key] = $this->sanitize_for_serialization($value, $seen);
			}
			return $clean;
		}
	
		if ($data instanceof \Closure) {
			return '[closure]';
		}
	
		if (is_object($data)) {
			// Avoid recursion on already-processed objects
			$oid = spl_object_hash($data);
			if (isset($seen[$oid])) {
				return '[circular_reference]';
			}
			$seen[$oid] = true;
	
			$object_vars = [];
			foreach ((array) $data as $key => $value) {
				$object_vars[$key] = $this->sanitize_for_serialization($value, $seen);
			}
	
			// Optionally include class name info
			return [
				'__class__' => get_class($data),
				'__properties__' => $object_vars,
			];
		}
	
		return $data;
	}


	/**
	 * Recursively redact values from a strack trace arguments if they match a sepecific set of keys.
	 *
	 * Accepts all value types, only processes arrays, all others returned as as.
	 *
	 * @since 1.0.0
	 *
	 * @param mixed $args Data to possibly redact (array, object, scalar, etc.).
	 */
	public function redact_stack_trace_args( $data ) {
		

		if ( is_array( $data ) ) {

			$redact_keys = ['password','pass','pwd','token','secret','api_key','apikey','authorization','cookie'];
			
			$redacted_data = [];
			
			foreach ( $data as $key => $value ) {
				if ( is_string( $key ) && in_array( strtolower( $key ), $redact_keys, true ) ) {
					$redacted_data[ $key ] = '*** REDACTED ***';
				} else {
					$redacted_data[ $key ] = $this->redact_stack_trace_args( $value );
				}
			}

			return $redacted_data;

		}

		// If $args is not an array, return 
		return $data;

	}
	
	/**
	 * Determine whether a matching stack trace already exists in the database.
	 *
	 * Looks up a record in the custom PHP error stack trace table that matches
	 * a specific error ID, trace type, and trace data.
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 *
	 * @param int    $error_id ID of the related error.
	 * @param string $type     Stack trace type identifier.
	 * @param string $trace    Serialized or encoded stack trace data.
	 * @return int|false       The existing stack_trace_id if a match is found,
	 *                         or false if no matching record exists.
	 */
	public function is_existing_stack_trace( int $error_id, string $type, string $trace ) {
		
		global $wpdb;

		// Prepare and execute the query to look for an existing stack trace with the same error ID, type, and data.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table, Caching would not be appropriate in this context.
		$result = $wpdb->get_var(
			$wpdb->prepare(
				"
					SELECT stack_trace_id FROM %i
					WHERE error_id = %d
					AND stack_trace_type = %s
					AND stack_trace_data = %s
					LIMIT 1
				",
				array (
					$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
					$error_id,
					$type,
					$trace
				)
			)
		);

		// Return the stack_trace_id if found, or false if not.
		return $result ?: false;
	}


	/**
	 * Save a new stack trace to the database.
	 *
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 *
	 * @param int    $error_id ID of the related error.
	 * @param string $type     Stack trace type identifier.
	 * @param string $trace    Serialized or encoded stack trace data.
	 * @return int|false       The existing stack_trace_id if a match is found,
	 *                         or false if no matching record exists.
	 */
	public function save_new_stack_trace( int $error_id, string $type, string $trace) {
		
		global $wpdb;
		
		// Insert the stack trace into the database
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->insert(
			$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
			array (
				'error_id' => $error_id,
				'stack_trace_type' => $type,
				'stack_trace_data' => $trace
			)
		);
		
		// Return the stack_trace_id of the inserted entry
		return $wpdb->insert_id;
		
	}
	
	
	/**
	 * Create a new php error occurrence record in the database
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 * @global wp_version $wp_version Current WordPress version number.
	 * @param array 	$error 		The current error array
	 */
	public function log_php_error_occurrence($error) { 
		
		global $wp_version;
		global $wpdb;
		
		$error['stack_trace_available'] = 0;
		
		// If exists, clean and serialize stack trace array
		if ( isset( $error['stack_trace_array'] ) && is_array( $error['stack_trace_array'] ) && !empty ( $error['stack_trace_array'] ) ) {
			
			// Add arg types to array
			$error['stack_trace_array'] = $this->add_arg_types_to_stacktrace( $error['stack_trace_array'] ); 
			
			// Convert any resources in stack trace to description
			$error['stack_trace_array'] = $this->convert_resources_to_description( $error['stack_trace_array'] );
			
			// Serialise for storage
			try {
				$clean_stack_trace = $this->sanitize_for_serialization( $error['stack_trace_array'] );
				$redacted_stack_trace = $this->redact_stack_trace_args( $clean_stack_trace );
				$stack_trace_array_serialized = serialize( $redacted_stack_trace );
			} catch (\Throwable $e) {
				$stack_trace_array_serialized = null;
			}

			
		} else {
			$stack_trace_array_serialized = null;
		}
		
		// If serialised stack trace array, check if stack trace array exists in database and retrieve ID, or else save as new stack trace
		if ( null != $stack_trace_array_serialized ) {
			$existing_stack_trace_array = $this->is_existing_stack_trace($error['error_id'], 'array', $stack_trace_array_serialized);
			if ( false != $existing_stack_trace_array ) {
				$error['stack_trace_array_id'] = $existing_stack_trace_array;
				$error['stack_trace_available'] = 1;
			} else {
				$new_stack_trace_array = $this->save_new_stack_trace($error['error_id'], 'array', $stack_trace_array_serialized);
				if ( false != $new_stack_trace_array ) {
					$error['stack_trace_array_id'] = $new_stack_trace_array;
					$error['stack_trace_available'] = 1;
				} else {
					$error['stack_trace_array_id'] = null;
				}
			}
		} else {
			$error['stack_trace_array_id'] = null;
		}
		
		// If exists, check if string stack trace exists in database and retrieve ID, or else save as new stack trace
		if ( isset( $error['stack_trace_string'] ) && null != $error['stack_trace_string'] ) {
			$existing_stack_trace_string = $this->is_existing_stack_trace($error['error_id'], 'string', $error['stack_trace_string']);
			if ( false != $existing_stack_trace_string ) {
				$error['stack_trace_string_id'] = $existing_stack_trace_string;
				$error['stack_trace_available'] = 1;
			} else {
				$new_stack_trace_string = $this->save_new_stack_trace($error['error_id'], 'string', $error['stack_trace_string']);
				if ( false != $new_stack_trace_string ) {
					$error['stack_trace_string_id'] = $new_stack_trace_string;
					$error['stack_trace_available'] = 1;
				} else {
					$error['stack_trace_string_id'] = null;
				}
			}
		} else {
			$error['stack_trace_string_id'] = null;
		}
		
		// Insert the error occurrence into the database
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->insert(
			$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
			array(
				'error_id' => $error['error_id'],
				'source_version' => $error['source_version'],
				'wp_version' => $wp_version,
				'timestamp' => current_time( 'mysql' ),
				'additional_data' => null,
				'php_version' => PHP_VERSION,
				'stack_trace_array_id' => $error['stack_trace_array_id'],
				'stack_trace_string_id' => $error['stack_trace_string_id'],
				'stack_trace_available' => $error['stack_trace_available'],
				'method' => $error['method']
			),
			array (
				'%d',
				'%d',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%d',
				'%d',
				'%d',
				'%s',
			)
		);
		
	}
	

	/**
	 * Calculates the error source (either 'plugin', 'theme' or 'wp_core' based on the error file path ) and source_slug if applicable (only applicable for plugin and theme errors) and return the updated error array
	 *
	 * @since 1.0.0
	 *
	 * @param array 	$error 		The current error array
	 */
	public function caclulate_error_source($error) {
		
		// Does the file path contain the WP Plugin Directory?
		if ( strpos( $error['file'], WP_PLUGIN_DIR ) !== false ) {
			
			$error['source'] = 'plugin';
			
			$slug_start_pos = strpos( $error['file'], WP_PLUGIN_DIR ) + strlen( WP_PLUGIN_DIR ) + 1;
			$slug_end_pos = strpos( $error['file'], '/', $slug_start_pos );
			$slug_end_pos = strpos( $error['file'], '/', $slug_start_pos );
			$error['source_slug'] = substr( $error['file'], $slug_start_pos, $slug_end_pos - $slug_start_pos );
			
		// Does the file path contain the WP Theme Directory?
		} else if ( strpos( $error['file'], get_theme_root() ) !== false ) {
			
			$error['source'] = 'theme';
			
			$slug_start_pos = strpos( $error['file'], get_theme_root() ) + strlen( get_theme_root() ) + 1;
			$slug_end_pos = strpos( $error['file'], '/', $slug_start_pos );
			$error['source_slug'] = substr( $error['file'], $slug_start_pos, $slug_end_pos - $slug_start_pos );
			
		// Does the error message contain the WP Plugin Directory?
		} else if ( strpos( $error['message'], WP_PLUGIN_DIR ) !== false ) {
			$error['source'] = 'plugin';
			
			$slug_start_pos = strpos( $error['message'], WP_PLUGIN_DIR ) + strlen( WP_PLUGIN_DIR ) + 1;
			$slug_end_pos = strpos( $error['message'], '/', $slug_start_pos );
			$error['source_slug'] = substr( $error['message'], $slug_start_pos, $slug_end_pos - $slug_start_pos );
			
		// Does the error message contain the WP Theme Directory?
		} else if ( strpos( $error['file'], get_theme_root() ) !== false ) {
			$error['source'] = 'theme';
			
			$slug_start_pos = strpos( $error['message'], get_theme_root() ) + strlen( get_theme_root() ) + 1;
			$slug_end_pos = strpos( $error['message'], '/', $slug_start_pos );
			$error['source_slug'] = substr( $error['message'], $slug_start_pos, $slug_end_pos - $slug_start_pos );
		
		// Else label error as 	'wp_core'
		} else {
			
			global $wp_version;
			
			$error['source'] = 'wp_core';
			$error['source_slug'] = null;
			$error['source_version'] = $wp_version;
			
		}
		
		return $error;
		
	}
	

	/**
	 * Gets the version number of a plugin based on the given plugin slug
	 *
	 * @since 1.0.0
	 *
	 * @param string 	$plugin_slug 		The plugin slug
	 */
	public function get_plugin_version($plugin_slug) { 
		
		// Get currently active plugins
		$active_plugins = get_option('active_plugins');
		
		// Match plugin slug to get plugin index file
		foreach ( $active_plugins as $plugin ) {
			if ( strpos( $plugin, $plugin_slug . '/' ) !== false ) {
				$plugin_index = $plugin;
			}
		}
		
		if ( isset( $plugin_index ) ) {
			
			// Get the contents of the main plugin file
			$plugin_file = WP_PLUGIN_DIR . '/' . $plugin_index;
			$plugin_content = file_get_contents($plugin_file);
			
			// Match the version field
			preg_match('/^.*Version:\s*(\S+)/mi', $plugin_content, $matches);
			
			// If result found, return value
			if (!empty($matches[1])) {
				$plugin_version = $matches[1];
				return $plugin_version;
			}
			
		}
			
		return false;
		
	}
	

	/**
	 * Gets the version number of a theme based on the given theme slug
	 *
	 * @since 1.0.0
	 *
	 * @param string 	$theme_slug 		The theme slug
	 */
	public function get_theme_version( $theme_slug ) { 
		
		// Get the theme data
		$theme_data = wp_get_theme( $theme_slug );	
		
		if ( $theme_data !== false ) {
		
			// Get the version
			$theme_version = $theme_data->get('Version');
			
			// Return the value
			return $theme_version;
			
		}
		
		return false;
		
	}
	
	
	/**
	 * The main function to process a php error.
	 * 
	 * @since 1.0.0
	 *
	 * @param array 	$error 		The current error array
	 */
	public function process_php_error( $error ) { 
		// Check that error level should be logged, otherwise return false
		if ( !isset($this->settings['error_levels']) || !is_array($this->settings['error_levels']) || !in_array( $this->error_levels[ $error['type'] ], $this->settings['error_levels'] ) ) {
			return false;
		}
		
		// Attempt to fetch error details from database if existing error
		$error = $this->get_php_error_details( $error );
		
		// If existing error exists and ignore error is set to true, return false
		if ( isset( $error['ignore_error'] ) && $error['ignore_error'] == 1 ) {
			return false;
		}
		
		// If existing error record does not exist, calculate source of error 
		if ( false == $error['existing'] ) {
			$error = $this->caclulate_error_source( $error );
		}
		
		// Check if error source should be logged
		$log_error = false;
		if ( $error['source'] == 'plugin' ) {
			// Should this plugin error be logged?
			if ( isset( $this->settings['scope_wp_plugins'] ) && 'all' == $this->settings['scope_wp_plugins'] ) {
				$log_error = true;
			} else if ( isset( $this->settings['scope_wp_plugins'] ) && 'all_except' == $this->settings['scope_wp_plugins'] ) {
				$log_error = true;
				if ( isset( $this->settings['scope_wp_plugins_excepted'] ) && is_array( $this->settings['scope_wp_plugins_excepted'] ) ) {
					foreach ( $this->settings['scope_wp_plugins_excepted'] as $excepted_slug ) {
						if ( $excepted_slug == $error['source_slug'] ) {
							$log_error = false;
						}
					}
				}
			} else if ( isset( $this->settings['scope_wp_plugins'] ) && 'selected' == $this->settings['scope_wp_plugins'] ) {
				if ( isset( $this->settings['scope_wp_plugins_selected'] ) && is_array( $this->settings['scope_wp_plugins_selected'] ) ) {
					foreach ( $this->settings['scope_wp_plugins_selected'] as $selected_slug ) {
						if ( $selected_slug == $error['source_slug'] ) {
							$log_error = true;
						}
					}
				}
			}
		} else if ( $error['source'] == 'theme' ) {
			// Should this theme error be logged?
			if ( isset( $this->settings['scope_wp_themes'] ) && 'all' == $this->settings['scope_wp_themes'] ) {
				$log_error = true;
			} else if ( isset( $this->settings['scope_wp_themes'] ) && 'all_except' == $this->settings['scope_wp_themes'] ) {
				$log_error = true;
				if ( isset( $this->settings['scope_wp_themes_excepted'] ) && is_array( $this->settings['scope_wp_themes_excepted'] ) ) {
					foreach ( $this->settings['scope_wp_themes_excepted'] as $excepted_slug ) {
						if ( $excepted_slug == $error['source_slug'] ) {
							$log_error = false;
						}
					}
				}
			} else if ( isset( $this->settings['scope_wp_themes'] ) && 'selected' == $this->settings['scope_wp_themes'] ) {
				if ( isset( $this->settings['scope_wp_themes_selected'] ) && is_array( $this->settings['scope_wp_themes_selected'] ) ) {
					foreach ( $this->settings['scope_wp_themes_selected'] as $selected_slug ) {
						if ( $selected_slug == $error['source_slug'] ) {
							$log_error = true;
						}
					}
				}
			}
		} else {
			// Should WP Core Errors be logged?
			if ( isset( $this->settings['scope_wp_core'] ) && true == $this->settings['scope_wp_core'] ) {
				$log_error = true;
			}
		}
		
		// If error should be logged, proceed 
		if ( true == $log_error ) {
		
			// If existing error record does not exist, record new error
			if ( false == $error['existing'] ) {
				$error = $this->add_new_php_error( $error );
			}
			
			// Get source version number (if applicable)
			$error['source_version'] = null;
			if ( $error['source'] == 'plugin' ) {
				$error['source_version'] = $this->get_plugin_version( $error['source_slug'] );
			} else if ( $error['source'] == 'theme' ) {
				$error['source_version'] = $this->get_theme_version( $error['source_slug'] );
			}
			
			// Record error occurrence
			$this->log_php_error_occurrence( $error );
			
			// Do action after processing php error
			do_action( 'logtastic_php_error_processed', $error );
			
		}
		
		// Return
		return true;
		
	}
	
	/**
	 * Get the plugin settings from the database using get_option()
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function get_settings() {
		
		$current_options_serialized = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_php_error_log' );
		
		if ( !empty( $current_options_serialized ) ) {
			
			$current_options = maybe_unserialize( $current_options_serialized );
			
			return $current_options;
			
		} else {
			
			return false;
			
		}
		
	}

	
	/**
	 * Function to check for and log fatal errors, called as part of wp shutdown action.
	 * 
	 * @since 1.0.0
	 */
	public function log_fatal_error() {
		
		// Get last php error
		$error = error_get_last();

		// Is this an uncaught exception?
		$isUncaught = isset( $error['message'] ) && strpos( $error['message'], 'Uncaught ' ) === 0;
		
		// Check if the last error was a fatal error (non-fatal errors will have been logged by the log_recoverable_error function) and not an uncaught exception (this will have been logged by the log_exception function)
		if ( $error && in_array( $error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR] ) && !$isUncaught ) {
			
			// Add data to array to indicate severity and logging function which captured the error
			$error['severity'] = 'fatal';
			$error['method'] = 'log_fatal_error';
			
			// Attempt to capture stack trace if setting enabled
			if ( isset( $this->settings['capture_stacktrace'] ) && 1 == $this->settings['capture_stacktrace'] ) {
			
				// Check if the string contains 'Stack trace:' and capture everything that follows
				if (preg_match('/Stack trace:(.*)/s', $error['message'], $matches)) {
					// $matches[1] will contain everything after 'Stack trace:'
					$stack_trace_string = trim($matches[1]);
					$error['stack_trace_string'] = $stack_trace_string;
				} else {
					$error['stack_trace_string'] = null;
				}
			} else {
				$error['stack_trace_string'] = null;
			}
			
			// Pass the error to the process_php_error function
			$this->process_php_error( $error );
			
		}
		
		// Prevent default error handling?
		if ( isset( $this->settings['disable_default_error_handler'] ) && 1 == $this->settings['disable_default_error_handler'] ) {
			return true;
		} else {
			return false;
		}
		
	}
	

	/**
	 * Function to log recoverable errors, called using set_error_handler.
	 * 
	 * @since 1.0.0
	 */
	public function log_recoverable_error( $errno, $errstr, $errfile, $errline ) {
		
		// Capture stack trace if setting enabled
		if ( isset( $this->settings['capture_stacktrace'] ) && 1 == $this->settings['capture_stacktrace'] ) {
			
			// Capture stack trace arguments if setting enabled
			if ( isset( $this->settings['capture_stacktrace_args'] ) && 1 == $this->settings['capture_stacktrace_args'] ) {
			
				// Get the backtrace array with arguments
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_debug_backtrace -- included for error logging functionality
				$trace = debug_backtrace(0);
				
			} else {
				
				// Get the backtrace array without arguments
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_debug_backtrace -- included for error logging functionality
				$trace = debug_backtrace(2);
				
			}
			
			// Remove the first item from the array (this function)
			array_shift($trace);
			
		} else {
			
			$trace = null;
			
		}
		
		// Populate error array		
		$error = [
			'severity' 	=> 'recoverable',
			'type' 		=>	(int) $errno,
			'file'		=>	$errfile,
			'line'		=>	(int) $errline,
			'message'	=>	$errstr,
			'stack_trace_array'		=>	$trace,
			'stack_trace_string'	=>	null,
			'method' => 'log_recoverable_error'
		];

		// Pass the error to the process_php_error function
		$this->process_php_error( $error );		

		// If there is a previously defined error handler, pass to the error to this
		if ( is_callable( $this->previous_error_handler ) ) {
            call_user_func(
                $this->previous_error_handler,
                $errno,
                $errstr,
                $errfile,
                $errline
            );
        }
		
		// Prevent default error handling?
		if ( isset( $this->settings['disable_default_error_handler'] ) && 1 == $this->settings['disable_default_error_handler'] ) {
			return true;
		} else {
			return false;
		}
		
	}
	

	/**
	 * Function to log uncaught exceptions, called using set_exception_handler.
	 * 
	 * @since 1.0.0
	 */
	public function log_exception( $exception ) {
		
		// Populate error array		
		$error = [
			'severity' 	=> 'exception',
			'type' 		=>	1,
			'file'		=>	$exception->getFile(),
			'line'		=>	$exception->getLine(),
			'message'	=>	'Uncaught ' . get_class( $exception ) . ': ' . $exception->getMessage(),
			'method'	=>  'log_exception'
			
		];
		
		// Capture stack trace if setting enabled
		if ( isset( $this->settings['capture_stacktrace'] ) && 1 == $this->settings['capture_stacktrace'] ) {
			
			// Capture stack trace arguments if setting enabled
			if ( isset( $this->settings['capture_stacktrace_args'] ) && 1 == $this->settings['capture_stacktrace_args'] ) {
			
				// Get the backtrace array with arguments
				$error['stack_trace_array'] = $exception->getTrace();
				$error['stack_trace_string'] = $exception->getTraceAsString();
				
			} else {
				
				// Get the backtrace array without arguments
				$error['stack_trace_array'] = $exception->getTrace();
				foreach ($error['stack_trace_array'] as &$item) {
					// Set 'args' to null for each item
					$item['args'] = null;
				}
				$error['stack_trace_string'] = null;
				
			}
			
		} else {
			
			$error['stack_trace_array'] = null;
			$error['stack_trace_string'] = null;
			
		}
		
		// Pass the error to the process_php_error function
		$this->process_php_error( $error );		
		
		// If there is a previously defined global exception handler, pass the exception to this, otherwise throw the exception
		if ( is_callable( $this->previous_exception_handler ) ) {
        	call_user_func( $this->previous_exception_handler, $exception );
		} else {
			throw $exception;
		}
		
	}
	
}