<?php

/**
 * The error logging functionality for the JavaScript Error Log.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;


// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Base class for logging JavaScript errors.
 *
 * @since 1.0.0
 */
class Logtastic_JS_Error_Logger {
	
	/**
	 * The current plugin settings.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $settings;


	/**
	 * The JS error table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_TABLE_NAME = 'logtastic_js_errors';


	/**
	 * The JS error occurrences table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_OCCURRENCES_TABLE_NAME = 'logtastic_js_error_occurrences';


	/**
	 * The JS error stack traces table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_STACK_TRACES_TABLE_NAME = 'logtastic_js_error_stack_traces';
	
	
	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	function __construct() {
		
		$this->settings = $this->get_settings();
		
	}
	
	/**
	 * Attempt to find a matching error in the database, based on error level, file, line and message.
	 * If a match is found, return error_id, source_type, source_slug and ignore_error
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 * @param array 	$error 		The current error array
	 */
	public function get_js_error_details($error) { 
		
		global $wpdb;

		// Prepare and execute the query
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table, Caching would not be appropriate in this context.
		$result = $wpdb->get_row(
			$wpdb->prepare(
				"
					SELECT error_id, source, source_slug, ignore_error FROM %i
					WHERE error_type = %s
					AND file = %s
					AND line = %d
					AND col = %d
					AND message = %s
					LIMIT 1
				",
				array (
					$wpdb->prefix . self::ERROR_TABLE_NAME,
					$error['type'],
					$error['file'],
					$error['line'],
					$error['col'],
					$error['message']
				)
			)
		);
		
		// Update the $error array based on the results
		if ($result) {
			// Record found, mark as existing = false and add additional details to error array 
			$error['existing'] = true;
			$error['error_id'] = $result->error_id;
			if ( isset( $result->source ) ) {
				$error['source'] = $result->source;
			}
			if ( isset( $result->source_slug ) ) {
				$error['source_slug'] = $result->source_slug;
			}
			if ( isset( $result->ignore_error ) ) {
				$error['ignore_error'] = $result->ignore_error;
			}
		} else {
			// No record found, mark as existing = false 
			$error['existing'] = false;
		}
		
		// Return the updated error array
		return $error;
	}
	
	
	/**
	 * Create a new js error record in the database
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 * @param array 	$error 		The current error array
	 */
	public function add_new_js_error($error) { 
		
		global $wpdb;
		
		// Insert new error
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct query as custom table
		$wpdb->insert(
			$wpdb->prefix . self::ERROR_TABLE_NAME,
			array(
				'error_type' => $error['type'],
				'file' => $error['file'],
				'line' => $error['line'],
				'col' => $error['col'],
				'source' => $error['source'],
				'source_slug' => $error['source_slug'],
				'message' => $error['message'],
				'timestamp' => current_time( 'mysql' ),
			),
			array(
				'%d',
				'%s',
				'%d',
				'%d',
				'%s',
				'%s',
				'%s',
				'%s'
			)
		);
		
		// Add the error ID of the inserted error to the error array
		$error['error_id'] = $wpdb->insert_id;
		
		// Return the updated error array
		return $error;
		
	}


	/**
	 * Determine whether a matching stack trace already exists in the database.
	 *
	 * Looks up a record in the custom JS error stack trace table that matches
	 * a specific error ID, trace type, and trace data.
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 *
	 * @param int    $error_id ID of the related error.
	 * @param string $type     Stack trace type identifier.
	 * @param string $trace    Serialized or encoded stack trace data.
	 * @return int|false       The existing stack_trace_id if a match is found,
	 *                         or false if no matching record exists.
	 */
	public function is_existing_stack_trace( int $error_id, string $type, string $trace ) {
		
		global $wpdb;

		// Prepare and execute the query to look for an existing stack trace with the same error ID, type, and data.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table, Caching would not be appropriate in this context.
		$result = $wpdb->get_var(
			$wpdb->prepare(
				"
					SELECT stack_trace_id FROM %i
					WHERE error_id = %d
					AND stack_trace_type = %s
					AND stack_trace_data = %s
					LIMIT 1
				",
				array (
					$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
					$error_id,
					$type,
					$trace
				)
			)
		);

		// Return the stack_trace_id if found, or false if not.
		return $result ?: false;
	}


	/**
	 * Save a new stack trace to the database.
	 *
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 *
	 * @param int    $error_id ID of the related error.
	 * @param string $type     Stack trace type identifier.
	 * @param string $trace    Serialized or encoded stack trace data.
	 * @return int|false       The existing stack_trace_id if a match is found,
	 *                         or false if no matching record exists.
	 */
	public function save_new_stack_trace($error_id, $type, $trace) {
		
		global $wpdb;
		
		// Insert the stack trace into the database
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->insert(
			$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
			array (
				'error_id' => $error_id,
				'stack_trace_type' => $type,
				'stack_trace_data' => $trace
			)
		);
		
		// Return the stack_trace_id of the inserted entry
		return $wpdb->insert_id;
		
	}
	
	
	/**
	 * Create a new js error occurrence record in the database
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 * @global wp_version $wp_version Current WordPress version number.
	 * @param array 	$error 		The current error array
	 */
	public function log_js_error_occurrence($error) { 
		
		global $wp_version;
		global $wpdb;
		
		$error['stack_trace_available'] = 0;

		// Not applicable stack trace array id, but may be used in future
		$error['stack_trace_array_id'] = null;
		
		// If exists, check if string stack trace exists in database and retrieve ID, or else save as new stack trace
		if ( isset( $error['stack_trace_string'] ) && null != $error['stack_trace_string'] ) {
			$existing_stack_trace_string = $this->is_existing_stack_trace($error['error_id'], 'string', $error['stack_trace_string']);
			if ( false != $existing_stack_trace_string ) {
				$error['stack_trace_string_id'] = $existing_stack_trace_string;
				$error['stack_trace_available'] = 1;
			} else {
				$new_stack_trace_string = $this->save_new_stack_trace($error['error_id'], 'string', $error['stack_trace_string']);
				if ( false != $new_stack_trace_string ) {
					$error['stack_trace_string_id'] = $new_stack_trace_string;
					$error['stack_trace_available'] = 1;
				} else {
					$error['stack_trace_string_id'] = null;
				}
			}
		} else {
			$error['stack_trace_string_id'] = null;
		}

		// Serialise additional occurrence data
		$error['occurrence_data'] = serialize( $error['occurrence_data'] );
		
		// Insert the error occurrence into the database
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->insert(
			$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
			array(
				'error_id' => $error['error_id'],
				'source_version' => $error['source_version'],
				'wp_version' => $wp_version,
				'timestamp' => current_time( 'mysql' ),
				'additional_data' => null,
				'php_version' => PHP_VERSION,
				'stack_trace_array_id' => $error['stack_trace_array_id'],
				'stack_trace_string_id' => $error['stack_trace_string_id'],
				'stack_trace_available' => $error['stack_trace_available'],
				'method' => $error['method'],
				'additional_data' => $error['occurrence_data']
			),
			array (
				'%d',
				'%d',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%d',
				'%d',
				'%d',
				'%s',
				'%s',
			)
		);
		
	}

	
	/**
	 * Calculates the error source (either 'plugin', 'theme' or 'wp_core' based on the error file path ) and source_slug if applicable (only applicable for plugin and theme errors) and return the updated error array
	 *
	 * @since 1.0.0
	 *
	 * @param array 	$error 		The current error array
	 */
	public function caclulate_error_source($error) {
		
		// Clear the error file to remove the http:// or https://
		$error_file_clean = preg_replace('#^https?://#', '', $error['file']);
		
		// Clean the plugin directory url to remove the http:// or https://
		$plugin_url_clean = preg_replace('#^https?://#', '', WP_PLUGIN_URL);
		
		// Clear the theme directory url to remove the http:// or https://
		$theme_url_clean = preg_replace('#^https?://#', '', get_theme_root_uri());
		
		// Does the file path contain the WP Plugin Directory?
		if ( strpos( $error_file_clean, $plugin_url_clean ) !== false ) {
			
			$error['source'] = 'plugin';
			
			$slug_start_pos = strpos( $error_file_clean, $plugin_url_clean ) + strlen( $plugin_url_clean ) + 1;
			$slug_end_pos = strpos( $error_file_clean, '/', $slug_start_pos );
			$error['source_slug'] = substr( $error_file_clean, $slug_start_pos, $slug_end_pos - $slug_start_pos );
			
		// Does the file path contain the WP Theme Directory?
		} else if ( strpos( $error_file_clean, $theme_url_clean ) !== false ) {
			
			$error['source'] = 'theme';
			
			$slug_start_pos = strpos( $error_file_clean, $theme_url_clean ) + strlen( $theme_url_clean ) + 1;
			$slug_end_pos = strpos( $error_file_clean, '/', $slug_start_pos );
			$error['source_slug'] = substr( $error_file_clean, $slug_start_pos, $slug_end_pos - $slug_start_pos );
			
		// Does the error message contain the WP Plugin Directory?
		} else if ( strpos( $error['message'], $plugin_url_clean ) !== false ) {
			$error['source'] = 'plugin';
			
			$slug_start_pos = strpos( $error['message'], $plugin_url_clean ) + strlen( $plugin_url_clean ) + 1;
			$slug_end_pos = strpos( $error['message'], '/', $slug_start_pos );
			$error['source_slug'] = substr( $error['message'], $slug_start_pos, $slug_end_pos - $slug_start_pos );
			
		// Does the error message contain the WP Theme Directory?
		} else if ( strpos( $error['file'], $theme_url_clean ) !== false ) {
			$error['source'] = 'theme';
			
			$slug_start_pos = strpos( $error['message'], $theme_url_clean ) + strlen( $theme_url_clean ) + 1;
			$slug_end_pos = strpos( $error['message'], '/', $slug_start_pos );
			$error['source_slug'] = substr( $error['message'], $slug_start_pos, $slug_end_pos - $slug_start_pos );
		
		// Else label error as 	'wp_core'
		} else {
			
			global $wp_version;
			
			$error['source'] = 'wp_core';
			$error['source_slug'] = null;
			$error['source_version'] = $wp_version;
			
		}
		
		return $error;
		
	}
	
	
	/**
	 * Gets the version number of a plugin based on the given plugin slug
	 *
	 * @since 1.0.0
	 *
	 * @param string 	$plugin_slug 		The plugin slug
	 */
	public function get_plugin_version($plugin_slug) { 
		
		// Get currently active plugins
		$active_plugins = get_option('active_plugins');
		
		// Match plugin slug to get plugin index file
		foreach ( $active_plugins as $plugin ) {
			if ( strpos( $plugin, $plugin_slug . '/' ) !== false ) {
				$plugin_index = $plugin;
			}
		}
		
		if ( isset( $plugin_index ) ) {
			
			// Get the contents of the main plugin file
			$plugin_file = WP_PLUGIN_DIR . '/' . $plugin_index;
			$plugin_content = file_get_contents($plugin_file);
			
			// Match the version field
			preg_match('/^.*Version:\s*(\S+)/mi', $plugin_content, $matches);
			
			// If result found, return value
			if (!empty($matches[1])) {
				$plugin_version = $matches[1];
				return $plugin_version;
			}
			
		}
			
		return false;
		
	}
	

	/**
	 * Gets the version number of a theme based on the given theme slug
	 *
	 * @since 1.0.0
	 *
	 * @param string 	$theme_slug 		The theme slug
	 */
	public function get_theme_version( $theme_slug ) { 
		
		// Get the theme data
		$theme_data = wp_get_theme( $theme_slug );	
		
		if ( $theme_data !== false ) {
		
			// Get the version
			$theme_version = $theme_data->get('Version');
			
			// Return the value
			return $theme_version;
			
		}
		
		return false;
		
	}
	
	
	/**
	 * The main function to process a JavaScript error.
	 *
	 * @since 1.0.0
	 *
	 * @param array 	$error 		The current error array
	 */
	public function process_js_error( $error ) { 
		
		// Check that error type should be logged, otherwise return false
		if ( !isset($this->settings['error_types']) || !is_array($this->settings['error_types']) || !in_array( $error['type'], $this->settings['error_types'] ) ) {
			return false;
		}
		
		// Attempt to fetch error details from database if existing error
		$error = $this->get_js_error_details( $error );
		
		// If existing error exists and ignore error is set to true, return false
		if ( isset( $error['ignore_error'] ) && $error['ignore_error'] == 1 ) {
			return false;
		}
		
		// If existing error record does not exist, calculate source of error 
		if ( false == $error['existing'] ) {
			$error = $this->caclulate_error_source( $error );
		}
		
		// Check if error source should be logged
		$log_error = false;
		if ( $error['source'] == 'plugin' ) {
			// Should this plugin error be logged?
			if ( isset( $this->settings['scope_wp_plugins'] ) && 'all' == $this->settings['scope_wp_plugins'] ) {
				$log_error = true;
			} else if ( isset( $this->settings['scope_wp_plugins'] ) && 'all_except' == $this->settings['scope_wp_plugins'] ) {
				$log_error = true;
				if ( isset( $this->settings['scope_wp_plugins_excepted'] ) && is_array( $this->settings['scope_wp_plugins_excepted'] ) ) {
					foreach ( $this->settings['scope_wp_plugins_excepted'] as $excepted_slug ) {
						if ( $excepted_slug == $error['source_slug'] ) {
							$log_error = false;
						}
					}
				}
			} else if ( isset( $this->settings['scope_wp_plugins'] ) && 'selected' == $this->settings['scope_wp_plugins'] ) {
				if ( isset( $this->settings['scope_wp_plugins_selected'] ) && is_array( $this->settings['scope_wp_plugins_selected'] ) ) {
					foreach ( $this->settings['scope_wp_plugins_selected'] as $selected_slug ) {
						if ( $selected_slug == $error['source_slug'] ) {
							$log_error = true;
						}
					}
				}
			}
		} else if ( $error['source'] == 'theme' ) {
			// Should this theme error be logged?
			if ( isset( $this->settings['scope_wp_themes'] ) && 'all' == $this->settings['scope_wp_themes'] ) {
				$log_error = true;
			} else if ( isset( $this->settings['scope_wp_themes'] ) && 'all_except' == $this->settings['scope_wp_themes'] ) {
				$log_error = true;
				if ( isset( $this->settings['scope_wp_themes_excepted'] ) && is_array( $this->settings['scope_wp_themes_excepted'] ) ) {
					foreach ( $this->settings['scope_wp_themes_excepted'] as $excepted_slug ) {
						if ( $excepted_slug == $error['source_slug'] ) {
							$log_error = false;
						}
					}
				}
			} else if ( isset( $this->settings['scope_wp_themes'] ) && 'selected' == $this->settings['scope_wp_themes'] ) {
				if ( isset( $this->settings['scope_wp_themes_selected'] ) && is_array( $this->settings['scope_wp_themes_selected'] ) ) {
					foreach ( $this->settings['scope_wp_themes_selected'] as $selected_slug ) {
						if ( $selected_slug == $error['source_slug'] ) {
							$log_error = true;
						}
					}
				}
			}
		} else {
			// Should WP Core Errors be logged?
			if ( isset( $this->settings['scope_wp_core'] ) && true == $this->settings['scope_wp_core'] ) {
				$log_error = true;
			}
		}
		
		// If error should be logged, proceed 
		if ( true == $log_error ) {
		
			// If existing error record does not exist, record new error
			if ( false == $error['existing'] ) {
				$error = $this->add_new_js_error( $error );
			}
			
			// Get source version number (if applicable)
			$error['source_version'] = null;
			if ( $error['source'] == 'plugin' ) {
				$error['source_version'] = $this->get_plugin_version( $error['source_slug'] );
			} else if ( $error['source'] == 'theme' ) {
				$error['source_version'] = $this->get_theme_version( $error['source_slug'] );
			}
			
			// Record error occurrence
			$this->log_js_error_occurrence( $error );
			
			// Do action after processing js error
			do_action( 'logtastic_js_error_processed', $error );
			
		}
		
		// Return
		return true;
		
	}
	
	
	/**
	 * Get the plugin settings from the database using get_option()
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function get_settings() {
		
		$current_options_serialized = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_js_error_log' );
		
		if ( !empty( $current_options_serialized ) ) {
			
			$current_options = maybe_unserialize( $current_options_serialized );
			
			return $current_options;
			
		} else {
			
			return false;
			
		}
		
	}


	/**
	 * JavaScript error handler, receives error data via Ajax
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public function ajax_js_error_handler() {
	
		// Verify nonce
		check_ajax_referer('logtastic-log-js-error');

		if ( empty( $_POST['data'] ) ) {
			wp_send_json_error( [ 'message' => __( 'Bad request', 'logtastic' ) ], 400 );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- $_POST['data'] not sanitized prior to JSON decode. Individual values santized below following JSON decode, if $_POST['data'] contains valid JSON.
		$data_unsanitized = json_decode( wp_unslash( $_POST['data'] ), true );
		
		if ( !empty( $data_unsanitized )) {

			// Create empty array to stored extraced error data
			$error = array();
			
			// Extract error type from JSON
			if ( isset( $data_unsanitized['type'] ) ) {
				if ( $data_unsanitized['type'] == 'runtime' ) {
					$error['type'] = 1;
				} else if ( $data_unsanitized['type'] == 'promiseRejection' ) {
					$error['type'] = 2;
				} else {
					$error['type'] = 0;
				} 
			} else {
				$error['type'] = 0;
			}

			// Extract error source from JSON
			if ( isset( $data_unsanitized['source'] ) ) {
				$error['file'] = sanitize_text_field ( $data_unsanitized['source'] );
			} else {
				$error['file'] = 'Unknown';
			}

			// Extract line number from JSON
			if ( isset( $data_unsanitized['line'] ) ) {
				$error['line'] = (int) $data_unsanitized['line'];
			} else {
				$error['line'] = null;
			}

			// Extract column number from JSON
			if ( isset( $data_unsanitized['column'] ) ) {
				$error['col'] = (int) $data_unsanitized['column'];
			} else {
				$error['col'] = null;
			}

			// Extract error message from JSO
			if ( isset( $data_unsanitized['message'] ) ) {
				$error['message'] = sanitize_text_field ( $data_unsanitized['message'] );
			} else {
				$error['message'] = null;
			}

			// Extract logging method from JSON
			if ( isset( $data_unsanitized['type'] ) ) {
				$error['method'] = sanitize_text_field ( $data_unsanitized['type'] );
			} else {
				$error['method'] = null;
			}

			// Initiate array to store occurence data
			$error['occurrence_data'] = array();

			// Record session info is setting enabled
			if ( isset( $this->settings['capture_session_info'] ) && 1 == $this->settings['capture_session_info'] ) {
				if ( isset( $data_unsanitized['url'] ) ) {
					$error['occurrence_data']['session_info']['url'] = sanitize_text_field ( $data_unsanitized['url']);
				}
				if ( isset( $data_unsanitized['userAgent'] ) ) {
					$error['occurrence_data']['session_info']['user_agent'] = sanitize_text_field ( $data_unsanitized['userAgent']);
				}
			}

			// Record stack trace if setting enabled 
			if ( isset( $this->settings['capture_stacktrace'] ) && 1 == $this->settings['capture_stacktrace'] ) {
				if ( isset( $data_unsanitized['stack'] ) ) {
					$error['stack_trace_string'] = sanitize_textarea_field ( $data_unsanitized['stack'] );
					$error['stack_trace_array'] = null;
				}
			} else {
				$error['stack_trace_array'] = null;
				$error['stack_trace_string'] = null;
			}

			// Unset $data_unsanitized to prevent further use of unsanitized data
			unset( $data_unsanitized );

		}
		
		if ( $error ) {
			
			$this->process_js_error($error);
			
		}
		
		wp_send_json_success();

		exit;
	}
	

	
	/**
	 * Register the JavaScript for the js error logger.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
	
		// Enqueue javascript error logger file
		wp_enqueue_script( LOGTASTIC_PLUGIN_SLUG . '-js-error-logger', LOGTASTIC_PLUGIN_DIR_URL . '/js/js-error-logger.js', array(), LOGTASTIC_PLUGIN_VERSION, false );

		// Define additional variables array for javascript error logger file
		$js_error_logger_js_args = [
			'ajax_url' 			=> admin_url( 'admin-ajax.php' ),
			'log_js_error_nonce'=> wp_create_nonce( 'logtastic-log-js-error' )
		];

		// Localize javascript error logger file
		wp_add_inline_script( LOGTASTIC_PLUGIN_SLUG . '-js-error-logger', 'const logtasticEnvData = ' . json_encode( $js_error_logger_js_args ), 'before' );

	}
	

	/**
	 * Set the js error logger JavaScript as a dependency for all other enqueued scripts.
	 *
	 * @since    1.0.0
	 */
	public function add_js_error_logger_as_dependency_for_all_other_scripts() {
		
		global $wp_scripts;
		
		// Check that 'js-error-logger' is enqueued, otherwise exit
		if ( !in_array( 'js-error-logger', $wp_scripts->queue ) ) {
			return;
		}
		
		// Loop over all registered scripts, set 'js-error-logger' as dependency
		foreach ($wp_scripts->registered as $handle => $script) {
			if ( $handle !== 'js-error-logger' && !in_array('js-error-logger', $script->deps, true) ) {
				$wp_scripts->registered[$handle]->deps[] = 'js-error-logger';
			}
		}
		
	}
	
}

