<?php
/**
 * Logtastic: Main plugin class
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;


// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}


final class Logtastic {
	
	/**
	 * Stores the full name of this plugin
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_full_name    The full name of this plugin, correctly formatted for display.
	 */
	protected $plugin_full_name;
	
	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_slug    The string used to uniquely identify this plugin.
	 */
	protected $plugin_slug;
	
	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * An array of options/settings for the plugin
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array    $settings    Plugin options/settings
	 */
	protected $settings;	
	
	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Plugin_Name_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;
	
	
	
	/**
	 * Define the core functionality of the plugin.
	 *
	 * @since    1.0.0
	 */
	function __construct() {
		
		$this->plugin_full_name = LOGTASTIC_PLUGIN_FULL_NAME;
		$this->plugin_slug = LOGTASTIC_PLUGIN_SLUG;
		$this->version = LOGTASTIC_PLUGIN_VERSION;
		
		$this->settings = $this->get_settings();
		
		$this->load_core_dependencies();
		
		$this->loader = new Logtastic_Loader();

		// If local installed version not set or does not match current plugin version, update database structure & default settings
		if ( empty( $this->settings['logtastic_installed_version'] ) || $this->settings['logtastic_installed_version'] != LOGTASTIC_PLUGIN_FEATURE_VERSION . '_' . LOGTASTIC_PLUGIN_VERSION ) {
			
			// Check and update database structure
			$db_ok = $this->update_db_structure();

			// Check and update plugin settings
			$settings_ok = $this->check_and_update_default_settings();

			// If database and plugin settings checked/updated successfully, update installed version in settings and save changes
			if ( $db_ok && $settings_ok ) {
				$this->settings['logtastic_installed_version'] = LOGTASTIC_PLUGIN_FEATURE_VERSION . '_' . LOGTASTIC_PLUGIN_VERSION;
				update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME, $this->settings, true );
			}

		}

		// Load Admin Functions (If Is Admin)
		if ( is_admin() ) {
			$this->load_admin();
		}
		
		// Load PHP Error Logging Functions (If Enabled)
		if ( ! empty( $this->settings['enabled_logs']['php-error-log'] ) ) {
			$this->load_php_error_logger();
		}
		
		// Load JS Error Logging Functions (If Enabled)
		if ( ! empty( $this->settings['enabled_logs']['js-error-log'] ) ) {
			$this->load_js_error_logger();
		}
		
	}
	
	
	/**
	 * Get the plugin settings from the database using get_option()
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function get_settings() {
		
		$current_options_serialized = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME );
		
		// If the options exist, return them
		if ( !empty( $current_options_serialized ) ) {
			
			$current_options = maybe_unserialize( $current_options_serialized );
			
			return $current_options;
		
		// If the options don't exist, return false
		} else {
			
			return false;
			
		}
			
	}


	/**
	 * Check and pdate database structure
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function update_db_structure() {
		
		// Load DB utilities
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/utilities/class-logtastic-db-utilities.php';
		$db_utilities = new Logtastic_DB_Utilities( $this->settings );

		// Check and update database structure
		$result = $db_utilities->check_and_update_db_structure_all_active_logs();

		return $result;
			
	}


	/**
	 * Check and update the default plugin options/settings
	 *
	 * @since      	1.0.0
	 * @param array $default_settings	The default plugin settings
	 */
	private function check_and_update_default_settings() {

		// Load default settings
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/default-settings/class-logtastic-default-settings_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';
		$default_settings = Logtastic_Default_Settings::$default_settings_general;
		
		// If there are existing options, compare against the default options and add any missing options
		if ( !empty( $this->settings ) ) {
			
			$updated_settings = $this->settings;
			
			foreach ( $default_settings as $setting_name => $setting_default_value ) {
				
				if ( !isset( $this->settings[$setting_name] ) ) {
					$updated_settings[$setting_name] = $setting_default_value;
				}
				
			}

			$this->settings = $updated_settings;
			
		// If there are not existing options, set the default options
		} else {

			$this->settings = $default_settings;
			
		}

		return true;
		
	}

	
	
	/**
	 * Load the required dependencies for this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_core_dependencies() {
	
		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/class-logtastic-loader.php';
	
	}
	
	
	/**
	 * Load the required admin functionality for this plugin and add related actions to the loader.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_admin() {
			
		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/class-logtastic-admin.php';
		
		$plugin_admin = new Logtastic_Admin( $this->settings );
		
		// General Admin Actions
		$this->loader->add_action( 'init', $plugin_admin, 'define_access', 1 );
		$this->loader->add_action( 'admin_notices', $plugin_admin, 'loading_method_error_check' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'admin_pages', 99 );
		$this->loader->add_action( 'admin_body_class', $plugin_admin, 'add_plugin_admin_body_class' );
		$this->loader->add_action( 'wp_ajax_logtastic_enable_disable_log', $plugin_admin, 'enable_disable_log_via_ajax' );
		// PHP Error Log Admin Actions
		if ( isset( $this->settings['enabled_logs']['php-error-log'] ) && true == $this->settings['enabled_logs']['php-error-log'] ) {
			$this->loader->add_action( 'wp_ajax_logtastic_load_php_error_details', $plugin_admin, 'ajax_load_php_error_details' );
			$this->loader->add_action( 'wp_ajax_logtastic_load_php_error_stack_trace', $plugin_admin, 'ajax_load_php_error_stack_trace' );
			$this->loader->add_action( 'wp_ajax_logtastic_unignore_php_error', $plugin_admin, 'ajax_unignore_php_error' );
		}
		// JS Error Log Admin Actions
		if ( isset( $this->settings['enabled_logs']['js-error-log'] ) && true == $this->settings['enabled_logs']['js-error-log'] ) {
			$this->loader->add_action( 'wp_ajax_logtastic_load_js_error_details', $plugin_admin, 'ajax_load_js_error_details' );
			$this->loader->add_action( 'wp_ajax_logtastic_load_js_error_stack_trace', $plugin_admin, 'ajax_load_js_error_stack_trace' );
			$this->loader->add_action( 'wp_ajax_logtastic_unignore_js_error', $plugin_admin, 'ajax_unignore_js_error' );
		}
	
	}
	

	/**
	 * Load the PHP Error Logging functionality.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_php_error_logger() {
			
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/logging-classes/class-logtastic-php-error-logger.php';
		
		$php_error_logger = new Logtastic_PHP_Error_Logger( $this->settings );
		
		$log_fatal_errors = false;
		$log_recoverable_errors = false;
		
		if ( isset( $php_error_logger->settings['error_levels'] ) ) {
			foreach ( $php_error_logger->settings['error_levels'] as $error_level ) {
				if ( $error_level == "e_error" || $error_level == "e_user_error" ) {
					$log_fatal_errors = true;
				} else {
					$log_recoverable_errors = true;
				}
			}
		}
		
		// If enabled, log fatal errors
		if ( true == $log_fatal_errors ) {
			$this->loader->add_action( 'shutdown', $php_error_logger, 'log_fatal_error' );
		}
		
		// If enabled, log non-fatal errors
		if ( true == $log_recoverable_errors ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_set_error_handler -- included for error logging functionality
			$php_error_logger->previous_error_handler = set_error_handler( array($php_error_logger, 'log_recoverable_error') );
		}
		
		// If enabled, log exceptions
		if ( true == $log_fatal_errors ) {

			// If capture stack trace arguments is true, ensure exception arguments are not ignored 
			if ( !empty( $php_error_logger->settings['capture_stacktrace'] ) && !empty( $php_error_logger->settings['capture_stacktrace_args'] ) ) {
				// phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- required to capture stack trace arguments
				ini_set('zend.exception_ignore_args', 0);
			}
			$php_error_logger->previous_exception_handler = set_exception_handler( array($php_error_logger, 'log_exception') );
		}
		
		// Load scheduled task for retention schedule, if set, or clear scheduled task if retention schedule is set to '0' or not defined
		if ( isset( $php_error_logger->settings['retention_schedule'] ) && '0' != $php_error_logger->settings['retention_schedule'] ) {
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/scheduled-tasks/class-logtastic-php-error-log-scheduled-tasks.php';
			$php_error_log_scheduled_tasks = new Logtastic_PHP_Error_Log_Scheduled_Tasks();
			$this->loader->add_action( 'logtastic_scheduled_task_process_php_error_log_retention_schedule', $php_error_log_scheduled_tasks, 'process_php_error_log_retention_schedule' );
			if ( ! wp_next_scheduled( 'logtastic_scheduled_task_process_php_error_log_retention_schedule' ) ) {
				wp_schedule_event( time(), 'daily', 'logtastic_scheduled_task_process_php_error_log_retention_schedule' );
			}
		} else {
			$scheduled_timestamp = wp_next_scheduled( 'logtastic_scheduled_task_process_php_error_log_retention_schedule' );
			if ( $scheduled_timestamp ) {
				wp_unschedule_event( $scheduled_timestamp, 'logtastic_scheduled_task_process_php_error_log_retention_schedule' );
			}
		}
	}
	
	
	/**
	 * Load the JS Error Logging functionality.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_js_error_logger() {
			
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/logging-classes/class-logtastic-js-error-logger.php';
		
		$js_error_logger = new Logtastic_JS_Error_Logger( $this->settings );
		
		// Ajax functions to logs errors
		$this->loader->add_action( 'wp_ajax_logtastic_log_js_error', $js_error_logger, 'ajax_js_error_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_logtastic_log_js_error', $js_error_logger, 'ajax_js_error_handler' );
		
		// Javascript to log errors on front-end
		$this->loader->add_action( 'wp_enqueue_scripts', $js_error_logger, 'enqueue_scripts', 100 );
		
		// Javascript to log errors on admin
		$this->loader->add_action( 'admin_enqueue_scripts', $js_error_logger, 'enqueue_scripts', 100 );
		
		// Add JS error logger as dependency for all other front-end scripts to ensure it loads first
		$this->loader->add_action( 'wp_print_scripts', $js_error_logger, 'add_js_error_logger_as_dependency_for_all_other_scripts', 100 );
		
		// Load scheduled task for retention schedule, if set, or clear scheduled task if retention schedule is set to '0' or not defined
		if ( isset( $js_error_logger->settings['retention_schedule'] ) && '0' != $js_error_logger->settings['retention_schedule'] ) {
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/scheduled-tasks/class-logtastic-js-error-log-scheduled-tasks.php';
			$js_error_log_scheduled_tasks = new Logtastic_JS_Error_Log_Scheduled_Tasks();
			$this->loader->add_action( 'logtastic_scheduled_task_process_js_error_log_retention_schedule', $js_error_log_scheduled_tasks, 'process_js_error_log_retention_schedule' );
			if ( ! wp_next_scheduled( 'logtastic_scheduled_task_process_js_error_log_retention_schedule' ) ) {
				wp_schedule_event( time(), 'daily', 'logtastic_scheduled_task_process_js_error_log_retention_schedule' );
			}
		} else {
			$scheduled_timestamp = wp_next_scheduled( 'logtastic_scheduled_task_process_js_error_log_retention_schedule' );
			if ( $scheduled_timestamp ) {
				wp_unschedule_event( $scheduled_timestamp, 'logtastic_scheduled_task_process_js_error_log_retention_schedule' );
			}
		}
		
	}
	
	
	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}
	
	
}
