<?php

/**
 * The functonality required during plugin activation
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}


/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 */
class Logtastic_Activator {

	/**
	 * An array of options/settings for the plugin
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array    $settings    Plugin options/settings
	 */
	protected $settings;
	

	/**
	 * Function to check and update the default setting passed in the array
	 *
	 * @since      	1.0.0
	 * @param array $default_settings	The default plugin settings
	 */
	private function check_and_update_default_settings( $default_settings ) {
		
		// Check for existing options 
		$current_options_serialized = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME );
		
		// If there are existing options, compare against the default options and add any missing options
		if ( !empty( $current_options_serialized ) ) {
			
			$current_options = maybe_unserialize( $current_options_serialized );
			
			$updated_options = $current_options;
			
			foreach ( $default_settings as $setting_name => $setting_default_value ) {
				
				if ( !isset( $current_options[$setting_name] ) ) {
					$updated_options[$setting_name] = $setting_default_value;
				}
				
			}
			
			update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME, $updated_options, true );

			$this->settings = $updated_options;
			
		// If there are not existing options, set the default options
		} else {
			
			update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME, $default_settings, true );

			$this->settings = $default_settings;
			
		}
		
	}


	/**
	 * The main plugin activation function
	 *
	 * @since      	  1.0.0
	 * @param string  $options_name		The name of the plugin's options
	 */
	public function activate( $options_name ) {
		
		// Load default settings
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/default-settings/class-logtastic-default-settings_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';
		$default_settings = Logtastic_Default_Settings::$default_settings_general;
		
		// Check if settings exist, otherwise create and set default values
		$this->check_and_update_default_settings( $default_settings );

		// Check and update database table structure for any active logs
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/utilities/class-logtastic-db-utilities.php';
		$db_utilities = new Logtastic_DB_Utilities( $this->settings );
		$db_utilities->check_and_update_db_structure_all_active_logs();
		
	}
	
}

