<?php
/**
 * The admin page to display the settings page for the JavaScript error log.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @subpackage        Logtastic/admin
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */
 
namespace Inspired_Plugins\Logtastic;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

?>

<div class="settings-section-body">
	
	<h3 class="logtastic-admin-section-title"><?php esc_html_e( 'JS Error Log Settings', 'logtastic' ); ?></h3>
	
	<p><?php esc_html_e( 'Configure the settings for the JavaScript Error Log.', 'logtastic' ); ?></p>

	<hr>
	
	<form method="post" data-form-conditional="true">
		
		<input type="hidden" id="_wpnonce" name="_wpnonce" value="<?php echo esc_attr( $settings_nonce ); ?>">
		<input type="hidden" id="action" name="action" value="update-settings">
		
		<table class="form-table" role="presentation">
			<tbody>
				<tr>
					<th scope="row"><?php esc_html_e( 'Error Types', 'logtastic' ); ?></th>
					<td>
						<fieldset>
							
							<legend class="screen-reader-text"><span><?php esc_html_e( 'Error Types', 'logtastic' ); ?></span></legend>
							
							<p><?php esc_html_e( 'Select which types of JavaScript errors to log', 'logtastic' ); ?>:</p>

							<br>
							
							<label for="js_error_type_runtime" class="logtastic-admin-settings-label-225">
								<input 
									name="js_error_types[]" 
									type="checkbox" id="js_error_type_runtime" 
									value="1" 
									<?php checked( ! empty( $log_settings_array['error_types'] ) && in_array( '1', (array) $log_settings_array['error_types'] ) ); ?>
								>
								<?php esc_html_e( 'Runtime Errors', 'logtastic' ); ?>
							</label>

							<div class="logtastic-admin-tooltip tooltip-medium">
								? <span class="screen-reader-text"><?php esc_html_e( 'Help', 'logtastic' ); ?></span>
								<div class="logtastic-admin-tooltip-content">
									<?php esc_html_e( 'Errors that occur while code is executing (e.g., calling an undefined function or accessing a missing variable).', 'logtastic' ); ?>
								</div>
							</div>

							<br>
							
							<label for="js_error_type_promise_rejection" class="logtastic-admin-settings-label-225">
								<input 
									name="js_error_types[]" 
									type="checkbox" 
									id="js_error_type_promise_rejection" 
									value="2" 
									<?php checked( ! empty( $log_settings_array['error_types'] ) && in_array( '2', (array) $log_settings_array['error_types'] ) ); ?>
								>
								<?php esc_html_e( 'Unhandled Promise Rejections', 'logtastic' ); ?>
							</label>

							<div class="logtastic-admin-tooltip tooltip-medium">
								? <span class="screen-reader-text"><?php esc_html_e( 'Help', 'logtastic' ); ?></span>
								<div class="logtastic-admin-tooltip-content">
									<?php esc_html_e( 'Failures from promises (async operations) that aren’t caught with a .catch() or try…catch.', 'logtastic' ); ?>
								</div>
							</div>

							<br>

							<br>

							<small class="settings-select-all"><a id="select-all-error-types" href="#" data-select-all="true"><?php esc_html_e( 'Select All', 'logtastic' ); ?></a> / <a id="unselect-all-error-types"  href="#" data-select-none="true"><?php esc_html_e( 'Unselect All', 'logtastic' ); ?></a></small>
							
						</fieldset>
						
					</td>
					
				</tr>
				
				<tr>
					<th scope="row"><?php esc_html_e( 'Scope', 'logtastic' ); ?></th>
					<td>
						
						<fieldset>
						
							<legend class="screen-reader-text"><span><?php esc_html_e( 'Scope', 'logtastic' ); ?></span></legend>
							
							<p><?php esc_html_e( 'Select which sources to log errors for', 'logtastic' ); ?>:</p>
							<br>
							
							<div class="notice notice-info inline">
								<p><strong><?php esc_html_e( 'Note', 'logtastic' ); ?>:</strong> <?php esc_html_e( 'Logtastic will track the source of JavaScript errors based on the location of the JavaScript file (eg. /plugin/example.js will be recorded as the source being the relevant plugin), however, inline JavaScript will always be recorded as WordPress core.', 'logtastic' ); ?></p>
							</div>
							<br>
							
							<label for="js_error_scope_wp_core">
								<input 
									name="js_error_scope_wp_core" 
									type="checkbox" 
									id="js_error_scope_wp_core" 
									value="1" 
									<?php checked( ! empty( $log_settings_array['scope_wp_core'] ) ); ?>
								>
								<?php esc_html_e( 'WordPress Core Errors', 'logtastic' ); ?>
							</label>

							<div class="logtastic-admin-tooltip tooltip-medium">
								? <span class="screen-reader-text"><?php esc_html_e( 'Help', 'logtastic' ); ?></span>
								<div class="logtastic-admin-tooltip-content">
									<?php esc_html_e( 'Log errors originating from WordPress core files.', 'logtastic' ); ?>
								</div>
							</div>

							<br>
							
							<label for="js_error_scope_wp_themes">
								<input 
									name="js_error_scope_wp_themes" 
									type="checkbox" 
									id="js_error_scope_wp_themes" 
									value="1"
									<?php checked( !empty( $log_settings_array['scope_wp_themes'] ) ); ?>
								>
								<?php esc_html_e( 'Theme Errors', 'logtastic' ); ?>
							</label>

							<div class="logtastic-admin-tooltip tooltip-medium">
								? <span class="screen-reader-text"><?php esc_html_e( 'Help', 'logtastic' ); ?></span>
								<div class="logtastic-admin-tooltip-content">
									<?php esc_html_e( 'Log errors originating from WordPress theme files.', 'logtastic' ); ?>
								</div>
							</div>

							<br>
						
							<div class="logtastic-admin-sub-options" data-form-conditional-field-name="js_error_scope_wp_themes" data-form-conditional-field-value="1">	
								<label>
									<input 
										type="radio" 
										name="js_error_scope_wp_themes_definition"
										value="all"
										<?php checked( isset( $log_settings_array['scope_wp_themes'] ) && ( $log_settings_array['scope_wp_themes'] === 'all' || $log_settings_array['scope_wp_themes'] === false ) ); ?>
									>
									<?php esc_html_e( 'All themes', 'logtastic' ); ?>
								</label>

								<br>
								
								<label>
									<input 
										type="radio" 
										name="js_error_scope_wp_themes_definition" 
										value="all_except"
										<?php checked( isset( $log_settings_array['scope_wp_themes'] ) && 'all_except' == $log_settings_array['scope_wp_themes'] ); ?>
									>
									<?php esc_html_e( 'All themes except selected', 'logtastic' ); ?>
								</label>
								<br>
								
								<div class="logtastic-admin-sub-options" data-form-conditional-field-name="js_error_scope_wp_themes_definition" data-form-conditional-field-value="all_except">
									<i><?php esc_html_e( 'Select themes to exclude from error logging', 'logtastic' ); ?>:</i>
									<br>
									<?php foreach ( $all_themes as $logtastic_index => $logtastic_theme ) { ?>
										<label>
											<input 
												name="js_error_scope_select_themes_excepted[]" 
												type="checkbox" 
												value="<?php echo esc_attr ( $logtastic_index ); ?>"
												<?php checked( isset( $log_settings_array['scope_wp_themes_excepted'] ) && is_array( $log_settings_array['scope_wp_themes_excepted'] ) && in_array( $logtastic_index, $log_settings_array['scope_wp_themes_excepted'] ) ); ?>
											>
											<?php echo esc_html( $logtastic_theme['Name'] ); ?>
										</label>
										<br>
									<?php } ?>
								</div>
								
								<label>
									<input 
										type="radio" 
										name="js_error_scope_wp_themes_definition" 
										value="selected"
										<?php checked( isset( $log_settings_array['scope_wp_themes'] ) && 'selected' == $log_settings_array['scope_wp_themes'] ); ?>
									>
									<?php esc_html_e( 'Selected themes only', 'logtastic' ); ?>
								</label>
								<br>
								
								<div class="logtastic-admin-sub-options" data-form-conditional-field-name="js_error_scope_wp_themes_definition" data-form-conditional-field-value="selected">
									<i><?php esc_html_e( 'Select themes to include in error logging', 'logtastic' ); ?>:</i>
									<br>
									<?php foreach ( $all_themes as $logtastic_index => $logtastic_theme ) { ?>
										<label>
											<input 
												name="js_error_scope_select_themes_selected[]" 
												type="checkbox" 
												value="<?php echo esc_attr ( $logtastic_index ); ?>"
												<?php checked( isset( $log_settings_array['scope_wp_themes_selected'] ) && is_array( $log_settings_array['scope_wp_themes_selected'] ) && in_array( $logtastic_index, $log_settings_array['scope_wp_themes_selected'] ) ); ?>
											>
											<?php echo esc_html( $logtastic_theme['Name'] ); ?>
										</label>
										<br>
									<?php } ?>
								</div>
								
							</div>
							
							<label for="js_error_scope_wp_plugins">
								<input 
									name="js_error_scope_wp_plugins" 
									type="checkbox" 
									id="js_error_scope_wp_plugins" 
									value="1"
									<?php checked( !empty( $log_settings_array['scope_wp_plugins'] ) ); ?>
								>
								<?php esc_html_e( 'Plugin Errors', 'logtastic' ); ?>
							</label>
							<div class="logtastic-admin-tooltip tooltip-medium">
								? <span class="screen-reader-text"><?php esc_html_e( 'Help', 'logtastic' ); ?></span>
								<div class="logtastic-admin-tooltip-content">
									<?php esc_html_e( 'Log errors originating from WordPress plugin files.', 'logtastic' ); ?>
								</div>
							</div>
							<br>
							
							<div class="logtastic-admin-sub-options" data-form-conditional-field-name="js_error_scope_wp_plugins" data-form-conditional-field-value="1">
								
								<label>
									<input 
										type="radio" 
										name="js_error_scope_wp_plugins_definition" 
										value="all"
										<?php checked( isset( $log_settings_array['scope_wp_plugins'] ) && ( 'all' == $log_settings_array['scope_wp_plugins'] || false == $log_settings_array['scope_wp_plugins'] ) ); ?>
									>
									<?php esc_html_e( 'All plugins', 'logtastic' ); ?>
								</label>
								<br>
								<label>
									<input 
										type="radio" 
										name="js_error_scope_wp_plugins_definition" 
										value="all_except"
										<?php checked( isset( $log_settings_array['scope_wp_plugins'] ) && 'all_except' == $log_settings_array['scope_wp_plugins'] ); ?>
									>
									<?php esc_html_e( 'All plugins except selected', 'logtastic' ); ?>
								</label>
								<br>
								
								<div class="logtastic-admin-sub-options" data-form-conditional-field-name="js_error_scope_wp_plugins_definition" data-form-conditional-field-value="all_except">
									<i><?php esc_html_e( 'Select plugins to exclude from error logging', 'logtastic' ); ?>:</i>
									<br>
									<?php foreach ( $all_plugins as $logtastic_index => $logtastic_plugin ) { ?>
										<label class="logtastic-admin-php-error-type-label">
											<input 
												name="js_error_scope_select_plugins_excepted[]" 
												type="checkbox" 
												value="<?php echo esc_attr( explode('/', $logtastic_index)[0] ); ?>"
												<?php checked( isset( $log_settings_array['scope_wp_plugins_excepted'] ) && is_array( $log_settings_array['scope_wp_plugins_excepted'] ) && in_array( explode('/', $logtastic_index)[0], $log_settings_array['scope_wp_plugins_excepted'] ) ); ?>
											>
											<?php echo esc_html( $logtastic_plugin['Name'] ); ?>
										</label>
										<br>
									<?php } ?>
								</div>
								
								<label>
									<input 
										type="radio" 
										name="js_error_scope_wp_plugins_definition" 
										value="selected"
										<?php checked( isset( $log_settings_array['scope_wp_plugins'] ) && 'selected' == $log_settings_array['scope_wp_plugins'] ); ?>
									>
									<?php esc_html_e( 'Selected plugins only', 'logtastic' ); ?>
								</label>
								<br>
								
								<div class="logtastic-admin-sub-options" data-form-conditional-field-name="js_error_scope_wp_plugins_definition" data-form-conditional-field-value="selected">
									<i><?php esc_html_e( 'Select plugins to include in error logging', 'logtastic' ); ?>:</i>
									<br>
									<?php foreach ( $all_plugins as $logtastic_index => $logtastic_plugin ) { ?>
										<label>
											<input 
												name="js_error_scope_select_plugins_selected[]" 
												type="checkbox" 
												value="<?php echo esc_attr( explode('/', $logtastic_index)[0] ); ?>"
												<?php checked( isset( $log_settings_array['scope_wp_plugins_selected'] ) && is_array( $log_settings_array['scope_wp_plugins_selected'] ) && in_array( explode('/', $logtastic_index)[0], $log_settings_array['scope_wp_plugins_selected'] ) ); ?>
											>
											<?php echo esc_html( $logtastic_plugin['Name'] ); ?>
										</label>
										<br>
									<?php } ?>
								</div>
								
							</div>
							
						</fieldset>
					</td>
				</tr>
				
				<tr>
					<th scope="row"><?php esc_html_e( 'Stack Trace', 'logtastic' ); ?></th>
					<td>
						<fieldset>
						
							<legend class="screen-reader-text"><span><?php esc_html_e( 'Stack Trace', 'logtastic' ); ?></span></legend>
							
							<p><?php esc_html_e( 'Where possible, the JavaScript Error Log can attempt to capture a full stack trace (also known as a backtrace) for each error. The stack trace records a list of the functions that were called leading up to an error being generated. This can be useful tool when debugging errors.', 'logtastic' ); ?></p>
							
							<br>
							
							<br>
							
							<p><?php esc_html_e( 'Should a stack trace be captured for each error?', 'logtastic' ); ?></p>
							
							<br>
							
							<label>
								<input 
									type="radio" 
									name="js_error_log_capture_stacktrace" 
									value="1"
									<?php checked( !empty( $log_settings_array['capture_stacktrace'] ) ); ?>
								>
								<?php esc_html_e( 'Yes', 'logtastic' ); ?>
							</label>
							<br>
							
							<label>
								<input 
									type="radio" 
									name="js_error_log_capture_stacktrace" 
									value="0"
									<?php checked( empty( $log_settings_array['capture_stacktrace'] ) ); ?>
								>
								<?php esc_html_e( 'No', 'logtastic' ); ?>
							</label>
							<br>
							
						</fieldset>
					</td>
				</tr>
				
				
				<tr>
					<th scope="row"><?php esc_html_e( 'Session Information', 'logtastic' ); ?></th>
					<td>
						<fieldset>
						
							<legend class="screen-reader-text"><span><?php esc_html_e( 'Session Information', 'logtastic' ); ?></span></legend>
							
							<p><?php esc_html_e( "Where possible, the JavaScript Error Log can attempt to capture additional session information for each error, including the full url of the page on which the error occurred and information about the user's web browser.", 'logtastic' ); ?></p>
							
							<br>
							
							<br>
							
							<p><?php esc_html_e( 'Should session information be captured for each error?', 'logtastic' ); ?></p>
							
							<br>
							
							<label>
								<input 
									type="radio" 
									name="js_error_log_capture_session_info" 
									value="1"
									<?php checked( !empty( $log_settings_array['capture_session_info'] ) ); ?>
								>
								<?php esc_html_e( 'Yes', 'logtastic' ); ?>
							</label>
							<br>
							
							<label>
								<input 
									type="radio" 
									name="js_error_log_capture_session_info" 
									value="0"
									<?php checked( empty( $log_settings_array['capture_session_info'] ) ); ?>
								>
								<?php esc_html_e( 'No', 'logtastic' ); ?>
							</label>
							<br>
							
						</fieldset>
					</td>
				</tr>
				
				
				
				<tr>
					<th scope="row"><?php esc_html_e( 'Log Retention', 'logtastic' ); ?></th>
					<td>
						<fieldset>
						
							<legend class="screen-reader-text"><span><?php esc_html_e( 'Log Retention', 'logtastic' ); ?></span></legend>
							
							<p><?php esc_html_e( 'Choose how long to keep records for', 'logtastic' ); ?>:</p>
							<br>
							
							<label class="logtastic-admin-php-error-type-label">
								<input 
									type="radio" 
									name="js_error_log_retention_schedule" 
									value="0"
									<?php checked( empty( $log_settings_array['retention_schedule'] ) ); ?>
								>
								<?php esc_html_e( 'Keep records indefinitely', 'logtastic' ); ?>
							</label>
							<br>
							<label class="logtastic-admin-php-error-type-label">
								<input 
									type="radio" 
									name="js_error_log_retention_schedule"
									value="30"
									<?php checked( isset( $log_settings_array['retention_schedule'] ) && 30 == $log_settings_array['retention_schedule'] ); ?>
								>
								<?php esc_html_e( 'Delete records after 30 days', 'logtastic' ); ?>
							</label>
							<br>
							<label class="logtastic-admin-php-error-type-label">
								<input 
									type="radio" 
									name="js_error_log_retention_schedule"
									value="7"
									<?php checked( isset( $log_settings_array['retention_schedule'] ) && 7 == $log_settings_array['retention_schedule'] ); ?>
								>
								<?php esc_html_e( 'Delete records after 7 days', 'logtastic' ); ?>
							</label>
							<br>
							<label class="logtastic-admin-php-error-type-label">
								<input 
									type="radio" 
									name="js_error_log_retention_schedule"
									value="custom"
									disabled
								>
								<?php esc_html_e( 'Custom', 'logtastic' ); ?>
							</label>
							<div class="pro logtastic-admin-tooltip tooltip-medium">
								<?php esc_html_e( 'Pro', 'logtastic' ); ?> <span class="screen-reader-text"><?php esc_html_e( 'Pro Feature', 'logtastic' ); ?></span>
								<div class="logtastic-admin-tooltip-content">
									<?php
										printf(
											wp_kses(
												/* translators: %s: URL to the Logtastic Pro website */
												__( 'This feature is only available in the <a href="%s" target="_blank">Logtastic Pro version</a>.', 'logtastic' ),
												[ 'a' => [ 'href' => [], 'target' => [] ] ]
											),
											esc_url( LOGTASTIC_PLUGIN_URL )
										);
									?>
								</div>
							</div>
							<br>
							
						</fieldset>
					</td>
				</tr>
				
				<tr>
					<th scope="row"><?php esc_html_e( 'Ignored Errors', 'logtastic' ); ?></th>
					<td>
						
						<fieldset>
						
							<legend class="screen-reader-text"><span><?php esc_html_e( 'Ignored Errors', 'logtastic' ); ?></span></legend>
							
							<?php if ( false == $ignored_errors ) { ?>
								
								<p><?php esc_html_e( 'No errors are currently being ignored. You can select errors to ignore and exclude from logging via the main JavaScript Error Log page.', 'logtastic' ); ?></p>
								
							<?php } else { ?>
								
								<p id="ignored-errors-description">

									<?php

										$logtastic_count = count( $ignored_errors );

										printf(
											esc_html (
												/* translators: %s: number of ignored errors (wrapped in a span) */
												_n(
													'The following %s error is currently being ignored and excluded from logging:',
													'The following %s errors are currently being ignored and excluded from logging:',
													$logtastic_count,
													'logtastic'
												)
											),
											'<span id="ignored-error-count">' . esc_html( number_format_i18n( $logtastic_count ) ) . '</span>'
										);

									?>

								</p>

								<br>
								
								<div id="ignored-errors-table-wrapper" class="settings-ignore-list-table-wrapper">
								
									<table class="widefat view-log-table view-log-table-js-errors settings-ignore-list-table">
										<thead>
											<tr>
												<th><?php esc_html_e( 'Error Type', 'logtastic' ); ?></th>
												<th><?php esc_html_e( 'Source', 'logtastic' ); ?></th>
												<th class="column-error-detail"><?php esc_html_e( 'Error', 'logtastic' ); ?></th>
												<th></th>
											</tr>
										</thead>
										<tbody>
											<?php foreach ($ignored_errors as $error) { ?>
												<tr>
													<td class="error column-error-type has-label" data-colname="<?php esc_attr_e( 'Error Type', 'logtastic' ); ?>">
														<span class="error-level-label error-level-<?php echo esc_attr( $error['error_type'] ); ?>">
															<?php echo esc_html( Logtastic_JS_Error_Log_Admin::get_error_label( $error['error_type'] ) ); ?>
														</span>
													</td>
													<td class="error column-error-source has-label" data-colname="<?php esc_attr_e( 'Source', 'logtastic' ); ?>">
														<div class="source-label">
															<?php if ( 'plugin' == $error['source'] ) { ?>
																<span class="dashicons dashicons-admin-plugins source-icon"></span>
																<span class="source-text"><?php echo esc_html( Logtastic_JS_Error_Log_Admin::get_plugin_name_static( $error['source_slug'] ) ); ?></span>
															<?php } else if ( 'theme' == $error['source'] ) { ?>
																<span class="dashicons dashicons-admin-appearance source-icon"></span>
																<span class="source-text"><?php echo esc_html( Logtastic_JS_Error_Log_Admin::get_theme_name_static( $error['source_slug'] ) ); ?></span>
															<?php } else { ?>
																<span class="dashicons dashicons-wordpress source-icon"></span>
																<span class="source-text"><?php esc_html_e( 'WordPress Core', 'logtastic' ); ?></span>
															<?php } ?>
														</div>
													</td>
													<td class="error column-error-detail column-primary" data-colname="<?php esc_attr_e( 'Error', 'logtastic' ); ?>">
														<strong class="row-title">
																<?php Logtastic_JS_Error_Log_Admin::extract_and_display_error_title($error); ?>
														</strong>
														<div class="file">
															<strong><?php esc_html_e( 'File', 'logtastic' ); ?>:</strong> 
															<span><?php echo esc_html( $error['file'] ); ?></span>
														</div>
														<div class="line-column">
															<strong><?php esc_html_e( 'Line', 'logtastic' ); ?>:</strong>
															<span><?php echo esc_html ( $error['line'] ); ?></span>
															<strong><?php esc_html_e( 'Column', 'logtastic' ); ?>:</strong>
															<span><?php echo esc_html ($error['col'] ); ?></span>
														</div>
													</td>
													<td class="error column-error-stop-ignoring">
														<?php if ( !empty ( $this->access_permissions ) && in_array( 'js-error-log-edit', $this->access_permissions ) ) { ?>
															<button class="button button-default stop-ignoring-error" data-error-id="<?php echo esc_attr( $error['error_id'] ); ?>" onclick="logtasticStopIgnoringJSError(event)"><?php esc_html_e( 'Stop Ignoring', 'logtastic' ); ?></button>
														<?php } ?>
													</td>
												</tr>
											<?php } ?>
										</tbody>
									</table>
									
								</div>
								
							<?php } ?>
							
						</fieldset>
							
					</td>
				</tr>
				
			</tbody>
			
		</table>
		
		<p class="submit"><input type="submit" name="submit" id="submit" class="button button-primary" value="<?php esc_attr_e( 'Save Changes', 'logtastic' ); ?>"></p>
		
	</form>
	
</div>