/**
 * Localization
 **/

// Localization constants
const { __, _n, _x, sprintf } = wp.i18n;


/**
 * General
 **/

// Display admin notification
function logtasticDisplayNotification( type, message, dismissible = true ) {

	jQuery(function ($) {

		// One-time event delegation for dismiss button
		if (!logtasticDisplayNotification._bound) {
		$(document).on('click', '.notice.is-dismissible .notice-dismiss', function () {
			const $notice = $(this).closest('.notice.is-dismissible');
			$notice.fadeTo(100, 0, function () {
			$notice.slideUp(100, function () { $notice.remove(); });
			});
		});
		logtasticDisplayNotification._bound = true;
		}

		// Remove any existing notices
   		$('#wpbody-content .wrap:first .logtastic-notice').remove();

		// Build classes
		const classes = ['notice', 'logtastic-notice'];
		if (type === 'warning') classes.push('notice-warning');
		else if (type === 'info') classes.push('notice-info');
		else if (type === 'success') classes.push('notice-success');
		else if (type === 'error') classes.push('notice-error');
		if (dismissible) classes.push('is-dismissible');

		// Escape the message to avoid XSS
    	const safeMessage = $('<div/>').text(message).html();

		// Build HTML
		const html =
		'<div class="' + classes.join(' ') + '">' +
			'<p>' + safeMessage + '</p>' +
			(dismissible
			? '<button type="button" class="notice-dismiss"><span class="screen-reader-text">' + __( 'Dismiss this notice.', 'logtastic' ) + '</span></button>'
			: ''
			) +
		'</div>';

		// Pick an anchor (h1/h2/wp-heading-inline); fallback to prepend
		const $anchor = $(
			'#wpbody-content .wrap:first .wp-heading-inline, ' +
			'#wpbody-content .wrap:first h1:first, ' +
			'#wpbody-content .wrap:first h2:first'
		).first();

		if ($anchor.length) {
			$anchor.after(html);
		} else {
			// Fallback if no heading exists
			$('#wpbody-content .wrap:first').prepend(html);
		}

	});
	
	window.scrollTo({
		top: 0,
		behavior: 'smooth'
	});
	
}

// Javascript Tabs
document.addEventListener("DOMContentLoaded", function () {
	document.querySelectorAll("[data-logtastic-tab-target-id][data-logtastic-tab-group-id]").forEach(tabLink => {
		tabLink.addEventListener("click", function (event) {
			event.preventDefault();
			
			let tabGroupId = this.getAttribute("data-logtastic-tab-group-id");
			let targetTabId = this.getAttribute("data-logtastic-tab-target-id");

			// Remove active class from all tab links in the same group
			document.querySelectorAll(`[data-logtastic-tab-group-id='${tabGroupId}'][data-logtastic-tab-target-id]`).forEach(tab => {
				tab.classList.remove("active");
			});

			// Remove active class from all tab content sections in the same group
			document.querySelectorAll(`[data-logtastic-tab-group-id='${tabGroupId}'][data-logtastic-tab-id]`).forEach(tabContent => {
				tabContent.classList.remove("active");
			});

			// Add active class to the clicked tab
			this.classList.add("active");

			// Find and activate the corresponding content tab
			let targetContent = document.querySelector(`[data-logtastic-tab-group-id='${tabGroupId}'][data-logtastic-tab-id='${targetTabId}']`);
			if (targetContent) {
				targetContent.classList.add("active");
			}
		});
	});
});

// Admin responsive nav toggle
jQuery( document ).ready( function() {
	
	jQuery( 'button#logtastic-admin-header-nav-toggle' ).on( 'click', function() {
		jQuery( 'nav.logtastic-admin-header-nav-wrapper' ).toggleClass('visible');
	});

});

// Admin action button handler
jQuery( document ).ready( function() {
	
	jQuery( 'button[data-logtastic-action]' ).on( 'click', function() {
		
		var button = jQuery( this );
		var action = jQuery( button ).data( 'logtastic-action' );
		var nonce = jQuery( button ).data( 'nonce' );
		
		switch( action ) {
			
			// Enable/disable log
			case 'enable-disable-log':
				
				var tr = jQuery( button ).closest( 'tr' );
				var log_id = jQuery( tr ).data( 'log-id' );
				var log_name = jQuery( tr ).data( 'log-name' );
				
				if ( jQuery( button ).hasClass( 'enabled' ) ) {
					var current_status = 'enabled';
				} else {
					var current_status = 'disabled';
				}
				
				// Enable log
				if ( current_status == 'disabled' ) {
					
					// Disable button to prevent additional clicks
					jQuery( button ).prop( 'disabled', true );
					
					// Enable the log
					logtasticUpdateLogEnabledStatus( log_id, true, nonce, function(result) {
						
						if (result == true ) {
							
							// Change button class to enabled
							jQuery( button ).removeClass( 'disabled' );
							jQuery( button ).addClass( 'enabled' );
							
							// Change tr class to enabled
							jQuery( tr ).removeClass( 'disabled' );
							jQuery( tr ).addClass( 'enabled' );
							
							// Display success message
							logtasticDisplayNotification( 'success', sprintf( __( '%s enabled.', 'logtastic' ), log_name ) );
							
							// Show nav tab for log settings
							jQuery( '.log-settings-tabs a[data-log-id="' + log_id + '"]' ).addClass( 'enabled' );
							
							// Reload Main WP Admin Nav
							jQuery( '#adminmenu #toplevel_page_logtastic' ).load( jQuery(location).attr("href") + ' #adminmenu #toplevel_page_logtastic > *' );
							
							// Reload plugin admin header
							jQuery( '.logtastic-admin-header-nav.primary' ).load( jQuery(location).attr("href") + ' .logtastic-admin-header-nav.primary > *' );
							
							// Re-enable button to prevent additional clicks
							jQuery( button ).prop( 'disabled', false );
							
						} else {
							
							// Display error message
							logtasticDisplayNotification( 'error', sprintf( __( 'Failed to enable %s.', 'logtastic' ), log_name ) );
							
							// Re-enable button to prevent additional clicks
							jQuery( button ).prop( 'disabled', false );
							
						}
							
					});
					
				// Disable the log
				} else {
					
					// Disable button to prevent additional clicks
					jQuery( button ).prop( 'disabled', true );
					
					// Disable the log
					logtasticUpdateLogEnabledStatus( log_id, false, nonce, function(result) {
						
						if (result == true ) {
							
							// Change button class to enabled
							jQuery( button ).removeClass( 'enabled' );
							jQuery( button ).addClass( 'disabled' );
							
							// Change tr class to enabled
							jQuery( tr ).removeClass( 'enabled' );
							jQuery( tr ).addClass( 'disabled' );
							
							// Display success message
							logtasticDisplayNotification( 'success', sprintf( __( '%s disabled.', 'logtastic' ), log_name ) );
							
							// Hide nav tab for log settings
							jQuery( '.log-settings-tabs a[data-log-id="' + log_id + '"]' ).removeClass( 'enabled' );
							
							// Reload Main WP Admin Nav
							jQuery( '#adminmenu #toplevel_page_logtastic' ).load( jQuery(location).attr("href") + ' #adminmenu #toplevel_page_logtastic > *' );
							
							// Reload plugin admin header
							jQuery( '.logtastic-admin-header-nav.primary' ).load( jQuery(location).attr("href") + ' .logtastic-admin-header-nav.primary > *' );
							
							// Re-enable button to prevent additional clicks
							jQuery( button ).prop( 'disabled', false );
							
						} else {
						
							// Display error message
							logtasticDisplayNotification( 'error', sprintf( __( 'Failed to disable %s.', 'logtastic' ), log_name ) );
							
							// Re-enable button to prevent additional clicks
							jQuery( button ).prop( 'disabled', false );
							
						}
							
					});
					
				}
				
				break;
			
			default:
				logtasticDisplayNotification( 'error', __( 'Unknown action.', 'logtastic' ) );
				
		}
		
	} );
	
} );

// Sync bulk action values (above and below tables)
function logtasticBulkActionValueSync(event) {

	// Get the select that triggered the event
	let changedSelect = event.target; 

	// Get its selected value
	let selectedValue = changedSelect.value; 
	
	// Get its name attribute
	let fieldName = changedSelect.name;
		
	// Find all select elements with the same name and update their value
	document.querySelectorAll(`select[name="${fieldName}"]`).forEach(otherSelect => {
		if (otherSelect !== changedSelect) { // Prevent updating itself
			otherSelect.value = selectedValue;
		}
	});

}

// Bulk action button handler
function logtasticBulkAction(event) {
		
	// Prevent the default link action
	event.preventDefault();
	
	// Get all checked checkboxes with name 'selected-error'
	let selectedItems = document.querySelectorAll('input[name="selected-error[]"]:checked');
	
	if (selectedItems.length === 0) {
		
		// Show error message if no items are checked
		logtasticDisplayNotification( 'error', __( 'No items selected. Please select at least one item.', 'logtastic' ) );
		
	} else {
		
		// Collect values of checked checkboxes
		let selectedItemIDs = Array.from(selectedItems).map(checkbox => checkbox.value);
		
		// Find the closest parent div with class "bulkactions"
		let parentDiv = event.target.closest(".bulkactions");
		
		// Find the action select element inside the same div
		let selectElement = parentDiv.querySelector('select[name="bulk-action-select"]');
		
		// Get the selected action value
		let selectedValue = selectElement ? selectElement.value : null;
		
		// If selected action value is delete-errors or ignore-errors
		if ( selectedValue === 'delete-errors' || selectedValue === 'ignore-errors' ) {
			
			// Prepare confirmation message
			let confirmationText = sprintf(
				_n(
					'This will delete the selected %d error and all associated data.\n\n',
					'This will delete the selected %d errors and all associated data.\n\n',
					selectedItems.length,
					'logtastic'
				),
				selectedItems.length
			);

			
			// Prepare confirmation message - additional content
			confirmationText += 'This operation cannot be undone.\n\n';
			
			// Prepare confirmation message - if selected action value == ignore, add additional message regarding errors being ignored
			if (selectedValue === 'ignore-errors') {

				confirmationText +=	_n(
					'In addition, all future occurrences of this error will be ignored - you can remove this error from the ignore list at any time from the log settings page.',
					'In addition, all future occurrences of these errors will be ignored - you can remove these errors from the ignore list at any time from the log settings page.',
					selectedItems.length,
					'logtastic'
				);
				
			}
			
			// Confirm and submit
			if ( confirm( confirmationText ) == true ) {
				
				// Get the form element
				const form = document.getElementById('action');
				
				if (form) {
					
					// Set the values of the hidden inputs - action type
					if (selectedValue === 'ignore-errors') {
						form.querySelector('input[name="action"]').value = 'ignore';
					} else if (selectedValue === 'delete-errors') {
						form.querySelector('input[name="action"]').value = 'delete';
					}
					
					// Set the values of the hidden inputs - add comma separated list of error ids
					form.querySelector('input[name="error_ids"]').value = selectedItemIDs.join(',');
					
					// Submit the form
					form.submit();
					
				}
				
			}
			
			
		} else {
			
			// Show error message if no valid action is selected
			logtasticDisplayNotification( 'error', __( 'Please select an action to perform.', 'logtastic' ) );

		}
		
		
	}
	
}

// Function to encode control characters
function logtasticEncodeControlChars(str) {
	return str
		.replace(/\n/g, '\\n')  // replace newlines with literal '\n'
		.replace(/\t/g, '\\t')  // replace tabs with literal '\t'
		.replace(/\r/g, '\\r')  // replace carriage return with literal '\r'
}

// Overflow checker
const logtasticHasOverflow = (ele) => ele.scrollHeight > ele.clientHeight || ele.scrollWidth > ele.clientWidth;
const logtasticHasOverflowVertical = (ele) => ele.scrollHeight > ele.clientHeight;
const logtasticHasOverflowHorizontal = (ele) => ele.scrollWidth > ele.clientWidth;

// Function to escable html in a string
function logtasticEscapeHTML(str) {
	return String(str).replace(/[&<>"']/g, function (match) {
		const escapeMap = {
			'&': '&amp;',
			'<': '&lt;',
			'>': '&gt;',
			'"': '&quot;',
			"'": '&#39;'
		};
		return escapeMap[match];
	});
}

// Add class to admin nav tabs if has horizontal overflow - page load
jQuery( document ).ready( function() {
	const navTabs = jQuery( '.inspired-logtastic .nav-tab-wrapper' );
	if (navTabs.length) {
		const el=navTabs[0];
		if ( logtasticHasOverflowHorizontal( el ) ) {
			navTabs.addClass('overflown');
		}
	}
});

// Add class to admin nav tabs if has horizontal overflow - resize
jQuery(window).on('resize', function() {
	const navTabs = jQuery( '.inspired-logtastic .nav-tab-wrapper' );
	if (navTabs.length) {
		navTabs.removeClass('overflown');
		const el=navTabs[0];
		if ( logtasticHasOverflowHorizontal( el ) ) {
			navTabs.addClass('overflown');
		}
	}
});

// Admin nav tabs responsive toggle
jQuery( document ).ready( function() {
	// Select All
	jQuery( '#logtastic-admin-settings-nav-tabs-toggle' ).on( 'click', function(event) {
		
		// Prevent default action
		event.preventDefault();
		
		// If already toggled toggled
		if ( jQuery( this ).hasClass('toggled') ) {
			
			// Remove dropdown nav
			jQuery('.settings-tabs-responsive').remove();

			// Remove class toggled
			jQuery( this ).removeClass('toggled');

		} else {
			
			// Construct dropdown nav
    		const generalSettingsTabs = jQuery('.inspired-logtastic .nav-tab-wrapper .general-settings-tabs').children().not('.nav-tab-active');
			const logSettingsTabs = jQuery('.inspired-logtastic .nav-tab-wrapper .log-settings-tabs').children('.enabled').not('.nav-tab-active');
			const responsiveContainer = jQuery('<div class="settings-tabs-responsive"></div>');
			const hr = jQuery('<hr>');
        	jQuery('.inspired-logtastic .nav-tab-wrapper').after(responsiveContainer);
			generalSettingsTabs.clone(true, true).appendTo(responsiveContainer);
			if (logSettingsTabs.length) {
				hr.appendTo(responsiveContainer);
			}
    		logSettingsTabs.clone(true, true).appendTo(responsiveContainer);

			// Add class toggled
			jQuery( this ).addClass('toggled');

		}
	});
});


/**
 * Enable/Disable Logs Page
 **/

// Update log enabled status
function logtasticUpdateLogEnabledStatus( log_id, new_status, nonce, callback ) {
	
	jQuery.ajax( {
		url: ajaxurl,
		type: 'POST',
		data: {
			action: 'logtastic_enable_disable_log',
			log: log_id,
			status: new_status,
			_wpnonce: nonce
		},
		success: function( response ) {
			if ( response.success ) {
				callback(true);
			} else {
				callback(false);
			}
		},
		error: function(xhr, status, error) {
			callback(false);
		}
	} );
	
}


/**
 * Settings Pages
 **/

// Settings form conditional logic
function logtasticSettingsFormConditionalLogic(form, event) {
	
	// Check that passed form value is a valid form
	if (form instanceof HTMLFormElement) {
	
		// If no event passed, process conditional logic for all all form fields
		if (event === undefined) {

			const conditional_elements = form.querySelectorAll('[data-form-conditional-field-name]');
			conditional_elements.forEach((element) => {
				logtasticSettingsFormElementConitionalLogic(form, element);
			});

		// Else process conditional logic for specific field that trigger change event
		} else {

			const conditional_elements = form.querySelectorAll('[data-form-conditional-field-name="' + event.srcElement.name + '"]');
			conditional_elements.forEach((element) => {
				logtasticSettingsFormElementConitionalLogic(form, element);
			});

		}
		
	}
	
}

// Settings form conditional logic - hide / show element
function logtasticSettingsFormElementConitionalLogic(form, element) {

	const conditional_field_name = element.getAttribute('data-form-conditional-field-name');
	const conditional_field_value = logtasticGetFieldValueByName(form, conditional_field_name);
	const conditional_match_value = element.getAttribute('data-form-conditional-field-value');

	if ( conditional_field_value ==  conditional_match_value ) {
		// Show conditional field
		element.style.display = 'block';
	} else {
		// Hide conditional field
		element.style.display = 'none';
	}
	
}

// Settings form conditional logic - get field value by field name
function logtasticGetFieldValueByName(form, fieldName) {

	// Get all elements with the given name
	const field = form.elements[fieldName];

	// Check if the field exists
	if (!field) return null;

	// Handle radio buttons or checkboxes with the same name
	if (field instanceof NodeList || Array.isArray(field)) {
		if (field[0].type === "radio") {
			// Find the checked radio button
			const checkedRadio = Array.from(field).find(input => input.checked);
			return checkedRadio ? checkedRadio.value : null;
		} else if (field[0].type === "checkbox") {
			// Get all checked checkboxes
			return Array.from(field)
				.filter(input => input.checked)
				.map(input => input.value);
		}
	}

	// Handle a single checkbox
	if (field.type === "checkbox") {
		return field.checked ? field.value : null;
	}

	// Handle select elements
	if (field.type === "select-multiple") {
		// For multiple select, get all selected options
		return Array.from(field.options)
			.filter(option => option.selected)
			.map(option => option.value);
	} else if (field.type === "select-one") {
		return field.value;
	}

	// Handle regular inputs (text, number, email, etc.)
	return field.value;

}

// Settings form conditional logic - run conditional logic on page load
jQuery( document ).ready( function() {
	logtasticSettingsFormConditionalLogic();
});

// Settings form conditional logic - run conditional logic on form change
window.onload = function () {
	const conditional_forms = document.querySelectorAll('form[data-form-conditional="true"]');
	// Add a change event listener to each form
	conditional_forms.forEach((form) => {
		form.addEventListener('change', (event) => {
			// Pass the form and event that triggered the event to the function
			logtasticSettingsFormConditionalLogic(form, event);
		});
		logtasticSettingsFormConditionalLogic(form);
	});
};

// Settings Select All/None Links
jQuery( document ).ready( function() {
	// Select All
	jQuery( 'a[data-select-all]' ).on( 'click', function(event) {
		// Prevent default action
		event.preventDefault();
		// Parent fieldset
		let parentFieldset = jQuery( this ).closest( 'fieldset' );
		// Mark all checkboxes in parent fieldset as checked
		parentFieldset.find('input[type="checkbox"]').prop('checked', true);
	});
	// Select None
	jQuery( 'a[data-select-none]' ).on( 'click', function(event) {
		// Prevent default action
		event.preventDefault();
		// Parent fieldset
		let parentFieldset = jQuery( this ).closest( 'fieldset' );
		// Mark all checkboxes in parent fieldset as checked
		parentFieldset.find('input[type="checkbox"]').prop('checked', false);
	});
});


/**
 * General - View Log Page Functions
 **/

// Close Details Modal
function logtasticCloseDetailsModal() {
	document.body.classList.remove('show-details');
	document.querySelectorAll('div.details-modal-body').forEach(div => {
		div.classList.remove('active');
	});
}

// Generate date array for occurrences chart
function logtasticGenerateDateArray(startDateString) {
	
	let startDate = new Date(startDateString);
	let endDate = new Date();
	let result  = [];
	let dayMillisec = 24 * 60 * 60 * 1000;
	
	for (let date = startDate; date <= endDate; date = new Date(date.getTime() + dayMillisec)) {
		// Convert start date to a YYYY-MM-DD string
		let startDateString = date.toISOString().split('T')[0];
		result.push(startDateString);
	}
	
	return result;
}


// Function to copy previous input value to clipboard
function logtasticCopyPreviousSiblingInputToClipboard(event) {
	event.preventDefault();
	event.stopPropagation();
	let inputField = event.target.previousElementSibling;
	inputField.select();
	inputField.setSelectionRange(0, 99999);
	document.execCommand('copy');
}


/**
 * PHP Error Log / JS Error Log - Shared Functions
 **/


// Limit an error message to 50 characters
function logtasticTruncateErrorMsg(str, maxLength = 50) {
	return str.length > maxLength ? str.substring(0, maxLength) + "..." : str;
}


// Occurrence pagination - button navigation
function logtasticOccurrencesPaginationNavigation(event, context) {
		
	event.preventDefault();
	
	let targetModal = event.target.closest('div.details-modal-body');
	
	// Disable pagination buttons
	const navigationButtons = targetModal.querySelectorAll('div.tablenav button');
	navigationButtons.forEach(button => {
		button.disabled = true;
	});
	
	// Empty table
	const tableBody = targetModal.querySelector('.occurrences-table tbody');
	while (tableBody.firstChild) {
		tableBody.removeChild(tableBody.firstChild);
	}
	
	// Show loading screen
	const occurrencesTableWrapper = targetModal.querySelector('div.occurrences-table-wrapper');
	occurrencesTableWrapper.classList.add('loading');
	
	let targetPage = event.target.dataset.targetPage;

	if ( context == 'PHP') {

		logtasticLoadPHPErrorData( targetModal, targetPage );

	} else if ( context == 'JS') {
	
		logtasticLoadJSErrorData( targetModal, targetPage );

	}
	
}


// Occurrence pagination - form submit
function logtasticOccurrencesPaginationSubmit(event, context) {
		
	event.preventDefault();
	
	let targetModal = event.target.closest('div.details-modal-body');
	
	// Disable pagination buttons
	const navigationButtons = targetModal.querySelectorAll('div.tablenav button');
	navigationButtons.forEach(button => {
		button.disabled = true;
	});
	
	// Empty table
	const tableBody = targetModal.querySelector('.occurrences-table tbody');
	while (tableBody.firstChild) {
		tableBody.removeChild(tableBody.firstChild);
	}
	
	// Show loading screen
	const occurrencesTableWrapper = targetModal.querySelector('div.occurrences-table-wrapper');
	occurrencesTableWrapper.classList.add('loading');
	
	let targetPage = event.target.elements.paged.value;
	
	if ( context == 'PHP') {

		logtasticLoadPHPErrorData( targetModal, targetPage );

	} else if ( context == 'JS') {
	
		logtasticLoadJSErrorData( targetModal, targetPage );

	}
	
}


// Show additional details modal
function logtasticShowAdditionalDetails() {
		
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Get the additional details modal ID
	const additionalDetailsModalId = clickedElement.dataset.modalId;
	
	// Find target modal
	const additionalDetailsModal = document.querySelector('div.additional-details-modal[data-modal-id="' + additionalDetailsModalId + '"]');
	
	// Add class 'show' to target modal
	additionalDetailsModal.classList.add('show');
	
}


// Close additional details modal
function logtasticCloseAdditionalDetails() {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Find the parent additional details modal
	const additionalDetailsModal = clickedElement.closest('div.additional-details-modal');	
	
	// Remove class 'show' from target modal
	additionalDetailsModal.classList.remove('show');
	
}


// Close stack trace modal
function logtasticCloseStackTrace() {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Find the parent additional details modal
	const stackTraceModal = clickedElement.closest('div.stack-trace-modal');
	
	// Remove class 'show' from target modal
	stackTraceModal.classList.remove('show');
	
}


// Switch stack trace view ( raw / visual )
function logtasticSwitchStackTrace(target) {
		
	let stackTraceModal = document.querySelector('div.stack-trace-modal');
	let rawStackTraceElement = stackTraceModal.querySelector('#stack-trace-body-content-raw');
	let rawStackTraceSwitcher = stackTraceModal.querySelector('#stack-trace-body-content-switcher-raw');
	let visualStackTraceElement = stackTraceModal.querySelector('#stack-trace-body-content-visual');
	let visualStackTraceSwitcher = stackTraceModal.querySelector('#stack-trace-body-content-switcher-visual');
	
	if ( target == 'raw' ) {
		
		visualStackTraceSwitcher.classList.remove('active');
		rawStackTraceSwitcher.classList.add('active');
		visualStackTraceElement.classList.remove('active');
		rawStackTraceElement.classList.add('active');
		
	} else if ( target == 'visual' ) {
		
		rawStackTraceSwitcher	.classList.remove('active');
		visualStackTraceSwitcher.classList.add('active');
		rawStackTraceElement.classList.remove('active');
		visualStackTraceElement.classList.add('active');
		
	}
	
}


// Function to load an error details modal on page load if view_error_id is set
jQuery(document).ready( function() {
		
	let errorID = logtastic_admin.view_error_id;

	if ( ( logtastic_admin.page == 'logtastic_php-error-log' || logtastic_admin.page == 'logtastic_js-error-log' ) && errorID > 0 ) {
	
		// Find target details modal
		let targetModal = document.querySelector('div.details-modal-body[data-error-id="' + errorID + '"]');
		
		if (targetModal) {
			
			// Show error details modal
			document.body.classList.add('show-details');
			targetModal.classList.add('active');
			
			// Load error data via ajax
			if ( logtastic_admin.page == 'logtastic_php-error-log' ) {
				logtasticLoadPHPErrorData( targetModal );
			} else if ( logtastic_admin.page == 'logtastic_js-error-log' ) {
				logtasticLoadJSErrorData( targetModal );
			}
			
		}

	}
	
});


// Occurrences table expandable rows on mobile
jQuery( document ).ready( function() {
	jQuery( '.occurrences-table-wrapper' ).on( 'click', '.occurrences-table tbody tr td:first-child', function() {
		if (window.matchMedia('(max-width: 782px)').matches) {
            jQuery(this).closest('tr').toggleClass('expanded');
        }
	});
});


/**
 * PHP Error Log 
 **/


// Load PHP error data via ajax and update modal content
function logtasticLoadPHPErrorData( targetModal, occurrencesPage = 1 ) {
		
	// Get the error ID
	const errorID =	 targetModal.dataset.errorId;
	
	// Load data via ajax
	jQuery.ajax({
		type : 'post',
		dataType : 'json',
		url : logtastic_admin.ajax_url,
		data : { 
			action: 'logtastic_load_php_error_details',
			_ajax_nonce: logtastic_admin.load_php_error_data_nonce,
			error_id: errorID,
			occurrences_page: occurrencesPage
		}
	}).done(function(response) {

		// Was request successful?
		if ( true == response.success ) {
		
			// Check if total occurrence count has changed since last loaded
			let prevOccurrenceCount = targetModal.dataset.occurrenceCount;
			let newOccurrenceCount = response?.data?.total_occurrences;
			if ( newOccurrenceCount > prevOccurrenceCount ) {
				occurrenceCountChanged = true;
			} else {
				occurrenceCountChanged = false;
			}
			
			// Get error source
			let errorSource = targetModal.dataset.errorSource;
			
			// If data is not already loaded for target modal, or the occurrence count has changed, update the error details
			if ( targetModal.dataset.loaded != 'true' || occurrenceCountChanged == true ) {
				
				// Update source version
				if ( errorSource == 'wp_core' ) {
					let sourceVersionString = response?.data?.wp_version?.range_string_full;
					if ( sourceVersionString && sourceVersionString !== false ) {
						targetModal.querySelector( '.source-version' ).textContent = sourceVersionString;
					}
				} else {
					let sourceVersionString = response?.data?.source_version?.range_string_full;
					if ( sourceVersionString && sourceVersionString !== false ) {
						targetModal.querySelector( '.source-version' ).textContent = sourceVersionString;
					}
				}
				
				// Update Wordpress Version
				let wpVersionStringShort = response.data.wp_version.range_string_short;
				if ( wpVersionStringShort && wpVersionStringShort !== false ) {
					targetModal.querySelector( '.wp-version-number' ).textContent = wpVersionStringShort;
				}
				let wpVersionStringText = response?.data?.wp_version?.range_string_version_text;
				if ( wpVersionStringText && wpVersionStringText !== false ) {
					targetModal.querySelector( '.wp-version-text' ).textContent = wpVersionStringText;
				}
				
				// Update PHP Version
				let phpVersionStringShort = response.data.php_version.range_string_short;
				if ( phpVersionStringShort && phpVersionStringShort !== false ) {
					targetModal.querySelector( '.php-version-number' ).textContent = phpVersionStringShort;
				}
				let phpVersionStringText = response?.data?.php_version?.range_string_version_text;
				if ( phpVersionStringText && phpVersionStringText !== false ) {
					targetModal.querySelector( '.php-version-text' ).textContent = phpVersionStringText;
				}
				
				// Show Environment Data
				targetModal.querySelector( '.sidebar-section.environment .loading' ).classList.add('complete');
				targetModal.querySelector( '.sidebar-section.environment .loaded' ).classList.add('complete');
				
				// Get the first (earliest) date from occurrences_by_date
				const startDate = Object.keys(response.data.occurrences_by_date)[0]; // The first date in the object
				
				// Generate an array of dates from the first occurrence date until the current date
				const datesArray = logtasticGenerateDateArray(startDate);
				
				// Create a new object with dates and their corresponding occurrences (0 if missing)
				const mergedData = datesArray.reduce((acc, date) => {
					acc[date] = response.data.occurrences_by_date[date] || 0;
					return acc;
				}, {});
				
				// Prepare labels/data for Chart
				const labels = [];
				const values = [];
				
				for (const [key, value] of Object.entries(mergedData)) {
					labels.push(key);
					values.push(value);
				}
				
				// Destroy existing chart if already defined
				const existingChart = Chart.getChart( 'occurrence-chart-' + errorID );
				if (existingChart) {
					existingChart.destroy();  // Destroy the chart instance
				}
				
				// Create Chart
				const chartElement = document.getElementById( 'occurrence-chart-' + errorID );
				const chart = new Chart(chartElement, {
				type: 'bar',
				data: {
					datasets: [{
						data: values,
						maxBarThickness: 5, // Set maximum bar thickness to 5px
					}],
					labels: labels
					},
				options: {
					responsive: true,
					plugins: {
					legend: {
						display: false, // Hide the legend
					},
					},
					scales: {
						x: {
							ticks: {
							display: false, // Hide the x-axis labels
							},
							grid: {
								display: false,  // Hide gridlines for the X-axis
							},
							border: {
							display: false, // Hide the x-axis line at the bottom
							},
						},
						y: {
							ticks: {
							display: false, // Hide the y-axis labels
							},
							grid: {
								display: false,  // Hide gridlines for the Y-axis
							},
							border: {
							display: false, // Hide the y-axis line at the bottom
							}
						}
					},
					layout: {
					padding: {
						top: 0,
						left: 0,
						right: 0,
						bottom: 0, // Remove all padding around the chart
					},
					}
				}
				});
				
				// Show chart
				targetModal.querySelector( '.sidebar-section.occurrences .occurrences-chart .loading' ).classList.add('complete');
				targetModal.querySelector( '.sidebar-section.occurrences .occurrences-chart .loaded' ).classList.add('complete');
				
				// Update latest occurrence date string
				let lastOccurredString = response?.data?.last_occurrence_string;
				if ( lastOccurredString && lastOccurredString !== false ) {
					targetModal.querySelector( '.sidebar-section.occurrences .last-occurred span' ).innerHTML = lastOccurredString;
				}
				
				// Update total occurrences value
				if ( newOccurrenceCount && newOccurrenceCount !== false ) {
					targetModal.querySelector( '.sidebar-section.occurrences .total-occurrences span' ).textContent = newOccurrenceCount;
				}
				
				// Update total occurrence data attribute
				if ( newOccurrenceCount && newOccurrenceCount !== false ) {
					targetModal.dataset.occurrenceCount = newOccurrenceCount;
				}
				
			}
			
			// Populate Occurrences Table Navigation
			const tableNav = targetModal.querySelector('.occurrences-table-navigation');
			tableNav.innerHTML = response.data.occurrences_nav_html;
			
			// Populate Occurrences Table
			const tableBody = targetModal.querySelector('.occurrences-table tbody');
			while (tableBody.firstChild) {
				tableBody.removeChild(tableBody.firstChild);
			}
			
			response.data.occurrences.forEach(item => {
				const row = document.createElement("tr");
				row.setAttribute("data-occurrence-id", item.occurrence_id); // Add occurrence_id as data attribute
				
				// Add Timestamp Column
				const dataCol = document.createElement("td");
				dataCol.innerHTML = item.date_time_string;
				row.appendChild(dataCol);
				
				if ( errorSource == 'plugin' || errorSource == 'theme' ) {
				
					// Add Source Version Column
					const sourceVerCol = document.createElement("td");
					if ( errorSource == 'plugin' ) {
						sourceVerCol.setAttribute("data-colname", __( 'Plugin Version', 'logtastic' ) );
					}
					if ( errorSource == 'theme' ) {
						sourceVerCol.setAttribute("data-colname", __( 'Theme Version', 'logtastic' ) );
					}
					sourceVerCol.textContent = item.source_version;
					row.appendChild(sourceVerCol);
					
				}
				
				// Add WP Version Column
				const wpVerCol = document.createElement("td");
				wpVerCol.setAttribute("data-colname", __( 'WP Version', 'logtastic' ) );
				wpVerCol.textContent = item.wp_version;
				row.appendChild(wpVerCol);
				
				// Add PHP Version Column
				const phpVerCol = document.createElement("td");
				phpVerCol.setAttribute("data-colname", __( 'PHP Version', 'logtastic' ) );
				phpVerCol.textContent = item.php_version;
				row.appendChild(phpVerCol);
				
				// Add user column
				const userCol = document.createElement("td");
				userCol.setAttribute("data-colname", __( 'User', 'logtastic' ) );
				userCol.innerHTML = `
					<div class="pro logtastic-admin-tooltip tooltip-small">
						Pro 
						<span class="screen-reader-text">${ __( 'Pro Feature', 'logtastic' ) }</span>
						<div class="logtastic-admin-tooltip-content">
							${ sprintf(
								__( 'This feature is only available in the <a href="%s" target="_blank">Logtastic Pro version</a>.', 'logtastic' ),
								logtastic_admin.plugin_website
							) }
						</div>
					</div>
				`;
				row.appendChild(userCol);
			
				// Add stack trace column
				const stackTraceCol = document.createElement("td");
				stackTraceCol.setAttribute("data-colname", __( 'Stack Trace', 'logtastic' ) );
				if ( item.stack_trace_available == "1" ) {
					stackTraceCol.innerHTML = '<a href="#" onclick="logtasticLoadPHPStackTrace(' + item.occurrence_id + ');">View Stack Trace</a>';
				} else {
					stackTraceCol.textContent = 'n/a';
				}
				row.appendChild(stackTraceCol);
			
				tableBody.appendChild(row);
				
				// Update loaded data attribute
				targetModal.dataset.loaded = 'true';
				
			});
			
			// Hide occurrences table loading screen
			const occurrencesTableWrapper = targetModal.querySelector('div.occurrences-table-wrapper');
			occurrencesTableWrapper.classList.remove('loading');

		} else {

			// Close details modal
			document.body.classList.remove('show-details');
			document.querySelectorAll('div.details-modal-body').forEach(div => {
				div.classList.remove('active');
			});

			// Show error notification
			if (response && typeof response.data === 'string' && response.data.trim() !== '') {
				logtasticDisplayNotification( 'error', response.data );
			} else {
				logtasticDisplayNotification( 'error', __( 'Unknown error loading requested error details.', 'logtastic' ) );
			}

		}

	}).fail(function (response) {

		// Close details modal
		document.body.classList.remove('show-details');
		document.querySelectorAll('div.details-modal-body').forEach(div => {
			div.classList.remove('active');
		});

		// Show error notification
		logtasticDisplayNotification( 'error', __( 'Error loading requested error details.', 'logtastic' ) );

	});
	
}


// View PHP Error Details Modal
function viewPHPErrorDetails(event) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');		
	
	if ( parentTr ) {
		
		// Get the id of the <tr> element
		const elementID = parentTr.id;
		
		// Extract the numerical part after the hyphen (the error ID)
		const errorID = elementID.split('-')[1];
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Find target details modal
			let targetModal = document.querySelector('div.details-modal-body[data-error-id="' + errorID + '"]');
			
			if (targetModal) {
				
				// Show error details modal
				document.body.classList.add('show-details');
				targetModal.classList.add('active');
				
				// Load error data via ajax
				logtasticLoadPHPErrorData( targetModal );
				
			}
			
		}
		
	}
	
}


// Navigate PHP Error Details Modal
function logtasticNavigatePHPErrorDetails( order ) {
	
	// Find target details modal
	let targetModal = document.querySelector('div.details-modal-body[data-nav-order="' + order + '"]');
	
	if (targetModal) {
		// Remove currently active details
		document.querySelectorAll('div.details-modal-body').forEach(div => {
			div.classList.remove('active');
		});
		// Set target as active
		targetModal.classList.add('active');
		
		// Load error data via ajax
		logtasticLoadPHPErrorData( targetModal );
		
	}
}


// Delete Single PHP Error
function logtasticDeleteSinglePHPError(event) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Define vars to hold error id and error message
	let errorID;
	let errorMsg;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');
	
	if ( parentTr ) {
		
		// Find the error message
		errorMsg = parentTr.querySelector('strong.row-title').textContent.trim();
		
		// Get the id of the <tr> element
		const elementID = parentTr.id;
		
		// Extract the numerical part after the hyphen (the error ID)
		errorID = elementID.split('-')[1];
		
	} else {
		
		// Find the parent modal body
		const parentModal = clickedElement.closest('.details-modal-body');
		
		if ( parentModal ) {
			
			errorID = parentModal.dataset.errorId;
			
			errorMsg = parentModal.querySelector('.details-intro h2').textContent.trim();
			
		}
		
	}
	
	
	if ( typeof errorID !== 'undefined' && typeof errorMsg !== 'undefined' ) {
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Confirmation 
			let confirmationText = __( 'This will delete the following error and all associated data:' , 'logtastic' );
			confirmationText += '\n\n' + errorMsg + '\n\n';
			confirmationText += __( 'This operation cannot be undone.' , 'logtastic' );
			if ( confirm( confirmationText ) == true ) {
				
				// Get the form element
				const form = document.getElementById('action');
				
				if (form) {
					// Set the values of the hidden inputs
					form.querySelector('input[name="action"]').value = 'delete';
					form.querySelector('input[name="error_ids"]').value = errorID;
					
					// Submit the form
					form.submit();
				}
				
				
			}
			
		}
		
	}
	
}


// Ignore Single PHP Error
function logtasticIgnoreSinglePHPError(event) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Define vars to hold error id and error message
	let errorID;
	let errorMsg;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');		
	
	if ( parentTr ) {
		
		// Find the error message
		errorMsg = parentTr.querySelector('strong.row-title').textContent.trim();
		
		// Get the id of the <tr> element
		const elementID = parentTr.id;
		
		// Extract the numerical part after the hyphen (the error ID)
		errorID = elementID.split('-')[1]; // Split the string and get the second part
		
	} else {
		
		// Find the parent modal body
		const parentModal = clickedElement.closest('.details-modal-body');
		
		if ( parentModal ) {
			
			errorID = parentModal.dataset.errorId;
			errorMsg = parentModal.querySelector('.details-intro h2').textContent.trim();
			
		}
		
	}
	
	if ( typeof errorID !== 'undefined' && typeof errorMsg !== 'undefined' ) {
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Confirmation 
			let confirmationText = __( 'This will delete the following error and all associated data:' , 'logtastic' );
			confirmationText += '\n\n' + errorMsg + '\n\n';
			confirmationText += __( 'This operation cannot be undone.' , 'logtastic' );
			confirmationText += '\n\n';
			confirmationText += __( 'In addition, all future occurrences of this error will be ignored - you can remove this error from the ignore list at any time from the log settings page.' , 'logtastic' );
			if ( confirm( confirmationText ) == true ) {
				
				// Get the form element
				const form = document.getElementById('action');
				
				if (form) {
					// Set the values of the hidden inputs
					form.querySelector('input[name="action"]').value = 'ignore';
					form.querySelector('input[name="error_ids"]').value = errorID;
					
					// Submit the form
					form.submit();
				}
				
				
			}
			
		}
		
	}
	
}


// Expand an arg display in the stack trace
function logtasticStackTraceExpandArgs(el) {
	
	// Find the parent arg content span
	const argContent = el.closest('.arg-content');	
	
	// Add class expanded to the arg content span
	argContent.classList.add('expanded');
	
}


// Display arguments in the php stack trace, similar to PHP var_dump() function
function logtasticStackTraceVarDump(value, indent = 0) {
	
	const padding = '  '.repeat(indent);
	let output = '';

	if (Array.isArray(value)) {
		if (indent > 0) {
			output += `Array(${value.length}) {\n`;
		}
		value.forEach((v, i) => {
			output += padding + '[' + i + '] => ' + logtasticStackTraceVarDump(v, indent + 1);
		});
		if (indent > 0) {
			let padding_close = '  '.repeat(indent - 1);
			output += padding_close + "}\n";
		}
	} else if (typeof value === 'object' && value !== null) {
		if ('__class__' in value && '__properties__' in value) {
			// Simulated PHP object
			const classValue = value.__class__;
			const propertiesValue = value.__properties__;

			if (indent > 0) {
				output += `Object(${classValue}) (${Object.keys(propertiesValue).length}) {\n`;
			}

			for (const key in propertiesValue) {
				if (Object.hasOwnProperty.call(propertiesValue, key)) {
					let cleanKey = key.replace(/\x00/g, '');
					if (cleanKey.startsWith('*')) {
						output += padding + '["' + cleanKey.slice(1) + '":protected] => ' + logtasticStackTraceVarDump(propertiesValue[key], indent + 1);
					} else {
						output += padding + '["' + key + '"] => ' + logtasticStackTraceVarDump(propertiesValue[key], indent + 1);
					}
				}
			}

			if (indent > 0) {
				let padding_close = '  '.repeat(indent - 1);
				output += padding_close + "}\n";
			}
		} else {
			// Standard key/value object
			if (indent > 0) {
				output += `Array(${Object.keys(value).length}) {\n`;
			}

			for (const key in value) {
				if (Object.hasOwnProperty.call(value, key)) {
					let cleanKey = key.replace(/\x00/g, '');
					if (cleanKey.startsWith('*')) {
						output += padding + '["' + cleanKey.slice(1) + '":protected] => ' + logtasticStackTraceVarDump(value[key], indent + 1);
					} else {
						output += padding + '["' + key + '"] => ' + logtasticStackTraceVarDump(value[key], indent + 1);
					}
				}
			}

			if (indent > 0) {
				let padding_close = '  '.repeat(indent - 1);
				output += padding_close + "}\n";
			}
		}
	} else if (typeof value === 'string') {
		output += `"${value}"\n`;
	} else {
		output += String(value) + "\n";
	}

	return output;
}


// Load PHP stack trace via ajax and display in modal for a given error occurrence
function logtasticLoadPHPStackTrace(occurrenceID) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the parent error modal and retrieve details
	const errorModal = event.target.closest('.details-modal-body');	 
	const errorID = errorModal.dataset.errorId;
	const errorLevel = errorModal.dataset.errorLevel;
	const errorTitleHTML = errorModal.querySelector('.details-modal-body-content .details-intro h2').innerHTML;
	const errorFile = errorModal.querySelector('.details-modal-body-content .details-intro .file span').innerHTML;
	const errorLine = errorModal.querySelector('.details-modal-body-content .details-intro .line span').innerHTML;
	
	// Get stack trace modal element
	let stackTraceModal = document.querySelector('div.stack-trace-modal');

	// Get stack trace modal body element
	let stackTraceModalBody = stackTraceModal.querySelector('div.stack-trace-body-content');
	
	// Get WP Home Path from Stack Trace Modal Data Attribute
	const wpHomePath = stackTraceModal.dataset.wpHomePath;
	
	// Add class 'loading' to modal to hide modal body content / show modal body loading div
	stackTraceModal.classList.add("loading");
	
	// Show modal
	stackTraceModal.classList.add("show");
	
	// Load data via ajax
	jQuery.ajax({
		type : 'post',
		dataType : 'json',
		url : logtastic_admin.ajax_url,
		data : { 
			action: 'logtastic_load_php_error_stack_trace',
			_ajax_nonce: logtastic_admin.load_php_error_data_nonce,
			occurrence_id: occurrenceID
		}
	}).done(function(response) {

		// Was request successful?
		if ( true == response.success ) {
		
			// Populate modal raw stacktrace (if available)
			let rawStackTraceContent = response?.data?.string;
			let rawStackTraceElement = stackTraceModal.querySelector('#stack-trace-body-content-raw');
			let rawStackTraceElementPre = stackTraceModal.querySelector('#stack-trace-body-content-raw pre');
			let rawStackTraceSwitcher = stackTraceModal.querySelector('#stack-trace-body-content-switcher-raw');
			
			// Remove active class from raw stack trace content
			rawStackTraceElement.classList.remove('active');
			
			// Remove active class from raw stack trace switcher
			rawStackTraceSwitcher.classList.remove('active');
			
			// Clear existing content
			rawStackTraceElementPre.innerHTML = '';
			
			if (rawStackTraceContent && rawStackTraceContent !== false) {
				
				// Split the content into lines
				const lines = rawStackTraceContent.split('\n');
			
				// Create a <code> for each line
				lines.forEach(line => {
					const codeLine = document.createElement('code');
					codeLine.textContent = line;
					rawStackTraceElementPre.appendChild(codeLine);
				});
			}
			
			// Populate modal visual stacktrace (if available)
			let visualStackTraceData = response?.data?.array;
			let visualStackTraceElement = stackTraceModal.querySelector('#stack-trace-body-content-visual');
			let visualStackTraceSwitcher = stackTraceModal.querySelector('#stack-trace-body-content-switcher-visual');
			
			// Remove active class from visual stack trace content
			visualStackTraceElement.classList.remove('active');
			
			// Clear existing content
			visualStackTraceElement.innerHTML = '';
			
			if (visualStackTraceData && visualStackTraceData !== false) {
				
				// Create UL Element
				const visualStackTraceUl = document.createElement('ul');
				visualStackTraceUl.className = 'stacktrace';
				
				let previousFile = null;
				
				// Add initial error to UL Element
				
				let displayFile = errorFile;
				let fileLi = document.createElement('li');
				fileLi.className = 'file';
				let fileLiHtml = '<div class="file-icon">php</div>';
				fileLiHtml += '<strong class="file">' + displayFile + '</strong>';
				fileLi.innerHTML = fileLiHtml;
				visualStackTraceUl.appendChild(fileLi);
				previousFile = errorFile;
				
				let errorLi = document.createElement('li');
				errorLi.className = 'data error';
				let errorLiHtml = '';
				errorLiHtml += '<span class="line">' + errorLine + '</span>';
				errorLiHtml += '<div class="trace">';
				errorLiHtml += '<div class="error-icon error-level-' + errorLevel + '"></div>';
				errorLiHtml += '<strong class="error-title">' + errorTitleHTML + '</strong>';
				errorLiHtml += '</div>';
				errorLi.innerHTML = errorLiHtml;
				visualStackTraceUl.appendChild(errorLi);
				
				// Loop over items in stacktrace and add to ul
				visualStackTraceData.forEach(function(log, index) {
					
					// File Li
					if ( log.file == null ) {
						let fileLi = document.createElement('li');
						fileLi.className = 'file internal';
						let fileLiHtml = '<div class="internal-function-icon"><span>{&nbsp;&nbsp;}</span></div>';
						fileLiHtml += '<strong class="file">[Internal Function]</strong>';
						fileLi.innerHTML = fileLiHtml;
						visualStackTraceUl.appendChild(fileLi);
						previousFile = null;
					} else if ( log.file != previousFile ) {
						let displayFile = '/' + log.file.substring(wpHomePath.length);
						let fileLi = document.createElement('li');
						fileLi.className = 'file';
						let fileLiHtml = '<div class="file-icon">php</div>';
						fileLiHtml += '<strong class="file">' + displayFile + '</strong>';
						fileLi.innerHTML = fileLiHtml;
						visualStackTraceUl.appendChild(fileLi);
						previousFile = log.file;
					}
					
					// Error Li
					let errorLi = document.createElement('li');
					errorLi.className = 'data';
					let errorLiHtml = '';
					if (log.line != null) {
						if ( ( index == 0 && log.file == errorFile && log.line == errorLine ) ) {
							errorLiHtml += '<span class="line"></span>';
						} else {
							errorLiHtml += '<span class="line">' + log.line + '</span>';
						}
					}
					errorLiHtml += '<div class="trace">';
					if (log.class != null) {
						errorLiHtml += '<span class="class">' + log.class + '</span>';
						if (log.type != null) {
							errorLiHtml += '<span class="type">' + log.type + '</span>';
						}
					}
					if (log.function != null) {
						errorLiHtml += '<span class="function">' + log.function + '</span>';
						if (log.args != null && log.args.length > 0) {
							errorLiHtml += '<span class="punctuation">( </span>';
							errorLiHtml += '<span class="arg-break margin-bottom-10"></span>';
							log.args.forEach(function(arg, index) {
								errorLiHtml += '<span class="arg">';
								if ( arg.type == 'object' ) {
									errorLiHtml += '<span class="arg-type">' + arg.type + '<span class="arg-type-detail">' + arg.content.__class__ + '</span></span>';
								} else if ( arg.type == 'NULL' ) {
									errorLiHtml += '<span class="arg-type null">' + arg.type + '</span>';
								} else if ( arg.type != null ) {
									errorLiHtml += '<span class="arg-type">' + arg.type + '</span>';
								}
								if ( arg.type == 'array' ) {
									errorLiHtml += '<span class="arg-content array">' + logtasticEscapeHTML( logtasticStackTraceVarDump( arg.content ) ) + '</span>';
								} else if ( arg.type == 'object' ) {
									errorLiHtml += '<span class="arg-content object">' + logtasticEscapeHTML( logtasticStackTraceVarDump( arg.content ) ) + '</span>';
								} else if ( arg.content != null ) {
									errorLiHtml += '<span class="arg-content general">' + logtasticEscapeHTML( arg.content ) + '</span>';
								} else if ( arg.type != 'NULL' ) {
									errorLiHtml += '<span class="arg-content null">&nbsp;</span>';
								}
								errorLiHtml += '</span>';
								if (index != log.args.length - 1) {
									errorLiHtml += '<span class="punctuation break-after">, </span>';
								}
								errorLiHtml += '<span class="arg-break"></span>';
							});
							errorLiHtml += '<span class="punctuation margin-top-10"> )</span>';
						} else {
							errorLiHtml += '<span class="punctuation">( </span>';
							errorLiHtml += '<span class="punctuation"> )</span>';
						}
					}
					
					errorLiHtml += '</div>';
					errorLi.innerHTML = errorLiHtml;
					visualStackTraceUl.appendChild(errorLi);
					
				});
				
				// Add UL to modal
				visualStackTraceElement.appendChild(visualStackTraceUl);
				
				// Add active class to visual stack trace content
				visualStackTraceElement.classList.add('active');
				
				// Add available and active classes to visual stack trace switcher
				visualStackTraceSwitcher.classList.add('available','active');
				
				
			} else {
				
				// Add active class to visual raw trace content
				rawStackTraceElement.classList.add('active');
				
				// Remove available class from visual stack trace switcher
				visualStackTraceSwitcher.classList.remove('available');
				
				// Add active class to raw stack trace switcher
				rawStackTraceSwitcher.classList.add('active');
			}
			
			
			// If no raw stacktrace, but there is a visual stacktrace, create a replica raw stacktrace
			if ( (!rawStackTraceContent || rawStackTraceContent == false) && (visualStackTraceData && visualStackTraceData !== false) ) {
				
				let itemCount = 0;
				
				visualStackTraceData.forEach(function(log, index) {
					
					let codeLine = document.createElement('code');
					
					let codeLineContent = '#' + index + ' ';
					
					if ( log.file == null ) {
						codeLineContent += '[Internal Function]';
					} else {
						codeLineContent += log.file;
					}
					
					if (log.line != null) {
						codeLineContent += '(' + log.line + ')';
					}
					
					codeLineContent += ': ';
					
					if (log.class != null) {
						codeLineContent += log.class;
						if (log.type != null) {
							codeLineContent += log.type;
						}
					}
					
					if (log.function != null) {
					codeLineContent += log.function;
						if (log.args != null) {
							codeLineContent += '(';
							log.args.forEach(function(arg, index) {
								if ( arg.type == 'object' ) {
									codeLineContent += 'Object(' + arg.content.__class__ + ')';
								} else if ( arg.type == 'NULL' ) {
									codeLineContent += 'NULL';
								} else if ( arg.type == 'resource' || arg.type == 'resource (closed)' ) {
									
									let resourceString = arg.content;
									
									// Use a regular expression to extract the number inside parentheses
									let numberMatch = resourceString.match(/\((\d+)\)/);
									
									// Check if a match is found and extract the number
									let extractedID = numberMatch ? numberMatch[1] : null;
									
									if ( extractedID !== null ) {
										codeLineContent += 'Resource id #' + extractedID;
									} else {
										codeLineContent += 'Resource';
									}	
									
								} else if ( arg.type == 'array' ) {
									codeLineContent += 'Array';
								} else if ( arg.type == 'string' ) {
									codeLineContent += "'" + logtasticEncodeControlChars(arg.content) + "'";
								} else {
									codeLineContent += arg.content;
								}
								if (index != log.args.length - 1) {
									codeLineContent += ', ';
								}
							});
							codeLineContent += ')';
						} else {
							codeLineContent += '()';
						}
					}
					
					
					codeLine.textContent = codeLineContent;
					rawStackTraceElementPre.appendChild(codeLine);
					
				});
				
			}

			// Hide loader
			stackTraceModal.classList.remove("loading");
			stackTraceModal.classList.add("loaded");
			
			// Check if any arg-content elements have overflow, and apply class
			visualStackTraceElement.querySelectorAll('.arg-content').forEach(function(el) {
				if (logtasticHasOverflowVertical(el)) {
					
					el.classList.add('has-overflow-vertical');
					
					// Create and append the Expand link
					const expandLinkWrapper = document.createElement('div');
					expandLinkWrapper.classList.add("expand-wrapper");
					const expandLink = document.createElement('a');
					expandLink.href = '#expand';
					expandLink.textContent = 'Expand';
					expandLink.classList.add("expand");
					expandLink.onclick = function(event) {
						event.preventDefault();
						logtasticStackTraceExpandArgs(el); 
					};
					expandLinkWrapper.appendChild(expandLink);
					el.appendChild(expandLinkWrapper);
					
				}
			});


			// Scroll stack trace content to top
			stackTraceModalBody.scrollTop = 0;
		
		} else {

			// Close stack trace modal
			stackTraceModal.classList.remove('show');
			
			// Close details modal
			document.body.classList.remove('show-details');
			document.querySelectorAll('div.details-modal-body').forEach(div => {
				div.classList.remove('active');
			});

			// Show error notification
			if (response && typeof response.data === 'string' && response.data.trim() !== '') {
				logtasticDisplayNotification( 'error', response.data );
			} else {
				logtasticDisplayNotification( 'error', __( 'Unknown error loading requested stack trace.', 'logtastic' ) );
			}
							

		}
	
	}).fail(function (response) {
		

		// Close stack trace modal
		stackTraceModal.classList.remove('show');
		
		// Close details modal
		document.body.classList.remove('show-details');
		document.querySelectorAll('div.details-modal-body').forEach(div => {
			div.classList.remove('active');
		});

		// Show error notification
		logtasticDisplayNotification( 'error', __( 'Error loading requested stack trace.', 'logtastic' ) );
		
	});
	
}


// Remove ignore flag from selected PHP error
function logtasticStopIgnoringPHPError(event) {
		
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Disable Button
	clickedElement.disabled = true;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');
	
	// Hide the error message (if previously errored)
	const prevErrorMsg = parentTr.getElementsByClassName('unignore-error');
	if ( prevErrorMsg[0] ) {
		prevErrorMsg[0].remove();
	}
	
	if ( parentTr ) {
		
		// Extract the numerical part after the hyphen (the error ID)
		const errorID = clickedElement.dataset.errorId;
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Load data via ajax
			jQuery.ajax({
				type : 'post',
				dataType : 'json',
				url : logtastic_admin.ajax_url,
				data : { 
					action: 'logtastic_unignore_php_error',
					error_id: errorID,
					_ajax_nonce: logtastic_admin.unignore_php_error_nonce
				}
			}).done(function(response) {
				
				if ( response.success == true ) {
					
					// Create success message
					let successMsg = document.createElement('span');
					successMsg.classList.add('unignore-success');
					successMsg.innerHTML = '<span class="dashicons dashicons-yes">';
					
					// Append success message
					clickedElement.after(successMsg);
					
					// Remove button
					clickedElement.remove();
					
					// Update Error Count in Paragraph Above Table
					const errorCountElement = document.getElementById('ignored-error-count');
					let currentCount = parseInt(errorCountElement.textContent, 10);
					currentCount -= 1;
					errorCountElement.textContent = currentCount;
					
					if ( currentCount > 0 ) {
					
						// Wait .5 seconds then remove row
						setTimeout(function() {
							// Remove Row
							parentTr.remove();
						}, 500); 
						
					} else {
						
						// Wait .5 seconds then remove table and update text
						setTimeout(function() {
							// Remove table
							let ignoredErrorsTable = document.getElementById('ignored-errors-table-wrapper');
							ignoredErrorsTable.remove();
							// Update text
							let ignoredErrorsDescription = document.getElementById('ignored-errors-description');
							ignoredErrorsDescription.textContent = __( 'No errors are currently being ignored. You can select errors to ignore and exclude from logging via the main PHP Error Log page.', 'logtastic' );
						}, 500); 
						
					}
					
					
				} else {
					// Create error message
					let errorMsg = document.createElement('span');
					errorMsg.classList.add('unignore-error');
					errorMsg.textContent = __( 'Error processing request', 'logtastic' );
					
					// Append Error Message
					clickedElement.after(errorMsg);
					
					// Re-enable Button
					clickedElement.disabled = false;
				}
				
				
				
				
			}).error(function(response) {
				
				// Create error message
				let errorMsg = document.createElement('span');
				errorMsg.classList.add('unignore-error');
				errorMsg.textContent = __( 'Error processing request', 'logtastic' );
				
				// Append Error Message
				clickedElement.after(errorMsg);
				
				// Re-enable Button
				clickedElement.disabled = false;
				
			});
			
		}
		
	}
	
}


/**
 * JavaScript Error Log 
 **/

let logtasticSessionInfoData = {};

function logtasticLoadJSErrorData( targetModal, occurrencesPage = 1 ) {
		
	// Get the error ID
	const errorID =	 targetModal.dataset.errorId;

	// Clear logtasticSessionInfoData
	logtasticSessionInfoData = {};
	
	// Load data via ajax
	jQuery.ajax({
		type : 'post',
		dataType : 'json',
		url : logtastic_admin.ajax_url,
		data : { 
			action: 'logtastic_load_js_error_details',
			_ajax_nonce : logtastic_admin.load_js_error_data_nonce,
			error_id: errorID,
			occurrences_page: occurrencesPage
		}
	}).done(function(response) {

		// Was request successful?
		if ( true == response.success ) {
		
			// Check if total occurrence count has changed since last loaded
			let prevOccurrenceCount = targetModal.dataset.occurrenceCount;
			let newOccurrenceCount = response?.data?.total_occurrences;
			if ( newOccurrenceCount > prevOccurrenceCount ) {
				occurrenceCountChanged = true;
			} else {
				occurrenceCountChanged = false;
			}
			
			// Get error source
			let errorSource = targetModal.dataset.errorSource;
			
			// If data is not already loaded for target modal, or the occurrence count has changed, update the error details
			if ( targetModal.dataset.loaded != 'true' || occurrenceCountChanged == true ) {
				
				// Update source version
				if ( errorSource == 'wp_core' ) {
					let sourceVersionString = response?.data?.wp_version?.range_string_full;
					if ( sourceVersionString && sourceVersionString !== false ) {
						targetModal.querySelector( '.source-version' ).textContent = sourceVersionString;
					}
				} else {
					let sourceVersionString = response?.data?.source_version?.range_string_full;
					if ( sourceVersionString && sourceVersionString !== false ) {
						targetModal.querySelector( '.source-version' ).textContent = sourceVersionString;
					}
				}
				
				// Update Wordpress Version
				let wpVersionStringShort = response.data.wp_version.range_string_short;
				if ( wpVersionStringShort && wpVersionStringShort !== false ) {
					targetModal.querySelector( '.wp-version-number' ).textContent = wpVersionStringShort;
				}
				let wpVersionStringText = response?.data?.wp_version?.range_string_version_text;
				if ( wpVersionStringText && wpVersionStringText !== false ) {
					targetModal.querySelector( '.wp-version-text' ).textContent = wpVersionStringText;
				}
				
				// Show Environment Data
				targetModal.querySelector( '.sidebar-section.environment .loading' ).classList.add('complete');
				targetModal.querySelector( '.sidebar-section.environment .loaded' ).classList.add('complete');
				
				// Get the first (earliest) date from occurrences_by_date
				const startDate = Object.keys(response.data.occurrences_by_date)[0]; // The first date in the object
				
				// Generate an array of dates from the first occurrence date until the current date
				const datesArray = logtasticGenerateDateArray(startDate);
				
				// Create a new object with dates and their corresponding occurrences (0 if missing)
				const mergedData = datesArray.reduce((acc, date) => {
					acc[date] = response.data.occurrences_by_date[date] || 0;
					return acc;
				}, {});
				
				// Prepare labels/data for Chart
				const labels = [];
				const values = [];
				
				for (const [key, value] of Object.entries(mergedData)) {
					labels.push(key);
					values.push(value);
				}
				
				// Destroy existing chart if already defined
				const existingChart = Chart.getChart( 'occurrence-chart-' + errorID );
				if (existingChart) {
					existingChart.destroy();  // Destroy the chart instance
				}
				
				// Create Chart
				const chartElement = document.getElementById( 'occurrence-chart-' + errorID );
				const chart = new Chart(chartElement, {
				type: 'bar',
				data: {
					datasets: [{
						data: values,
						maxBarThickness: 5, // Set maximum bar thickness to 2px
					}],
					labels: labels
					},
				options: {
					responsive: true,
					plugins: {
					legend: {
						display: false, // Hide the legend
					},
					},
					scales: {
						x: {
							ticks: {
							display: false, // Hide the x-axis labels
							},
							grid: {
								display: false,  // Hide gridlines for the X-axis
							},
							border: {
							display: false, // Hide the x-axis line at the bottom
							},
						},
						y: {
							ticks: {
							display: false, // Hide the y-axis labels
							},
							grid: {
								display: false,  // Hide gridlines for the Y-axis
							},
							border: {
							display: false, // Hide the y-axis line at the bottom
							}
						}
					},
					layout: {
					padding: {
						top: 0,
						left: 0,
						right: 0,
						bottom: 0, // Remove all padding around the chart
					},
					}
				}
				});
				
				// Show chart
				targetModal.querySelector( '.sidebar-section.occurrences .occurrences-chart .loading' ).classList.add('complete');
				targetModal.querySelector( '.sidebar-section.occurrences .occurrences-chart .loaded' ).classList.add('complete');
				
				// Update latest occurrence date string
				let lastOccurredString = response?.data?.last_occurrence_string;
				if ( lastOccurredString && lastOccurredString !== false ) {
					targetModal.querySelector( '.sidebar-section.occurrences .last-occurred span' ).innerHTML = lastOccurredString;
				}
				
				// Update total occurrences value
				if ( newOccurrenceCount && newOccurrenceCount !== false ) {
					targetModal.querySelector( '.sidebar-section.occurrences .total-occurrences span' ).textContent = newOccurrenceCount;
				}
				
				// Update total occurrence data attribute
				if ( newOccurrenceCount && newOccurrenceCount !== false ) {
					targetModal.dataset.occurrenceCount = newOccurrenceCount;
				}
				
			}
			
			// Populate Occurrences Table Navigation
			const tableNav = targetModal.querySelector('.occurrences-table-navigation');
			tableNav.innerHTML = response.data.occurrences_nav_html;
			
			// Populate Occurrences Table
			const tableBody = targetModal.querySelector('.occurrences-table tbody');
			while (tableBody.firstChild) {
				tableBody.removeChild(tableBody.firstChild);
			}
			
			response.data.occurrences.forEach(item => {
				const row = document.createElement("tr");
				row.setAttribute("data-occurrence-id", item.occurrence_id); // Add occurrence_id as data attribute
				
				// Add Timestamp Column
				const dataCol = document.createElement("td");
				dataCol.innerHTML = item.date_time_string;
				row.appendChild(dataCol);
				
				if ( errorSource == 'plugin' || errorSource == 'theme' ) {
				
					// Add Source Version Column
					const sourceVerCol = document.createElement("td");
					if ( errorSource == 'plugin' ) {
						sourceVerCol.setAttribute("data-colname", __( 'Plugin Version', 'logtastic' ) );
					}
					if ( errorSource == 'theme' ) {
						sourceVerCol.setAttribute("data-colname", __( 'Theme Version', 'logtastic' ) );
					}
					sourceVerCol.textContent = item.source_version;
					row.appendChild(sourceVerCol);
					
				}
				
				// Add WP Version Column
				const wpVerCol = document.createElement("td");
				wpVerCol.setAttribute("data-colname", __( 'WP Version', 'logtastic' ) );
				wpVerCol.textContent = item.wp_version;
				row.appendChild(wpVerCol);
				
				// Add user column
				const userCol = document.createElement("td");
				userCol.setAttribute("data-colname", __( 'User', 'logtastic' ) );
				userCol.innerHTML = `
					<div class="pro logtastic-admin-tooltip tooltip-small">
						Pro 
						<span class="screen-reader-text">${ __( 'Pro Feature', 'logtastic' ) }</span>
						<div class="logtastic-admin-tooltip-content">
							${ sprintf(
								__( 'This feature is only available in the <a href="%s" target="_blank">Logtastic Pro version</a>.', 'logtastic' ),
								logtastic_admin.plugin_website
							) }
						</div>
					</div>
				`;
				row.appendChild(userCol);
			
				// Add stack trace column
				const stackTraceCol = document.createElement("td");
				stackTraceCol.setAttribute("data-colname", __( 'Stack Trace', 'logtastic' ) );
				if ( item.stack_trace_available == "1" ) {
					stackTraceCol.innerHTML = '<a href="#" onclick="logtasticLoadJSStackTrace(' + item.occurrence_id + ');">View Stack Trace</a>';
				} else {
					stackTraceCol.textContent = 'n/a';
				}
				row.appendChild(stackTraceCol);

				// Add session info column
				const sessionInfoCol = document.createElement("td");
				sessionInfoCol.setAttribute("data-colname", __( 'Session Info', 'logtastic' ) );
				if ( item.additional_data?.session_info ) {
					sessionInfoCol.innerHTML = '<a href="#" onclick="logtasticLoadSessionInfo(' + item.occurrence_id + ');">View Session Info</a>';
					// Store session info in logtasticSessionInfoData object
					logtasticSessionInfoData[item.occurrence_id] = item.additional_data.session_info;
					
				} else {
					sessionInfoCol.textContent = 'n/a';
				}
				row.appendChild(sessionInfoCol);
			
				tableBody.appendChild(row);
				
				// Update loaded data attribute
				targetModal.dataset.loaded = 'true';
				
			});
			
			// Hide occurrences table loading screen
			const occurrencesTableWrapper = targetModal.querySelector('div.occurrences-table-wrapper');
			occurrencesTableWrapper.classList.remove('loading');

		} else {

			// Close details modal
			document.body.classList.remove('show-details');
			document.querySelectorAll('div.details-modal-body').forEach(div => {
				div.classList.remove('active');
			});

			// Show error notification
			if (response && typeof response.data === 'string' && response.data.trim() !== '') {
				logtasticDisplayNotification( 'error', response.data );
			} else {
				logtasticDisplayNotification( 'error', __( 'Unknown error loading requested error details.', 'logtastic' ) );
			}

		}


	}).fail(function (response) {
		
		// Close details modal
		document.body.classList.remove('show-details');
		document.querySelectorAll('div.details-modal-body').forEach(div => {
			div.classList.remove('active');
		});

		// Show error notification
		logtasticDisplayNotification( 'error', __( 'Error loading requested error details.', 'logtastic' ) );
		
	});		
	
}


// View JS Error Details Modal
function logtasticViewJSErrorDetails(event) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');		
	
	if ( parentTr ) {
		
		// Get the id of the <tr> element
		const elementID = parentTr.id;
		
		// Extract the numerical part after the hyphen (the error ID)
		const errorID = elementID.split('-')[1]; // Split the string and get the second part
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Find target details modal
			let targetModal = document.querySelector('div.details-modal-body[data-error-id="' + errorID + '"]');
			
			if (targetModal) {
				
				// Show error details modal
				document.body.classList.add('show-details');
				targetModal.classList.add('active');
				
				// Load error data via ajax
				logtasticLoadJSErrorData( targetModal );
				
			}
			
		}
		
	}
	
}


// Navigate JS Error Details Modal
function logtasticNavigateJSErrorDetails( order ) {
	
	// Find target details modal
	let targetModal = document.querySelector('div.details-modal-body[data-nav-order="' + order + '"]');
	
	if (targetModal) {
		// Remove currently active details
		document.querySelectorAll('div.details-modal-body').forEach(div => {
			div.classList.remove('active');
		});
		// Set target as active
		targetModal.classList.add('active');
		
		// Load error data via ajax
		logtasticLoadJSErrorData( targetModal );
		
	}
}

// Delete Single JS Error
function logtasticDeleteSingleJSError(event) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Define vars to hold error id and error message
	let errorID;
	let errorMsg;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');
	
	if ( parentTr ) {
		
		// Find the error message
		errorMsg = parentTr.querySelector('strong.row-title').textContent.trim();
		
		// Get the id of the <tr> element
		const elementID = parentTr.id;
		
		// Extract the numerical part after the hyphen (the error ID)
		errorID = elementID.split('-')[1]; // Split the string and get the second part
		
	} else {
		
		// Find the parent modal body
		const parentModal = clickedElement.closest('.details-modal-body');
		
		if ( parentModal ) {
			
			errorID = parentModal.dataset.errorId;
			
			errorMsg = parentModal.querySelector('.details-intro h2').textContent.trim();
			
		}
		
	}
	
	
	if ( typeof errorID !== 'undefined' && typeof errorMsg !== 'undefined' ) {
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Confirmation 
			let confirmationText = __( 'This will delete the following error and all associated data:' , 'logtastic' );
			confirmationText += '\n\n' + errorMsg + '\n\n';
			confirmationText += __( 'This operation cannot be undone.' , 'logtastic' );
			if ( confirm( confirmationText ) == true ) {
				
				// Get the form element
				const form = document.getElementById('action');
				
				if (form) {
					// Set the values of the hidden inputs
					form.querySelector('input[name="action"]').value = 'delete';
					form.querySelector('input[name="error_ids"]').value = errorID;
					
					// Submit the form
					form.submit();
				}
				
				
			}
			
		}
		
	}
	
}


// Ignore single JS error
function logtasticIgnoreSingleJSError(event) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Define vars to hold error id and error message
	let errorID;
	let errorMsg;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');		
	
	if ( parentTr ) {
		
		// Find the error message
		errorMsg = parentTr.querySelector('strong.row-title').textContent.trim();
		
		// Get the id of the <tr> element
		const elementID = parentTr.id;
		
		// Extract the numerical part after the hyphen (the error ID)
		errorID = elementID.split('-')[1]; // Split the string and get the second part
		
	} else {
		
		// Find the parent modal body
		const parentModal = clickedElement.closest('.details-modal-body');
		
		if ( parentModal ) {
			
			errorID = parentModal.dataset.errorId;
			errorMsg = parentModal.querySelector('.details-intro h2').textContent.trim();
			
		}
		
	}
	
	if ( typeof errorID !== 'undefined' && typeof errorMsg !== 'undefined' ) {
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Confirmation 
			let confirmationText = __( 'This will delete the following error and all associated data:' , 'logtastic' );
			confirmationText += '\n\n' + errorMsg + '\n\n';
			confirmationText += __( 'This operation cannot be undone.' , 'logtastic' );
			confirmationText += '\n\n';
			confirmationText += __( 'In addition, all future occurrences of this error will be ignored - you can remove this error from the ignore list at any time from the log settings page.' , 'logtastic' );
			if ( confirm( confirmationText ) == true ) {
				
				// Get the form element
				const form = document.getElementById('action');
				
				if (form) {
					// Set the values of the hidden inputs
					form.querySelector('input[name="action"]').value = 'ignore';
					form.querySelector('input[name="error_ids"]').value = errorID;
					
					// Submit the form
					form.submit();
				}
				
				
			}
			
		}
		
	}
	
}


// Load JS stack trace via ajax and display in modal for a given error occurrence
function logtasticLoadJSStackTrace(occurrenceID) {

	// Prevent the default link action
	event.preventDefault();
	
	// Get the parent error modal and retrieve details
	const errorModal = event.target.closest('.details-modal-body');	 
	const errorID = errorModal.dataset.errorId;
	const errorLevel = errorModal.dataset.errorLevel;
	const errorTitleHTML = errorModal.querySelector('.details-modal-body-content .details-intro h2').innerHTML;
	
	// Get stack trace modal element
	let stackTraceModal = document.querySelector('div.stack-trace-modal');

	// Get stack trace modal body element
	let stackTraceModalBody = stackTraceModal.querySelector('div.stack-trace-body-content');
	
	// Get WP Home Path from Stack Trace Modal Data Attribute
	const wpHomePath = stackTraceModal.dataset.wpHomePath;
	
	// Add class 'loading' to modal to hide modal body content / show modal body loading div
	stackTraceModal.classList.add("loading");
	
	// Show modal
	stackTraceModal.classList.add("show");
	
	// Load data via ajax
	jQuery.ajax({
		type : 'post',
		dataType : 'json',
		url : logtastic_admin.ajax_url,
		data : { 
			action: 'logtastic_load_js_error_stack_trace',
			_ajax_nonce : logtastic_admin.load_js_error_data_nonce,
			occurrence_id: occurrenceID
		}
	}).done(function(response) {

		// Was request successful?
		if ( true == response.success ) {
		
			// Populate modal raw stacktrace (if available)
			let rawStackTraceContent = response?.data?.string;
			let rawStackTraceElement = stackTraceModal.querySelector('#stack-trace-body-content-raw');
			let rawStackTraceElementPre = stackTraceModal.querySelector('#stack-trace-body-content-raw pre');
			let rawStackTraceSwitcher = stackTraceModal.querySelector('#stack-trace-body-content-switcher-raw');
			
			// Remove active class from raw stack trace content
			rawStackTraceElement.classList.remove('active');
			
			// Remove active class from raw stack trace switcher
			rawStackTraceSwitcher.classList.remove('active');
			
			// Clear existing content
			rawStackTraceElementPre.innerHTML = '';
			
			if (rawStackTraceContent && rawStackTraceContent !== false) {
				
				// Split the content into lines
				const lines = rawStackTraceContent.split('\n');
			
				// Create a <code> for each line
				lines.forEach(line => {
					const codeLine = document.createElement('code');
					codeLine.textContent = line;
					rawStackTraceElementPre.appendChild(codeLine);
				});
			}
				
			// Add active class to visual raw trace content
			rawStackTraceElement.classList.add('active');
			
			// Add active class to raw stack trace switcher
			rawStackTraceSwitcher.classList.add('active');
			
			// Hide loader
			stackTraceModal.classList.remove("loading");
			stackTraceModal.classList.add("loaded");

			// Scroll stack trace content to top
			stackTraceModalBody.scrollTop = 0;

		} else {

			// Close stack trace modal
			stackTraceModal.classList.remove('show');
			
			// Close details modal
			document.body.classList.remove('show-details');
			document.querySelectorAll('div.details-modal-body').forEach(div => {
				div.classList.remove('active');
			});

			// Show error notification
			if (response && typeof response.data === 'string' && response.data.trim() !== '') {
				logtasticDisplayNotification( 'error', response.data );
			} else {
				logtasticDisplayNotification( 'error', __( 'Unknown error loading requested stack trace.', 'logtastic' ) );
			}
							

		}
	
	}).fail(function (response) {
		
		// Close stack trace modal
		stackTraceModal.classList.remove('show');
		
		// Close details modal
		document.body.classList.remove('show-details');
		document.querySelectorAll('div.details-modal-body').forEach(div => {
			div.classList.remove('active');
		});

		// Show error notification
		logtasticDisplayNotification( 'error', __( 'Error loading requested stack trace.', 'logtastic' ) );
		
	});
	
	
}


// Display JS session info in a modal for a given error occurrence
function logtasticLoadSessionInfo(occurrenceID) {

	// Prevent the default link action
	event.preventDefault();
	
	// Get session info modal element
	let sessionInfoModal = document.querySelector('div.session-info-modal');
	
	// Add class 'loading' to modal to hide modal body content / show modal body loading div
	sessionInfoModal.classList.add("loading");
	
	// Show modal
	sessionInfoModal.classList.add("show");
	
	// Get session info data from logtasticSessionInfoData object
	sessionInfo = logtasticSessionInfoData[occurrenceID];

	// Update modal content
	let urlSpan = sessionInfoModal.querySelector('#session-info-url');
	if (urlSpan) {
		urlSpan.textContent = sessionInfo.url;
	}
	let userAgentSpan = sessionInfoModal.querySelector('#session-info-user-agent');
	if (userAgentSpan) {
		userAgentSpan.textContent = sessionInfo.user_agent;
	}

	// Hide loader
	sessionInfoModal.classList.remove("loading");
	sessionInfoModal.classList.add("loaded");
	
}


// Close session info modal
function logtasticCloseSessionInfo() {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Find the parent additional details modal
	const sessionInfoModal = clickedElement.closest('div.session-info-modal');	
	
	// Remove class 'show' from target modal
	sessionInfoModal.classList.remove('show');
	
}


// Remove ignore flag from selected JS error
function logtasticStopIgnoringJSError(event) {
	
	// Prevent the default link action
	event.preventDefault();
	
	// Get the clicked element
	const clickedElement = event.target;
	
	// Disable Button
	clickedElement.disabled = true;
	
	// Find the parent <tr> element
	const parentTr = clickedElement.closest('tr');
	
	// Hide the error message (if previously errored)
	const prevErrorMsg = parentTr.getElementsByClassName('unignore-error');
	if ( prevErrorMsg[0] ) {
		prevErrorMsg[0].remove();
	}
	
	if ( parentTr ) {
		
		// Extract the numerical part after the hyphen (the error ID)
		const errorID = clickedElement.dataset.errorId;
		
		// Validate that errorID exists and is numerical
		if (errorID && !isNaN(errorID)) {
			
			// Load data via ajax
			jQuery.ajax({
				type : 'post',
				dataType : 'json',
				url : logtastic_admin.ajax_url,
				data : { 
					action: 'logtastic_unignore_js_error',
					error_id: errorID,
					_ajax_nonce: logtastic_admin.unignore_js_error_nonce
				}
			}).done(function(response) {
				
				if ( response.success == true ) {
					
					// Create success message
					let successMsg = document.createElement('span');
					successMsg.classList.add('unignore-success');
					successMsg.innerHTML = '<span class="dashicons dashicons-yes">';
					
					// Append success message
					clickedElement.after(successMsg);
					
					// Remove button
					clickedElement.remove();
					
					// Update Error Count in Paragraph Above Table
					const errorCountElement = document.getElementById('ignored-error-count');
					let currentCount = parseInt(errorCountElement.textContent, 10);
					currentCount -= 1;
					errorCountElement.textContent = currentCount;
					
					if ( currentCount > 0 ) {
					
						// Wait .5 seconds then remove row
						setTimeout(function() {
							// Remove Row
							parentTr.remove();
						}, 500); 
						
					} else {
						
						// Wait .5 seconds then remove table and update text
						setTimeout(function() {
							// Remove table
							let ignoredErrorsTable = document.getElementById('ignored-errors-table-wrapper');
							ignoredErrorsTable.remove();
							// Update text
							let ignoredErrorsDescription = document.getElementById('ignored-errors-description');
							ignoredErrorsDescription.textContent = __( 'No errors are currently being ignored. You can select errors to ignore and exclude from logging via the main PHP Error Log page.', 'logtastic' );
						}, 500); 
						
					}
					
					
				} else {
					// Create error message
					let errorMsg = document.createElement('span');
					errorMsg.classList.add('unignore-error');
					errorMsg.textContent = __( 'Error processing request', 'logtastic' );
					
					// Append Error Message
					clickedElement.after(errorMsg);
					
					// Re-enable Button
					clickedElement.disabled = false;
				}
				
				
				
				
			}).error(function(response) {
				
				// Create error message
				let errorMsg = document.createElement('span');
				errorMsg.classList.add('unignore-error');
				errorMsg.textContent = __( 'Error processing request', 'logtastic' );
				
				// Append Error Message
				clickedElement.after(errorMsg);
				
				// Re-enable Button
				clickedElement.disabled = false;
				
			});
			
			
			
		}
		
	}
	
}