<?php
/**
 * The admin-specific functionality of the JavaSript error log.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @subpackage        Logtastic/admin
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Base class for displaying logged JS errors in WP Admin, including searchable/filterable tabular view of errors and detailed view of each error.
 *
 * @since 1.0.0
 */
class Logtastic_JS_Error_Log_Admin {

	/**
	 * The JS error table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_TABLE_NAME = 'logtastic_js_errors';

	/**
	 * The JS error occurrences table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_OCCURRENCES_TABLE_NAME = 'logtastic_js_error_occurrences';

	/**
	 * The JS error stack traces table name
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const ERROR_STACK_TRACES_TABLE_NAME = 'logtastic_js_error_stack_traces';
	
	/**
	 * The current list of errors.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	protected $errors;
	
	/**
	 * The total number of errors, based on the current query.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	protected $total_error_count;
	
	/**
	 * The number of errors to display per page.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	protected $results_per_page;
	
	/**
	 * The current page.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	protected $current_page;
	
	/**
	 * The maximum number of pages based on the current query.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	protected $max_page;
	
	/**
	 * The number of rows to offset the database query by, based on $current_page and $results_per_page.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	protected $offset;
	
	/**
	 * The error tyoe filter for the current query.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	protected $error_type_filter;
	
	/**
	 * The source filter for the current query.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	protected $source_filter;
	
	/**
	 * The source slug filter for the current query.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	protected $source_slug_filter;
	
	/**
	 * The search term for the current query.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	protected $search_term;
	
	/**
	 * The column to sort by for the current query.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $sort_by;
	
	/**
	 * The label for the current sort option.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $sort_by_label;
	
	/**
	 * The sort order for the current query (ASC or DESC).
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $sort_order;

	/**
	 * The label for the current sort order (AscendingSC or Descending).
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $sort_order_label;
	
	/**
	 * User friendly error type labels.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public static $error_type_labels;
	
	/**
	 * The list of unique error sources found in the database (all errors).
	 *
	 * @since 1.0.0
	 * @var array
	 */
	protected $db_error_sources;
	
	/**
	 * The list of unique error types found in the database (all errors).
	 *
	 * @since 1.0.0
	 * @var array
	 */
	protected $db_error_types;
	
	/**
	 * The list of all installed plugins.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $all_plugins;
	
	/**
	 * The list of all installed themes.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	public $all_themes;
	
	/**
	 * The absolute filesystem path to the root of the WordPress installation
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $wp_home_path;
	
	/**
	 * The number of occurrences to display per page.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	protected static $occurrences_per_page = 200;
	
	
	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		
		// Set home path variable
		$this->wp_home_path = get_home_path();

		// Get All Plugins
		$this->all_plugins = get_plugins();		
		
		// Get All Themes
		$this->all_themes = wp_get_themes();

		// Define error type labels
		self::define_error_type_labels();
		
		// Fire function to get all unique error sources from DB and populate var
		$this->get_error_sources_from_db();
		
		// Fire function to get all unique error levels from DB and populate var
		$this->get_error_types_from_db();
		
		// Process actions, if applicable
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce checked in process_actions() function
		if ( isset( $_POST['action'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce checked in process_actions() function
			$action = sanitize_text_field( wp_unslash( $_POST['action'] ) );
			$this->process_actions( $action );
		}
		
		// Set $results_per_page var
		$this->results_per_page = 20;
		
		// Fire function to get any filter variables from $_GET and populate vars
		$this->get_filter_vars();
		
		// Fire function to get error count from DB based on current query, and set max page var
		$this->get_error_count_from_db();
		
		// If requested current page is greater than the max number of pages available, set the current page to the max page available 
		if ( $this->current_page > 1 && $this->current_page > $this->max_page ) {
			$this->current_page = $this->max_page;
		}

		// Calculate $offset based on $current_page and $reults_per_page
		$this->calculate_offset();
		
		// Fire function to get errors from DB based on current query and populate $errors
		$this->get_errors_from_db();
		
	}


	/**
	 * Define the error type labels
	 *
	 * @since 1.0.0
	 */
	private static function define_error_type_labels() {
		self::$error_type_labels = [
			1 => __( 'Runtime Error', 'logtastic' ),
			2 => __( 'Unhandled Promise Rejection', 'logtastic' ),
		];
	}
	
	
	/**
	 * Sets all necessary filter vars for the DB query based on $_GET
	 *
	 * @since 1.0.0
	 */
	private function get_filter_vars() {
		
		// Set value for $current_page 
		$this->current_page = max( 1, ( int ) filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT ) );
		
		// Set value for $error_type_filter - will store error type filter - all or numeric key of error type (eg. 1, 2).
		$this->error_type_filter = 'all';
		$error_type_filter_sanitized = filter_input(INPUT_GET, 'error_type_filter', FILTER_VALIDATE_INT);
		if ( !empty( $error_type_filter_sanitized ) && array_key_exists( $error_type_filter_sanitized, self::$error_type_labels ) ) {
    		$this->error_type_filter = $error_type_filter_sanitized;
		}
		
		// Set value for $source_filter - will store source filter - all, plugin, theme
		// Set value for $source_slug_filter - will source either a specific plugin or theme slug or null
		$this->source_filter = 'all';
		$this->source_slug_filter = null;
		$source_filter_sanitized = sanitize_key( filter_input(INPUT_GET, 'source_filter') );
		if ( !empty( $source_filter_sanitized ) ) {
			if ( $source_filter_sanitized === 'all-plugins' && array_key_exists('plugin', $this->db_error_sources ) ) {
        		$this->source_filter = 'plugin';
			} elseif ( str_starts_with( $source_filter_sanitized, 'plugin_' ) && array_key_exists('plugin', $this->db_error_sources ) ) {
				$this->source_filter = 'plugin';
				$plugin_slug = substr( $source_filter_sanitized, 7 );
				if ( in_array( $plugin_slug, $this->db_error_sources['plugin'] ) ) {
					$this->source_slug_filter = $plugin_slug;
				}
			} else if ( $source_filter_sanitized === 'all-themes' && array_key_exists('theme', $this->db_error_sources ) ) {
				$this->source_filter = 'theme';
			} elseif ( str_starts_with( $source_filter_sanitized, 'theme_' ) && array_key_exists('theme', $this->db_error_sources ) ) {
				$this->source_filter = 'theme';
				$theme_slug = substr( $source_filter_sanitized, 6 );
				if ( in_array( $theme_slug, $this->db_error_sources['theme'] ) ) {
					$this->source_slug_filter = $theme_slug;
				}
			} else if ( $source_filter_sanitized === 'wp_core' && array_key_exists('wp_core', $this->db_error_sources ) ) {
				$this->source_filter = 'wp_core';
			}
		}
		
		// $sort_by - will be occurrence_count, first_occurred or last_occurred
		$this->sort_by = 'last_occurred';
		$allowed_sort_values = ['occurrence_count', 'first_occurred', 'last_occurred'];
		$sort_by_sanitized = sanitize_key( filter_input( INPUT_GET, 'sort_by' ) );
		if ( !empty( $sort_by_sanitized ) && in_array( $sort_by_sanitized, $allowed_sort_values ) ) {
    		$this->sort_by = $sort_by_sanitized;
		}

		// $sort_by_label
		if ( 'occurrence_count' == $this->sort_by ) {
			$this->sort_by_label = __( 'Occurrence Count', 'logtastic' );
		} else if ( 'first_occurred' == $this->sort_by ) {
			$this->sort_by_label = __( 'First Occurred', 'logtastic' );
		} else if ( 'last_occurred' == $this->sort_by ) {
			$this->sort_by_label = __( 'Last Occurred', 'logtastic' );
		}
		
		// $sort_order (will be ASC or DESC)
		$this->sort_order = 'DESC';
		$sort_order_sanitized = sanitize_key( filter_input( INPUT_GET, 'sort_order' ) );
		if ( 'asc' == $sort_order_sanitized ) {
			$this->sort_order = 'ASC';
		}

		// $sort_order_label
		if ( 'DESC' == $this->sort_order ) {
			$this->sort_order_label = __( 'Descending', 'logtastic' );
		} else if ( 'ASC' == $this->sort_order ) {
			$this->sort_order_label = __( 'Ascending', 'logtastic' );
		}
		
		// $search_term - will be string if present or null
		$this->search_term = null;
		$search_term_sanitized = sanitize_text_field( filter_input( INPUT_GET, 'search_term' ) );
		if ( !empty( $search_term_sanitized ) ) {
			$this->search_term = $search_term_sanitized;
		}
		
	}


	/**
	 * Calculate offset for databae query based on $current_page and $reults_per_page and update $offset
	 *
	 * @since 1.0.0
	 *
	 */
	private function calculate_offset() {
		// $offset = ( $this->current_page - 1 ) *  $this->results_per_page
		$this->offset = ( $this->current_page - 1 ) * $this->results_per_page;
	}
	
	
	/**
	 * Get errors from DB based on filter vars
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 */
	private function get_errors_from_db() {
		
		global $wpdb;
				
		// Begin SQL query definition
		$query = "
			SELECT 
				e.error_id,
				e.error_type,
				e.file,
				e.line,
				e.col,
				e.source,
				e.source_slug,
				e.message,
				e.ignore_error,
				COUNT(o.occurrence_id) AS occurrence_count,
				MIN(o.timestamp) AS first_occurred,
				MAX(o.timestamp) AS last_occurred
			FROM %i e
			LEFT JOIN %i o
				ON e.error_id = o.error_id
			WHERE e.ignore_error = 0
		";
		
		/* Add additional where statements to query, if required */
		
		// Error Level Filter Where Statement
		if ( 'all' !== $this->error_type_filter ) {
			$query .= "
				AND e.error_type = %d
			";
		}
		
		// Source Filter Where Statement
		if ( 'all' !== $this->source_filter ) {
			$query .= "
				AND e.source = %s
			";
		}
		
		// Source Slug WHERE statement
		if ( null !== $this->source_slug_filter ) {
			$query .= "
				AND e.source_slug = %s
			";
		}
		
		// Search Term WHERE statement
		if ( null !== $this->search_term ) {
			$query .= "
				AND e.message LIKE %s
			";
		}
		
		// Complete SQL query
		// phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- $this->sort_order whitelisted on line 352
		$query .= "
			GROUP BY e.error_id
			HAVING occurrence_count > 0
			ORDER BY %i {$this->sort_order}
			LIMIT %d OFFSET %d
		";
		
		// Define query variables array and include table names
		$query_vars = [
			$wpdb->prefix . self::ERROR_TABLE_NAME,
			$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME
		];
		
		// If error level filter not equal 'all', add error level to query vars
		if ( 'all' !== $this->error_type_filter ) {
			$query_vars[] = $this->error_type_filter;
		}
		
		// If source filter not equal 'all', add source
		if ( 'all' !== $this->source_filter ) {
			$query_vars[] = $this->source_filter;
		}
		
		// If source slug filter not equal null, add slug
		if ( null !== $this->source_slug_filter ) {
			$query_vars[] = $this->source_slug_filter;
		}
		
		// If search term not equal null, prepare like term and add
		if ( null !== $this->search_term ) {
			$query_vars[] = '%' . $wpdb->esc_like( $this->search_term ) . '%';
		}
		
		// Add remaining query vars ( Sort By, Sort Order and Offset )
		$query_vars[] = $this->sort_by;
		$query_vars[] = $this->results_per_page;
		$query_vars[] = $this->offset;

		// Prepare the query and get results
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- $query defined dynamically above with placeholders, $query_vars array dynamically poplulated above. Direct query as custom table. Caching not appropriate as fresh data always required.
		$errors = $wpdb->get_results( $wpdb->prepare($query, $query_vars), ARRAY_A);
		
		// Return results, or false if none found
		if (!empty($errors)) {
			$this->errors = $errors;
		} else {
			$this->errors = false;
		}
		
	}
	
	
	/**
	 * Get a single error from DB based on error id
	 *
	 * @since 1.0.0
	 *
	 * @param    string		$error_id 	The id of the error to fetch from the database
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	private function get_error_from_db( $error_id ) {
		
		global $wpdb;

		// Prepare the query and get error data
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$error = $wpdb->get_results(
			$wpdb->prepare(
				"
					SELECT 
						e.error_id,
						e.error_type,
						e.file,
						e.line,
						e.col,
						e.source,
						e.source_slug,
						e.message,
						e.ignore_error,
						COUNT(o.occurrence_id) AS occurrence_count,
						MIN(o.timestamp) AS first_occurred,
						MAX(o.timestamp) AS last_occurred
					FROM %i e
					LEFT JOIN %i o 
						ON e.error_id = o.error_id
					WHERE e.error_id = %d
				", 
				$wpdb->prefix . self::ERROR_TABLE_NAME,
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id
			), 
			ARRAY_A
		);
		
		// Return result, or false if none found
		if (!empty($error)) {
			return $error;
		} else {
			return false;
		}
		
	}
	
	
	/**
	 * Get ignored errors from DB 
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb WordPress database abstraction object.
	 */
	private static function get_ignored_errors_from_db() {
		
		global $wpdb;

		// Get ingnored errors from db
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$errors = $wpdb->get_results(
			$wpdb->prepare(
				"
					SELECT 
						error_id,
						error_type,
						file,
						line,
						col,
						source,
						source_slug,
						message,
						ignore_error
					FROM %i
					WHERE ignore_error = %d
				",
				$wpdb->prefix . self::ERROR_TABLE_NAME,
				1
			),
			ARRAY_A
		);
		
		// Return results, or false if none found
		if (!empty($errors)) {
			return $errors;
		} else {
			return false;
		}
		
	}
	
	
	/**
	 * Process any actions
	 *
	 * @since 1.0.0
	 *
	 * @param string 	$action 	The action sent via $_POST, either 'delete' or 'ignore'
	 */
	private function process_actions( $action ) {
		
		// Check for valid nonce
		check_admin_referer( 'error-log-action' );
		
		// Check for error ids
		if ( isset ( $_POST['error_ids'] ) && null != $_POST['error_ids'] ) {

			// Explode into array
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Santized in next step
			$error_ids_raw = explode( ',', wp_unslash( $_POST['error_ids'] ) );
			
			// Sanitize and filter to keep only valid integers
			$error_ids_sanitized = array_filter( array_map( 'intval', $error_ids_raw ), function( $id ) {
				return $id > 0;
			});
			
		}
		
		// Process each error id
		$success_count = 0;
		$fail_count = 0;
		if ( isset( $error_ids_sanitized ) && is_array( $error_ids_sanitized ) ) {
			if ( 'delete' == $action ) {
				foreach ( $error_ids_sanitized as $error_id ) {
					if ( is_numeric( $error_id ) ) {
						if ( true == $this->delete_error_and_occurrences_and_stack_traces( $error_id ) ) {
							$success_count++;
						} else {
							$fail_count++;
						}
					} else {
						$fail_count++;
					}
				}
				if ( $success_count > 0 ) {
					// Display success message
					$message_content = sprintf(
						/* translators: %d: number of errors successfully deleted */
						_n(
							'%d error successfully deleted.',
							'%d errors successfully deleted.',
							$success_count,
							'logtastic'
						),
						$success_count
					);
					Logtastic_Admin::admin_page_notice( 'success', $message_content, true, true );
				}
				if ( $fail_count > 0 ) {
					// Display fail message
					$message_content = sprintf(
						/* translators: %d: number of errors that failed to delete */
						_n(
							'Failed to delete %d error.',
							'Failed to delete %d errors.',
							$fail_count,
							'logtastic'
						),
						$fail_count
					);
					Logtastic_Admin::admin_page_notice( 'error', $message_content, true, true );
				}
			} else if ( 'ignore' == $action ) {
				foreach ( $error_ids_sanitized as $error_id ) {
					if ( is_numeric( $error_id ) ) {
						if ( true == $this->ignore_error( $error_id ) ) {
							$success_count++;
						} else {
							$fail_count++;
						}
					} else {
						$fail_count++;
					}
				}
				if ( $success_count > 0 ) {
					// Display success message
					$message_content = sprintf(
						/* translators: %d: number of errors successfully deleted and ignored */
						_n(
							'%d error successfully deleted and ignored.',
							'%d errors successfully deleted and ignored.',
							$success_count,
							'logtastic'
						),
						$success_count
					);
					Logtastic_Admin::admin_page_notice( 'success', $message_content, true, true );
				}
				if ( $fail_count > 0 ) {
					// Display fail message
					$message_content = sprintf(
						/* translators: %d: number of errors that failed to delete and ignore */
						_n(
							'Failed to delete and ignore %d error.',
							'Failed to delete and ignore %d errors.',
							$fail_count,
							'logtastic'
						),
						$fail_count
					);
					Logtastic_Admin::admin_page_notice( 'error', $message_content, true, true );
				}
			}
		}
	}
	
	
	/**
	 * Delete a single error from the database
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 *
	 * @param int 	$error_id 	The ID of the error in the database
	 */
	private function delete_error( $error_id ) {
		global $wpdb;
		
		// Sanitize the error_id to prevent SQL injection
		$error_id = intval($error_id);
		
		// Delete the error
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
		$deleted_error = $wpdb->delete($wpdb->prefix . self::ERROR_TABLE_NAME, ['error_id' => $error_id], ['%d']);
		
		// Return success or failure
		if ($deleted_error) {
			return true;
		} else {
			return false;
		}
	}
	
	
	/**
	 * Delete a single error and it's occurrences from the databases
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 *
	 * @param int 	$error_id 	The ID of the error in the database
	 */	
	private function delete_error_and_occurrences_and_stack_traces( $error_id ) {
		global $wpdb;
		
		// Sanitize the error_id to prevent SQL injection
		$error_id = intval($error_id);
		
		// Check for occurrences
		$error_occurrence_count = $this->count_error_occurrences_by_error_id( $error_id );
		
		// Delete the error occurrences
		if ( $error_occurrence_count > 0 ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
			$deleted_error_occurrences = $wpdb->delete($wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME, ['error_id' => $error_id], ['%d']);
		} else {
			$deleted_error_occurrences = true;
		}
		
		// Check for stack traces
		$stack_trace_count = $this->count_error_stack_traces_by_error_id( $error_id );
		
		// Delete the stack traces
		if ( $stack_trace_count > 0 ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
			$deleted_error_stack_traces = $wpdb->delete($wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME, ['error_id' => $error_id], ['%d']);
		} else {
			$deleted_error_stack_traces = true;
		}
		
		// Delete the error (if occurrences and stack traces successfully deleted)
		if ( $deleted_error_occurrences && $deleted_error_stack_traces ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
			$deleted_error = $wpdb->delete($wpdb->prefix . self::ERROR_TABLE_NAME, ['error_id' => $error_id], ['%d']);
		} else {
			$deleted_error = false;
		}
		
		// Return success or failure
		if ($deleted_error && $deleted_error_occurrences && $deleted_error_stack_traces) {
			return true;
		} else {
			return false;
		}
		
	}
	
	
	/**
	 * Mark a single error as ignored in the database, and delete its occurrences
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 *
	 * @param int 	$error_id 	The ID of the error in the database
	 */	
	private function ignore_error( $error_id ) {
		global $wpdb;
		
		// Sanitize the error_id to prevent SQL injection
		$error_id = intval($error_id);
		
		// Check for occurrences
		$error_occurrence_count = $this->count_error_occurrences_by_error_id( $error_id );
		
		// Delete the error occurrences
		if ( $error_occurrence_count > 0 ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
			$deleted_error_occurrences = $wpdb->delete($wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME, ['error_id' => $error_id], ['%d']);
		} else {
			$deleted_error_occurrences = true;
		}
		
		// Check for stack traces
		$stack_trace_count = $this->count_error_stack_traces_by_error_id( $error_id );
		
		// Delete the stack traces
		if ( $stack_trace_count > 0 ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
			$deleted_error_stack_traces = $wpdb->delete($wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME, ['error_id' => $error_id], ['%d']);
		} else {
			$deleted_error_stack_traces = true;
		}
		
		// Mark the error as ignored (if occurrences and stack traces successfully deleted)
		if ( $deleted_error_occurrences && $deleted_error_stack_traces ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
			$ignored_error = $wpdb->update(
				$wpdb->prefix . self::ERROR_TABLE_NAME,
				['ignore_error' => 1],
				['error_id' => $error_id],
				['%d'],
				['%d']
			);
		} else {
			$ignored_error = false;
		}
		
		// Return success or failure
		if ($ignored_error && $deleted_error_occurrences && $deleted_error_stack_traces) {
			return true;
		} else {
			return false;
		}
		
	}
	
	/**
	 * Count the number of errors in the database that match the current filter variables
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 */	
	private function get_error_count_from_db() {
		
		global $wpdb;
		
		// Begin SQL query definition
		$query = "
			SELECT COUNT(*) AS total_results FROM (
				SELECT 
					e.error_id
				FROM %i e
				LEFT JOIN %i o 
					ON e.error_id = o.error_id
				WHERE e.ignore_error = 0
		";
		
		/* Add additional where statements to query, if required */
		
		// Error Level Filter Where Statement
		if ( 'all' !== $this->error_type_filter ) {
			$query .= "
				AND e.error_type = %d
			";
		}
		
		// Source Filter Where Statement
		if ( 'all' !== $this->source_filter ) {
			$query .= "
				AND e.source = %s
			";
		}
		
		// Source Slug WHERE statement
		if ( null !== $this->source_slug_filter ) {
			$query .= "
				AND e.source_slug = %s
			";
		}
		
		// Search Term WHERE statement
		if ( null !== $this->search_term ) {
			$query .= "
				AND e.message LIKE %s
			";
		}
		
		// Complete SQL query
		$query .= "
				GROUP BY e.error_id
				HAVING COUNT(o.occurrence_id) > 0
			) AS subquery;
		";
		
		// Define query variables array and include table names
		$query_vars = [
			$wpdb->prefix . self::ERROR_TABLE_NAME,
			$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME
		];
		
		// If error level filter not equal 'all', add error level to query vars*
		if ( 'all' !== $this->error_type_filter ) {
			$query_vars[] = $this->error_type_filter;
		}
		
		// If source filter not equal 'all', add source
		if ( 'all' !== $this->source_filter ) {
			$query_vars[] = $this->source_filter;
		}
		
		// If source slug filter not equal null, add slug
		if ( null !== $this->source_slug_filter ) {
			$query_vars[] = $this->source_slug_filter;
		}
		
		// If search term not equal null, prepare like term and add
		if ( null !== $this->search_term ) {
			$query_vars[] = '%' . $wpdb->esc_like( $this->search_term ) . '%';
		}
		
		// Prepare query and get reesults 
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- $query defined dynamically above with placeholders, $query_vars array dynamically poplulated above. $query_vars contains only sanitised/whitelisted values (incl. LIKE term via $wpdb->esc_like()). Direct query as custom table. Caching not appropriate as fresh data always required.
		$count_errors = $wpdb->get_results( $wpdb->prepare( $query, $query_vars ), ARRAY_A );
		
		if (!empty($count_errors)) {
			$this->total_error_count = $count_errors[0]['total_results'];
			$this->max_page = ceil( $this->total_error_count / $this->results_per_page);
		} else {
			$this->total_error_count = 0;
			$this->max_page = 1;
		}
		
	}
	
	/**
	 * Count the number of error occurrences in the database that match the passed error ID.
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 *
	 * @param int 	$error_id 	The ID of the error in the database
	 * @return int
	 */	
	private function count_error_occurrences_by_error_id( $error_id ) {
		
		global $wpdb;
		
		// Sanitize the error_id to prevent SQL injection
		$error_id = intval($error_id);
		
		// Query to count occurrences for the given error_id
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$count = $wpdb->get_var(
			$wpdb->prepare(
				"
					SELECT COUNT(*) 
					FROM %i 
					WHERE error_id = %d
				",
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id
			)
		);
		
		return intval($count);
		
	}
	
	
	/**
	 * Count the number of stack traces in the database that match the passed error ID.
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 *
	 * @param int 	$error_id 	The ID of the error in the database
	 * @return int
	 */	
	private function count_error_stack_traces_by_error_id( $error_id ) {
		
		global $wpdb;
		
		// Sanitize the error_id to prevent SQL injection
		$error_id = intval($error_id);
		
		// Query to count occurrences for the given error_id
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$count = $wpdb->get_var(
			$wpdb->prepare(
				"
					SELECT COUNT(*) 
					FROM %i 
					WHERE error_id = %d
				",
				$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
				$error_id
			)
		);
		
		return intval($count);
		
	}


	/**
	 * Gets a list of distinct sources and source slugs from the errors database and updates $this->db_error_sources
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 */
	private function get_error_sources_from_db() {
		
		global $wpdb;
		
		// Get error source from DB
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$sources = $wpdb->get_results(
			$wpdb->prepare(
				"
					SELECT 
						e.source,
						GROUP_CONCAT(DISTINCT e.source_slug ORDER BY e.source_slug ASC) AS source_slugs
					FROM %i e
					LEFT JOIN %i o 
						ON e.error_id = o.error_id
					WHERE e.ignore_error = 0
					GROUP BY e.source
					HAVING COUNT(o.occurrence_id) > 0
					ORDER BY e.source ASC;
				",
				$wpdb->prefix . self::ERROR_TABLE_NAME,
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME
			),
			ARRAY_A
		);
		
		// Update $this->error_sources with results
		if ( !empty( $sources ) ) {
			foreach ( $sources as $source ) {
				if ( !empty ( $source['source_slugs'] ) ) {
					$slugs = explode( ',', $source['source_slugs'] );
				} else {
					$slugs = null;
				}
				$this->db_error_sources[$source['source']] = $slugs;
			}
		}
		
	}
	

	/**
	 * Gets a list of distinct error types from the errors database and updates $this->db_error_types
	 *
	 * @since 1.0.0
	 *
	 * @global wpdb $wpdb 		WordPress database abstraction object.
	 */
	private function get_error_types_from_db() {
		
		global $wpdb;

		// Get error levels from db
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$error_types = $wpdb->get_results(
			$wpdb->prepare(
				"
					SELECT 
						e.error_type
					FROM %i e
					LEFT JOIN %i o 
						ON e.error_id = o.error_id
					WHERE e.ignore_error = 0
					GROUP BY e.error_type
					HAVING COUNT(o.occurrence_id) > 0;
				",
				$wpdb->prefix . self::ERROR_TABLE_NAME,
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME
			),
			ARRAY_A
		);
		
		// Update $this->error_sources with results
		if (!empty($error_types) ) {
			foreach ( $error_types as $error_type ) {
				$this->db_error_types[$error_type['error_type']] = true;
			}
			ksort($this->db_error_types);
		}
		
	}

	
	/**
	 * Returns the array of errors stored in $this->errors
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_errors() {
		return $this->errors;
	}
	

	/**
	 * Returns a specific error by error ID
	 *
	 * @since 1.0.0
	 * 
	 * @param int 	$error_id 	The ID of the error in the database
	 * @return array
	 */
	public function get_error( $error_id ) {
		return $this->get_error_from_db( $error_id );
	}
	

	/**
	 * Returns an array of errors marked as ignored
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public static function get_ignored_errors() {
		return self::get_ignored_errors_from_db();
	}
	

	/**
	 * Returns the total number of errors, based on the current query.
	 *
	 * @since 1.0.0
	 *
	 * @return int
	 */
	public function get_error_count() {
		return $this->total_error_count;
	}
	

	/**
	 * Returns the corresponding error type label for the given error type
	 *
	 * @since 1.0.0
	 * 
	 * @param int 	$error_type 	The error type for which to return the label
	 * @return string
	 */
	public static function get_error_label($error_type) {
		if ( empty( self::$error_type_labels ) ) {
			// Define error type labels
			self::define_error_type_labels();
		}
		return self::$error_type_labels[$error_type];
	}
	

	/**
	 * Build the URL for the "Occurrence Count" column header with current filters applied
	 *
	 * @return string Admin URL with query args.
	 */
	public function get_count_column_header_url() {
		
		// Base args
		$args = [
			'page'       => 'logtastic_js-error-log',
			'sort_by'    => 'occurrence_count'
		];

		// Add sort order to args
		if ( 'occurrence_count' == $this->sort_by && 'DESC' == $this->sort_order ) {
			$args['sort_order'] = 'asc';
		} else {
			$args['sort_order'] = 'desc';
		}

		// Add error_type_filter to args
		if ( 'all' != $this->error_type_filter ) {
			$args['error_type_filter'] = $this->error_type_filter;
		}

		// Add source_filter to args 
		if ( 'all' != $this->source_filter ) {
			if ( null == $this->source_slug_filter ) {
				$args['source_filter'] = 'all-' . $this->source_filter .'s';
			} else {
				$args['source_filter'] = $this->source_filter . '_' . $this->source_slug_filter;
			}
		}

		// Add search_term to args
		if ( null != $this->search_term ) {
			$args['search_term'] = $this->search_term;
		}

		// Build and return full URL
		return add_query_arg( $args, admin_url( 'admin.php' ) );

	}
	

	/**
	 * Build the URL for the "First Occurred" column header with current filters applied
	 *
	 * @since 1.0.0
	 * 
	 * @return string
	 */
	public function get_first_occurred_column_header_url() {

		// Base args
		$args = [
			'page'       => 'logtastic_js-error-log',
			'sort_by'    => 'first_occurred'
		];

		// Add sort order to args
		if ( 'first_occurred' == $this->sort_by && 'DESC' == $this->sort_order ) {
			$args['sort_order'] = 'asc';
		} else {
			$args['sort_order'] = 'desc';
		}

		// Add error_type_filter to args
		if ( 'all' != $this->error_type_filter ) {
			$args['error_type_filter'] = $this->error_type_filter;
		}

		// Add source_filter to args 
		if ( 'all' != $this->source_filter ) {
			if ( null == $this->source_slug_filter ) {
				$args['source_filter'] = 'all-' . $this->source_filter .'s';
			} else {
				$args['source_filter'] = $this->source_filter . '_' . $this->source_slug_filter;
			}
		}

		// Add search_term to args
		if ( null != $this->search_term ) {
			$args['search_term'] = $this->search_term;
		}

		// Build and return full URL
		return add_query_arg( $args, admin_url( 'admin.php' ) );
	
	}
	

	/**
	 * Build the URL for the "Last Occurred" column header with current filters applied
	 *
	 * @since 1.0.0
	 * 
	 * @return string
	 */
	public function get_last_occurred_column_header_url() {

		// Base args
		$args = [
			'page'       => 'logtastic_js-error-log',
			'sort_by'    => 'last_occurred'
		];

		// Add sort order to args
		if ( 'last_occurred' == $this->sort_by && 'DESC' == $this->sort_order ) {
			$args['sort_order'] = 'asc';
		} else {
			$args['sort_order'] = 'desc';
		}

		// Add error_type_filter to args
		if ( 'all' != $this->error_type_filter ) {
			$args['error_type_filter'] = $this->error_type_filter;
		}

		// Add source_filter to args 
		if ( 'all' != $this->source_filter ) {
			if ( null == $this->source_slug_filter ) {
				$args['source_filter'] = 'all-' . $this->source_filter .'s';
			} else {
				$args['source_filter'] = $this->source_filter . '_' . $this->source_slug_filter;
			}
		}

		// Add search_term to args
		if ( null != $this->search_term ) {
			$args['search_term'] = $this->search_term;
		}

		// Build and return full URL
		return add_query_arg( $args, admin_url( 'admin.php' ) );
	
	}


	/**
	 * Construct and echo the HTML for the error table pagination
	 *
	 * @since 1.0.0
	 */
	public function display_pagination( $location = 'default' ) {
		
		// If 1 or fewer pages, return false
		if ( $this->max_page <= 1 ) {
			return false;
		}
		
		// Construct pagination base url - define base args
		$args = [
			'page'       => 'logtastic_js-error-log',
			'sort_by'    => $this->sort_by
		];
		// Construct pagination base url - add sort order to args
		if ( 'ASC' == $this->sort_order ) {
			$args['sort_order'] = 'asc';
		}
		// Construct pagination base url - add error_type_filter to args
		if ( 'all' != $this->error_type_filter ) {
			$args['error_type_filter'] = $this->error_type_filter;
		}
		// Construct pagination base url - add source_filter to args 
		if ( 'all' != $this->source_filter ) {
			if ( null == $this->source_slug_filter ) {
				$args['source_filter'] = 'all-' . $this->source_filter .'s';
			} else {
				$args['source_filter'] = $this->source_filter . '_' . $this->source_slug_filter;
			}
		}
		// Construct pagination base url - add search_term to args
		if ( null != $this->search_term ) {
			$args['search_term'] = $this->search_term;
		}
		// Construct pagination base url - build full url
		$path = add_query_arg( $args, admin_url( 'admin.php' ) );
		
		// Echo pagination HTML
		echo '<span class="pagination-links">';
		
		// Echo pagination HTML - First & Prev Page Buttons
		if ( $this->current_page == 1 ) {
			echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">«</span> ';
			echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">‹</span> ';
		} else {
			echo '<a class="first-page button" href="' . esc_url( $path ) . '"><span class="screen-reader-text">' . esc_html__( 'First page', 'logtastic' ) . '</span><span aria-hidden="true">«</span></a> ';
			echo '<a class="prev-page button" href="' . esc_url( $path . '&amp;paged=' . $this->current_page - 1 ) . '"><span class="screen-reader-text">' . esc_html__( 'Previous page', 'logtastic' ) . '</span><span aria-hidden="true">‹</span></a> ';
		}
		
		// Echo pagination HTML - Paging Form & Input
		echo '<form action="admin.php" method="get">';
		echo '<input type="hidden" name="page" value="logtastic_js-error-log" />';
		if ( 'all' != $this->error_type_filter ) {
			echo '<input type="hidden" name="error_type_filter" value="' . esc_attr( $this->error_type_filter ) . '" />';
		}
		if ( 'all' != $this->source_filter ) {
			if ( null == $this->source_slug_filter ) {
				echo '<input type="hidden" name="source_filter" value="all-' . esc_attr( $this->source_filter ) .'s" />';
			} else {
				echo '<input type="hidden" name="source_filter" value="' . esc_attr( $this->source_filter ) . '_' . esc_attr( $this->source_slug_filter ) . '" />';
			}
		}
		if ( null != $this->search_term ) {
			echo '<input type="hidden" name="search_term" value="' . esc_attr( $this->search_term ) . '" />';
		}
		echo '<input type="hidden" name="sort_by" value="' . esc_attr( $this->sort_by ) . '" />';
		if ( 'ASC' == $this->sort_order ) {
			echo '<input type="hidden" name="sort_order" value="asc" />';
		}
		echo '<span class="paging-input"><label for="current-page-selector-' . esc_attr( $location ) . '" class="screen-reader-text">' . esc_html__( 'Current page', 'logtastic' ) . '</label><input class="current-page" id="current-page-selector-' . esc_attr( $location ) . '" type="text" name="paged" value="' . esc_attr( $this->current_page ) . '" size="' . esc_attr( strlen((string)$this->current_page) ) . '" aria-describedby="table-paging"><span class="tablenav-paging-text"> ' . esc_html__( 'of', 'logtastic' ) . ' <span class="total-pages">' . esc_html ( $this->max_page ) . '</span></span></span>';
		echo '</form>';
		
		// Echo pagination HTML - Next & Last Page Buttons
		if ( $this->current_page == $this->max_page ) {
			echo ' <span class="tablenav-pages-navspan button disabled" aria-hidden="true">›</span>';
			echo ' <span class="tablenav-pages-navspan button disabled" aria-hidden="true">»</span>';
		} else {
			echo ' <a class="next-page button" href="' . esc_url( $path . '&amp;paged=' . $this->current_page + 1 ) . '"><span class="screen-reader-text">' . esc_html__( 'Next page', 'logtastic' ) . '</span><span aria-hidden="true">›</span></a>';
			echo ' <a class="last-page button" href="' . esc_url( $path . '&amp;paged=' . $this->max_page ) . '"><span class="screen-reader-text">' . esc_html__( 'Last page', 'logtastic' ) . '</span><span aria-hidden="true">»</span></a>';
		}
		
		echo '</span>';
		
	}
	
	
	/**
	 * Construct and echo the HTML for the error table search form
	 *
	 * @since 1.0.0
	 */
	public function display_table_search() {
		
		// Don't display if no errors found by current query and search term is empty
		if ( null == $this->search_term && 0 == $this->total_error_count ) {
			return false;
		}
		
		// Begin form layout
		echo '<form id="error-search" method="get">';
		
		// Hidden fields for any existing variables
		echo '<input type="hidden" name="page" value="logtastic_js-error-log" />';
		echo '<input type="hidden" name="sort_by" value="' . esc_attr( $this->sort_by ) . '" />';
		if ( 'ASC' == $this->sort_order ) {
			echo '<input type="hidden" name="sort_order" value="asc" />';
		}
		if ( 'all' != $this->error_type_filter ) {
			echo '<input type="hidden" name="error_type_filter" value="' . esc_attr( $this->error_type_filter ) . '" />';
		}
		if ( 'all' != $this->source_filter ) {
			if ( null == $this->source_slug_filter ) {
				echo '<input type="hidden" name="source_filter" value="all-' . esc_attr( $this->source_filter ) .'s" />';
			} else {
				echo '<input type="hidden" name="source_filter" value="' . esc_attr( $this->source_filter ) . '_' . esc_attr( $this->source_slug_filter ) . '" />';
			}
		}

		// Display search field and submit button
		echo '<p class="search-box">';
		echo '<label class="screen-reader-text" for="error-search-input">' . esc_html__( 'Search Error Messages:', 'logtastic' ) . '</label>';
		if ( null != $this->search_term ) {
			echo '<input type="search" id="error-search-input" name="search_term" value="' . esc_attr( $this->search_term ) . '">';
		} else {
			echo '<input type="search" id="error-search-input" name="search_term" value="">';
		}
		echo '<button type="submit" id="search-submit" class="button" aria-label="' . esc_attr__( 'Search Error Messages', 'logtastic' ) . '"><span>' . esc_attr__( 'Search Error Messages', 'logtastic' ) . '</span></button>';
		echo '</p>';

		// End form layout
		echo '</form>';
		
	}
	
	
	/**
	 * Construct and echo the HTML for the error table filter form
	 *
	 * @since 1.0.0
	 */
	public function display_table_filters() {
		
		// Don't display if no error levels or error source available
		if ( $this->db_error_types == null && $this->db_error_sources == null ) {
			return false;
		}
		
		// Begin form layout
		echo '<form action="admin.php" method="get">';
		
		// Hidden fields for any existing variables
		echo  '<input type="hidden" name="page" value="logtastic_js-error-log" />';
		if ( null != $this->search_term ) {
			echo '<input type="hidden" name="search_term" value="' . esc_attr( $this->search_term ) . '" />';
		}
		echo '<input type="hidden" name="sort_by" value="' . esc_attr( $this->sort_by ) . '" />';
		if ( 'ASC' == $this->sort_order ) {
			echo '<input type="hidden" name="sort_order" value="asc" />';
		}
		
		// Error Type Filter
		if ( $this->db_error_types != null ) {
			echo '	<label for="filter-by-error-type" class="screen-reader-text">' . esc_html__( 'Filter by error type', 'logtastic' ) . '</label>';
			echo '	<select name="error_type_filter" id="filter-by-error-type">';
			echo '		<option value="all">' . esc_html__( 'All error types', 'logtastic' ) . '</option>';
			foreach ( $this->db_error_types as $error_type => $boolean ) {
				if ( $this->error_type_filter == $error_type ) {
					echo '		<option selected="selected" value="' . esc_attr( $error_type ) . '">' . esc_html( self::get_error_label( $error_type ) ) .'</option>';
				} else {
					echo '		<option value="' . esc_attr( $error_type ) . '">' . esc_html( self::get_error_label( $error_type ) ) .'</option>';
				}
			}
			echo '	</select>';
		}
		
		// Source Filter
		if ( $this->db_error_sources != null ) {
			echo '	<label for="filter-by-source" class="screen-reader-text">' . esc_html__( 'Filter by error source', 'logtastic' ) . '</label>';
			echo '	<select name="source_filter" id="filter-by-source">';
			echo '		<option value="all">' . esc_html__( 'All sources', 'logtastic' ) . '</option>';
			
			// Source Filter - WordPress Core
			if ( array_key_exists('wp_core', $this->db_error_sources ) ) {
				echo '		<optgroup label="' . esc_attr__( 'WordPress Core', 'logtastic' ) . '">';
				if ( 'wp_core' == $this->source_filter ) {
					echo '			<option selected="selected" value="wp_core">' . esc_html__( 'WordPress Core', 'logtastic' ) . '</option>';
				} else {
					echo '			<option value="wp_core">' . esc_html__( 'WordPress Core', 'logtastic' ) . '</option>';	
				}
				echo '		</optgroup>';
			}
			
			// Source Filter - Plugins
			if ( array_key_exists('plugin', $this->db_error_sources ) ) {
				echo '		<optgroup label="' . esc_attr__( 'Plugins', 'logtastic' ) . '">';
				if ( 'plugin' == $this->source_filter && null == $this->source_slug_filter ) {
					echo '			<option selected="selected" value="all-plugins">' . esc_html__( 'All plugins', 'logtastic' ) . '</option>';
				} else {
					echo '			<option value="all-plugins">' . esc_html__( 'All plugins', 'logtastic' ) . '</option>';
				}
				if ( null != $this->db_error_sources['plugin'] ) {
					foreach ( $this->db_error_sources['plugin'] as $db_plugin_slug ) {
						foreach ( $this->all_plugins as $plugin_index => $plugin_data ) {	
							if ( strpos( $plugin_index, $db_plugin_slug . '/' ) === 0 ) {
								if ( 'plugin' == $this->source_filter && $db_plugin_slug == $this->source_slug_filter ) {
									echo '			<option selected="selected" value="plugin_' . esc_attr( $db_plugin_slug ) . '">' . esc_html( $plugin_data['Name'] ) . '</option>';
								} else {
									echo '			<option value="plugin_' . esc_attr( $db_plugin_slug ) . '">' . esc_html( $plugin_data['Name'] ) . '</option>';
								}
								break;
							}
						}
					}
				}
				echo '		</optgroup>';
			}
			
			// Source Filter - Themes
			if ( array_key_exists('theme', $this->db_error_sources ) ) {
				echo '		<optgroup label="Themes">';
				if ( 'theme' == $this->source_filter && null == $this->source_slug_filter ) {
					echo  '			<option selected="selected" value="all-themes">' . esc_html__( 'All themes', 'logtastic' ) . '</option>';
				} else {
					echo  '			<option value="all-themes">' . esc_html__( 'All themes', 'logtastic' ) . '</option>';
				}
				if ( null != $this->db_error_sources['theme'] ) {
					foreach ( $this->db_error_sources['theme'] as $db_theme_slug ) {
						foreach ( $this->all_themes as $theme_index => $theme_data ) {
							
							if ( $theme_index == $db_theme_slug ) {
								if ( 'theme' == $this->source_filter && $db_theme_slug == $this->source_slug_filter ) {
									echo  '			<option selected="selected" value="theme_' . esc_attr( $db_theme_slug ) . '">' . esc_html( $theme_data['Name'] ) . '</option>';
								} else {
									echo  '			<option value="theme_' . esc_attr( $db_theme_slug ) . '">' . esc_html( $theme_data['Name'] ) . '</option>';
								}
								break;
							}
						}
					}
				}
				
				echo '		</optgroup>';
			}
			
			echo '	</select>';
			
		}
		
		// Submit button
		echo '	<input type="submit" name="filter_action" id="filter-submit" class="button" value="' . esc_attr__( 'Filter', 'logtastic' ) . '" />';
		
		// End form layout
		echo  '</form>';
		
	}


	/**
	 * Construct and echo the HTML for the error table bulk actions
	 *
	 * @since 1.0.0
	 */
	public function display_table_bulk_actions( $location = 'default' ) {
		
		// Don't display if no errors in table
		if ( $this->total_error_count == 0 ) {
			return false;
		}
		
		// Display bulk actions selector
		echo '<label for="bulk-action-selector-' . esc_attr( $location ) . '" class="screen-reader-text">' . esc_html__( 'Select bulk action', 'logtastic' ) . '</label>';
		echo '<select name="bulk-action-select" id="bulk-action-selector-' . esc_attr( $location ) . '" onchange="logtasticBulkActionValueSync(event)">';
		echo '	<option value="-1">' . esc_html__( 'Bulk actions', 'logtastic' ) . '</option>';
		echo '	<option value="delete-errors">' . esc_html__( 'Delete', 'logtastic' ) . '</option>';
		echo '	<option value="ignore-errors">' . esc_html__( 'Delete & Ingore', 'logtastic' ) . '</option>';
		echo '</select>';
		echo '<button name="bulk-action-apply" id="doaction-' . esc_attr( $location ) . '" onclick="logtasticBulkAction(event)" class="button action">' . esc_html__( 'Apply', 'logtastic' ) . '</button>';
		
	}
	

	/**
	 * Get the plugin name for the given plugin slug. Optionally can be passed an array of plugins, otherwise will fetch using get_plugins()
	 *
	 * @since 1.0.0
	 * 
	 * @param    string		$slug 			The plugin slug to look up
	 * @param    array		$all_plugins 	An array of all installed plugins (as returned by get_plugins())
	 */
	public static function get_plugin_name_static( $slug, $all_plugins = null ) {
		
		if ( null == $all_plugins ) {
			$all_plugins = get_plugins();
		}
			
		foreach ( $all_plugins as $plugin_index => $plugin_data ) {	
			if ( strpos( $plugin_index, $slug . '/' ) === 0 ) {
				return $plugin_data['Name'];
				break;
			}
		}
		
		return false;
		
	}
	

	/**
	 * Get the plugin name for the given plugin slug
	 *
	 * @since 1.0.0
	 * 
	 * @param    string		$slug 			The plugin slug to look up
	 */
	public function get_plugin_name( $slug ) {
			
		return self::get_plugin_name_static( $slug, $this->all_plugins );
		
	}
	

	/**
	 * Get the theme name for the given theme slug. Optionally can be passed an array of themes, otherwise will fetch using wp_get_themes()
	 *
	 * @since 1.0.0
	 * 
	 * @param    string		$slug 			The theme slug to look up
	 * @param    array		$all_themes 	An array of all installed themes (as returned by wp_get_themes())
	 */
	public static function get_theme_name_static( $slug, $all_themes = null ) {
		
		if ( null == $all_themes ) {
			$all_themes = wp_get_themes();
		}
			
		foreach ( $all_themes as $theme_index => $theme_data ) {	
			if ( $slug == $theme_index ) {
				return $theme_data['Name'];
				break;
			}
		}
		
		return false;
		
	}
	

	/**
	 * Get the theme name for the given theme slug. 
	 *
	 * @since 1.0.0
	 * 
	 * @param    string		$slug 			The theme slug to look up
	 */
	public function get_theme_name( $slug ) {
			
		return self::get_theme_name_static( $slug, $this->all_themes );
		
	}
	
	/**
	 * Sort an array of version numbers in ascending order
	 *
	 * @since 1.0.0
	 * 
	 * @param    array		$versions 		An array of version numbers to sort
	 */
	public static function sort_versions_ascending( array $versions ) {
		usort( $versions, function ( $a, $b ) {
			return version_compare( $a, $b );
		} );
		return $versions;
	}
	
	
	/**
	 * Get the source version range for a single error from DB based on error id
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$error_id 	The id of the error to fetch from the database
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	public static function get_source_version_range( int $error_id ) {
			
		global $wpdb;
		
		// Get results from DB
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$results = $wpdb->get_results( 
			$wpdb->prepare( 
				"
					SELECT DISTINCT source_version
					FROM %i
					WHERE error_id = %d
					AND source_version IS NOT NULL
				",
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id 
			) 
		);
		
		// Add results to array
		$range_array = [];
		foreach ( $results as $row ) {
			$range_array[] = $row->source_version; 
		}
		
		// Return array, sorted by version number, lowest to highest
		return self::sort_versions_ascending( $range_array );
		
	}
	

	/**
	 * Get the WP version range for a single error from DB based on error id
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$error_id 	The id of the error to fetch from the database
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	public static function get_wp_version_range( int $error_id ) {
		
		global $wpdb;
		
		// Get results from database
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$results = $wpdb->get_results( 
			$wpdb->prepare( 
				"
					SELECT DISTINCT wp_version
					FROM %i
					WHERE error_id = %d
					AND wp_version IS NOT NULL
				",
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id
			)
		);
		
		// Add results to array
		$range_array = [];
		foreach ( $results as $row ) {
			$range_array[] = $row->wp_version; 
		}
		
		// Return array, sorted by version number, lowest to highest
		return self::sort_versions_ascending( $range_array );
		
	}
	

	/**
	 * Get the PHP version range for a single error from DB based on error id
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$error_id 	The id of the error to fetch from the database
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	public static function get_php_version_range( int $error_id ) {
			
		global $wpdb;
		
		// Get results from DB
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$results = $wpdb->get_results( 
			$wpdb->prepare( 
				"
					SELECT DISTINCT php_version
					FROM %i
					WHERE error_id = %d
					AND php_version IS NOT NULL
				",
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id
			)
		);
		
		// Add results to array
		$range_array = [];
		foreach ( $results as $row ) {
			$range_array[] = $row->php_version; 
		}
		
		// Return array, sorted by version number, lowest to highest
		return self::sort_versions_ascending( $range_array );
		
	}
	
	/**
	 * Get the count of occurrences by date for a single error from DB based on error id
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$error_id 	The id of the error to fetch from the database
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	public static function get_occurrence_count_by_date( int $error_id ) {
			
		global $wpdb;
		
		// Get results from DB
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$results = $wpdb->get_results( 
			$wpdb->prepare( 
				"
					SELECT DATE(timestamp) AS error_date, COUNT(*) AS occurrence_count
					FROM %i
					WHERE error_id = %d
					GROUP BY error_date
					ORDER BY error_date ASC;
				",
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id
			)
		);
		
		// Convert results to an associative array
		$date_occurrences = [];
		foreach ( $results as $row ) {
			$date_occurrences[$row->error_date] = intval( $row->occurrence_count );
		}
		
		// Return array, sorted by version number, lowest to highest
		return $date_occurrences;
		
	}


	/**
	 * Get the timestamp of the latest occurrence for a single error from DB based on error id
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$error_id 	The id of the error to fetch from the database
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	public static function get_latest_occurrence_timestamp( int $error_id ) {
			
		global $wpdb;
		
		// Get result from DB
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
		$result = $wpdb->get_results( 
			$wpdb->prepare( 
				"
					SELECT timestamp
					FROM %i
					WHERE error_id = %d
					ORDER BY occurrence_id DESC
					LIMIT 1;
				",
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id
			)
		);
		
		// Return timestamp from result
		return $result[0]->timestamp;
		
	}
	

	/**
	 * Get occurrences for a single error from DB based on error id (and optional offset, limit and sort order)
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$error_id 	The id of the error to fetch from the database
	 * @param    int		$offset 	The offset for the results 
	 * @param    int		$limit 		The number of results to return
	 * @param    string		$sort_order	The sort order for the query (either ASC or DESC)
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	public static function get_occurrences( int $error_id, int $offset = 0, int $limit = 20, string $sort_order = 'DESC' ) {
		
		global $wpdb;

		// Validate source order
		$sort_order = strtoupper( $sort_order );
		if ( 'ASC' !== $sort_order && 'DESC' !== $sort_order ) {
			$sort_order = 'DESC';
		}

		// Get occurrences from DB
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $sort_order cannot be passed as prepared variable. Is sanitised/validated above.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Direct query as custom table. Caching not appropriate as fresh data always required. $sort_order is whitelisted to ASC/DESC above; other params are passed via placeholders.
		$occurrences = $wpdb->get_results(
			$wpdb->prepare( 
				"
					SELECT occurrence_id, source_version, wp_version, php_version, timestamp, stack_trace_available, additional_data
					FROM %i 
					WHERE error_id = %d
					ORDER BY %i {$sort_order}
					LIMIT %d OFFSET %d
				",
				$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
				$error_id,
				'occurrence_id',
				$limit,
				$offset
			),
			ARRAY_A 
		);
		// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		// Unserialize additional data
		if ( !empty( $occurrences ) ) {
			foreach ( $occurrences as &$occurrence ) {
				if ( isset( $occurrence['additional_data'] ) && !is_null( $occurrence['additional_data'] ) ) {
					$occurrence['additional_data'] = maybe_unserialize( $occurrence['additional_data'] );
				}
			}
			unset( $occurrence );
		}
		
		// Return results, or false if none found
		if (!empty( $occurrences ) ) {
			return $occurrences;
		} else {
			return false;
		}
		
	}
	
	/**
	 * Get the file path for a single error from DB based on error id
	 *
	 * @since 1.0.0
	 *
	 * @param    int		$error_id 	The id of the error to fetch from the database
	 * @global 	 wpdb 		$wpdb 		WordPress database abstraction object.
	 */
	public static function get_error_file_path( int $error_id ) {
			
		global $wpdb;
		
		// Get results from DB
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not required.
		$results = $wpdb->get_results( 
			$wpdb->prepare( 
				"
					SELECT file
					FROM %i
					WHERE error_id = %d
				",
				$wpdb->prefix . self::ERROR_TABLE_NAME,
				$error_id 
			)
		);
		
		// Return response
		if ( isset( $results[0]->file ) ) {
			return $results[0]->file;
		} else {
			return false;
		}
	}
	

	/**
	 * Return a date based on the WP Date Format option
	 *
	 * @since 1.0.0
	 *
	 * @param    string		$date	 	The date to be formatted
	 */
	public static function format_date ( $date ) {
		
		if ( null == $date ) {
			return null;
		}
		
		$wp_date_format = get_option( 'date_format' );
		
		$date_object = date_create( $date );
			
		return date_format( $date_object, $wp_date_format );
		
	}
	
	/**
	 * Return a time based on the WP Time Format option
	 *
	 * @since 1.0.0
	 *
	 * @param    string		$date	 	The date to be formatted
	 */
	public static function format_time ( $date ) {
		
		if ( null == $date ) {
			return null;
		}
		
		$wp_time_format = get_option( 'time_format' );
		
		$date_object = date_create( $date );
		
		return date_format( $date_object, $wp_time_format );
		
	}
	
	/**
	 * Return a date and time (separated by @) based on the WP Date & Time Format options
	 *
	 * @since 1.0.0
	 *
	 * @param    string		$date	 	The date to be formatted
	 */
	public static function format_date_and_time ( $date ) {
		
		if ( null == $date ) {
			return null;
		}
		
		$wp_date_format = get_option( 'date_format' );
		
		$wp_time_format = get_option( 'time_format' );
		
		$date_object = date_create( $date );
		
		return date_format( $date_object, $wp_date_format ) . ' @ ' . date_format( $date_object, $wp_time_format );
		
	}
	
	
	/**
	 * Load error details for a specific error and return as JSON (called via Ajax)
	 *
	 * @since 1.0.0
	 */
	public static function ajax_load_js_error_details() {

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Nonce and user access permissions verified in class-logtastic-admin.php before calling this function
		
		// Check if error id is present in request and is numeric, otherwise send error message
		if ( empty( $_REQUEST ) || empty( $_REQUEST['error_id'] ) || !is_numeric( $_REQUEST['error_id'] ) ) {
			
			// Send JSON error response
			wp_send_json_error( __( 'Invalid Error ID', 'logtastic' ) );
			
		} else {
			
			// Validate error ID
			$error_id = intval( $_REQUEST['error_id'] ) ;
			
			// If Valid error ID, proceed, otherwise show error message
			if ( $error_id ) {
				
				// Declare empty array to store error details
				$error_details = [];
				
				// Get source version range
				$error_details['source_version']['range_array'] = self::get_source_version_range( $error_id );

				// If source version range found, add to error details array
				if ( is_array( $error_details['source_version']['range_array'] ) && !is_null( $error_details['source_version']['range_array'] ) ) {

					// Pick out first item from range array
					$error_details['source_version']['range_first'] = reset( $error_details['source_version']['range_array'] );

					// Pick out last item from range array and add to error details
					$error_details['source_version']['range_last'] = end( $error_details['source_version']['range_array'] );
					
					// Construction version string
					if ( count( $error_details['source_version']['range_array'] ) > 1 ) {
						$error_details['source_version']['range_string_short'] = $error_details['source_version']['range_first'] . ' - ' . $error_details['source_version']['range_last'];
						$error_details['source_version']['range_string_full'] = __( 'Versions', 'logtastic' ) . ': ' . $error_details['source_version']['range_first'] . ' - ' . $error_details['source_version']['range_last'];
						$error_details['source_version']['range_string_version_text'] = __( 'Versions', 'logtastic' );
					} else {
						$error_details['source_version']['range_string_short'] = $error_details['source_version']['range_first'];
						$error_details['source_version']['range_string_full'] = __( 'Version', 'logtastic' ) . ': ' . $error_details['source_version']['range_first'];
						$error_details['source_version']['range_string_version_text'] = __( 'Version', 'logtastic' );
					}
				
				// Else add blank placeholders to version details array	
				} else {
					$error_details['source_version']['range_first'] = '-';
					$error_details['source_version']['range_last'] = '-';
					$error_details['source_version']['range_string_short'] = '-';
					$error_details['source_version']['range_string_full'] = __( 'Version', 'logtastic' ) . ': -';
					$error_details['source_version']['range_string_version_text'] = __( 'Version', 'logtastic' );
				}
				
				// Get WordPress version range
				$error_details['wp_version']['range_array'] = self::get_wp_version_range( $error_id );

				// If WordPress version range found, add to error details array
				if ( is_array( $error_details['wp_version']['range_array'] ) && !is_null( $error_details['wp_version']['range_array'] ) ) {

					// Pick out first item from range array
					$error_details['wp_version']['range_first'] = reset( $error_details['wp_version']['range_array'] );

					// Pick out last item from range array and add to error details
					$error_details['wp_version']['range_last'] = end( $error_details['wp_version']['range_array'] );
					
					// Construction version string
					if ( count( $error_details['wp_version']['range_array'] ) > 1 ) {
						$error_details['wp_version']['range_string_short'] = $error_details['wp_version']['range_first'] . ' - ' . $error_details['wp_version']['range_last'];
						$error_details['wp_version']['range_string_full'] = __( 'Versions', 'logtastic' ) . ': ' . $error_details['wp_version']['range_first'] . ' - ' . $error_details['wp_version']['range_last'];
						$error_details['wp_version']['range_string_version_text'] = __( 'Versions', 'logtastic' );
					} else {
						$error_details['wp_version']['range_string_short'] = $error_details['wp_version']['range_first'];
						$error_details['wp_version']['range_string_full'] = __( 'Version', 'logtastic' ) . ': ' . $error_details['wp_version']['range_first'];
						$error_details['wp_version']['range_string_version_text'] = __( 'Version', 'logtastic' );
					}

				// Else add blank placeholders to version details array		
				} else {
					$error_details['wp_version']['range_first'] = '-';
					$error_details['wp_version']['range_last'] = '-';
					$error_details['wp_version']['range_string_short'] = '-';
					$error_details['wp_version']['range_string_full'] = __( 'Version', 'logtastic' ) . ': -';
					$error_details['wp_version']['range_string_version_text'] = __( 'Version', 'logtastic' );
				}
				
				// Get occurrence count by date and add to error details array
				$error_details['occurrences_by_date'] = self::get_occurrence_count_by_date( $error_id ); 
				
				// Calculate last occurrence date/time
				$error_details['last_occurrence_date'] = self::get_latest_occurrence_timestamp( $error_id );
				
				// Format last occurrence data/time for display
				$error_details['last_occurrence_string'] = self::format_date( $error_details['last_occurrence_date'] ) . '<br />' . self::format_time( $error_details['last_occurrence_date'] );
				
				// Calculate total occurrences
				$error_details['total_occurrences'] = array_sum( $error_details['occurrences_by_date'] );
				
				// Calculate total pages
				$error_details['occurrences_total_pages'] = ceil( $error_details['total_occurrences'] / self::$occurrences_per_page );
					
				// Calculate current page
				if ( isset ( $_REQUEST['occurrences_page'] ) && is_numeric( $_REQUEST['occurrences_page'] ) ) {
					$target_page = intval( $_REQUEST['occurrences_page'] );
					if ( $target_page <= 1 ) {
						$error_details['current_page'] = 1;
					} else if ( $target_page >= $error_details['occurrences_total_pages'] ) {
						$error_details['current_page'] = $error_details['occurrences_total_pages'];
					} else {
						$error_details['current_page'] = $target_page;
					}
				} else {
					$error_details['current_page'] = 1;
				}
				
				// Get full data for most recent occurrences
				$offset = ( $error_details['current_page'] - 1 ) * self::$occurrences_per_page;
				$error_details['occurrences'] = self::get_occurrences( $error_id, $offset, $limit = self::$occurrences_per_page );
					
				// Add formatted timestamp for each occurrence
				foreach ($error_details['occurrences'] as &$occurrence) {
					$occurrence['date_time_string'] = self::format_date( $occurrence['timestamp'] ) . ' <br /> ' . self::format_time( $occurrence['timestamp'] );
				}
				
				// Get count of visible occurrences in current view
				$error_details['current_view_occurrences_count'] = count( $error_details['occurrences'] );
				
				// Prepare Occurrences Table Navigation
				$occurrence_from = $offset + 1;
				$occurrence_to =  $offset + $error_details['current_view_occurrences_count'];
				$prev_page = $error_details['current_page'] - 1;
				$next_page = $error_details['current_page'] + 1;
				
				$error_details['occurrences_nav_html'] =	'<div class="tablenav-pages">';
				$error_details['occurrences_nav_html'] .= 	'	<span class="displaying-num">';
				if ( $occurrence_from != $occurrence_to ) {
					$error_details['occurrences_nav_html'] .=	'		' . __( 'Displaying occurrences ', 'logtastic' ) . $occurrence_from . ' - ' . $occurrence_to . ' ' . __( 'of', 'logtastic' ) . ' ' . $error_details['total_occurrences'];
				} else {
					$error_details['occurrences_nav_html'] .=	'		' . __( 'Displaying occurrences ', 'logtastic' ) . $occurrence_from . ' ' . __( 'of', 'logtastic' ) . ' ' . $error_details['total_occurrences'];
				}
				$error_details['occurrences_nav_html'] .=	'	</span>';
				
				if ( $error_details['occurrences_total_pages'] > 1 ) {
				
					$error_details['occurrences_nav_html'] .= 	'	<span class="pagination-links">';
					
					// First and previous buttons
					if ( $error_details['current_page'] == 1 ) {
						$error_details['occurrences_nav_html'] .= 	'		<span class="tablenav-pages-navspan button disabled" aria-hidden="true">«</span>';
						$error_details['occurrences_nav_html'] .= 	'		<span class="tablenav-pages-navspan button disabled" aria-hidden="true">‹</span>';
					} else {
						$error_details['occurrences_nav_html'] .= 	'		<button class="first-page button" data-target-page="1" onclick="logtasticOccurrencesPaginationNavigation(event, \'JS\');">«</button>';
						$error_details['occurrences_nav_html'] .= 	'		<button class="prev-page button" data-target-page="' . $prev_page . '" onclick="logtasticOccurrencesPaginationNavigation(event, \'JS\');">‹</button>';
					}
					
					// Construct pagination HTML - Paging Form & Input
					$error_details['occurrences_nav_html'] .= 	'		<form onsubmit="logtasticOccurrencesPaginationSubmit(event, \'JS\');">';
					
					$error_details['occurrences_nav_html'] .= 	'			<span class="paging-input"><label for="current-page-selector" class="screen-reader-text">' . __( 'Current Page', 'logtastic' ) . '</label><input class="current-page" id="current-page-selector" type="text" name="paged" value="' . $error_details['current_page'] . '" size="' . strlen((string)$error_details['current_page']) . '" aria-describedby="table-paging"><span class="tablenav-paging-text"> of <span class="total-pages">' . $error_details['occurrences_total_pages'] . '</span></span></span>';
					
					$error_details['occurrences_nav_html'] .= 	'		</form>';
					
					// Next and last buttons
					if ( $error_details['current_page'] == $error_details['occurrences_total_pages'] ) {
						$error_details['occurrences_nav_html'] .= 	'		<span class="tablenav-pages-navspan button disabled" aria-hidden="true">›</span>';
						$error_details['occurrences_nav_html'] .= 	'		<span class="tablenav-pages-navspan button disabled" aria-hidden="true">»</span>';
					} else {
						$error_details['occurrences_nav_html'] .= 	'		<button class="next-page button" data-target-page="' . $next_page . '" onclick="logtasticOccurrencesPaginationNavigation(event, \'JS\');">›</button>';
						$error_details['occurrences_nav_html'] .= 	'		<button class="last-page button" data-target-page="' . $error_details['occurrences_total_pages'] . '" onclick="logtasticOccurrencesPaginationNavigation(event, \'JS\');">»</button>';
					}
					
					$error_details['occurrences_nav_html'] .= 	'	</span>';
					
				}
				
				$error_details['occurrences_nav_html'] .= 	'</div>';
				
				// Send JSON success response and error data
				wp_send_json_success( $error_details );
				
			} else {
				
				// Send JSON error response
				wp_send_json_error( __( 'Invalid Error ID', 'logtastic' ) );
				
			}
			
		}

		// phpcs:enable WordPress.Security.NonceVerification.Recommended

	}

	
	/**
	 * Load stack trace for a specific occurrence and return as JSON (called via Ajax)
	 *
	 * @since 1.0.0
	 */
	public static function ajax_load_js_error_stack_trace() {

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Nonce and user access permissions verified in class-logtastic-admin.php before calling this function
		
		// Check if occurrence id is present in request and is numeric, otherwise send error message
		if ( empty( $_REQUEST ) || empty( $_REQUEST['occurrence_id'] ) || !is_numeric( $_REQUEST['occurrence_id'] ) ) {
			
			// Send JSON error response
			wp_send_json_error( __( 'Invalid Occurrence ID', 'logtastic' ) );
			
		} else {
			
			// Validate occurrence ID
			$occurrence_id = intval( $_REQUEST['occurrence_id'] ) ;
			
			// If valid occurrence ID, proceed, otherwise show error message			
			if ( $occurrence_id ) {
				
				global $wpdb;
				
				// Get the results from the database
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate as fresh data always required.
				$results = $wpdb->get_results(
					$wpdb->prepare(
						"
							SELECT 
								st_array.stack_trace_data AS stack_trace_array,
								st_string.stack_trace_data AS stack_trace_string
							FROM %i AS oe
							LEFT JOIN %i AS st_array ON st_array.stack_trace_id = oe.stack_trace_array_id
							LEFT JOIN %i AS st_string ON st_string.stack_trace_id = oe.stack_trace_string_id
							WHERE oe.occurrence_id = %d
						",
						$wpdb->prefix . self::ERROR_OCCURRENCES_TABLE_NAME,
						$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
						$wpdb->prefix . self::ERROR_STACK_TRACES_TABLE_NAME,
						$occurrence_id
					)
				);
				
				// If no result found, return error
				if ( !isset( $results[0] ) ) {
					wp_send_json_error( __( 'Invalid Occurrence ID', 'logtastic' ) );
				}
				
				// Define array and add results to array
				$stack_trace = array();
				
				if ( isset( $results[0]->stack_trace_array ) ) {
					$stack_trace['array'] = maybe_unserialize($results[0]->stack_trace_array);							
				}
				
				if ( isset( $results[0]->stack_trace_string ) ) {
					$stack_trace['string'] = $results[0]->stack_trace_string;
				}

				// Send JSON success response and stack trace data
				wp_send_json_success( $stack_trace );
				
			} else {
				
				// Send JSON error response
				wp_send_json_error( __( 'Invalid Occurrence ID', 'logtastic' ) );
				
			}
			
		}

		// phpcs:enable WordPress.Security.NonceVerification.Recommended
		
	}
	
	
	/**
	 * Remove the ignore flag from a specific error (called via Ajax)
	 *
	 * @since 1.0.0
	 */
	public static function ajax_unignore_js_error() {

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Nonce and user access permissions verified in class-logtastic-admin.php before calling this function
		
		// Check if error id is present in request and is numeric, otherwise send error message
		if ( empty( $_REQUEST ) || empty( $_REQUEST['error_id'] ) || !is_numeric( $_REQUEST['error_id'] ) ) {
			
			// Send JSON error response
			wp_send_json_error( __( 'Invalid Error ID', 'logtastic' ) );
			
		} else {

			// Validate error ID
			$error_id = intval( $_REQUEST['error_id'] ) ;

			// If Valid error ID, proceed, otherwise show error message
			if ( $error_id ) {
			
				global $wpdb;
				
				// Mark error as unignored
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query as custom table. Caching not appropriate.
				$unignored_error = $wpdb->update(
					$wpdb->prefix . self::ERROR_TABLE_NAME,       			// Table name
					['ignore_error' => 0], 									// Column to update
					['error_id' => $error_id], 								// WHERE clause
					['%d'],            										// Data format for update values
					['%d']             										// Data format for WHERE condition
				);
				
				// Check if update successful
				if ( $unignored_error ) {
					
					// Send JSON success response
					wp_send_json_success( __( 'Success', 'logtastic' ) );
					
				} else {
					
					// Send JSON error response
					wp_send_json_error( __( 'Invalid Error ID', 'logtastic' ) );
					
				}

			} else {
				
				// Send JSON error response
				wp_send_json_error( __( 'Invalid Error ID', 'logtastic' ) );

			}
			
		}

		// phpcs:enable WordPress.Security.NonceVerification.Recommended
		
	}


	/**
	 * Extract an error title from the Error Message and echo safely
	 *
	 * @since 1.0.0
	 *
	 * @param    array		$error	 	The error data array
	 */
	public static function extract_and_display_error_title($error) {
		if ( null != $error['message'] ) {
			$error_message_parts = explode(' in ' . $error['file'], $error['message'] );
			$error_title = $error_message_parts[0];
			$allowed_html_tags = array( 
				'code' => array(),
				'strong' => array(),
				'b' => array(),
				'i' => array(),
				'em' => array(),
				'a' => array(
					'href' => true,
					'title' => true,
					'target' => true
				)
			);
			echo wp_kses( $error_title, $allowed_html_tags );
		}
	}


	/**
	 * Create and display a sharing url for the given error
	 *
	 * @since 1.0.0
	 *
	 * @param    array		$error	 	The error data array
	 */
	public function create_and_display_error_url( $error ) {
		
		if ( null != $error['error_id'] ) {

			$error_id = (int) $error['error_id'];

			if ( $error_id ) {
			
				// Define args
				$args = [
					'page'       => 'logtastic_js-error-log',
					'view-error' => $error_id
				];

				// Build full URL
				$error_url = add_query_arg( $args, admin_url( 'admin.php' ) );

				// Escape and echo
				echo esc_url( $error_url );

			}

		}
	}


	/**
	 * Trim error source path to remove site url
	 *
	 * @since 1.0.0
	 *
	 * @param    array		$source_url	 	The full error source url
	 */
	public function trim_error_source_path( $source_url ) {
		
		if ( null != $source_url ) {

			$trimmed = str_replace( site_url(), '', $source_url );

			return $trimmed;

		}

		return false;

	}
	
}