<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link              https://logtastic.net/
 * @since             1.0.0
 * @package           Logtastic
 * @subpackage        Logtastic/admin
 * @author            Inspired Plugins
 * @copyright         2025 Morley Digital Limited
 * @license           GPL-2.0-or-later
 */

namespace Inspired_Plugins\Logtastic;


// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}


// Main admin class
class Logtastic_Admin {
	

	/**
	 * An array of options/settings for the plugin
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array		$settings		Plugin options/settings
	 */
	protected $settings;


	/**
	 * An array of access permissions for the current user
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array		$access_permissions		Access permissions
	 */
	protected $access_permissions;
	

	/**
	 * An array of definitions for the admin page structure.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array 		$admin_page_definitions		Admin page definitions
	 */
	protected $admin_page_definitions;
	

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $settings       Plugin options/settings
	 */
	public function __construct( $settings ) {
		

		// Add settings passed to __construct to $settings variable
		$this->settings = $settings;

	}


	/**
	 * Define the current user's access permissions for the admin area
	 *
	 * @since    1.0.0
	 */
	public function define_access() {
		
		// Define user's access permissions
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-admin-access_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';
		$this->access_permissions = Logtastic_Admin_Access::define_access( $this->settings );
	
	}
	

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		
		// Enqueue main admin style sheet
		wp_enqueue_style( LOGTASTIC_PLUGIN_SLUG, plugin_dir_url( __FILE__ ) . 'css/logtastic-admin.css', array(), LOGTASTIC_PLUGIN_VERSION, 'all' );
	
	}
	

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- $_GET['tab'] and $_GET['page'] and $_GET['view-error'] used to localize admin javascript based on page being displayed, Nonce not required

		// Get current admin screen
		$screen = get_current_screen() ? get_current_screen()->id : '';

		// If not a Logtastic plugin page, return false (don't load admin JS)
		if ( strpos( $screen, 'logtastic' ) === false ) {
			return;
		}

		// Get current settings tab
		$settings_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : '';

		// Enqueue Chart.js
		wp_enqueue_script( 'chartjs', plugin_dir_url( __FILE__ ) . 'js/chart-js/chart.umd.min.js', array(), '4.5.1', true );
		
		// Enqueue main admin javascript file
		wp_enqueue_script( LOGTASTIC_PLUGIN_SLUG . '-admin', plugin_dir_url( __FILE__ ) . 'js/logtastic-admin.js', array( 'jquery', 'wp-i18n', 'chartjs' ), LOGTASTIC_PLUGIN_VERSION, false );

		// Define additional variables array for main admin javascript file
		$logtastic_admin_js_args = [
			'ajax_url' 			=> admin_url( 'admin-ajax.php' ),
			'screen' 			=> $screen,
        	'page'  			=> isset( $_GET['page'] ) ? sanitize_key( $_GET['page'] ) : '',
			'plugin_website'	=> LOGTASTIC_PLUGIN_URL
		];

		// Add error ID to additional variables array for relevant screens
		if ( 'logtastic_page_logtastic_php-error-log' == $screen || 'logtastic_page_logtastic_js-error-log' == $screen ) {
			$error_id = isset( $_GET['view-error'] ) ? absint( wp_unslash( $_GET['view-error'] ) ) : 0;
			$logtastic_admin_js_args['view_error_id'] = $error_id;
		}

		// Add required nonces to additional variables array for relevant screens
		if ( 'logtastic_page_logtastic_php-error-log' == $screen ) {
			$logtastic_admin_js_args['load_php_error_data_nonce'] = wp_create_nonce( 'logtastic-admin-php-error-log-load-error-data' );
		}
		if ( 'logtastic_page_logtastic_js-error-log' == $screen ) {
			$logtastic_admin_js_args['load_js_error_data_nonce'] = wp_create_nonce( 'logtastic-admin-js-error-log-load-error-data' );
		}
		if ( 'logtastic_page_logtastic_settings' == $screen && 'php-error-log' == $settings_tab ) {
			$logtastic_admin_js_args['unignore_php_error_nonce'] = wp_create_nonce( 'logtastic-admin-php-error-log-unignore-error' );
		}
		if ( 'logtastic_page_logtastic_settings' == $screen && 'js-error-log' == $settings_tab ) {
			$logtastic_admin_js_args['unignore_js_error_nonce'] = wp_create_nonce( 'logtastic-admin-js-error-log-unignore-error' );
		}

		// Localize the main admin javascript file
		wp_add_inline_script( LOGTASTIC_PLUGIN_SLUG . '-admin', 'const logtastic_admin = ' . json_encode( $logtastic_admin_js_args ), 'before' );

		// phpcs:enable WordPress.Security.NonceVerification.Recommended

	}
	

	/**
	 * Register the admin nenu pages
	 *
	 * @since    1.0.0
	 */
	public function admin_pages() { 

		// Load admin page definitions
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/default-settings/class-logtastic-default-settings_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';
		$this->admin_page_definitions = Logtastic_Default_Settings::get_admin_page_definitions();
		
		// Loop over each admin page defined in $admin_page_definitions array and call add_menu_page function
		foreach ( $this->admin_page_definitions as $slug => $settings ) {

			// Does current user have required permissions for this menu item?
			if ( empty( $settings['permission-requirements'] ) || ( !empty ( $this->access_permissions ) && array_intersect( $settings['permission-requirements'], $this->access_permissions ) ) ) {

				add_menu_page(
					$settings['page-title'],
					$settings['menu-title'],
					$settings['required_capability'],
					$slug,
					array($this, $settings['callback']),
					$settings['icon'],
					$settings['position'],
				);

				// Loop over each admin sub page defined in $admin_page_definitions array and call add_submenu_page function if the relevant requirements are met
				foreach ( $settings['sub_pages'] as $sub_slug => $sub_settings ) {

					// Does current user have required permissions for this menu item?
					if ( empty( $sub_settings['permission-requirements'] ) || ( !empty ( $this->access_permissions) && array_intersect( $sub_settings['permission-requirements'], $this->access_permissions ) ) ) {

						$menu_item_active = true;

						// If sub page requires a specific log to be enabled, prevent the page from being added if the requirement is not met
						if ( isset( $sub_settings['enabled_setting'] ) ) {
							if ( empty( $this->settings['enabled_logs'][$sub_settings['enabled_setting']] ) ) {
								$menu_item_active = false;
							}
						}

						if ( true == $menu_item_active ) {
							add_submenu_page(
								$slug,
								$sub_settings['page-title'],
								$sub_settings['menu-title'],
								$sub_settings['required_capability'],
								$sub_slug,
								array($this, $sub_settings['callback']),
							);
						}

					}

				}

			}

		}
		
	}


	/**
	 * Update admin body classes with custom class if current admin page belongs to this plugin
	 *
	 * @since    1.0.0
	 * @param    string		$classes 	A string of classes applied to the body element
	 * @return   string		$classes 	The updated string of classes applied to the bodoy element
	 */
	public function add_plugin_admin_body_class( $classes ) {
		
		global $plugin_page;
		
		$is_this_plugin_page = false;
		
		if ( isset( $plugin_page ) ) {
			
			foreach ( $this->admin_page_definitions as $slug => $settings ) {
				
				if ( $slug == $plugin_page ) {
					$is_this_plugin_page = true;
					break;
				}
				
				foreach ( $settings['sub_pages'] as $sub_slug => $sub_settings ) {
					if ( $sub_slug == $plugin_page ) {
						$is_this_plugin_page = true;
						break;
					}
				}
				
				if ( true == $is_this_plugin_page ) {
					break;
				}
				
			}
			
		}
		
		if ( true == $is_this_plugin_page ) {
			
			$classes .= ' inspired-logtastic ';
			
		}
		
		return $classes;
		
	}


	/**
	 * Display a notice in the admin area
	 *
	 * @since    1.0.0
	 * @param    string		$type 				The type of notification (info, success, error, warning)
	 * @param    string		$content 			The content of the notification
	 * @param    bool		$wrap_in_p_tags 	Whether or not the content should be wrapped in <p> tags
	 * @param    bool		$dismissible 		Whether or not the 
	 */
	public static function admin_page_notice( $type, $content, $wrap_in_p_tags = false, $dismissable = false ) {
		
		// Define allowed HTML tags for content without <p> tag
		$allowed_html_tags = array( 
			'code' => array(),
			'strong' => array(),
			'b' => array(),
			'i' => array(),
			'em' => array(),
			'span' => array(),
			'ul' => array(),
			'li' => array(), 
			'a' => array(
				'href' => true,
				'title' => true,
				'target' => true
			)
		);

		// Define allowed HTML tags for content including <p> tag
		$allowed_html_tags_inc_p = array( 
			'code' => array(),
			'strong' => array(),
			'b' => array(),
			'i' => array(),
			'em' => array(),
			'p' => array(),
			'span' => array(),
			'ul' => array(),
			'li' => array(), 
			'a' => array(
				'href' => true,
				'title' => true,
				'target' => true
			)
		);

		// Opening div tag with custom class for type
		echo '<div class="notice notice-' . esc_html( $type );
		if ( true == $dismissable ) { 
			echo ' is-dismissible';
		}
		echo '">';

		// Output content, either wrapped in p tags or not - sanitised with wp_kses 
		if ( true == $wrap_in_p_tags ) { 
			echo '<p>' . wp_kses( $content, $allowed_html_tags ) . '</p>';
		} else {
			echo wp_kses( $content, $allowed_html_tags_inc_p );
		}
		echo '</div>';
	}


	/**
	 * Display the plugin's admin home page
	 *
	 * @since    1.0.0
	 */
	public function admin_page_home() {
		
		global $plugin_page;
		global $title;

		// Load page parts
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-header.php';
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/general-pages/admin-page-home.php';
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-footer.php';
		
	}
	

	/**
	 * Display the plugin's PHP error log page
	 *
	 * @since    1.0.0
	 */
	public function admin_page_php_error_log() {
		
		global $plugin_page;
		global $title;

		// Load the PHP Errog Log Admin class
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-php-error-log-admin.php';
		$php_error_log_admin = new Logtastic_PHP_Error_Log_Admin();
		
		// Fetch Errors
		$errors = $php_error_log_admin->get_errors();

		// If a specific error is passed in the URL, fetch it (if not already in the errors array)
		$view_error_id = filter_input(INPUT_GET, 'view-error', FILTER_VALIDATE_INT);

		if ($view_error_id) {
			$requested_error_in_errors_array = false;

			foreach ($errors as $error) {
				if ((int) $error['error_id'] === $view_error_id) {
					$requested_error_in_errors_array = true;
					break; // Stop looping once found
				}
			}

			if (!$requested_error_in_errors_array) {
				$requested_error = $php_error_log_admin->get_error($view_error_id);
			}
		}

		// Populate the error modals list
		if ( isset( $requested_error ) && $requested_error ) {
			$error_modals = array_merge( $errors, $requested_error);
		} else {
			$error_modals = $errors;
		}

		// Load page parts
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-header.php';
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/log-pages/admin-page-php-error-log.php';
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-footer.php';
		
	}


	/**
	 * Display the plugin's JavaScript error log page
	 *
	 * @since    1.0.0
	 */
	public function admin_page_js_error_log() {
		
		global $plugin_page;
		global $title;

		// Load the JS Errog Log Admin class
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-js-error-log-admin.php';
		$js_error_log_admin = new Logtastic_JS_Error_Log_Admin();

		// Fetch Errors
		$errors = $js_error_log_admin->get_errors();
		
		// If a specific error is passed in the URL, fetch it (if not already in the errors array)
		$view_error_id = filter_input(INPUT_GET, 'view-error', FILTER_VALIDATE_INT);

		if ($view_error_id) {
			$requested_error_in_errors_array = false;

			foreach ($errors as $error) {
				if ((int) $error['error_id'] === $view_error_id) {
					$requested_error_in_errors_array = true;
					break; // Stop looping once found
				}
			}

			if (!$requested_error_in_errors_array) {
				$requested_error = $js_error_log_admin->get_error($view_error_id);
			}
		}

		// Populate the error modals list
		if ( isset( $requested_error ) && $requested_error ) {
			$error_modals = array_merge( $errors, $requested_error);
		} else {
			$error_modals = $errors;
		}

		// Load page parts
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-header.php';
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/log-pages/admin-page-js-error-log.php';
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-footer.php';
		
	}


	/**
	 * Display the plugin's settings page header and footer, and call the relevant settings page function based on the 'tab' get variable
	 *
	 * @since    1.0.0
	 */
	public function admin_page_settings() {
		
		global $plugin_page;
		global $title;
		
		// Load admin page header
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-header.php';

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- $_GET['tab'] used to display relevant admin content, Nonce verification not required

		// Display the relevant settings page
		if ( isset( $_GET['tab'] ) && $_GET['tab'] == 'php-error-log' ) {
			$this->admin_page_settings_php_error_log();
		} else if ( isset( $_GET['tab'] ) && $_GET['tab'] == 'js-error-log' ) {
			$this->admin_page_settings_js_error_log();
		} else if ( isset( $_GET['tab'] ) && $_GET['tab'] == 'security' ) {
			$this->admin_page_settings_security();
		} else if ( isset( $_GET['tab'] ) && $_GET['tab'] == 'general' ) {
			$this->admin_page_settings_general();
		} else {
			$this->admin_page_settings_enable_disable_logs();
		}

		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		// Load admin page footer
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-footer.php';
		
	}


	/**
	 * Display the plugin's general settings page content
	 *
	 * @since    1.0.0
	 */
	public function admin_page_settings_general() {

		// Set active settings tab
		$active_settings_tab = 'general';

		// Load the settings tabs
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-settings-tabs.php';

		// Generate Nonce for saving updates
		$settings_nonce = wp_create_nonce( 'logtastic-admin-general-update-settings' );

		// Possibly update options
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in $this->admin_page_settings_security_update_options() function
		if ( isset( $_POST['action'] ) && 'update-settings' == $_POST['action'] ) {
			$this->settings = $this->admin_page_settings_general_update_options( $this->settings );
		}
		
		// Load the page content
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/settings-pages/admin-page-settings-general.php';

	}


	/**
	 * Update the General settings/options
	 *
	 * @since    1.0.0
	 */
	private function admin_page_settings_general_update_options( $settings ) {

		// Check for valid nonce
		if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'logtastic-admin-general-update-settings' ) ) {

			// Check user has necessary permissions to edit settings
			if ( current_user_can('read') && in_array( 'plugin-settings', $this->access_permissions ) ) {

				// Store copy of original settings
				$settings_original = $settings;
				
				// Update settings value for loading_method and adjust plugin functionality accordingly
				if ( isset( $_POST['loading_method'] ) ) {

					$loading_method = sanitize_text_field( wp_unslash( $_POST['loading_method'] ) );

					// If loading method if set to standard
					if ( 'standard' == $loading_method ) {

						// Load MU Loader Utilities class
						require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/utilities/class-logtastic-mu-loader-utilities.php';
						$mu_loader_utilities = new Logtastic_MU_Loader_Utilities();

						// Remove MU loader file if exists
						$remove_mu_loader = $mu_loader_utilities->delete_mu_loader( LOGTASTIC_PLUGIN_MU_LOADER_FILE_NAME );

						// If file successfully removed, or doesn't exist, update $settings
						if ( $remove_mu_loader ) {

							// Update settings array
							$settings['loading_method'] = 'standard';

						// If failed to remove file
						} else {

							// Update settings array
							$settings['loading_method'] = 'priority';

							// Show error message
							$this->admin_page_notice( 'error', __( 'Error removing MU loader file', 'logtastic' ), true, true);

						}

					// If loading method if set to priority
					} else if ( 'priority' == $loading_method ) {

						// Load MU Loader Utilities class
						require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/utilities/class-logtastic-mu-loader-utilities.php';
						$mu_loader_utilities = new Logtastic_MU_Loader_Utilities();

						// Create MU loader file if doesn't already exist
						$mu_loader_file = $mu_loader_utilities->create_mu_loader( LOGTASTIC_PLUGIN_MU_LOADER_FILE_NAME, $mu_loader_utilities->build_loader_content( LOGTASTIC_PLUGIN_SLUG . '/' . LOGTASTIC_PLUGIN_FILE_NAME ) );

						// If file successfully created, or already exists, update $settings
						if ( $mu_loader_file ) {

							// Update settings array
							$settings['loading_method'] = 'priority';
						
						// If failed to create file
						} else {

							// Update settings array
							$settings['loading_method'] = 'standard';

							// Show error message
							$this->admin_page_notice( 'error', __( 'Error creating MU loader file', 'logtastic' ), true, true);
						
						}

					// Invalid value
					} else {
						$this->admin_page_notice( 'error', __( 'Invalid value supplied for plugin loading method', 'logtastic' ), true, true);
					}

				}

				// Check if options being saved are the same as current db entry
				if ( $settings == $settings_original ) {
					
					// Show message
					$this->admin_page_notice( 'success', __( 'No changes to save', 'logtastic' ), true, true);
					
				} else {
				
					// Update option in database
					$update_option = update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME , $settings, true );
					
					// Show success / fail message
					if ( true == $update_option ) {
						// Success message
						$this->admin_page_notice( 'success', __( 'Settings updated', 'logtastic' ), true, true);
					} else {
						// Error, error saving to database
						$this->admin_page_notice( 'error', __( 'Error saving updated settings to database, please try again', 'logtastic' ), true, true);
					}
					
				}

			} else {

				$this->admin_page_notice( 'error', __( 'Insufficient user permissions', 'logtastic' ), true, true);

			}

		} else {

			// Error, invalid Nonce
			$this->admin_page_notice( 'error', __( 'Invalid Nonce', 'logtastic' ), true, true);

		}

		return $settings;

	}


	/**
	 * Display the plugin's enable/disable logs page content
	 *
	 * @since    1.0.0
	 */
	public function admin_page_settings_enable_disable_logs() {

		// Set active settings tab
		$active_settings_tab = 'enable-disable-logs';

		// Load the settings tabs
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-settings-tabs.php';
		
		// Load the page content
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/settings-pages/admin-page-settings-enable-disable-logs.php';

	}


	/**
	 * Display the plugin's php error log settings page content
	 *
	 * @since    1.0.0
	 */
	public function admin_page_settings_php_error_log() {

		// Set active settings tab
		$active_settings_tab = 'php-error-log';

		// Load the settings tabs
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-settings-tabs.php';
		
		// Get All Plugins
		$all_plugins = get_plugins();
		
		// Get All Themes
		$all_themes = wp_get_themes();
		
		// Get Ignored Errors
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-php-error-log-admin.php';
		$ignored_errors = Logtastic_PHP_Error_Log_Admin::get_ignored_errors();	
		
		// Generate Nonce for saving updates
		$settings_nonce = wp_create_nonce( 'logtastic-admin-php-error-log-update-settings' );
		
		// Generate Nonce for unignoring errors
		$unignore_nonce = wp_create_nonce( 'logtastic-admin-php-error-log-unignore-error' );
		
		// Get the existing options
		$log_settings_array = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_php_error_log');
		
		// Possibly update options
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in $this->admin_page_settings_php_error_log_update_options() function
		if ( isset( $_POST['action'] ) && 'update-settings' == $_POST['action'] ) {
			$log_settings_array = $this->admin_page_settings_php_error_log_update_options( $log_settings_array, $all_plugins, $all_themes );
		}

		if ( !isset( $this->settings['loading_method'] ) || ( isset( $this->settings['loading_method'] ) && $this->settings['loading_method'] != 'priority' ) ) {
			$this->admin_page_notice( 'warning', __( 'To improve the accuracy of the PHP Error Log, please enable <strong>priority loading</strong> from the plugins General Settings tab.', 'logtastic' ), true, true);
		}
		
		// Load settings page HTML
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/settings-pages/admin-page-settings-php-error-log.php';

	}


	/**
	 * Update the PHP error log settings/options
	 *
	 * @since    1.0.0
	 */
	private function admin_page_settings_php_error_log_update_options( $log_settings_array, $all_plugins, $all_themes ) {

		// Check for valid nonce
		if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'logtastic-admin-php-error-log-update-settings' ) ) {

			// Check user has necessary permissions to edit settings
			if ( current_user_can('read') && in_array( 'plugin-settings', $this->access_permissions ) ) {

				// Store copy of original log settings
				$log_settings_array_original = $log_settings_array;
					
				// Update settings value for error levels
				if ( isset( $_POST['php_error_levels'] ) && is_array( $_POST['php_error_levels'] ) ) {
					$allowed_values = [ 'e_error', 'e_warning', 'e_parse', 'e_notice', 'e_core_error', 'e_core_warning', 'e_compile_error', 'e_compile_warning', 'e_user_error', 'e_user_warning', 'e_user_notice', 'e_recoverable_error', 'e_deprecated', 'e_user_deprecated' ];
					$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['php_error_levels'] ) );
					$log_settings_array['error_levels'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
				} else {
					$log_settings_array['error_levels'] = array();
				}
				
				// Update settings value for scope - WP Core
				if ( isset( $_POST['php_error_scope_wp_core'] ) ) { 
					$log_settings_array['scope_wp_core'] = true;
				} else {
					$log_settings_array['scope_wp_core'] = false;
				}
				
				// Update settings value for scope - WP Themes
				if ( isset( $_POST['php_error_scope_wp_themes'] ) ) { 
					if ( isset( $_POST['php_error_scope_wp_themes_definition'] ) ) {
						if ( $_POST['php_error_scope_wp_themes_definition'] == 'all' ) {
							$log_settings_array['scope_wp_themes'] = 'all';
						} else if ( $_POST['php_error_scope_wp_themes_definition'] == 'all_except' ) {
							$log_settings_array['scope_wp_themes'] = 'all_except';
							// record selected...
							if ( isset( $_POST['php_error_scope_select_themes_excepted'] ) && is_array( $_POST['php_error_scope_select_themes_excepted'] ) ) {
								// Create array of allowed theme slugs
								$allowed_values = array_keys($all_themes);
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['php_error_scope_select_themes_excepted'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_themes_excepted'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_themes_excepted'] = array();
							}
						} else if ( $_POST['php_error_scope_wp_themes_definition'] == 'selected' ) {
							$log_settings_array['scope_wp_themes'] = 'selected';
							// record selected...
							if ( isset( $_POST['php_error_scope_select_themes_selected'] ) && is_array( $_POST['php_error_scope_select_themes_selected'] ) ) {
								// Create array of allowed theme slugs
								$allowed_values = array_keys($all_themes);
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['php_error_scope_select_themes_selected'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_themes_selected'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_themes_selected'] = array();
							}
						}
					} else {
						$log_settings_array['scope_wp_themes'] = false;
					}
				} else {
					$log_settings_array['scope_wp_themes'] = false;
				}
				
				// Update settings value for scope - WP Plugins
				if ( isset( $_POST['php_error_scope_wp_plugins'] ) ) { 
					if ( isset( $_POST['php_error_scope_wp_plugins_definition'] ) ) {
						if ( $_POST['php_error_scope_wp_plugins_definition'] == 'all' ) {
							$log_settings_array['scope_wp_plugins'] = 'all';
						} else if ( $_POST['php_error_scope_wp_plugins_definition'] == 'all_except' ) {
							$log_settings_array['scope_wp_plugins'] = 'all_except';
							// record selected...
							if ( isset( $_POST['php_error_scope_select_plugins_excepted'] ) && is_array( $_POST['php_error_scope_select_plugins_excepted'] ) ) {
								// Create array of allowed plugin slugs
								$allowed_values = array_map( function( $key ) {
									return explode( '/', $key )[0];
								}, array_keys( $all_plugins ) );
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['php_error_scope_select_plugins_excepted'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_plugins_excepted'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_plugins_excepted'] = array();
							}
						} else if ( $_POST['php_error_scope_wp_plugins_definition'] == 'selected' ) {
							$log_settings_array['scope_wp_plugins'] = 'selected';
							// record selected...
							if ( isset( $_POST['php_error_scope_select_plugins_selected'] ) && is_array( $_POST['php_error_scope_select_plugins_selected'] ) ) {
								// Create array of allowed plugin slugs
								$allowed_values = array_map( function( $key ) {
									return explode( '/', $key )[0];
								}, array_keys( $all_plugins ) );
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['php_error_scope_select_plugins_selected'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_plugins_selected'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_plugins_selected'] = array();
							}
						}
					} else {
						$log_settings_array['scope_wp_plugins'] = false;
					}
				} else {
					$log_settings_array['scope_wp_plugins'] = false;
				}
				
				// Update settings value for log retention period
				if ( isset( $_POST['php_error_log_retention_schedule'] ) && ( $_POST['php_error_log_retention_schedule'] == 0 || $_POST['php_error_log_retention_schedule'] == 7 || $_POST['php_error_log_retention_schedule'] == 30 ) ) { 
					$log_settings_array['retention_schedule'] = absint( wp_unslash( $_POST['php_error_log_retention_schedule'] ) );
				}
				
				// Update settings value for default error logging
				if ( isset( $_POST['php_error_log_disable_default_error_handler'] ) && ( $_POST['php_error_log_disable_default_error_handler'] == 0 || $_POST['php_error_log_disable_default_error_handler'] == 1 ) ) { 
					$log_settings_array['disable_default_error_handler'] = absint( wp_unslash( $_POST['php_error_log_disable_default_error_handler'] ) );
				}
				
				// Update settings for capture stack trace
				if ( isset( $_POST['php_error_log_capture_stacktrace'] ) && ( $_POST['php_error_log_capture_stacktrace'] == 0 || $_POST['php_error_log_capture_stacktrace'] == 1 ) ) { 
					$log_settings_array['capture_stacktrace'] = absint( wp_unslash( $_POST['php_error_log_capture_stacktrace'] ) );
				}
				
				// Update settings for capture stack trace arguments
				if ( isset( $_POST['php_error_log_capture_stacktrace_args'] ) && ( $_POST['php_error_log_capture_stacktrace_args'] == 0 || $_POST['php_error_log_capture_stacktrace_args'] == 1 ) ) { 
					$log_settings_array['capture_stacktrace_args'] = absint( wp_unslash( $_POST['php_error_log_capture_stacktrace_args'] ) );
				}
				
				// Check if options being saved are the same as current db entry
				if ( $log_settings_array == $log_settings_array_original ) {
					
					// Show message
					$this->admin_page_notice( 'success', __( 'No changes to save', 'logtastic' ), true, true);
					
				} else {
				
					// Update option in database
					$update_option = update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_php_error_log', $log_settings_array, true );
					
					// Show success / fail message
					if ( true == $update_option ) {
						// Success message
						$this->admin_page_notice( 'success', __( 'Settings updated', 'logtastic' ), true, true);
					} else {
						// Error, error saving to database
						$this->admin_page_notice( 'error', __( 'Error saving updated settings to database, please try again', 'logtastic' ), true, true);
					}
					
				}

			} else {

				$this->admin_page_notice( 'error', __( 'Insufficient user permissions', 'logtastic' ), true, true);

			}
			
		} else {

			// Error, invalid Nonce
			$this->admin_page_notice( 'error', __( 'Invalid Nonce', 'logtastic' ), true, true);

		}

		return $log_settings_array;

	}

	/**
	 * Display the plugin's JS error log settings page content
	 *
	 * @since    1.0.0
	 */
	public function admin_page_settings_js_error_log() {

		// Set active settings tab
		$active_settings_tab = 'js-error-log';
		
		// Load the settings tabs
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-settings-tabs.php';
		
		// Get All Plugins
		$all_plugins = get_plugins();
		
		// Get All Themes
		$all_themes = wp_get_themes();
		
		// Get Ignored Errors
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-js-error-log-admin.php';
		$ignored_errors = Logtastic_JS_Error_Log_Admin::get_ignored_errors();
			
		// Generate Nonce for saving updates
		$settings_nonce = wp_create_nonce( 'logtastic-admin-js-error-log-update-settings' );
		
		// Generate Nonce for unignoring errors
		$unignore_nonce = wp_create_nonce( 'logtastic-admin-js-error-log-unignore-error' );
		
		// Get the existing options
		$log_settings_from_db = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_js_error_log');
		
		// Deserialize the options
		$log_settings_array = maybe_unserialize( $log_settings_from_db );
		
		// Possibly update options
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in $this->admin_page_settings_php_error_log_update_options() function
		if ( isset( $_POST['action'] ) && 'update-settings' == $_POST['action'] ) {
			$log_settings_array = $this->admin_page_settings_js_error_log_update_options( $log_settings_array, $all_plugins, $all_themes );
		}
		
		// Load settings page HTML
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/settings-pages/admin-page-settings-js-error-log.php';

	}


	/**
	 * Update the JS error log settings/options
	 *
	 * @since    1.0.0
	 */
	private function admin_page_settings_js_error_log_update_options( $log_settings_array, $all_plugins, $all_themes ) {

		// Store copy of original log settings
		$log_settings_array_original = $log_settings_array;

		// Check for valid nonce
		if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'logtastic-admin-js-error-log-update-settings' ) ) {	

			// Check user has necessary permissions to edit settings
			if ( current_user_can('read') && in_array( 'plugin-settings', $this->access_permissions ) ) {

				// Update settings value for error types
				if ( isset( $_POST['js_error_types'] ) && is_array( $_POST['js_error_types'] ) ) {
					$allowed_values = [ 1, 2 ];
					$submitted_values_sanitized = array_map( 'intval', $_POST['js_error_types'] );
					$log_settings_array['error_types'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
				} else {
					$log_settings_array['error_types'] = array();
				}
				
				// Update settings value for scope - WP Core
				if ( isset( $_POST['js_error_scope_wp_core'] ) ) { 
					$log_settings_array['scope_wp_core'] = true;
				} else {
					$log_settings_array['scope_wp_core'] = false;
				}
				
				// Update settings value for scope - WP Themes
				if ( isset( $_POST['js_error_scope_wp_themes'] ) ) { 
					if ( isset( $_POST['js_error_scope_wp_themes_definition'] ) ) {
						if ( $_POST['js_error_scope_wp_themes_definition'] == 'all' ) {
							$log_settings_array['scope_wp_themes'] = 'all';
						} else if ( $_POST['js_error_scope_wp_themes_definition'] == 'all_except' ) {
							$log_settings_array['scope_wp_themes'] = 'all_except';
							// record selected...
							if ( isset( $_POST['js_error_scope_select_themes_excepted'] ) && is_array( $_POST['js_error_scope_select_themes_excepted'] ) ) {
								// Create array of allowed theme slugs
								$allowed_values = array_keys($all_themes);
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['js_error_scope_select_themes_excepted'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_themes_excepted'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_themes_excepted'] = array();
							}
						} else if ( $_POST['js_error_scope_wp_themes_definition'] == 'selected' ) {
							$log_settings_array['scope_wp_themes'] = 'selected';
							// record selected...
							if ( isset( $_POST['js_error_scope_select_themes_selected'] ) && is_array( $_POST['js_error_scope_select_themes_selected'] ) ) {
								// Create array of allowed theme slugs
								$allowed_values = array_keys($all_themes);
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['js_error_scope_select_themes_selected'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_themes_selected'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_themes_selected'] = array();
							}
						}
					} else {
						$log_settings_array['scope_wp_themes'] = false;
					}
				} else {
					$log_settings_array['scope_wp_themes'] = false;
				}
				
				// Update settings value for scope - WP Plugins
				if ( isset( $_POST['js_error_scope_wp_plugins'] ) ) { 
					if ( isset( $_POST['js_error_scope_wp_plugins_definition'] ) ) {
						if ( $_POST['js_error_scope_wp_plugins_definition'] == 'all' ) {
							$log_settings_array['scope_wp_plugins'] = 'all';
						} else if ( $_POST['js_error_scope_wp_plugins_definition'] == 'all_except' ) {
							$log_settings_array['scope_wp_plugins'] = 'all_except';
							// record selected...
							if ( isset( $_POST['js_error_scope_select_plugins_excepted'] ) && is_array( $_POST['js_error_scope_select_plugins_excepted'] ) ) {
								// Create array of allowed plugin slugs
								$allowed_values = array_map( function( $key ) {
									return explode( '/', $key )[0];
								}, array_keys( $all_plugins ) );
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['js_error_scope_select_plugins_excepted'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_plugins_excepted'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_plugins_excepted'] = array();
							}
						} else if ( $_POST['js_error_scope_wp_plugins_definition'] == 'selected' ) {
							$log_settings_array['scope_wp_plugins'] = 'selected';
							// record selected...
							if ( isset( $_POST['js_error_scope_select_plugins_selected'] ) && is_array( $_POST['js_error_scope_select_plugins_selected'] ) ) {
								// Create array of allowed plugin slugs
								$allowed_values = array_map( function( $key ) {
									return explode( '/', $key )[0];
								}, array_keys( $all_plugins ) );
								// Sanitize submitted values
								$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['js_error_scope_select_plugins_selected'] ) );
								// Record submitted allowed values
								$log_settings_array['scope_wp_plugins_selected'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );
							} else {
								$log_settings_array['scope_wp_plugins_selected'] = array();
							}
						}
					} else {
						$log_settings_array['scope_wp_plugins'] = false;
					}
				} else {
					$log_settings_array['scope_wp_plugins'] = false;
				}
				
				// Update settings value for log retention period
				if ( isset( $_POST['js_error_log_retention_schedule'] ) && ( $_POST['js_error_log_retention_schedule'] == 0 || $_POST['js_error_log_retention_schedule'] == 7 || $_POST['js_error_log_retention_schedule'] == 30 ) ) { 
					$log_settings_array['retention_schedule'] = absint( wp_unslash( $_POST['js_error_log_retention_schedule'] ) );
				}
				
				// Update settings for capture stack trace
				if ( isset( $_POST['js_error_log_capture_stacktrace'] ) && ( $_POST['js_error_log_capture_stacktrace'] == 0 || $_POST['js_error_log_capture_stacktrace'] == 1 ) ) { 
					$log_settings_array['capture_stacktrace'] = absint( wp_unslash( $_POST['js_error_log_capture_stacktrace'] ) );
				}
				
				// Update settings for capture session info
				if ( isset( $_POST['js_error_log_capture_session_info'] ) && ( $_POST['js_error_log_capture_session_info'] == 0 || $_POST['js_error_log_capture_session_info'] == 1 ) ) { 
					$log_settings_array['capture_session_info'] = absint( wp_unslash( $_POST['js_error_log_capture_session_info'] ) );
				}
				
				// Check if options being saved are the same as current db entry
				if ( $log_settings_array == $log_settings_array_original ) {
					
					// Show message
					$this->admin_page_notice( 'success', __( 'No changes to save', 'logtastic' ), true, true);
					
				} else {
					
					// Update option in database
					$update_option = update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_js_error_log', $log_settings_array, true );
					
					// Show success / fail message
					if ( true == $update_option ) {
						// Success message
						$this->admin_page_notice( 'success', __( 'Settings updated', 'logtastic' ), true, true);
					} else {
						// Error, error saving to database
						$this->admin_page_notice( 'error', __( 'Error saving updated settings to database, please try again', 'logtastic' ), true, true);
					}
					
				}

			} else {

				$this->admin_page_notice( 'error', __( 'Insufficient user permissions', 'logtastic' ), true, true);

			}
			
		} else {
			// Error, invalid Nonce
			$this->admin_page_notice( 'error', __( 'Invalid Nonce', 'logtastic' ), true, true);
		}

		return $log_settings_array;

	}

	
	/**
	 * Display the plugin's Help & Support admin page
	 *
	 * @since    1.0.0
	 */
	public function admin_page_help() {
		
		global $plugin_page;
		global $title;

		// Load admin page header
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-header.php';

		// Load page content
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/general-pages/admin-page-help.php';
		
		// Load admin page footer
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-footer.php';
		
	}


	/**
	 * Display the plugin's Security & Visibility settings page content
	 *
	 * @since    1.0.0
	 */
	public function admin_page_settings_security() {
		

		// Set active settings tab
		$active_settings_tab = 'security';

		// Load the settings tabs
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-layout/admin-page-settings-tabs.php';

		// Get array of editable roles
		$roles = get_editable_roles();

		// Filter to include only roles with the 'read' capability
		$roles = array_filter(
			$roles,
			function( $role ) {
				return ! empty( $role['capabilities']['read'] );
			}
		);

		// Generate Nonce for saving updates
		$settings_nonce = wp_create_nonce( 'logtastic-admin-security-update-settings' );

		// Possibly update options
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in $this->admin_page_settings_security_update_options() function
		if ( isset( $_POST['action'] ) && 'update-settings' == $_POST['action'] ) {
			$this->settings = $this->admin_page_settings_security_update_options( $this->settings, $roles );
		}
		
		// Load page content
		require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/partials/admin-pages/settings-pages/admin-page-settings-security_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';
		
	}


	/**
	 * Update the Security & Visibility settings/options
	 *
	 * @since    1.0.0
	 */
	private function admin_page_settings_security_update_options( $settings, $roles ) {

		// Check for valid nonce
		if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'logtastic-admin-security-update-settings' ) ) {

			// Check user has necessary permissions to edit settings
			if ( current_user_can('read') && in_array( 'plugin-settings', $this->access_permissions ) ) {

				// Store copy of original settings
				$settings_original = $settings;
				
				// Update settings value for access_roles
				if ( isset( $_POST['access_roles'] ) && is_array( $_POST['access_roles'] ) ) {

					$allowed_values = array_keys( $roles );

					$submitted_values_sanitized = array_map( 'sanitize_text_field', wp_unslash( $_POST['access_roles'] ) );

					$settings['access_roles'] = array_values( array_intersect( $submitted_values_sanitized, $allowed_values ) );

				} else {
					
					$this->admin_page_notice( 'error', __( 'You must select at least one role.', 'logtastic' ), true, true);

					return $settings;

				}

				// Check if options being saved are the same as current db entry
				if ( $settings == $settings_original ) {
					
					// Show message
					$this->admin_page_notice( 'success', __( 'No changes to save', 'logtastic' ), true, true);
					
				} else {
				
					// Update option in database
					$update_option = update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME , $settings, true );
					
					// Show success / fail message
					if ( true == $update_option ) {
						// Success message
						$this->admin_page_notice( 'success', __( 'Settings updated', 'logtastic' ), true, true);
					} else {
						// Error, error saving to database
						$this->admin_page_notice( 'error', __( 'Error saving updated settings to database, please try again', 'logtastic' ), true, true);
					}
					
				}

			} else {

				$this->admin_page_notice( 'error', __( 'Insufficient user permissions', 'logtastic' ), true, true);

			}

		} else {

			// Error, invalid Nonce
			$this->admin_page_notice( 'error', __( 'Invalid Nonce', 'logtastic' ), true, true);

		}

		return $settings;

	}
	
	
	/**
	 * Enable or disable a log via ajax and send a JSON response.
	 *
	 * @since    1.0.0
	 */
	public function enable_disable_log_via_ajax() {
		
		// Verify nonce
		check_ajax_referer('enable-disable-log');

		// Check user has necessary permissions to edit settings
		if ( current_user_can('read') && in_array( 'plugin-settings', $this->access_permissions ) ) {
		
			// Get the updates from the AJAX request
			if ( empty( $_REQUEST ) || !isset( $_REQUEST['log'] ) || empty( $_REQUEST['log'] ) || !isset( $_REQUEST['status'] ) || empty( $_REQUEST['status'] ) ) {
				wp_send_json_error( __( 'Missing request data', 'logtastic' ) );
			}
			
			// Get the existing options
			$options_array = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME );
			
			if ($options_array === false) {
				wp_send_json_error( __( 'Plugin options not found', 'logtastic' ) );
				return;
			}
			
			// Sanitise request vars
			$log = sanitize_text_field( wp_unslash( $_REQUEST['log'] ) );
			if ( is_bool( $_REQUEST['status'] ) || $_REQUEST['status'] === 'true' || $_REQUEST['status'] === 'false' ) {
				$status = filter_var( wp_unslash( $_REQUEST['status'] ), FILTER_VALIDATE_BOOLEAN );
			} else {
				wp_send_json_error( __( 'Invalid value for status', 'logtastic' ) );
				return;
			}

			// Update the option in array
			$options_array['enabled_logs'][$log] = $status;
			
			// Save updated options
			$updated = update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME, $options_array, true );
			
			// Check if options exist for the specific log, if not set default options
			switch( $log ) {
				
				case 'php-error-log':
					$log_settings = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_php_error_log');
					if ( false == $log_settings ) {
						// Load default settings
						require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/default-settings/class-logtastic-default-settings_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';
						$log_settings_array = Logtastic_Default_Settings::$default_settings_php_error_log;
						// Set default options
						$update_option = update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_php_error_log', $log_settings_array, true );
					}
					break;
				
				case 'js-error-log':
					$log_settings = get_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_js_error_log');
					if ( false == $log_settings ) {
						// Load default settings
						require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/default-settings/class-logtastic-default-settings_' . LOGTASTIC_PLUGIN_FEATURE_VERSION . '.php';
						$log_settings_array = Logtastic_Default_Settings::$default_settings_js_error_log;
						// Set default options
						$update_option = update_option( LOGTASTIC_PLUGIN_OPTIONS_NAME . '_js_error_log', $log_settings_array, true );
					}
					break;
				
			}

			// Create or update database tables for the activated log
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'includes/utilities/class-logtastic-db-utilities.php';
			$db_utilities = new Logtastic_DB_Utilities( $options_array );
			$db_utilities->check_and_update_db_structure_single_log( $log );
			
			// Send success or failure message
			if ( $updated ) {
				wp_send_json_success( __( 'Option updated successfully', 'logtastic' ) );
				return;
			} else {
				wp_send_json_error( __( 'Failed to update option', 'logtastic' ) );
				return;
			}

		} else {

			// Send JSON error response
			wp_send_json_error( __( 'Insufficient user permissions', 'logtastic' ) );

		}


	}


	/**
	 * Load php error details via ajax
	 *
	 * @since    1.0.0
	 */
	public function ajax_load_php_error_details() {

		// Verify nonce
		check_ajax_referer('logtastic-admin-php-error-log-load-error-data');

		// Check user has necessary permissions to view log data
		if ( current_user_can('read') && in_array( 'php-error-log-view', $this->access_permissions ) ) {

			// Return error details
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-php-error-log-admin.php';
			return Logtastic_PHP_Error_Log_Admin::ajax_load_php_error_details();

		} else {

			// Send JSON error response
			wp_send_json_error( __( 'Insufficient user permissions', 'logtastic' ) );

		}

	}
	
	
	/**
	 * Load php error stack trace via ajax
	 *
	 * @since    1.0.0
	 */
	public function ajax_load_php_error_stack_trace() {

		// Verify nonce
		check_ajax_referer('logtastic-admin-php-error-log-load-error-data');

		// Check user has necessary permissions to view log data
		if ( current_user_can('read') && in_array( 'php-error-log-view', $this->access_permissions ) ) {

			// Return stack trace data
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-php-error-log-admin.php';
			return Logtastic_PHP_Error_Log_Admin::ajax_load_php_error_stack_trace();

		} else {

			// Send JSON error response
			wp_send_json_error( __( 'Insufficient user permissions', 'logtastic' ) );

		}
		
	}
	

	/**
	 * Unignore php error via ajax
	 *
	 * @since    1.0.0
	 */
	public function ajax_unignore_php_error() {

		// Verify nonce
		check_ajax_referer('logtastic-admin-php-error-log-unignore-error');

		// Check user has necessary permissions to view log data
		if ( current_user_can('read') && in_array( 'php-error-log-edit', $this->access_permissions ) ) {

			// Unignore error
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-php-error-log-admin.php';
			return Logtastic_PHP_Error_Log_Admin::ajax_unignore_php_error();

		} else {

			// Send JSON error response
			wp_send_json_error( __( 'Insufficient user permissions', 'logtastic' ) );

		}
		
	}
	

	/**
	 * Load JS error details via ajax
	 *
	 * @since    1.0.0
	 */
	public function ajax_load_js_error_details() {

		// Verify nonce
		check_ajax_referer('logtastic-admin-js-error-log-load-error-data');

		// Check user has necessary permissions to view log data
		if ( current_user_can('read') && in_array( 'js-error-log-view', $this->access_permissions ) ) {

			// Return error details
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-js-error-log-admin.php';
			return Logtastic_JS_Error_Log_Admin::ajax_load_js_error_details();

		} else {

			// Send JSON error response
			wp_send_json_error( __( 'Insufficient user permissions', 'logtastic' ) );

		}

	}


	/**
	 * Load JS error stack trace via ajax
	 *
	 * @since    1.0.0
	 */
	public function ajax_load_js_error_stack_trace() {

		// Verify nonce
		check_ajax_referer('logtastic-admin-js-error-log-load-error-data');

		// Check user has necessary permissions to view log data
		if ( current_user_can('read') && in_array( 'js-error-log-view', $this->access_permissions ) ) {

			// Return stack trace data
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-js-error-log-admin.php';
			return Logtastic_JS_Error_Log_Admin::ajax_load_js_error_stack_trace();

		} else {

			// Send JSON error response
			wp_send_json_error( __( 'Insufficient user permissions', 'logtastic' ) );

		}

	}
	
	
	/**
	 * Unignore JS error via ajax
	 *
	 * @since    1.0.0
	 */
	public function ajax_unignore_js_error() {

		// Verify nonce
		check_ajax_referer('logtastic-admin-js-error-log-unignore-error');

		// Check user has necessary permissions to edit log data
		if ( current_user_can('read') && in_array( 'js-error-log-edit', $this->access_permissions ) ) {

			// Unignore error
			require_once LOGTASTIC_PLUGIN_DIR_PATH . 'admin/includes/class-logtastic-js-error-log-admin.php';
			return Logtastic_JS_Error_Log_Admin::ajax_unignore_js_error();

		} else {

			// Send JSON error response
			wp_send_json_error( __( 'Insufficient user permissions', 'logtastic' ) );

		}

	}
	

	/**
	 * Takes a mysql data/time and returns the date based on the WP Date Format preferrence 
	 *
	 * @since    1.0.0
	 * @param 	 string		$date    A mysql date/time string
	 * @return 	 string 	The date formatted based on the WP Date Format preferrence 
	 */	
	public function format_date ( $date ) {
		
		if ( null == $date ) {
			return null;
		}
		
		$wp_date_format = get_option( 'date_format' );
		
		$date_object = date_create( $date );
		
		return date_format( $date_object, $wp_date_format );
		
	}
	

	/**
	 * Takes a mysql data/time and returns the time based on the WP Time Format preferrence 
	 *
	 * @since    1.0.0
	 * @param 	 string		$date    A mysql date/time string
	 * @return 	 string 	The time formatted based on the WP Time Format preferrence 
	 */
	public function format_time ( $date ) {
		
		if ( null == $date ) {
			return null;
		}
		
		$wp_time_format = get_option( 'time_format' );
		
		$date_object = date_create( $date );
		
		return date_format( $date_object, $wp_time_format );
		
	}

	
	/**
	 * Takes a mysql data/time and returns the date and time based on the WP Date Format and WP Time Format preferrences, in the format date @ time
	 *
	 * @since    1.0.0
	 * @param 	 string		$date    A mysql date/time string
	 * @return 	 string 	The date and time based on the WP Date Format and WP Time Format preferrences, in the format date @ time
	 */
	public function format_date_and_time ( $date ) {
		
		if ( null == $date ) {
			return null;
		}
		
		$wp_date_format = get_option( 'date_format' );
		
		$wp_time_format = get_option( 'time_format' );
		
		$date_object = date_create( $date );
		
		return date_format( $date_object, $wp_date_format ) . ' @ ' . date_format( $date_object, $wp_time_format );
		
	}


	/**
	 * Checks that the plugin was loaded by the correct method. If a mismatch occurs, an error message is shown.
	 *
	 * @since    1.0.0
	 */
	public function loading_method_error_check () {

		if ( isset( $this->settings['loading_method'] ) && wp_unslash( $this->settings['loading_method'] ) == 'standard' ) {

			if ( defined( __NAMESPACE__ . '\PLUGIN_MU_LOADED' ) ) {

				// Plugin was loaded via MU loader file, but should not have been
				$this->admin_page_notice( 
					'error',
					'<p><strong>' . __( 'Logtastic Loading Error', 'logtastic' ) . '</strong></p><p>' . __( 'Logtastic is set to loading method "standard" however the plugin was loaded via the MU loader file.', 'logtastic' ) . '</p>', 
					false, 
					true
				);

			}

		} else if ( isset( $this->settings['loading_method'] ) && wp_unslash( $this->settings['loading_method'] ) == 'priority' ) {

			if ( !defined( __NAMESPACE__ . '\PLUGIN_MU_LOADED' ) ) {

				// Plugin should have been loaded via MU loader file, but was not
				$this->admin_page_notice( 
					'error',
					'<p><strong>' . __( 'Logtastic Loading Error', 'logtastic' ) . '</strong></p><p>' . __( 'Logtastic is set to loading method "priorty" however the MU loader file could not be loaded.', 'logtastic' ) . '</p>', 
					false, 
					true
				);

			}

		}
		
	}
	
}