<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/*
Plugin Name: Logistos Order Sync
Plugin URI:  https://logistos.in/woocommerce-plugin
Description: Syncs WooCommerce orders to Logistos. Sends WooCommerce order data to a remote endpoint on activation and on each new order.
Version:     Version: 1.7.0
Author:      Logistos
Author URI:  https://logistos.in
License:     GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: logistos
Requires at least: 5.8
Tested up to: 6.8
Requires PHP: 7.4
WC requires at least: 6.0
WC tested up to: 9.1
*/



function logistos_sync_log($msg, $ctx = []) {
    if (function_exists('wc_get_logger')) {
        wc_get_logger()->info(is_string($msg) ? $msg : wp_json_encode($msg), ['source' => 'logistos']);
    } else {
        error_log('[logistos] ' . (is_string($msg) ? $msg : wp_json_encode($msg)));
    }
}



/**
 * Register settings for Logistos endpoints.
 */
add_action( 'admin_init', function () {
    register_setting(
        'logistos_sync',                 // settings group
        'logistos_endpoint_credentials', // option name
        [ 'type' => 'string', 'sanitize_callback' => 'esc_url_raw' ]
    );

    register_setting(
        'logistos_sync',
        'logistos_endpoint_orders',
        [ 'type' => 'string', 'sanitize_callback' => 'esc_url_raw' ]
    );
});

/**
 * Add settings page under "Settings" menu.
 */
add_action( 'admin_menu', function () {
    add_options_page(
        __( 'Logistos – Order Sync', 'logistos' ),
        __( 'Logistos Sync', 'logistos' ),
        'manage_woocommerce',           // capability
        'logistos-sync-settings',       // slug
        'logistos_render_settings_page' // callback
    );
});

/**
 * Render the settings page.
 */
function logistos_render_settings_page() {
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        return;
    }

    $cred_url   = get_option( 'logistos_endpoint_credentials', 'https://admin.logistos.in/integration/integrate_woo_commerce' );
    $orders_url = get_option( 'logistos_endpoint_orders',      'https://admin.logistos.in/integration/woo_commerce_orders' );
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Logistos – Order Sync', 'logistos' ); ?></h1>
        <p><?php esc_html_e( 'Configure the external endpoints that receive credentials and order data from this site.', 'logistos' ); ?></p>

        <form method="post" action="options.php">
            <?php
            settings_fields( 'logistos_sync' );
            do_settings_sections( 'logistos_sync' );
            ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="logistos_endpoint_credentials">
                            <?php esc_html_e( 'Credentials endpoint URL', 'logistos' ); ?>
                        </label>
                    </th>
                    <td>
                        <input type="url" class="regular-text" id="logistos_endpoint_credentials"
                               name="logistos_endpoint_credentials"
                               value="<?php echo esc_attr( $cred_url ); ?>">
                        <p class="description">
                            <?php esc_html_e( 'This endpoint receives WooCommerce REST API keys on plugin activation.', 'logistos' ); ?>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="logistos_endpoint_orders">
                            <?php esc_html_e( 'Orders endpoint URL', 'logistos' ); ?>
                        </label>
                    </th>
                    <td>
                        <input type="url" class="regular-text" id="logistos_endpoint_orders"
                               name="logistos_endpoint_orders"
                               value="<?php echo esc_attr( $orders_url ); ?>">
                        <p class="description">
                            <?php esc_html_e( 'This endpoint receives order data on activation and on each new order.', 'logistos' ); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

// Hook into WooCommerce order creation
add_action('woocommerce_thankyou', 'logistos_send_order_details_to_endpoint', 10, 1);
add_action('woocommerce_checkout_order_created', 'logistos_send_order_details_to_endpoint', 10, 1);
add_action('woocommerce_new_order', function($order_id){ logistos_send_order_details_to_endpoint($order_id); }, 10, 1);

// Send all orders on plugin activation


// ---------- DEFERRED ACTIVATION RUNNER ----------
register_activation_hook(__FILE__, function () {
    update_option('logistos_do_activation_jobs', 1);
});

add_action('woocommerce_init', function () {
    if (get_option('logistos_do_activation_jobs')) {
        logistos_sync_log('Running activation jobs…');
        logistos_generate_woocommerce_api_key();
        logistos_send_all_orders_on_activation();
        delete_option('logistos_do_activation_jobs');
        logistos_sync_log('Activation jobs done.');
    }
});


add_action('logistos_activation_runner', 'logistos_run_activation_jobs');
function logistos_run_activation_jobs() {
    // Make sure WooCommerce is available
    if (!class_exists('WooCommerce')) {
        logistos_sync_log('WooCommerce not loaded during activation runner; skipping.');
        return;
    }

    logistos_sync_log('Running activation jobs…');
    logistos_generate_woocommerce_api_key();     // send keys
    logistos_send_all_orders_on_activation();    // push historical orders
    logistos_sync_log('Activation jobs done.');
}

function logistos_generate_woocommerce_api_key() {
    // Prefer the current user if they can manage WooCommerce
    $user = wp_get_current_user();
    if ( $user && $user->ID && user_can( $user, 'manage_woocommerce' ) ) {
        $user_id = $user->ID;
    } else {
        // Fallback for activation/cron: first admin or shop_manager
        $candidates = get_users( [
            'role__in' => [ 'administrator', 'shop_manager' ],
            'number'   => 1,
            'orderby'  => 'ID',
            'order'    => 'ASC',
            'fields'   => [ 'ID' ],
        ] );
        if ( empty( $candidates ) ) {
            logistos_sync_log( 'logistos_generate_woocommerce_api_key: no admin/shop_manager user exists.' );
            return;
        }
        $user_id = (int) $candidates[0]->ID;
    }

    // Generate plain keys (these are what you will send to your central system)
    $consumer_key_plain    = 'ck_' . ( function_exists( 'wc_rand_hash' ) ? wc_rand_hash() : wp_generate_password( 24, false ) );
    $consumer_secret_plain = 'cs_' . ( function_exists( 'wc_rand_hash' ) ? wc_rand_hash() : wp_generate_password( 24, false ) );

    // Hash the consumer key for storage, WooCommerce will compare hashes on incoming REST API calls
    if ( function_exists( 'wc_api_hash' ) ) {
        $consumer_key_hashed = wc_api_hash( $consumer_key_plain );
    } else {
        $consumer_key_hashed = hash_hmac( 'sha256', $consumer_key_plain, wp_salt() );
    }

    global $wpdb;
    $table = $wpdb->prefix . 'woocommerce_api_keys';

    $data = [
        'user_id'        => $user_id,
        'description'    => 'API key for logistos plugin integration',
        'permissions'    => 'read_write',
        'consumer_key'   => $consumer_key_hashed,
        'consumer_secret'=> $consumer_secret_plain,
        'truncated_key'  => substr( $consumer_key_plain, -7 ),
        'last_access'    => null,
    ];

    $ok = $wpdb->insert( $table, $data );

    if ( ! $ok ) {
        logistos_sync_log( 'API key DB insert FAILED for user ' . $user_id );
        return;
    }

    logistos_sync_log( 'API key row insert: 1 for user ' . $user_id );

    // Now send the *plain* keys to your central system
    logistos_send_api_credentials_to_central_system(
        $consumer_key_plain,
        $consumer_secret_plain,
        home_url()
    );
}


function logistos_send_api_credentials_to_central_system($consumer_key, $consumer_secret, $store_url) {
    $api_data = [
        'consumer_key'    => $consumer_key,
        'consumer_secret' => $consumer_secret,
        'store_url'       => $store_url,
    ];

    // Use your ngrok or server URL
   $endpoint_url = get_option( 'logistos_endpoint_credentials' );
// if not set, you can optionally fallback or just abort:
if ( empty( $endpoint_url ) ) {
    logistos_sync_log( 'Credentials endpoint not configured; aborting send.' );
    return;
}


    $response = wp_remote_post($endpoint_url, [
        'timeout' => 20,
        'headers' => ['Content-Type' => 'application/json'],
        'body'    => wp_json_encode($api_data),
    ]);

    if (is_wp_error($response)) {
        logistos_sync_log('logistos_send_api_credentials_to_central_system HTTP error: ' . $response->get_error_message());
    } else {
        logistos_sync_log(['keys_post_code' => wp_remote_retrieve_response_code($response)]);
    }
}






function logistos_send_all_orders_on_activation() {
    if (!function_exists('wc_get_orders')) { logistos_sync_log('wc_get_orders not available; abort.'); return; }

    try {
        $orders = wc_get_orders([
            'limit'   => 20,
            'status'  => ['wc-completed','wc-processing','wc-pending','wc-on-hold'],
            'orderby' => 'date',
            'order'   => 'DESC',
            'return'  => 'objects',
        ]);
    } catch (Exception $e) {
        logistos_sync_log('wc_get_orders() failed: ' . $e->getMessage());
        return;
    }

    foreach ($orders as $o) {
        if (!is_a($o, 'WC_Order')) { logistos_sync_log('Skipping invalid order entry.'); continue; }
        logistos_send_order_to_external_api($o);
    }
}


/**
 * Function to send an order's details to the external API
 */
/** ---------- Fire per-order ---------- */
function logistos_send_order_details_to_endpoint($order_id) {
    if (!$order_id) return;
    $order = wc_get_order($order_id);
    if (!$order) { logistos_sync_log('logistos_send_order_details_to_endpoint: order not found'); return; }
    logistos_send_order_to_external_api($order);
}

/**
 * Helper function to send an order's data to an external API
 */
/** ---------- Outbound order POST ---------- */
function logistos_get_state_name($country_code, $state_code) {
    $states = WC()->countries->get_states($country_code);
    if (isset($states[$state_code])) {
        return $states[$state_code];
    }
    return $state_code; // fallback to code if not found
}
function logistos_send_order_to_external_api( $order ) {

    // Units from WooCommerce settings
    $weight_unit    = get_option( 'woocommerce_weight_unit', 'kg' );
    $dimension_unit = get_option( 'woocommerce_dimension_unit', 'cm' );

    $items         = [];
    $total_weight  = 0;

    foreach ( $order->get_items() as $item_id => $item ) {
        $product = $item->get_product();

        // Safe defaults
        $weight = $product ? (float) $product->get_weight()  : 0;
        $length = $product ? (float) $product->get_length()  : 0;
        $width  = $product ? (float) $product->get_width()   : 0;
        $height = $product ? (float) $product->get_height()  : 0;

        $qty = (float) $item->get_quantity();
        $total_weight += $weight * $qty;

        $items[] = [
            'item_id'       => (int) $item_id,
            'product_id'    => $product ? (int) $product->get_id() : null,
            'name'          => $item->get_name(),
            'sku'           => $product ? $product->get_sku() : null,
            'qty'           => $qty,
            'total'         => (float) $item->get_total(),
            'subtotal'      => (float) $item->get_subtotal(),
            'tax'           => (float) $item->get_total_tax(),

            // 🔹 dimensions & weight
            'weight'        => $weight,
            'length'        => $length,
            'width'         => $width,
            'height'        => $height,
            'weight_unit'   => $weight_unit,
            'dim_unit'      => $dimension_unit,
        ];
    }
// Store base / pickup address (from WooCommerce settings)
$base_country  = WC()->countries->get_base_country();
$base_state    = WC()->countries->get_base_state();
$base_postcode = WC()->countries->get_base_postcode();
$base_city     = WC()->countries->get_base_city();

$store_address   = get_option( 'woocommerce_store_address' );
$store_address_2 = get_option( 'woocommerce_store_address_2' );

    $order_data = [
        'id'         => (int) $order->get_id(),
        'number'     => $order->get_order_number(),
        'status'     => $order->get_status(),
        'currency'   => $order->get_currency(),
        'total'      => (float) $order->get_total(),
        'discount_total' => (float) $order->get_discount_total(), // optional but useful
        'shipping_total' => (float) $order->get_shipping_total(), // optional but useful
        'total_tax'      => (float) $order->get_total_tax(),      // optional but useful

        'created_at' => $order->get_date_created()
            ? $order->get_date_created()->date( DATE_ATOM )
            : null,

        // 🔹 order-level weight summary
        'total_weight'   => $total_weight,
        'weight_unit'    => $weight_unit,
        'pickup'     => [
                'name'      => get_bloginfo( 'name' ),
                'phone'     => get_option( 'woocommerce_store_phone', '' ), // if you store it as an option
                'address_1' => $store_address,
                'address_2' => $store_address_2,
                'city'      => $base_city,
                'state'     => logistos_get_state_name( $base_country, $base_state ),
                'postcode'  => $base_postcode,
                'country'   => $base_country,
            ],
        'billing'    => [
            'first_name' => $order->get_billing_first_name(),
            'last_name'  => $order->get_billing_last_name(),
            'email'      => $order->get_billing_email(),
            'state'      => logistos_get_state_name( $order->get_billing_country(), $order->get_billing_state() ),
            'phone'      => $order->get_billing_phone(),
            'city'       => $order->get_billing_city(),
            'country'    => $order->get_billing_country(),
            'postcode'   => $order->get_billing_postcode(),
            'address_1'  => $order->get_billing_address_1(),
            'address_2'  => $order->get_billing_address_2(),
        ],

        // 🔹 shipping address (very important for courier side)
        'shipping'   => [
            'first_name' => $order->get_shipping_first_name(),
            'last_name'  => $order->get_shipping_last_name(),
            'phone'      => $order->get_billing_phone(), // Woo has no native shipping phone
            'state'      => logistos_get_state_name( $order->get_shipping_country(), $order->get_shipping_state() ),
            'city'       => $order->get_shipping_city(),
            'country'    => $order->get_shipping_country(),
            'postcode'   => $order->get_shipping_postcode(),
            'address_1'  => $order->get_shipping_address_1(),
            'address_2'  => $order->get_shipping_address_2(),
        ],

        'payment_method'      => $order->get_payment_method(),
        'payment_method_title'=> $order->get_payment_method_title(),

        'items'      => $items,

        'store'      => [
            'site_url'  => get_bloginfo( 'url' ),
            'site_name' => get_bloginfo( 'name' ),
        ],
    ];

    // Endpoint
    $endpoint_url = get_option( 'logistos_endpoint_orders' );
    if ( empty( $endpoint_url ) ) {
        $endpoint_url = 'https://admin.logistos.in/channel/woo_commerce_orders/';
        logistos_sync_log( 'Orders endpoint empty in DB; using default: ' . $endpoint_url );
    }

    $res = wp_remote_post(
        $endpoint_url,
        [
            'timeout' => 25,
            'headers' => [ 'Content-Type' => 'application/json' ],
            'body'    => wp_json_encode( $order_data ),
        ]
    );

    if ( is_wp_error( $res ) ) {
        logistos_sync_log( 'order post failed: ' . $res->get_error_message() );
    } else {
        logistos_sync_log(
            [
                'order_sent' => $order->get_id(),
                'code'       => wp_remote_retrieve_response_code( $res ),
                'body'       => wp_remote_retrieve_body( $res ),
            ]
        );
    }
}