<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class LoginEase_Settings {
    const OPTION_KEY = 'loginease_settings';

    public static function add_admin_menu() {
        add_options_page(
            __('LoginEase', 'loginease'),
            __('LoginEase', 'loginease'),
            'manage_options',
            'loginease_settings',
            [ __CLASS__, 'settings_page' ]
        );
    }

    public static function settings_init() {
        register_setting(
            'loginease_settings_group',
            self::OPTION_KEY,
            [
                'sanitize_callback' => [ __CLASS__, 'sanitize' ],
                'default' => [
                    'token_expiration' => 15,
                    'max_logs' => 1000,
                    'disable_password_login' => 0,
                ],
            ]
        );

        add_settings_section(
            'loginease_settings_section',
            __('General Settings', 'loginease'),
            null,
            'loginease_settings'
        );

        add_settings_field(
            'token_expiration',
            __('Login Link Validity Duration (minutes)', 'loginease'),
            [ __CLASS__, 'render_token_expiration' ],
            'loginease_settings',
            'loginease_settings_section'
        );

        add_settings_field(
            'max_logs',
            __('Maximum Number of Logs Retained', 'loginease'),
            [ __CLASS__, 'render_max_logs' ],
            'loginease_settings',
            'loginease_settings_section'
        );

        add_settings_field(
            'disable_password_login',
            __('Disable Password Login', 'loginease'),
            [ __CLASS__, 'render_disable_password_login_field' ],
            'loginease_settings',
            'loginease_settings_section'
        );
    }

    public static function sanitize( $input ) {
        $sanitized = [];
        $sanitized['token_expiration'] = isset( $input['token_expiration'] ) ? absint( $input['token_expiration'] ) : 15;
        $sanitized['max_logs'] = isset( $input['max_logs'] ) ? absint( $input['max_logs'] ) : 1000;
        $sanitized['disable_password_login'] = ! empty( $input['disable_password_login'] ) ? 1 : 0;
        return $sanitized;
    }

    public static function render_token_expiration() {
        $options = get_option( self::OPTION_KEY );
        $value = isset( $options['token_expiration'] ) ? $options['token_expiration'] : 15;
        ?>
        <input type="number" min="5" max="60" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[token_expiration]" value="<?php echo esc_attr( $value ); ?>" />
        <p class="description"><?php esc_html_e( 'Duration (in minutes) for which the login link is valid.', 'loginease' ); ?></p>
        <?php
    }

    public static function render_max_logs() {
        $options = get_option( self::OPTION_KEY );
        $value = isset( $options['max_logs'] ) ? $options['max_logs'] : 1000;
        ?>
        <input type="number" min="100" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[max_logs]" value="<?php echo esc_attr( $value ); ?>" />
        <p class="description"><?php esc_html_e( 'The plugin will delete oldest logs to keep this number.', 'loginease' ); ?></p>
        <?php
    }

    public static function render_disable_password_login_field() {
        $options = get_option( self::OPTION_KEY );
        ?>
        <input type="checkbox" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[disable_password_login]" value="1" <?php checked( ! empty( $options['disable_password_login'] ) ); ?> />
        <p class="description"><?php esc_html_e( 'Hide the password field, remember me checkbox, login button and disable classic login.', 'loginease' ); ?></p>
        <?php
    }

    public static function settings_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'LoginEase', 'loginease' ); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'loginease_settings_group' );
                do_settings_sections( 'loginease_settings' );
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }
}