<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class LoginEase_Login
 *
 * Handles passwordless login, custom login slug,
 * blocking default WP login & password reset if disabled,
 * enqueuing external CSS/JS files.
 */
class LoginEase_Login {
    private static $opts = null;

    public static function init_hooks() {
        self::$opts = get_option( 'loginease_settings', [] );

        add_filter( 'wp_redirect', [ __CLASS__, 'prevent_login_redirect' ], 0, 2 );
        add_action( 'init', [ __CLASS__, 'maybe_block_default_login_and_admin' ], 1 );
        add_action( 'admin_init', [ __CLASS__, 'maybe_block_wp_admin' ], 1 );
        add_action( 'login_init', [ __CLASS__, 'maybe_block_wp_login' ], 1 );
        add_action( 'login_init', [ __CLASS__, 'block_lost_password_if_passwordless' ], 0 );
        add_action( 'init', [ __CLASS__, 'add_custom_login_rewrite' ], 5 );
        add_filter( 'query_vars', [ __CLASS__, 'add_query_vars' ] );
        add_action( 'template_redirect', [ __CLASS__, 'maybe_handle_custom_slug' ], 1 );
        add_filter( 'authenticate', [ __CLASS__, 'loginease_authenticate' ], 30, 3 );
        add_filter( 'login_message', [ __CLASS__, 'add_sent_message' ] );
        add_filter( 'body_class', [ __CLASS__, 'body_class_notice' ] );
        add_action( 'login_form_login', [ __CLASS__, 'process_login_request' ] );
        add_action( 'login_form', [ __CLASS__, 'add_loginease_button' ], 20 );
        add_action( 'login_enqueue_scripts', [ __CLASS__, 'enqueue_styles_scripts' ] );
        add_action( 'login_enqueue_scripts', [ __CLASS__, 'disable_lost_password_link' ] );
        add_action( 'login_enqueue_scripts', [ __CLASS__, 'hide_password_login_fields' ], 20 );
        add_filter( 'login_url', [ __CLASS__, 'filter_login_url' ], 10, 3 );
        add_filter( 'allow_password_reset', [ __CLASS__, 'disable_password_reset_if_passwordless' ], 10, 2 );
    }

    private static function get_options() {
        if ( null === self::$opts ) {
            self::$opts = get_option( 'loginease_settings', [] );
        }
        return self::$opts;
    }

    public static function is_custom_login_enabled() {
        $options = self::get_options();
        return ! empty( $options['enable_custom_login'] );
    }

    public static function get_custom_login_slug() {
        $options = self::get_options();
        $slug = isset( $options['custom_login_slug'] ) && $options['custom_login_slug'] !== '' ? $options['custom_login_slug'] : 'connexion';
        return sanitize_title( $slug );
    }

    public static function block_lost_password_if_passwordless() {
        $opts = self::get_options();
        if ( empty( $opts['disable_password_login'] ) ) {
            return;
        }
        $action = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : '';
        if ( in_array( $action, [ 'lostpassword', 'resetpass' ], true ) ) {
            wp_safe_redirect( self::get_login_url() );
            exit;
        }
    }

    public static function disable_password_reset_if_passwordless( $allow, $user_id ) {
        $opts = self::get_options();
        if ( ! empty( $opts['disable_password_login'] ) ) {
            return false;
        }
        return $allow;
    }

    public static function disable_lost_password_link() {
        $opts = self::get_options();
        if ( empty( $opts['disable_password_login'] ) ) {
            return;
        }
        add_filter( 'gettext', function ( $translated_text, $text, $domain ) {
            if ( 'Lost your password?' === $text || 'Forgot your password?' === $text ) {
                return '';
            }
            return $translated_text;
        }, 10, 3 );
    }

    public static function hide_password_login_fields() {
        $opts = self::get_options();
        if ( empty( $opts['disable_password_login'] ) ) {
            return;
        }
        $current_action = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : 'login';
        if ( 'login' !== $current_action ) {
            return;
        }
        // Plus de CSS/JS inline ici, tout est externalisé et chargé via enqueue
    }

    public static function enqueue_styles_scripts() {
    $opts = self::get_options();

    $disable_password_login = ! empty( $opts['disable_password_login'] );
    $current_action = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : 'login';

    if ( $disable_password_login && in_array( $current_action, [ 'lostpassword', 'resetpass' ], true ) ) {
        wp_safe_redirect( self::get_login_url() );
        exit;
    }
    if ( in_array( $current_action, [ 'lostpassword', 'resetpass' ], true ) ) {
        return;
    }

    // Construction correcte de la base URL du plugin
    $plugin_url_base = plugin_dir_url( dirname( __FILE__ ) );

    // Enqueue sans numéro de version (false)
    wp_enqueue_style(
        'loginease-style',
        $plugin_url_base . 'assets/css/loginease-style.css',
        []
    );

    wp_enqueue_script(
        'loginease-script',
        $plugin_url_base . 'assets/js/loginease-script.js',
        [],
    );
}

    public static function add_custom_login_rewrite() {
        $opts = self::get_options();
        if ( empty( $opts['enable_custom_login'] ) ) {
            return;
        }
        $slug = self::get_custom_login_slug();

        add_rewrite_tag( '%loginease_custom%', '([^&]+)' );
        add_rewrite_rule( '^' . preg_quote( $slug, '/' ) . '/?$', 'index.php?loginease_custom=1', 'top' );

        global $wp_rewrite;
        if ( isset( $wp_rewrite ) && is_object( $wp_rewrite ) && property_exists( $wp_rewrite, 'rules' ) ) {
            $rule_key = '^' . $slug . '/?$';
            $rule_value = 'index.php?loginease_custom=1';
            $rules = (array) $wp_rewrite->rules;
            if ( ! isset( $rules[ $rule_key ] ) ) {
                $wp_rewrite->rules = [ $rule_key => $rule_value ] + $rules;
            }
        }
    }

    public static function add_query_vars( $vars ) {
        $vars[] = 'loginease_custom';
        return $vars;
    }

    public static function maybe_handle_custom_slug() {
        $opts = self::get_options();
        if ( empty( $opts['enable_custom_login'] ) ) {
            return;
        }
        $slug = self::get_custom_login_slug();
        $qv = get_query_var( 'loginease_custom' );
        $matched = ( (string) $qv === '1' );
        if ( ! $matched ) {
            $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
            $request_path = wp_parse_url( $request_uri, PHP_URL_PATH );
            $site_path = wp_parse_url( site_url(), PHP_URL_PATH );
            $site_path = $site_path ? ( '/' . trim( $site_path, '/' ) ) : '';
            $expected = rtrim( $site_path, '/' ) . '/' . $slug . '/';
            if ( strpos( $expected, '/' ) !== 0 ) {
                $expected = '/' . ltrim( $expected, '/' );
            }
            if ( untrailingslashit( $request_path ) === untrailingslashit( $expected ) ) {
                $matched = true;
            }
        }
        if ( $matched ) {
            $target = site_url( 'wp-login.php' );
            $args = [ 'loginease' => $slug ];
            if ( isset( $_GET['redirect_to'] ) && $_GET['redirect_to'] !== '' ) {
                $args['redirect_to'] = wp_unslash( $_GET['redirect_to'] );
            }
            if ( isset( $_GET['loginease_token'] ) && $_GET['loginease_token'] !== '' ) {
                $args['loginease_token'] = wp_unslash( $_GET['loginease_token'] );
            }
            $target = add_query_arg( $args, $target );
            wp_safe_redirect( $target, 302 );
            exit;
        }
    }

    public static function maybe_block_default_login_and_admin() {
        $opts = self::get_options();
        if ( empty( $opts['enable_custom_login'] ) ) {
            return;
        }
        if ( defined( 'WP_CLI' ) && WP_CLI ) {
            return;
        }
        if ( function_exists( 'is_user_logged_in' ) && is_user_logged_in() ) {
            return;
        }
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
        $path = wp_parse_url( $request_uri, PHP_URL_PATH );
        $basename = strtolower( basename( $path ) );
        $disable_password_login = ! empty( $opts['disable_password_login'] );
        $action = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : '';
        if ( $disable_password_login && in_array( $action, [ 'lostpassword', 'resetpass' ], true ) ) {
            self::send_404_and_exit();
        }
        if ( in_array( $basename, [ 'admin-ajax.php', 'admin-post.php' ], true ) ) {
            return;
        }
        if ( isset( $_GET['loginease_token'] ) && $_GET['loginease_token'] !== '' ) {
            return;
        }
        if ( isset( $_GET['loginease'] ) && $_GET['loginease'] === ( isset( $opts['custom_login_slug'] ) ? sanitize_title( $opts['custom_login_slug'] ) : 'connexion' ) ) {
            return;
        }
        if ( $basename === 'wp-login.php' || strpos( '/' . ltrim( (string) $path, '/' ), '/wp-admin' ) === 0 ) {
            self::send_404_and_exit();
        }
    }

    public static function maybe_block_wp_admin() {
        $opts = self::get_options();
        if ( empty( $opts['enable_custom_login'] ) ) {
            return;
        }
        if ( defined( 'WP_CLI' ) && WP_CLI ) {
            return;
        }
        if ( function_exists( 'is_user_logged_in' ) && is_user_logged_in() ) {
            return;
        }
        $basename = '';
        if ( isset( $_SERVER['PHP_SELF'] ) ) {
            $basename = strtolower( basename( wp_unslash( $_SERVER['PHP_SELF'] ) ) );
            if ( in_array( $basename, [ 'admin-ajax.php', 'admin-post.php' ], true ) ) {
                return;
            }
        }
        if (
            ( isset( $_GET['loginease'] ) && $_GET['loginease'] === self::get_custom_login_slug() )
            || ( isset( $_GET['loginease_token'] ) && $_GET['loginease_token'] !== '' )
        ) {
            return;
        }
        self::send_404_and_exit();
    }

    public static function maybe_block_wp_login() {
        $opts = self::get_options();
        if ( empty( $opts['enable_custom_login'] ) ) {
            return;
        }
        if ( defined( 'WP_CLI' ) && WP_CLI ) {
            return;
        }
        if ( function_exists( 'is_user_logged_in' ) && is_user_logged_in() ) {
            return;
        }
        if ( isset( $_GET['loginease_token'] ) && $_GET['loginease_token'] !== '' ) {
            return;
        }
        if ( isset( $_GET['loginease'] ) && $_GET['loginease'] === self::get_custom_login_slug() ) {
            return;
        }
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
        $path_info = '';
        if ( ! empty( $_SERVER['PATH_INFO'] ) ) {
            $path_info = trim( wp_unslash( $_SERVER['PATH_INFO'] ), '/' );
        } else {
            $pos = strpos( $request_uri, 'wp-login.php' );
            if ( $pos !== false ) {
                $after = substr( $request_uri, $pos + strlen( 'wp-login.php' ) );
                $path_info = trim( wp_parse_url( $after, PHP_URL_PATH ), '/' );
            }
        }
        if ( $path_info !== '' && $path_info === self::get_custom_login_slug() ) {
            return;
        }
        self::send_404_and_exit();
    }

    public static function prevent_login_redirect( $location, $status ) {
        $opts = self::get_options();
        if ( empty( $opts['enable_custom_login'] ) ) {
            return $location;
        }
        if ( function_exists( 'is_user_logged_in' ) && is_user_logged_in() ) {
            return $location;
        }
        if ( defined( 'WP_CLI' ) && WP_CLI ) {
            return $location;
        }
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
        $path = wp_parse_url( $request_uri, PHP_URL_PATH );
        $normalized = '/' . ltrim( (string) $path, '/' );
        if ( strpos( $normalized, '/wp-admin' ) === 0 ) {
            if ( false !== stripos( $location, 'wp-login.php' ) ) {
                self::send_404_and_exit();
            }
            if ( function_exists( 'wp_login_url' ) ) {
                $login_url = wp_login_url();
                if ( $login_url && 0 === strpos( $location, $login_url ) ) {
                    self::send_404_and_exit();
                }
            }
        }
        return $location;
    }

    private static function send_404_and_exit() {
        if ( ! headers_sent() ) {
            status_header( 404 );
            nocache_headers();
        }
        if ( function_exists( 'get_404_template' ) ) {
            $template = get_404_template();
            if ( $template && file_exists( $template ) ) {
                include $template;
                exit;
            }
        }
        if ( function_exists( 'wp_die' ) ) {
            wp_die( '<h1>404 Not Found</h1>', '404 Not Found', [ 'response' => 404 ] );
        }
        if ( isset( $_SERVER['SERVER_PROTOCOL'] ) ) {
            header( sanitize_text_field( wp_unslash( $_SERVER['SERVER_PROTOCOL'] ) ) . ' 404 Not Found', true, 404 );
        } else {
            header( 'HTTP/1.0 404 Not Found', true, 404 );
        }
        echo '<h1>404 Not Found</h1>';
        exit;
    }

    public static function add_sent_message( $message ) {
        if ( isset( $_GET['loginease_status'] ) ) {
            $status = sanitize_text_field( wp_unslash( $_GET['loginease_status'] ) );
            switch ( $status ) {
                case 'sent':
                    $message .= '<p class="message loginease-sent-message">' . esc_html__( 'A magic login link has been sent to your email.', 'loginease' ) . '</p>';
                    break;
                case 'invalid':
                    $message .= '<p class="message loginease-error-message">' . esc_html__( 'Invalid or expired login link.', 'loginease' ) . '</p>';
                    break;
            }
        }
        return $message;
    }

    public static function body_class_notice( $classes ) {
        if ( isset( $_GET['loginease_status'] ) && sanitize_text_field( wp_unslash( $_GET['loginease_status'] ) ) === 'sent' ) {
            $classes[] = 'loginease-email-sent';
        }
        return $classes;
    }

    public static function filter_login_url( $login_url, $redirect = '', $force_reauth = false ) {
        if ( ! self::is_custom_login_enabled() ) {
            return $login_url;
        }
        $slug = self::get_custom_login_slug();
        $url = site_url( '/' . $slug . '/' );
        if ( ! empty( $redirect ) ) {
            $url = add_query_arg( 'redirect_to', rawurlencode( $redirect ), $url );
        }
        return $url;
    }

    public static function get_login_url( $redirect = '' ) {
        return wp_login_url( $redirect );
    }

    public static function process_login_request() {
        $opts = self::get_options();
        $disable_password_login = ! empty( $opts['disable_password_login'] );
        if ( function_exists( 'is_user_logged_in' ) && is_user_logged_in() ) {
            return;
        }
        if ( $disable_password_login && isset( $_POST['wp-submit'] ) ) {
            wp_safe_redirect( wp_login_url() );
            exit;
        }
        if ( isset( $_POST['loginease_submit'] ) && check_admin_referer( 'login', '_wpnonce' ) ) {
            $login_or_email_raw = isset( $_POST['log'] ) ? wp_unslash( $_POST['log'] ) : '';
            $login_or_email = sanitize_text_field( $login_or_email_raw );
            $redirect_url = add_query_arg( 'loginease_status', 'sent', self::get_login_url() );
            if ( empty( $login_or_email ) ) {
                wp_safe_redirect( add_query_arg( 'login', 'empty', self::get_login_url() ) );
                exit;
            }
            $user = is_email( $login_or_email ) ? get_user_by( 'email', $login_or_email ) : get_user_by( 'login', $login_or_email );
            if ( ! $user ) {
                wp_safe_redirect( $redirect_url );
                exit;
            }
            $token = self::generate_token_for_user( $user->ID );
            self::send_login_email( $user, $token );
            wp_safe_redirect( $redirect_url );
            exit;
        }
        if ( isset( $_GET['loginease_token'] ) ) {
            $token = sanitize_text_field( wp_unslash( $_GET['loginease_token'] ) );
            self::login_with_token( $token );
        }
    }

    public static function loginease_authenticate( $user, $username, $password ) {
        if ( isset( $_POST['loginease_submit'] ) ) {
            if ( empty( $username ) ) {
                return new WP_Error( 'empty_username', __( 'Missing username or email.', 'loginease' ) );
            }
            // Return null to prevent default password login attempt and allow magic login process.
            return null;
        }
        return $user;
    }

    private static function generate_token_for_user( $user_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'loginease_tokens';
        $now = current_time( 'mysql' );
        $wpdb->query( $wpdb->prepare( "DELETE FROM $table WHERE expire < %s OR used = 1", $now ) );
        $token = wp_generate_password( 20, false, false );
        $opts = self::get_options();
        $minutes = isset( $opts['token_expiration'] ) ? (int) $opts['token_expiration'] : 15;
        $expire_time = date_i18n( 'Y-m-d H:i:s', current_time( 'timestamp' ) + ( $minutes * 60 ) );
        $wpdb->insert(
            $table,
            [
                'user_id' => $user_id,
                'token'   => $token,
                'expire'  => $expire_time,
                'used'    => 0,
            ],
            [ '%d', '%s', '%s', '%d' ]
        );
        return $token;
    }

    private static function send_login_email( $user, $token ) {
        $opts = self::get_options();
        $minutes = isset( $opts['token_expiration'] ) ? (int) $opts['token_expiration'] : 15;
        $expire_date = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), current_time( 'timestamp' ) + $minutes * 60 );
        $login_url = add_query_arg( 'loginease_token', rawurlencode( $token ), self::get_login_url() );
        $subject = sprintf( __( 'Your login link for %s', 'loginease' ), get_bloginfo( 'name' ) );
        $message = sprintf(
            __( "Hello,\n\nHere is your secure login link:\n\n%1\$s\n\nValid until %2\$s\n\nRegards,\n%3\$s", 'loginease' ),
            $login_url,
            $expire_date,
            get_bloginfo( 'name' )
        );
        wp_mail( $user->user_email, $subject, $message );
    }

    public static function login_with_token( $token ) {
        global $wpdb;
        $table = $wpdb->prefix . 'loginease_tokens';
        $now = current_time( 'mysql' );
        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM $table WHERE token = %s AND expire > %s AND used = 0",
                $token,
                $now
            )
        );
        if ( ! $row ) {
            wp_safe_redirect( add_query_arg( 'loginease_status', 'invalid', self::get_login_url() ) );
            exit;
        }
        $wpdb->update( $table, [ 'used' => 1 ], [ 'id' => $row->id ], [ '%d' ], [ '%d' ] );
        wp_set_current_user( $row->user_id );
        wp_set_auth_cookie( $row->user_id );
        if ( class_exists( 'LoginEase_Logger' ) ) {
            LoginEase_Logger::log( $row->user_id, 'Magic link' );
            LoginEase_Logger::cleanup();
        }
        $redirect_to = isset( $_REQUEST['redirect_to'] ) ? esc_url_raw( wp_unslash( $_REQUEST['redirect_to'] ) ) : admin_url();
        wp_safe_redirect( $redirect_to );
        exit;
    }

    public static function add_loginease_button() {
        $opts = self::get_options();
        $disable_password_login = ! empty( $opts['disable_password_login'] );
        if ( isset( $_GET['action'] ) && sanitize_text_field( wp_unslash( $_GET['action'] ) ) !== 'login' ) {
            return;
        }
        wp_nonce_field( 'login', '_wpnonce' );
        $btn_class = $disable_password_login ? 'button button-primary' : 'button button-secondary';
        echo '<input type="submit" name="loginease_submit" id="loginease_submit" class="' . esc_attr( $btn_class ) . '" value="' . esc_attr__( 'Send me a login link', 'loginease' ) . '" />';
    }
}