<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Prevent direct access
}

/**
 * Class LoginEase_Logger
 *
 * Handles login history logging and displays it in a user profile metabox.
 * Also enqueues required admin styles for proper display.
 */
class LoginEase_Logger {

    /**
     * Log a login event for a user.
     *
     * @param int $user_id User ID.
     * @param string $method Method of login, default 'Password'.
     */
    public static function log( $user_id, $method = 'Password' ) {
        global $wpdb;
        $table_logs = $wpdb->prefix . 'loginease_logs';
        $ip = self::get_ip_address();
        $date = current_time( 'mysql' );
        $wpdb->insert(
            $table_logs,
            [
                'user_id' => $user_id,
                'ip'      => $ip,
                'date'    => $date,
                'method'  => $method,
            ],
            [ '%d', '%s', '%s', '%s' ]
        );
    }

    /**
     * Cleanup old logs to maintain under max_logs limit.
     */
    public static function cleanup() {
        global $wpdb;
        $table_logs = $wpdb->prefix . 'loginease_logs';
        $options = get_option( 'loginease_settings', [ 'max_logs' => 1000 ] );
        $max_logs = isset( $options['max_logs'] ) ? (int) $options['max_logs'] : 1000;

        $count = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM $table_logs" ) );
        if ( $count > $max_logs ) {
            $to_delete = $count - $max_logs;
            // Use prepare for LIMIT clause
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM $table_logs ORDER BY date ASC LIMIT %d",
                    $to_delete
                )
            );
        }
    }

    /**
     * Get visitor IP address.
     *
     * @return string IP address or empty string if none found.
     */
    private static function get_ip_address() {
        if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
            return sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
        } elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
            $ip_list = explode( ',', wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
            return sanitize_text_field( trim( $ip_list[0] ) );
        } elseif ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
            return sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
        }

        return '';
    }

    /**
     * Adds a metabox in user profiles to show recent login history.
     *
     * @param WP_User $user User object.
     */
    public static function add_login_logs_metabox( $user ) {
        if ( ! current_user_can( 'read', $user->ID ) ) {
            return;
        }

        global $wpdb;
        $table_logs = $wpdb->prefix . 'loginease_logs';

        $logs = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT ip, date, method
                 FROM $table_logs
                 WHERE user_id = %d
                 ORDER BY date DESC
                 LIMIT 20",
                $user->ID
            )
        );
        ?>
        <h2><?php esc_html_e( 'Login History', 'loginease' ); ?></h2>
        <p class="loginease-login-history-description">
            <?php esc_html_e( "Only the most recent logins are shown. Older logins may not appear depending on the log retention settings.", 'loginease' ); ?>
        </p>

        <?php if ( ! empty( $logs ) ) : ?>
            <table class="widefat fixed striped loginease-history-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'IP Address', 'loginease' ); ?></th>
                        <th><?php esc_html_e( 'Date & Time', 'loginease' ); ?></th>
                        <th><?php esc_html_e( 'Login Method', 'loginease' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ( $logs as $log ) : ?>
                        <tr>
                            <td><?php echo esc_html( $log->ip ); ?></td>
                            <td><?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $log->date ) ) ); ?></td>
                            <td><?php echo esc_html( $log->method ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else : ?>
            <p><?php esc_html_e( 'No login history found.', 'loginease' ); ?></p>
        <?php endif;
    }
}