<?php
/**
 * Plugin Name: Login Security with Telegram Alerts
 * Plugin URI: https://gabirosca.com/login-security-with-telegram-alerts-wordpress-security-plugin/
 * Description: Login Security with Telegram Alerts protects WordPress logins, stops brute force, logs activity, and sends real-time Telegram alerts.
 * Version: 1.0.0
 * Author: Gabriel Rosca
 * Author URI: https://gabirosca.com
 * Donate link: https://ko-fi.com/gabrielrosca
 * Text Domain: login-security-with-telegram-alerts
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.2
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

// Define plugin constants
define('LSEC_PLUGIN_FILE', __FILE__);
define('LSEC_PLUGIN_DIR', plugin_dir_path(LSEC_PLUGIN_FILE));
define('LSEC_PLUGIN_URL', plugin_dir_url(LSEC_PLUGIN_FILE));
define('LSEC_VERSION', '1.0.0');

// Include necessary files (order matters for class dependencies)
require_once LSEC_PLUGIN_DIR . 'includes/lsec-helper-functions.php';
require_once LSEC_PLUGIN_DIR . 'includes/class-lsec-login-tracker.php';
require_once LSEC_PLUGIN_DIR . 'includes/class-lsec-telegram-notifier.php';
require_once LSEC_PLUGIN_DIR . 'includes/class-lsec-advanced-security.php';
require_once LSEC_PLUGIN_DIR . 'includes/class-lsec-file-integrity.php';
require_once LSEC_PLUGIN_DIR . 'includes/class-lsec-admin-settings.php';

// Register activation hook
register_activation_hook(LSEC_PLUGIN_FILE, 'lsec_create_tables');

/**
 * Main plugin initialization function.
 * This function initializes plugin components.
 * It's hooked to 'plugins_loaded' to ensure all WordPress functions are available.
 * Note: Translation loading is handled automatically by WordPress for plugins in the repository.
 */
function lsec_init() {
    // Check for database upgrades
    lsec_check_db_upgrade();

    // Initialize core components
    $login_tracker      = new LSEC_Login_Tracker();
    $telegram_notifier  = new LSEC_Telegram_Notifier();
    $advanced_security  = new LSEC_Advanced_Security();
    $admin_settings     = new LSEC_Admin_Settings($login_tracker, $telegram_notifier); // Pass dependencies

    // Actions and filters that are essential and don't fit naturally into a class
    // This hook is for checking the manual IP blacklist *before* login form is processed.
    add_action('init', 'lsec_check_manual_blacklist', 1);
    // Adds a settings link on the plugins page.
    add_filter('plugin_action_links_' . plugin_basename(LSEC_PLUGIN_FILE), 'lsec_add_settings_link');
    // Adds plugin row meta links (like Support Development).
    add_filter('plugin_row_meta', 'lsec_add_plugin_row_meta', 10, 2);
}
add_action('plugins_loaded', 'lsec_init');

/**
 * Adds a settings link to the plugin actions on the Plugins page.
 *
 * @param array $links Array of action links.
 * @return array Modified array of action links.
 */
function lsec_add_settings_link($links) {
    $settings_link = '<a href="' . esc_url(admin_url('options-general.php?page=login-security-telegram-alerts-settings')) . '">' . esc_html__('Settings', 'login-security-with-telegram-alerts') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}

/**
 * Adds plugin row meta links (like Support Development) on the Plugins page.
 *
 * @param array  $links Array of meta links.
 * @param string $file  Plugin file path.
 * @return array Modified array of meta links.
 */
function lsec_add_plugin_row_meta($links, $file) {
    if (plugin_basename(LSEC_PLUGIN_FILE) === $file) {
        $support_link = '<a href="https://ko-fi.com/gabrielrosca" target="_blank" rel="noopener noreferrer">' . esc_html__('Support Development', 'login-security-with-telegram-alerts') . '</a>';
        $links[] = $support_link;
    }
    return $links;
}



/**
 * Custom wp_die message for IP blocking.
 * Displays a full-page message for access denied scenarios.
 *
 * @param string $message_text The message to display.
 * @param string $title_text   The title of the page.
 * @param array  $args         Arguments for wp_die.
 */
function lsec_centered_wp_die_message($message_text, $title_text = '', $args = []) {
    $page_title    = $title_text ? esc_html($title_text) : esc_html__('Notification', 'login-security-with-telegram-alerts');
    $default_args = ['response' => 403];
    $final_args    = wp_parse_args($args, $default_args);

    if (!headers_sent()) {
        header('Content-Type: text/html; charset=' . get_option('blog_charset'));
        status_header($final_args['response']);
        nocache_headers();
    }

    $message_html  = '<!DOCTYPE html><html lang="' . esc_attr(get_locale()) . '">';
    $message_html .= '<head>';
    $message_html .= '<meta charset="' . esc_attr(get_bloginfo('charset')) . '">';
    $message_html .= '<meta name="viewport" content="width=device-width, initial-scale=1">';
    $message_html .= '<title>' . $page_title . '</title>';
    $message_html .= '<style> body { background-color: #f0f0f1; color: #444; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif; margin: 0; padding: 0; display: flex; justify-content: center; align-items: center; min-height: 100vh; text-align: center; } .message-container { background-color: #fff; padding: 30px 40px; border-radius: 5px; box-shadow: 0 1px 3px rgba(0,0,0,0.13); max-width: 500px; width: auto; min-width: 300px; box-sizing: border-box; } .message-container h1 { margin:0 0 15px; font-size:22px; color:#333; font-weight:600; } .message-container p { margin:0; font-size:15px; line-height:1.5; } </style>';
    $message_html .= '</head><body>';
    $message_html .= '<div class="message-container">';
    if (!empty($title_text)) {
        $message_html .= '<h1>' . $page_title . '</h1>';
    }
    $message_html .= '<p>' . wp_kses_post($message_text) . '</p>';
    $message_html .= '</div></body></html>';
    echo $message_html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- HTML is intentionally unescaped here for direct output as a full page.
    die();
}