<?php
/**
 * Helper functions for Login Security with Telegram Alerts plugin.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Gets the client's IP address, attempting to detect the real IP even behind proxies.
 *
 * @return string The client's IP address.
 */
function lsec_get_client_ip() {
    $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) && !empty($_SERVER[$key])) {
            $header_value = sanitize_text_field(wp_unslash($_SERVER[$key]));
            $ips = array_map('trim', explode(',', $header_value));
            $ip_candidate = $ips[0];
            if (filter_var($ip_candidate, FILTER_VALIDATE_IP)) {
                return $ip_candidate;
            }
        }
    }
    return '127.0.0.1';
}

/**
 * Parses a string of IP addresses (one per line) into an array of valid IPs.
 *
 * @param string $ip_string A string containing IP addresses separated by newlines.
 * @return array An array of unique, valid IP addresses.
 */
function lsec_parse_ip_list($ip_string) {
    if (!is_string($ip_string)) { return []; }
    $ips = explode("\n", $ip_string);
    $valid_ips = [];
    foreach ($ips as $ip) {
        $trimmed_ip = trim($ip);
        if (filter_var($trimmed_ip, FILTER_VALIDATE_IP)) {
            $valid_ips[] = $trimmed_ip;
        }
    }
    return array_unique($valid_ips);
}

/**
 * Checks if a given IP address is present in a list of IP addresses.
 *
 * @param string $current_ip    The IP address to check.
 * @param array  $ip_list_array An array of IP addresses.
 * @return bool True if the IP is in the list, false otherwise.
 */
function lsec_is_ip_in_list($current_ip, $ip_list_array) {
    if (empty($ip_list_array) || !is_array($ip_list_array)) { return false; }
    return in_array($current_ip, $ip_list_array, true);
}

/**
 * Checks if the current IP address is whitelisted.
 *
 * @param string $ip_address The IP address to check.
 * @return bool True if the IP is whitelisted, false otherwise.
 */
function lsec_is_ip_whitelisted($ip_address) {
    $whitelist_str = get_option('lsec_manual_whitelist_ips', '');
    $whitelisted_ips = lsec_parse_ip_list($whitelist_str);
    return lsec_is_ip_in_list($ip_address, $whitelisted_ips);
}

/**
 * Checks if the current IP address is blacklisted.
 *
 * @param string $ip_address The IP address to check.
 * @return bool True if the IP is blacklisted, false otherwise.
 */
function lsec_is_ip_blacklisted($ip_address) {
    $blacklist_str = get_option('lsec_manual_blacklist_ips', '');
    $blacklisted_ips = lsec_parse_ip_list($blacklist_str);
    return lsec_is_ip_in_list($ip_address, $blacklisted_ips);
}

/**
 * Checks if the client's IP is in the manual blacklist and denies access if it is.
 * This function is hooked very early in the 'init' action.
 */
function lsec_check_manual_blacklist() {
    $current_ip = lsec_get_client_ip();
    if (lsec_is_ip_blacklisted($current_ip)) {
        lsec_centered_wp_die_message(
            esc_html__('Your IP address has been blocked from accessing this site.', 'login-security-with-telegram-alerts'),
            esc_html__('Access Denied', 'login-security-with-telegram-alerts'),
            ['response' => 403, 'exit' => true]
        );
    }
}

/**
 * Creates the necessary database tables for the plugin.
 * This function is called on plugin activation.
 */
function lsec_create_tables() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

    $table_failed_attempts = $wpdb->prefix . 'lsec_login_attempts';
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is prefixed and not user input.
    $sql_failed_attempts = "CREATE TABLE {$table_failed_attempts} (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        ip_address varchar(100) NOT NULL,
        username varchar(60) DEFAULT NULL,
        attempts int NOT NULL,
        last_attempt datetime NOT NULL,
        PRIMARY KEY  (id),
        KEY ip_address (ip_address)
    ) {$charset_collate};";
    dbDelta($sql_failed_attempts);

    $table_all_activity = $wpdb->prefix . 'lsec_all_login_activity';
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is prefixed and not user input.
    $sql_all_activity = "CREATE TABLE {$table_all_activity} (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        user_id bigint(20) unsigned DEFAULT NULL,
        username varchar(255) NOT NULL,
        ip_address varchar(100) NOT NULL,
        login_time datetime NOT NULL,
        status varchar(30) NOT NULL,
        user_agent text DEFAULT NULL,
        role varchar(255) DEFAULT NULL,
        PRIMARY KEY  (id),
        KEY ip_address (ip_address),
        KEY login_time (login_time),
        KEY status (status)
    ) {$charset_collate};";
    dbDelta($sql_all_activity);

    $table_user_devices = $wpdb->prefix . 'lsec_user_devices';
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is prefixed and not user input.
    $sql_user_devices = "CREATE TABLE {$table_user_devices} (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        user_id bigint(20) unsigned NOT NULL,
        device_hash varchar(32) NOT NULL,
        user_agent text DEFAULT NULL,
        ip_address varchar(100) NOT NULL,
        last_login datetime NOT NULL,
        PRIMARY KEY  (id),
        KEY user_id (user_id),
        KEY device_hash (device_hash),
        UNIQUE KEY user_device (user_id, device_hash)
    ) {$charset_collate};";
    dbDelta($sql_user_devices);

    $table_content_access = $wpdb->prefix . 'lsec_content_access';
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is prefixed and not user input.
    $sql_content_access = "CREATE TABLE {$table_content_access} (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        post_id bigint(20) unsigned NOT NULL,
        user_id bigint(20) unsigned NOT NULL,
        created_at datetime NOT NULL,
        PRIMARY KEY  (id),
        KEY post_id (post_id),
        KEY user_id (user_id),
        UNIQUE KEY post_user (post_id, user_id)
    ) {$charset_collate};";
    dbDelta($sql_content_access);

    $options_defaults = [
        'lsec_max_attempts'             => 5,
        'lsec_lockout_duration'         => 20,
        'lsec_notify_roles'             => ['administrator'],
        'lsec_enable_geolocation'       => 1,
        'lsec_notify_on_failed_attempt' => 1,
        'lsec_notify_admin_on_lockout'  => 1,
        'lsec_manual_blacklist_ips'     => '',
        'lsec_manual_whitelist_ips'     => '',
        'lsec_bot_token'                => '',
        'lsec_chat_id'                  => '',
        'lsec_custom_admin_url'         => '',
        'lsec_max_devices_per_user'     => 0,
        'lsec_access_denied_message'    => __('Sorry, you do not have permission to view this content.', 'login-security-with-telegram-alerts'),
        'lsec_enable_content_restriction' => 0,
    ];
    foreach ($options_defaults as $option_name => $default_value) {
        if (false === get_option($option_name)) {
            update_option($option_name, $default_value);
        }
    }
    // Clean up old option if it exists
    delete_option('lsec_notify_new_device_login');

    // Store version for future upgrades
    update_option('lsec_db_version', '1.3');
}

/**
 * Checks and upgrades database schema if needed.
 * Called on plugins_loaded to handle updates for existing installations.
 */
function lsec_check_db_upgrade() {
    $current_db_version = get_option('lsec_db_version', '1.0');

    // If DB version is less than 1.1, we need to create the user_devices table
    if (version_compare($current_db_version, '1.1', '<')) {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        $table_user_devices = $wpdb->prefix . 'lsec_user_devices';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is prefixed and not user input.
        $sql_user_devices = "CREATE TABLE {$table_user_devices} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            user_id bigint(20) unsigned NOT NULL,
            device_hash varchar(32) NOT NULL,
            user_agent text DEFAULT NULL,
            ip_address varchar(100) NOT NULL,
            last_login datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY user_id (user_id),
            KEY device_hash (device_hash),
            UNIQUE KEY user_device (user_id, device_hash)
        ) {$charset_collate};";
        dbDelta($sql_user_devices);

        // Add new options if they don't exist
        if (false === get_option('lsec_custom_admin_url')) {
            update_option('lsec_custom_admin_url', '');
        }
        if (false === get_option('lsec_max_devices_per_user')) {
            update_option('lsec_max_devices_per_user', 0);
        }

        // Update DB version
        update_option('lsec_db_version', '1.1');
    }

    // If DB version is less than 1.2, we need to create the content_access table
    if (version_compare($current_db_version, '1.2', '<')) {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        $table_content_access = $wpdb->prefix . 'lsec_content_access';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is prefixed and not user input.
        $sql_content_access = "CREATE TABLE {$table_content_access} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            post_id bigint(20) unsigned NOT NULL,
            user_id bigint(20) unsigned NOT NULL,
            created_at datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY post_id (post_id),
            KEY user_id (user_id),
            UNIQUE KEY post_user (post_id, user_id)
        ) {$charset_collate};";
        dbDelta($sql_content_access);

        // Add new options if they don't exist
        if (false === get_option('lsec_access_denied_message')) {
            update_option('lsec_access_denied_message', __('Sorry, you do not have permission to view this content.', 'login-security-with-telegram-alerts'));
        }
        if (false === get_option('lsec_enable_content_restriction')) {
            update_option('lsec_enable_content_restriction', 0);
        }

        // Update DB version
        update_option('lsec_db_version', '1.2');
    }

    // IMMEDIATE FIX: Force convert all checkbox options to integers RIGHT NOW
    // This runs every time until we're certain all values are integers
    $boolean_options = [
        'lsec_enable_geolocation',
        'lsec_notify_on_failed_attempt',
        'lsec_notify_admin_on_lockout',
        'lsec_enable_content_restriction',
    ];

    foreach ($boolean_options as $option_name) {
        $value = get_option($option_name);
        // Force to integer if it's not already an integer
        if (!is_int($value)) {
            $int_value = (int) filter_var($value, FILTER_VALIDATE_BOOLEAN);
            update_option($option_name, $int_value, false);
        }
    }

    // Update DB version
    if (version_compare($current_db_version, '1.3', '<')) {
        update_option('lsec_db_version', '1.3');
    }
}

/**
 * Retrieves geolocation information for a given IP address.
 * Caches results using transients to improve performance and reduce external API calls.
 *
 * @param string $ip The IP address to geolocate.
 * @return string Formatted location string or a message indicating failure/private IP.
 */
function lsec_get_ip_geolocation($ip) {
    if (empty($ip) || !filter_var($ip, FILTER_VALIDATE_IP)) {
        return '<i>' . esc_html__('Invalid IP Address', 'login-security-with-telegram-alerts') . '</i>';
    }
    // Check for private/reserved IP ranges (RFC 1918, RFC 3927, loopback, etc.)
    if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        return lsec_get_private_network_type($ip);
    }

    $location_str    = '<i>' . esc_html__('Geolocation lookup failed or IP is private.', 'login-security-with-telegram-alerts') . '</i>';
    $transient_key   = 'lsec_geo_' . md5($ip);
    $cached_location = get_transient($transient_key);

    if (false !== $cached_location) {
        return $cached_location;
    }

    // Attempt with ip-api.com (more generous rate limits for non-HTTPS)
    $response_ip_api_com = wp_remote_get("http://ip-api.com/json/{$ip}?fields=status,message,country,regionName,city,isp,org,as", ['timeout' => 3]);
    if (!is_wp_error($response_ip_api_com) && wp_remote_retrieve_response_code($response_ip_api_com) === 200) {
        $data = json_decode(wp_remote_retrieve_body($response_ip_api_com), true);
        if ($data && isset($data['status']) && $data['status'] === 'success') {
            $loc_parts       = array_filter([$data['city'] ?? null, $data['regionName'] ?? null, $data['country'] ?? null]);
            $network_parts = array_filter([$data['isp'] ?? null, $data['org'] ?? null, $data['as'] ?? null]);
            $current_location_str = implode(', ', array_map('esc_html', $loc_parts));
            if (!empty($network_parts)) {
                $current_location_str .= ' (' . implode(' / ', array_map('esc_html', $network_parts)) . ')';
            }
            if (!empty(trim($current_location_str))) {
                $location_str = $current_location_str;
            } else {
                $location_str = '<i>' . esc_html__('Location details not available', 'login-security-with-telegram-alerts') . '</i>';
            }
            set_transient($transient_key, $location_str, HOUR_IN_SECONDS * 6); // Cache for 6 hours
            return $location_str;
        }
    }

    // Fallback to ipapi.co if ip-api.com fails or is rate-limited (uses HTTPS)
    $response_ipapi = wp_remote_get("https://ipapi.co/{$ip}/json/", ['timeout' => 5]);
    if (!is_wp_error($response_ipapi) && wp_remote_retrieve_response_code($response_ipapi) === 200) {
        $data = json_decode(wp_remote_retrieve_body($response_ipapi), true);
        if ($data && !(isset($data['error']) && $data['error'])) {
            $loc_parts       = array_filter([$data['city'] ?? null, $data['region'] ?? null, $data['country_name'] ?? null]);
            $network_parts = array_filter([$data['org'] ?? null, !empty($data['asn']) ? ('AS' . $data['asn']) : null]);
            $current_location_str = implode(', ', array_map('esc_html', $loc_parts));
            if (!empty($network_parts)) {
                $current_location_str .= ' (' . implode(' / ', array_map('esc_html', $network_parts)) . ')';
            }
            if (!empty(trim($current_location_str))) {
                $location_str = $current_location_str;
            } else {
                $location_str = '<i>' . esc_html__('Location details not available', 'login-security-with-telegram-alerts') . '</i>';
            }
            set_transient($transient_key, $location_str, HOUR_IN_SECONDS * 6); // Cache for 6 hours
            return $location_str;
        }
    }

    // Cache the failure message for 1 hour to avoid repeated failed lookups
    set_transient($transient_key, $location_str, HOUR_IN_SECONDS);
    return $location_str;
}

/**
 * Determines the type of private network an IP address belongs to.
 *
 * @param string $ip The IP address to check.
 * @return string A descriptive string for the private network type.
 */
function lsec_get_private_network_type($ip) {
    if ($ip === '127.0.0.1' || $ip === '::1') {
        return '🔧 ' . esc_html__('Local Development Server', 'login-security-with-telegram-alerts');
    }
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
        if (strpos($ip, '10.') === 0) {
            return '🏠 ' . esc_html__('Private Network (10.0.0.0/8)', 'login-security-with-telegram-alerts');
        }
        if (strpos($ip, '192.168.') === 0) {
            return '🏠 ' . esc_html__('Private Network (192.168.0.0/16)', 'login-security-with-telegram-alerts');
        }
        $ip_parts = explode('.', $ip);
        if (isset($ip_parts[0]) && (int)$ip_parts[0] === 172 && isset($ip_parts[1]) && (int)$ip_parts[1] >= 16 && (int)$ip_parts[1] <= 31) {
            return '🏠 ' . esc_html__('Private Network (172.16.0.0/12)', 'login-security-with-telegram-alerts');
        }
        if (strpos($ip, '169.254.') === 0) {
            return '🔌 ' . esc_html__('Link-Local Address (IPv4)', 'login-security-with-telegram-alerts');
        }
    } elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
        if (strtolower(substr($ip, 0, 2)) === 'fc' || strtolower(substr($ip, 0, 2)) === 'fd') {
            return '🌐 ' . esc_html__('IPv6 Unique Local Address', 'login-security-with-telegram-alerts');
        }
        if (strtolower(substr($ip, 0, 4)) === 'fe80') {
            return '🔌 ' . esc_html__('IPv6 Link-Local Address', 'login-security-with-telegram-alerts');
        }
    }
    return '🏠 ' . esc_html__('Internal/Private Network', 'login-security-with-telegram-alerts');
}