<?php
/**
 * Handles all Telegram notification functionalities.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class LSEC_Telegram_Notifier {

    public function __construct() {
        add_action('wp_ajax_lsec_send_test_telegram', [$this, 'ajax_send_test_telegram']);
        add_action('lsec_send_successful_login_notification', [$this, 'send_successful_login_notification'], 10, 2);
        add_action('lsec_send_failed_login_notification', [$this, 'send_failed_login_telegram_notification'], 10, 4);
        add_action('lsec_send_account_lockout_notification', [$this, 'send_account_lockout_notification'], 10, 4);
    }

    /**
     * Sends a message to Telegram using the configured bot token and chat ID.
     *
     * @param string $bot_token The Telegram bot token.
     * @param string $chat_id   The Telegram chat ID.
     * @param string $message   The message to send.
     * @return WP_Error|array The response from wp_remote_post or WP_Error on failure.
     */
    private function send_telegram_message($bot_token, $chat_id, $message) {
        $telegram_url = "https://api.telegram.org/bot{$bot_token}/sendMessage";
        $args = [
            'body'    => json_encode([
                'chat_id'              => $chat_id,
                'text'                 => $message,
                'parse_mode'           => 'Markdown',
                'disable_web_page_preview' => true,
                'disable_notification' => false,
            ]),
            'headers' => ['Content-Type' => 'application/json; charset=utf-8'],
            'timeout' => 10,
        ];
        return wp_remote_post($telegram_url, $args);
    }

    /**
     * AJAX handler for sending a test Telegram message from the settings page.
     */
    public function ajax_send_test_telegram() {
        check_ajax_referer('lsec_send_test_telegram_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('You do not have permission to perform this action.', 'login-security-with-telegram-alerts')]);
            return;
        }

        $bot_token = get_option('lsec_bot_token');
        $chat_id   = get_option('lsec_chat_id');

        if (empty($bot_token) || empty($chat_id)) {
            wp_send_json_error(['message' => esc_html__('Bot Token or Chat ID is not set. Please configure them in the settings and save changes.', 'login-security-with-telegram-alerts')]);
            return;
        }

        $site_name    = get_bloginfo('name');
        $test_message = "🔔 *Test Message from Login Security Plugin*\n\n";
        $test_message .= "This is a test notification from your website: *{$site_name}*.\n";
        $test_message .= "If you received this, your Telegram notification settings are working correctly!\n\n";
        $test_message .= "🕒 *Time:* " . current_time('Y-m-d H:i:s T');

        $response = $this->send_telegram_message($bot_token, $chat_id, $test_message);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => esc_html($response->get_error_message())]);
        } else {
            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            $decoded_body  = json_decode($response_body, true);

            if ($response_code === 200 && isset($decoded_body['ok']) && $decoded_body['ok'] === true) {
                wp_send_json_success(['message' => esc_html__('Test message sent successfully! Please check your Telegram.', 'login-security-with-telegram-alerts')]);
            } else {
                $error_description = isset($decoded_body['description']) ? esc_html($decoded_body['description']) : esc_html__('Unknown error from Telegram API.', 'login-security-with-telegram-alerts');
                wp_send_json_error(['message' => $error_description . ' (Code: ' . intval($response_code) . ')']);
            }
        }
    }

    /**
     * Sends a Telegram notification for a successful login.
     * Hooked to 'lsec_send_successful_login_notification' action.
     *
     * @param string  $user_login The user's login.
     * @param WP_User $user       The WP_User object.
     */
    public function send_successful_login_notification($user_login, $user) {
        $notify_roles = get_option('lsec_notify_roles', ['administrator']);
        if (empty($notify_roles) || !is_object($user) || empty($user->roles) || !array_intersect($user->roles, (array) $notify_roles)) {
            return;
        }

        $bot_token = get_option('lsec_bot_token');
        $chat_id   = get_option('lsec_chat_id');
        if (empty($bot_token) || empty($chat_id)) {
            return;
        }

        $site_name  = get_bloginfo('name');
        $site_url   = get_site_url();
        $user_roles_arr = (array) $user->roles;
        $user_role_display = !empty($user_roles_arr) ? implode(', ', array_map('ucfirst', $user_roles_arr)) : esc_html__('N/A', 'login-security-with-telegram-alerts');
        $public_ip = lsec_get_client_ip();
        $ip_link   = "[{$public_ip}](https://ip-api.com/#{$public_ip})";

        $message   = "✅ *Successful Login on {$site_name}*\n";
        $message  .= "🌐 {$site_url}\n\n";
        $message  .= "👤 *Username:* `{$user_login}`\n";
        $message  .= "📧 *Email:* `{$user->user_email}`\n";
        $message  .= "👥 *Role(s):* {$user_role_display}\n";
        $message  .= "🕒 *Time:* " . current_time('Y-m-d H:i:s T') . "\n";
        $message  .= "🌍 *IP Address:* {$ip_link}\n";

        if (get_option('lsec_enable_geolocation', 1)) {
            $location = lsec_get_ip_geolocation($public_ip);
            if ($location && strpos($location, '<i>') === false) { // Exclude if geolocation failed
                $message .= "📍 *Location:* {$location}\n";
            }
        }
        $this->send_telegram_message($bot_token, $chat_id, $message);
    }

    /**
     * Sends a Telegram notification for a failed login attempt.
     * Hooked to 'lsec_send_failed_login_notification' action.
     *
     * @param string $ip_address           The IP address of the failed attempt.
     * @param string $username_attempted   The username attempted.
     * @param int    $current_attempts_count Current number of attempts from this IP.
     * @param int    $max_attempts_allowed   Maximum allowed attempts before lockout.
     */
    public function send_failed_login_telegram_notification($ip_address, $username_attempted, $current_attempts_count, $max_attempts_allowed) {
        $bot_token = get_option('lsec_bot_token');
        $chat_id   = get_option('lsec_chat_id');
        if (empty($bot_token) || empty($chat_id)) {
            return;
        }

        $site_name = get_bloginfo('name');
        $site_url  = get_site_url();
        $ip_link   = "[{$ip_address}](https://ip-api.com/#{$ip_address})";

        $message   = "⚠️ *Failed Login Attempt on {$site_name}*\n";
        $message  .= "🌐 {$site_url}\n\n";
        $message  .= "👤 *Username Attempted:* `{$username_attempted}`\n";
        $message  .= "🌍 *IP Address:* {$ip_link}\n";
        if ($max_attempts_allowed > 0) {
            $message .= "🔄 *Attempt:* {$current_attempts_count} of {$max_attempts_allowed}\n";
        }
        $message .= "🕒 *Time:* " . current_time('Y-m-d H:i:s T') . "\n";

        if ($max_attempts_allowed > 0 && $current_attempts_count >= $max_attempts_allowed) {
            $lockout_duration = get_option('lsec_lockout_duration', 20);
            $message          .= "🚫 *Status:* IP will be blocked for {$lockout_duration} minutes.\n";
        }

        if (get_option('lsec_enable_geolocation', 1)) {
            $location = lsec_get_ip_geolocation($ip_address);
            if ($location && strpos($location, '<i>') === false) {
                $message .= "📍 *Location:* {$location}\n";
            }
        }
        $this->send_telegram_message($bot_token, $chat_id, $message);
    }

    /**
     * Sends a Telegram notification when an IP address or account is locked out.
     * Hooked to 'lsec_send_account_lockout_notification' action.
     *
     * @param string $ip_address           The IP address that was locked out.
     * @param string $username             The username associated with the lockout.
     * @param int    $failed_attempts      The number of failed attempts leading to lockout.
     * @param int    $lockout_duration_minutes The duration of the lockout in minutes.
     */
    public function send_account_lockout_notification($ip_address, $username, $failed_attempts, $lockout_duration_minutes) {
        $bot_token = get_option('lsec_bot_token');
        $chat_id   = get_option('lsec_chat_id');
        if (empty($bot_token) || empty($chat_id)) {
            return;
        }

        $site_name = get_bloginfo('name');
        $site_url  = get_site_url();
        $ip_link   = "[{$ip_address}](https://ip-api.com/#{$ip_address})";
        $location_info = '';

        if (get_option('lsec_enable_geolocation', 1)) {
            $location = lsec_get_ip_geolocation($ip_address);
            if ($location && strpos($location, '<i>') === false) {
                $location_info = "\n📍 *Location:* " . $location;
            }
        }

        // translators: %1$s: Site Name, %2$s: Site URL, %3$s: IP Address (Markdown link), %4$s: Username, %5$d: Lockout duration in minutes, %6$d: Number of failed attempts, %7$s: Location information (optional)
        $message_format = __("🔒 Account Lockout Alert: %1\$s 🔒\n🌐 %2\$s\n\nIP Address %3\$s attempting to login as user `%4\$s` has been locked out for %5\$d minutes after %6\$d failed attempts.%7\$s", 'login-security-with-telegram-alerts');
        $message = sprintf(
            $message_format,
            $site_name,
            $site_url,
            $ip_link,
            $username,
            $lockout_duration_minutes,
            $failed_attempts,
            $location_info
        );
        $this->send_telegram_message($bot_token, $chat_id, $message);
    }
}