<?php
/**
 * Handles login attempt tracking, IP blocking, and activity logging.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class LSEC_Login_Tracker {

    public function __construct() {
        // These hooks are now correctly referencing the class method using $this
        add_action('login_form_login', [$this, 'check_if_ip_blocked_before_login']);
        add_action('login_form_lostpassword', [$this, 'check_if_ip_blocked_before_login']);
        add_action('login_form_retrievepassword', [$this, 'check_if_ip_blocked_before_login']);
        add_action('login_form_register', [$this, 'check_if_ip_blocked_before_login']);
        add_action('wp_login_failed', [$this, 'track_failed_login'], 20, 2);
        add_action('wp_login', [$this, 'log_successful_login_activity'], 5, 2);
        add_action('wp_login', [$this, 'reset_attempts'], 15, 2);
    }

    /**
     * Checks if a given IP address is currently blocked due to excessive failed attempts.
     *
     * @return bool True if the IP is blocked, false otherwise.
     */
    public function is_ip_blocked() {
        global $wpdb;
        $current_ip = lsec_get_client_ip();
        if (lsec_is_ip_whitelisted($current_ip)) {
            return false;
        }

        $table_failed_attempts = $wpdb->prefix . 'lsec_login_attempts'; // Define table name
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check doesn't need caching.
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_failed_attempts)) == $table_failed_attempts) { // Check if table exists
            $max_attempts_allowed   = (int) get_option('lsec_max_attempts', 5);
            $lockout_duration_minutes = (int) get_option('lsec_lockout_duration', 20);
            $table_failed_attempts_escaped = esc_sql($table_failed_attempts);

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Real-time security check, caching would create vulnerabilities.
            $attempt_data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM `{$table_failed_attempts_escaped}` WHERE ip_address = %s", $current_ip ) );
            if ($attempt_data && $attempt_data->attempts >= $max_attempts_allowed) {
                $lockout_expires_at = strtotime($attempt_data->last_attempt) + ($lockout_duration_minutes * 60);
                if (time() < $lockout_expires_at) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Displays a wp_die message if the IP is blocked before login.
     * This function is hooked to various login forms.
     */
    public function check_if_ip_blocked_before_login() {
        if ($this->is_ip_blocked()) {
            $lockout_duration_minutes = (int) get_option('lsec_lockout_duration', 20);
            $message = sprintf(
                // translators: %d: lockout duration in minutes
                esc_html__('Too many failed login attempts from your IP address. Access is temporarily blocked for %d minutes.', 'login-security-with-telegram-alerts'),
                $lockout_duration_minutes
            );
            lsec_centered_wp_die_message($message, esc_html__('Login Attempt Limit Reached', 'login-security-with-telegram-alerts'));
        }
    }

    /**
     * Tracks failed login attempts and blocks IPs if limits are reached.
     *
     * @param string $username The username attempted.
     * @param WP_Error|null $error The WordPress error object, if any.
     */
    public function track_failed_login($username, $error = null) {
        global $wpdb;
        $ip_address = lsec_get_client_ip();

        if (lsec_is_ip_whitelisted($ip_address)) {
            return;
        }

        $max_attempts_allowed   = (int) get_option('lsec_max_attempts', 5);
        $table_failed_attempts = $wpdb->prefix . 'lsec_login_attempts'; // Define table name
        $table_failed_attempts_escaped = esc_sql($table_failed_attempts);
        $lockout_duration_minutes = (int) get_option('lsec_lockout_duration', 20);

        $this->record_login_activity($username, $ip_address, 'failed');
        $attempt_data   = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM `{$table_failed_attempts_escaped}` WHERE ip_address = %s", $ip_address ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped with esc_sql().
        $new_attempts_count = 1;
        $is_newly_locked    = false;

        if ($attempt_data) {
            if ($attempt_data->attempts >= $max_attempts_allowed) {
                $lockout_expires_at = strtotime($attempt_data->last_attempt) + ($lockout_duration_minutes * 60);
                if (time() < $lockout_expires_at) {
                    $this->record_login_activity($username, $ip_address, 'failed (blocked)');
                    return; // Already blocked and within lockout period, no need to update attempts.
                } else {
                    // Lockout period expired, reset attempts
                    $new_attempts_count = 1;
                }
            } else {
                $new_attempts_count = $attempt_data->attempts + 1;
            }

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Real-time security data update, caching not appropriate.
            $wpdb->update(
                $table_failed_attempts,
                [
                    'attempts'     => $new_attempts_count,
                    'last_attempt' => current_time('mysql', 1),
                    'username'     => $username,
                ],
                ['ip_address' => $ip_address]
            );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Real-time security data insert, caching not appropriate.
        } else {
            $wpdb->insert(
                $table_failed_attempts,
                [
                    'ip_address' => $ip_address,
                    'username'   => $username,
                    'attempts'   => $new_attempts_count,
                    'last_attempt' => current_time('mysql', 1),
                ]
            );
        }

        // Check if the IP is now newly locked out
        if ($new_attempts_count >= $max_attempts_allowed) {
            $is_newly_locked = true;
            $this->record_login_activity($username, $ip_address, 'failed (limit reached)');

            // Automatically add IP to blacklist when max attempts reached
            $this->add_ip_to_blacklist($ip_address);
        }

        // Trigger notifications if conditions are met
        if (get_option('lsec_notify_on_failed_attempt', 1)) {
            // Send failed attempt notification
            do_action('lsec_send_failed_login_notification', $ip_address, $username, $new_attempts_count, $max_attempts_allowed);
        }
        if ($is_newly_locked && get_option('lsec_notify_admin_on_lockout', 1)) {
            // Send lockout notification
            do_action('lsec_send_account_lockout_notification', $ip_address, $username, $new_attempts_count, $lockout_duration_minutes);
        }
    }

    /**
     * Logs successful login activity.
     *
     * @param string  $user_login The user's login.
     * @param WP_User $user       The WP_User object.
     */
    public function log_successful_login_activity($user_login, $user) {
        $ip_address = lsec_get_client_ip();
        $user_roles_arr = (array) $user->roles;
        $user_roles = !empty($user_roles_arr) ? implode(', ', array_map('ucfirst', $user_roles_arr)) : esc_html__('N/A', 'login-security-with-telegram-alerts');

        $this->record_login_activity($user_login, $ip_address, 'success', $user->ID, $user_roles);
        do_action('lsec_send_successful_login_notification', $user_login, $user);
    }

    /**
     * Records any login activity (successful or failed) into the activity log table.
     *
     * @param string      $username   The username involved in the activity.
     * @param string      $ip_address The IP address from which the activity originated.
     * @param string      $status     The status of the login (e.g., 'success', 'failed', 'failed (blocked)').
     * @param int|null    $user_id    The ID of the user, if available.
     * @param string|null $role       The role of the user, if available.
     */
    public function record_login_activity($username, $ip_address, $status, $user_id = null, $role = null) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check doesn't need caching.
        $table_all_activity = $wpdb->prefix . 'lsec_all_login_activity';
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_all_activity)) != $table_all_activity) {
            return;
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized in next line.
        $_user_agent_raw = isset($_SERVER['HTTP_USER_AGENT']) ? wp_unslash($_SERVER['HTTP_USER_AGENT']) : null;
        $user_agent = $_user_agent_raw ? sanitize_textarea_field($_user_agent_raw) : null;

        // Truncate user agent if it's too long for the 'text' column type
        if ($user_agent && mb_strlen($user_agent) > 65530) {
            $user_agent = mb_substr($user_agent, 0, 65530);
        }

        $data = [
            'username'   => sanitize_user($username, true),
            'ip_address' => $ip_address,
            'login_time' => current_time('mysql', 1),
            'status'     => sanitize_text_field($status),
            'user_agent' => $user_agent,
        ];
        if ($user_id !== null) {
            $data['user_id'] = absint($user_id);
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Activity log insert, real-time operation.
        if ($role !== null) {
            $data['role'] = sanitize_text_field($role);
        }
        $wpdb->insert($table_all_activity, $data);
    }

    /**
     * Resets failed login attempts for a successfully logged-in IP.
     *
     * @param string  $user_login The user's login.
     * @param WP_User $user       The WP_User object.
     */
    public function reset_attempts($user_login, $user) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'lsec_login_attempts';
        $ip_address = lsec_get_client_ip();

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Delete failed attempts on successful login, real-time operation.
        if (lsec_is_ip_whitelisted($ip_address)) {
            return;
        }
        $wpdb->delete($table_name, ['ip_address' => $ip_address]);
    }

    /**
     * Retrieves the count of failed login attempts for a given IP address.
     * Used by the settings page to determine if an IP is currently blocked.
     *
     * @param string $ip_address The IP address to check.
     * @return object|null Row object with attempt data, or null if not found.
     */
    public function get_failed_attempts_data($ip_address) {
        global $wpdb;
        $table_failed_attempts = $wpdb->prefix . 'lsec_login_attempts';
        $table_failed_attempts_escaped = esc_sql($table_failed_attempts);
        return $wpdb->get_row( $wpdb->prepare( "SELECT ip_address, last_attempt, attempts FROM `{$table_failed_attempts_escaped}` WHERE ip_address = %s", $ip_address ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped with esc_sql().
    }

    /**
     * Handles unblocking an IP address from the failed attempts table and blacklist.
     * This is triggered by a URL action from the admin activity log.
     */
    public function handle_unblock_ip() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce check is performed below after retrieving IP for nonce string.
        if (isset($_GET['lsec_unblock_ip'], $_GET['lsec_unblock_nonce']) && current_user_can('manage_options')) {
            $ip_address = sanitize_text_field(wp_unslash($_GET['lsec_unblock_ip'])); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- IP is retrieved to form part of the nonce string.
            if (lsec_is_ip_whitelisted($ip_address)) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Preliminary check before nonce.
                add_action('admin_notices', function() use ($ip_address) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Displaying a notice.
                    echo '<div class="notice notice-info is-dismissible"><p>' . sprintf(
                        // translators: %s: IP address
                        esc_html__('IP address %s is whitelisted and cannot be dynamically blocked or unblocked by this action.', 'login-security-with-telegram-alerts'),
                        esc_html($ip_address)
                    ) . '</p></div>';
                });
                return;
            }

            $nonce_value = sanitize_text_field(wp_unslash($_GET['lsec_unblock_nonce'])); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce value retrieved for verification.

            if (wp_verify_nonce($nonce_value, 'lsec_unblock_ip_action_' . $ip_address)) {
                $paged_activity    = isset($_GET['paged_activity']) ? intval(wp_unslash($_GET['paged_activity'])) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Accessed after nonce for redirect.
                $current_filter    = isset($_GET['lsec_log_filter']) ? sanitize_key(wp_unslash($_GET['lsec_log_filter'])) : 'all'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Accessed after nonce for redirect.
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- IP unblock operation, real-time security action.

                global $wpdb;
                $table_name = $wpdb->prefix . 'lsec_login_attempts';
                $wpdb->delete($table_name, ['ip_address' => $ip_address]);

                // Also remove from blacklist
                $this->remove_ip_from_blacklist($ip_address);

                $redirect_url_params = [
                    'page'            => 'login-security-telegram-alerts-settings',
                    'lsec_unblocked'  => '1',
                    'unblocked_ip'    => $ip_address,
                    'paged_activity'  => $paged_activity,
                    'lsec_log_filter' => $current_filter,
                ];
                $redirect_url = add_query_arg($redirect_url_params, admin_url('options-general.php')) . '#tab-activity-log-display';
                wp_safe_redirect($redirect_url);
                exit;
            } else {
                wp_die(esc_html__('Invalid nonce.', 'login-security-with-telegram-alerts'), esc_html__('Error', 'login-security-with-telegram-alerts'), ['response' => 403]);
            }
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These GET params are for displaying notices post-redirect.
        if (isset($_GET['lsec_unblocked']) && $_GET['lsec_unblocked'] == '1' && isset($_GET['page']) && $_GET['page'] === 'login-security-telegram-alerts-settings') {
            add_action('admin_notices', function() {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Displaying notice based on GET param.
                $unblocked_ip_msg = isset($_GET['unblocked_ip']) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    ? sprintf(
                        // translators: %s: IP address
                        esc_html__('IP %s unblocked successfully.', 'login-security-with-telegram-alerts'),
                        esc_html(sanitize_text_field(wp_unslash($_GET['unblocked_ip']))) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    )
                    : esc_html__('IP unblocked successfully.', 'login-security-with-telegram-alerts');
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                echo '<div class="notice notice-success is-dismissible"><p>' . $unblocked_ip_msg . '</p></div>';
            });
        }
    }

    /**
     * Handles blocking an IP address by adding it to the manual blacklist.
     * This is triggered by a URL action from the admin activity log.
     */
    public function handle_block_ip() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce check is performed below after retrieving IP for nonce string.
        if (isset($_GET['lsec_block_ip'], $_GET['lsec_block_nonce']) && current_user_can('manage_options')) {
            $ip_address = sanitize_text_field(wp_unslash($_GET['lsec_block_ip'])); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- IP is retrieved to form part of the nonce string.

            if (lsec_is_ip_whitelisted($ip_address)) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Preliminary check before nonce.
                add_action('admin_notices', function() use ($ip_address) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Displaying a notice.
                    echo '<div class="notice notice-info is-dismissible"><p>' . sprintf(
                        // translators: %s: IP address
                        esc_html__('IP address %s is whitelisted and cannot be blocked.', 'login-security-with-telegram-alerts'),
                        esc_html($ip_address)
                    ) . '</p></div>';
                });
                return;
            }

            $nonce_value = sanitize_text_field(wp_unslash($_GET['lsec_block_nonce'])); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce value retrieved for verification.

            if (wp_verify_nonce($nonce_value, 'lsec_block_ip_action_' . $ip_address)) {
                $paged_activity    = isset($_GET['paged_activity']) ? intval(wp_unslash($_GET['paged_activity'])) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Accessed after nonce for redirect.
                $current_filter    = isset($_GET['lsec_log_filter']) ? sanitize_key(wp_unslash($_GET['lsec_log_filter'])) : 'all'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Accessed after nonce for redirect.

                // Add IP to blacklist
                $this->add_ip_to_blacklist($ip_address);

                $redirect_url_params = [
                    'page'            => 'login-security-telegram-alerts-settings',
                    'lsec_blocked'    => '1',
                    'blocked_ip'      => $ip_address,
                    'paged_activity'  => $paged_activity,
                    'lsec_log_filter' => $current_filter,
                ];
                $redirect_url = add_query_arg($redirect_url_params, admin_url('options-general.php')) . '#tab-activity-log-display';
                wp_safe_redirect($redirect_url);
                exit;
            } else {
                wp_die(esc_html__('Invalid nonce.', 'login-security-with-telegram-alerts'), esc_html__('Error', 'login-security-with-telegram-alerts'), ['response' => 403]);
            }
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These GET params are for displaying notices post-redirect.
        if (isset($_GET['lsec_blocked']) && $_GET['lsec_blocked'] == '1' && isset($_GET['page']) && $_GET['page'] === 'login-security-telegram-alerts-settings') {
            add_action('admin_notices', function() {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Displaying notice based on GET param.
                $blocked_ip_msg = isset($_GET['blocked_ip']) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    ? sprintf(
                        // translators: %s: IP address
                        esc_html__('IP %s blocked successfully.', 'login-security-with-telegram-alerts'),
                        esc_html(sanitize_text_field(wp_unslash($_GET['blocked_ip']))) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    )
                    : esc_html__('IP blocked successfully.', 'login-security-with-telegram-alerts');
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                echo '<div class="notice notice-success is-dismissible"><p>' . $blocked_ip_msg . '</p></div>';
            });
        }
    }

    /**
     * Handles clearing the entire activity log table.
     * This is triggered by form submission from the admin settings page.
     */
    public function handle_clear_activity_log() {
        if (isset($_POST['lsec_clear_activity_log_submit'], $_POST['lsec_clear_activity_log_nonce']) && current_user_can('manage_options')) {
            if (check_admin_referer('lsec_clear_activity_log_action', 'lsec_clear_activity_log_nonce')) {
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check doesn't need caching.
                $table_name = $wpdb->prefix . 'lsec_all_login_activity';
                if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) == $table_name) {
                    $table_name_escaped = esc_sql($table_name);
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped with esc_sql() and only contains prefix + hardcoded string.
                    $wpdb->query("TRUNCATE TABLE `{$table_name_escaped}`");
                }
                wp_safe_redirect(add_query_arg('lsec_activity_log_cleared', '1', admin_url('options-general.php?page=login-security-telegram-alerts-settings#tab-activity-log-display')));
                exit;
            } else {
                wp_die(esc_html__('Invalid nonce.', 'login-security-with-telegram-alerts'), esc_html__('Error', 'login-security-with-telegram-alerts'), ['response' => 403]);
            }
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Value used for admin notice display after redirect.
        if (isset($_GET['lsec_activity_log_cleared']) && $_GET['lsec_activity_log_cleared'] == '1') {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Activity log cleared.', 'login-security-with-telegram-alerts') . '</p></div>';
            });
        }
    }

    /**
     * Handles clearing the failed login attempts log table.
     * This is triggered by form submission from the admin settings page.
     */
    public function handle_clear_failed_log() {
        if (isset($_POST['lsec_clear_failed_log_submit'], $_POST['lsec_clear_failed_log_nonce']) && current_user_can('manage_options')) {
            if (check_admin_referer('lsec_clear_failed_log_action', 'lsec_clear_failed_log_nonce')) {
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check doesn't need caching.
                $table_name = $wpdb->prefix . 'lsec_login_attempts';
                if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) == $table_name) {
                    $table_name_escaped = esc_sql($table_name);
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is properly escaped with esc_sql() and only contains prefix + hardcoded string.
                    $wpdb->query("TRUNCATE TABLE `{$table_name_escaped}`");
                }
                wp_safe_redirect(add_query_arg('lsec_failed_log_cleared', '1', admin_url('options-general.php?page=login-security-telegram-alerts-settings#tab-security-settings')));
                exit;
            } else {
                wp_die(esc_html__('Invalid nonce.', 'login-security-with-telegram-alerts'), esc_html__('Error', 'login-security-with-telegram-alerts'), ['response' => 403]);
            }
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Value used for admin notice display after redirect.
        if (isset($_GET['lsec_failed_log_cleared']) && $_GET['lsec_failed_log_cleared'] == '1') {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Failed login attempts log cleared.', 'login-security-with-telegram-alerts') . '</p></div>';
            });
        }
    }

    /**
     * Adds an IP address to the manual blacklist.
     * This prevents the IP from accessing the site entirely.
     *
     * @param string $ip_address The IP address to blacklist.
     * @return bool True if successfully added, false if already blacklisted or invalid.
     */
    public function add_ip_to_blacklist($ip_address) {
        // Validate IP address
        if (!filter_var($ip_address, FILTER_VALIDATE_IP)) {
            return false;
        }

        // Get current blacklist
        $blacklist_str = get_option('lsec_manual_blacklist_ips', '');
        $blacklisted_ips = lsec_parse_ip_list($blacklist_str);

        // Check if IP is already blacklisted
        if (in_array($ip_address, $blacklisted_ips, true)) {
            return false; // Already blacklisted
        }

        // Add IP to blacklist
        $blacklisted_ips[] = $ip_address;
        $new_blacklist_str = implode("\n", $blacklisted_ips);

        // Update option
        update_option('lsec_manual_blacklist_ips', $new_blacklist_str);

        return true;
    }

    /**
     * Removes an IP address from the manual blacklist.
     *
     * @param string $ip_address The IP address to remove from blacklist.
     * @return bool True if successfully removed, false if not in blacklist or invalid.
     */
    public function remove_ip_from_blacklist($ip_address) {
        // Validate IP address
        if (!filter_var($ip_address, FILTER_VALIDATE_IP)) {
            return false;
        }

        // Get current blacklist
        $blacklist_str = get_option('lsec_manual_blacklist_ips', '');
        $blacklisted_ips = lsec_parse_ip_list($blacklist_str);

        // Check if IP is in blacklist
        $key = array_search($ip_address, $blacklisted_ips, true);
        if ($key === false) {
            return false; // Not in blacklist
        }

        // Remove IP from blacklist
        unset($blacklisted_ips[$key]);
        $new_blacklist_str = implode("\n", $blacklisted_ips);

        // Update option
        update_option('lsec_manual_blacklist_ips', $new_blacklist_str);

        return true;
    }

    /**
     * Fetches activity log entries based on filter and pagination.
     *
     * @param string $filter 'all', 'success', or 'failed'.
     * @param int    $paged  Current page number.
     * @param int    $items_per_page Number of items per page.
     * @return array Contains 'entries' and 'total_count'.
     */
    public function get_activity_log_entries($filter = 'all', $paged = 1, $items_per_page = 50) {
        global $wpdb;
        $table_activity_log = $wpdb->prefix . 'lsec_all_login_activity';
        $table_activity_log_escaped = esc_sql($table_activity_log);

        $total_activity_count = 0;
        $activity_log_entries = [];
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check doesn't need caching.

        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_activity_log)) !== $table_activity_log) {
            return ['entries' => [], 'total_count' => 0];
        }

        $offset_activity = ($paged - 1) * $items_per_page;

        $sql_base_select = "SELECT * FROM `{$table_activity_log_escaped}`";
        $sql_base_count = "SELECT COUNT(*) FROM `{$table_activity_log_escaped}`";
        $where_conditions_sql = "";
        $params = [];

        if ($filter === 'success') {
            $where_conditions_sql = " WHERE status = %s"; // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- This is the format string for prepare.
            $params[] = 'success';
        } elseif ($filter === 'failed') {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- This is the format string for prepare.
            $where_conditions_sql = " WHERE status LIKE %s OR status LIKE %s OR status LIKE %s";
            $params[] = 'failed';
            $params[] = 'failed (blocked)%';
            $params[] = 'failed (limit reached)%';
        }

        $count_sql = $sql_base_count . $where_conditions_sql;
        // Always prepare the SQL query. The spread operator (...) correctly handles an empty $params array.
        // $wpdb->prepare can return null or false on error (e.g., if $count_sql is invalid or $params mismatch placeholders).

        if (empty($params)) {
            // If there are no parameters, $count_sql (e.g., "SELECT COUNT(*) FROM `table`") is static and has no placeholders.
            // It should be used directly with $wpdb->get_var() to avoid the "must have a placeholder" warning from wpdb::prepare().
            // Table name is from $wpdb->prefix and is considered safe in this context.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- No placeholders in static query, table name is escaped with esc_sql().
            $total_activity_count = (int) $wpdb->get_var( $count_sql );
        } else {
            // $params is not empty, so $count_sql (e.g., "SELECT COUNT(*) FROM `table` WHERE ...") has placeholders.
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- $count_sql will be prepared on the next line with $wpdb->prepare().
            $prepared_count_sql = $wpdb->prepare($count_sql, ...$params);
            if ($prepared_count_sql) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- $prepared_count_sql is the result of $wpdb->prepare() with all parameters properly escaped.
                $total_activity_count = (int) $wpdb->get_var($prepared_count_sql);
            } else {
                // If preparation fails, default to 0 to prevent further errors.
                // Consider logging $wpdb->last_error for debugging.
                $total_activity_count = 0;
            }
        }

        if ($total_activity_count > 0) {
            $activity_sql_format_string = $sql_base_select . $where_conditions_sql . " ORDER BY login_time DESC LIMIT %d OFFSET %d";
            $activity_params = $params; // Start with filter params
            $activity_params[] = $items_per_page;
            $activity_params[] = $offset_activity;
            $prepared_activity_sql = $wpdb->prepare($activity_sql_format_string, ...$activity_params); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- $activity_sql_format_string is a dynamically constructed format string, all parameters are properly prepared.
            if ($prepared_activity_sql) {
                $activity_log_entries = $wpdb->get_results($prepared_activity_sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter -- $prepared_activity_sql is the result of $wpdb->prepare() with all parameters properly escaped.
            } else {
                $activity_log_entries = [];
            }
        }

        return ['entries' => $activity_log_entries, 'total_count' => $total_activity_count];
    }
}