<?php
/**
 * Handles WordPress core files integrity checking and file permissions management.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class LSEC_File_Integrity {

    /**
     * Checks WordPress core files for modifications.
     *
     * @return array Array containing 'status', 'message', 'modified_files', and 'missing_files'.
     */
    public function check_core_files() {
        // Require WordPress admin files
        if (!function_exists('get_core_checksums')) {
            require_once ABSPATH . 'wp-admin/includes/update.php';
        }

        global $wp_version, $wp_local_package;

        // Get checksums from WordPress.org API
        $checksums = get_core_checksums($wp_version, isset($wp_local_package) ? $wp_local_package : 'en_US');

        if (!is_array($checksums)) {
            return [
                'status'         => 'error',
                'message'        => __('Could not retrieve checksums from WordPress.org. Please check your internet connection.', 'login-security-with-telegram-alerts'),
                'modified_files' => [],
                'missing_files'  => [],
            ];
        }

        $modified_files = [];
        $missing_files = [];

        foreach ($checksums as $file => $checksum) {
            $filepath = ABSPATH . $file;

            // Skip if file doesn't exist
            if (!file_exists($filepath)) {
                $missing_files[] = $file;
                continue;
            }

            // Calculate file hash
            $file_hash = md5_file($filepath);

            // Compare with expected checksum
            if ($file_hash !== $checksum) {
                $modified_files[] = $file;
            }
        }

        // Determine status
        $total_issues = count($modified_files) + count($missing_files);

        if ($total_issues === 0) {
            $status = 'success';
            $message = __('All WordPress core files are intact and unmodified.', 'login-security-with-telegram-alerts');
        } else {
            $status = 'warning';
            $message = sprintf(
                // translators: %d: number of issues found
                _n('%d issue found with WordPress core files.', '%d issues found with WordPress core files.', $total_issues, 'login-security-with-telegram-alerts'),
                $total_issues
            );
        }

        return [
            'status'         => $status,
            'message'        => $message,
            'modified_files' => $modified_files,
            'missing_files'  => $missing_files,
        ];
    }

    /**
     * Checks file permissions for WordPress installation.
     *
     * @return array Array containing 'status', 'message', and 'issues'.
     */
    public function check_file_permissions() {
        $issues = [];

        // Check wp-config.php permissions
        $wp_config_file = ABSPATH . 'wp-config.php';
        if (file_exists($wp_config_file)) {
            $perms = substr(sprintf('%o', fileperms($wp_config_file)), -4);
            if ($perms !== '0400' && $perms !== '0440' && $perms !== '0600' && $perms !== '0640' && $perms !== '0644') {
                $issues[] = [
                    'file'        => 'wp-config.php',
                    'current'     => $perms,
                    'recommended' => '0600',
                    'description' => __('Configuration file should be readable only by owner for security.', 'login-security-with-telegram-alerts'),
                ];
            }
        }

        // Check .htaccess permissions
        $htaccess_file = ABSPATH . '.htaccess';
        if (file_exists($htaccess_file)) {
            $perms = substr(sprintf('%o', fileperms($htaccess_file)), -4);
            if ($perms !== '0644' && $perms !== '0604' && $perms !== '0600') {
                $issues[] = [
                    'file'        => '.htaccess',
                    'current'     => $perms,
                    'recommended' => '0644',
                    'description' => __('Htaccess file permissions should be 644.', 'login-security-with-telegram-alerts'),
                ];
            }
        }

        // Check wp-content directory
        $wp_content_dir = WP_CONTENT_DIR;
        if (is_dir($wp_content_dir)) {
            $perms = substr(sprintf('%o', fileperms($wp_content_dir)), -4);
            if ($perms !== '0755' && $perms !== '0775') {
                $issues[] = [
                    'file'        => 'wp-content/',
                    'current'     => $perms,
                    'recommended' => '0755',
                    'description' => __('wp-content directory should be 755 for proper functionality.', 'login-security-with-telegram-alerts'),
                ];
            }
        }

        // Check uploads directory
        $upload_dir = wp_upload_dir();
        if (is_dir($upload_dir['basedir'])) {
            $perms = substr(sprintf('%o', fileperms($upload_dir['basedir'])), -4);
            if ($perms !== '0755' && $perms !== '0775') {
                $issues[] = [
                    'file'        => 'wp-content/uploads/',
                    'current'     => $perms,
                    'recommended' => '0755',
                    'description' => __('Uploads directory should be 755.', 'login-security-with-telegram-alerts'),
                ];
            }
        }

        // Check plugins directory
        $plugins_dir = WP_PLUGIN_DIR;
        if (is_dir($plugins_dir)) {
            $perms = substr(sprintf('%o', fileperms($plugins_dir)), -4);
            if ($perms !== '0755' && $perms !== '0775') {
                $issues[] = [
                    'file'        => 'wp-content/plugins/',
                    'current'     => $perms,
                    'recommended' => '0755',
                    'description' => __('Plugins directory should be 755.', 'login-security-with-telegram-alerts'),
                ];
            }
        }

        // Check themes directory
        $themes_dir = get_theme_root();
        if (is_dir($themes_dir)) {
            $perms = substr(sprintf('%o', fileperms($themes_dir)), -4);
            if ($perms !== '0755' && $perms !== '0775') {
                $issues[] = [
                    'file'        => 'wp-content/themes/',
                    'current'     => $perms,
                    'recommended' => '0755',
                    'description' => __('Themes directory should be 755.', 'login-security-with-telegram-alerts'),
                ];
            }
        }

        // Determine status
        if (count($issues) === 0) {
            $status = 'success';
            $message = __('All checked file permissions are correct.', 'login-security-with-telegram-alerts');
        } else {
            $status = 'warning';
            $message = sprintf(
                // translators: %d: number of permission issues
                _n('%d file permission issue found.', '%d file permission issues found.', count($issues), 'login-security-with-telegram-alerts'),
                count($issues)
            );
        }

        return [
            'status'  => $status,
            'message' => $message,
            'issues'  => $issues,
        ];
    }

    /**
     * Attempts to fix file permissions using WP_Filesystem.
     *
     * @param array $files Array of file paths to fix.
     * @return array Results of fix attempts.
     */
    public function fix_file_permissions($files) {
        $results = [];

        // Initialize WP_Filesystem
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }

        foreach ($files as $file_info) {
            $filepath = ABSPATH . $file_info['file'];
            $recommended_perms = octdec($file_info['recommended']);

            if (!$wp_filesystem->exists($filepath)) {
                $results[$file_info['file']] = [
                    'success' => false,
                    'message' => __('File does not exist.', 'login-security-with-telegram-alerts'),
                ];
                continue;
            }

            // Attempt to change permissions using WP_Filesystem
            $success = $wp_filesystem->chmod($filepath, $recommended_perms);

            if ($success) {
                $results[$file_info['file']] = [
                    'success' => true,
                    'message' => __('Permissions updated successfully.', 'login-security-with-telegram-alerts'),
                ];
            } else {
                $results[$file_info['file']] = [
                    'success' => false,
                    'message' => __('Failed to update permissions. You may need to change this manually via FTP.', 'login-security-with-telegram-alerts'),
                ];
            }
        }

        return $results;
    }
}
