<?php
/**
 * Handles the admin settings page for the plugin.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class LSEC_Admin_Settings {

    private $login_tracker;
    private $telegram_notifier;

    public function __construct(LSEC_Login_Tracker $login_tracker, LSEC_Telegram_Notifier $telegram_notifier) {
        $this->login_tracker     = $login_tracker;
        $this->telegram_notifier = $telegram_notifier;

        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'admin_enqueue_assets']);
        add_action('admin_init', [$this->login_tracker, 'handle_unblock_ip']);
        add_action('admin_init', [$this->login_tracker, 'handle_block_ip']);
        add_action('admin_init', [$this->login_tracker, 'handle_clear_failed_log']);
        add_action('admin_init', [$this->login_tracker, 'handle_clear_activity_log']);

        // IMPORTANT: Add this filter to capture the redirect after saving options
        add_filter('wp_redirect', [$this, 'filter_options_page_redirect'], 10, 2);
    }

    /**
     * Filters the redirect URL after a form submission to options.php
     * to ensure the correct tab remains active.
     *
     * @param string $location The redirect URL.
     * @param int    $status   The HTTP redirect status code.
     * @return string The modified redirect URL.
     */
    public function filter_options_page_redirect($location, $status) {
        // Check if we're on the options.php page and our plugin's options group was just saved.
        // Also ensure the 'option_page' is set in $_POST, indicating a settings submission.
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- We are checking a POST variable for context, not security.
        if (isset($_POST['option_page']) && $_POST['option_page'] === 'lsec_options_group') {
            // Check if this redirect is happening *after* WordPress has processed the settings update.
            // options.php adds 'settings-updated=true' to the URL after a successful save.
            if (strpos($location, 'settings-updated=true') !== false) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- We are reading a POST variable passed from JS.
                $current_tab_id = isset($_POST['lsec_current_tab']) ? sanitize_text_field(wp_unslash($_POST['lsec_current_tab'])) : '';

                // Ensure it's our plugin's settings page being redirected to
                if (!empty($current_tab_id) && strpos($location, 'page=login-security-telegram-alerts-settings') !== false) {
                    // Append the current tab ID as a query parameter
                    $location = add_query_arg('lsec_active_tab', $current_tab_id, $location);
                }
            }
        }
        return $location;
    }

    /**
     * Enqueues admin scripts and styles for the settings page.
     *
     * @param string $hook_suffix The current admin page hook.
     */
    public function admin_enqueue_assets($hook_suffix) {
        if ('settings_page_login-security-telegram-alerts-settings' !== $hook_suffix) {
            return;
        }
        $plugin_url = LSEC_PLUGIN_URL;
        $version    = LSEC_VERSION;
        wp_enqueue_style('lsec-admin-styles', $plugin_url . 'assets/css/style.css', [], $version);
        wp_enqueue_script('lsec-admin-script', $plugin_url . 'assets/js/script.js', ['jquery'], $version, true);

        $send_test_button_original_text = __('Send Test Telegram Message', 'login-security-with-telegram-alerts');
        wp_localize_script('lsec-admin-script', 'lsec_ajax_object', [
            'ajax_url'            => admin_url('admin-ajax.php'),
            'lsec_test_nonce'     => wp_create_nonce('lsec_send_test_telegram_nonce'),
            'sending_text'        => __('Sending...', 'login-security-with-telegram-alerts'),
            'success_message'     => __('Test message sent successfully! Please check your Telegram.', 'login-security-with-telegram-alerts'),
            'error_message'       => __('Failed to send test message. Please check your Bot Token and Chat ID.', 'login-security-with-telegram-alerts'),
            'request_failed_text' => __('Request failed. Please try again.', 'login-security-with-telegram-alerts'),
            'send_test_button_text' => $send_test_button_original_text,
        ]);
    }

    /**
     * Registers plugin settings with WordPress.
     */
    public function register_settings() {
        $option_group = 'lsec_options_group';

        register_setting($option_group, 'lsec_max_attempts', [
            'type'              => 'integer',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_positive_int'], // Now static
            'default'           => 5,
        ]);
        register_setting($option_group, 'lsec_lockout_duration', [
            'type'              => 'integer',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_positive_int'], // Now static
            'default'           => 20,
        ]);
        register_setting($option_group, 'lsec_bot_token', [
            'type'              => 'string',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_bot_token'], // Now static
            'default'           => '',
        ]);
        register_setting($option_group, 'lsec_chat_id', [
            'type'              => 'string',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_chat_id'], // Now static
            'default'           => '',
        ]);
        register_setting($option_group, 'lsec_notify_roles', [
            'type'              => 'array',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_roles'], // Now static
            'default'           => ['administrator'],
        ]);
        register_setting($option_group, 'lsec_enable_geolocation', [
            'type'              => 'integer',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_checkbox'],
            'default'           => 1,
        ]);
        register_setting($option_group, 'lsec_notify_on_failed_attempt', [
            'type'              => 'integer',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_checkbox'],
            'default'           => 1,
        ]);
        register_setting($option_group, 'lsec_notify_admin_on_lockout', [
            'type'              => 'integer',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_checkbox'],
            'default'           => 1,
        ]);
        register_setting($option_group, 'lsec_manual_blacklist_ips', [
            'type'              => 'string',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_ip_list_textarea'], // Now static
            'default'           => '',
        ]);
        register_setting($option_group, 'lsec_manual_whitelist_ips', [
            'type'              => 'string',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_ip_list_textarea'], // Now static
            'default'           => '',
        ]);
        register_setting($option_group, 'lsec_custom_admin_url', [
            'type'              => 'string',
            'sanitize_callback' => ['LSEC_Admin_Settings', 'sanitize_custom_admin_url'], // Now static
            'default'           => '',
        ]);
    }

    /**
     * Sanitizes a positive integer input.
     *
     * @param mixed $input The input value.
     * @return int The sanitized positive integer.
     */
    public static function sanitize_positive_int($input) {
        return max(1, intval($input));
    }

    /**
     * Sanitizes checkbox input (converts to integer).
     *
     * @param mixed $input The input value.
     * @return int 1 if checked, 0 otherwise.
     */
    public static function sanitize_checkbox($input) {
        // Convert to integer: 1 for truthy values, 0 for falsy values
        return ($input && $input !== '0') ? 1 : 0;
    }

    /**
     * Sanitizes the bot token, preserving existing token if hidden.
     *
     * @param string $input The input bot token.
     * @return string The sanitized bot token.
     */
    public static function sanitize_bot_token($input) {
        $current = get_option('lsec_bot_token');
        // If the input matches the masked format of the current token, return the current token.
        if ($current && $input === preg_replace('/(.{4}).+(.{4})/', '$1****$2', $current) && !empty($current)) {
            return $current;
        }
        return sanitize_text_field($input);
    }

    /**
     * Sanitizes the chat ID, preserving existing chat ID if hidden.
     *
     * @param string $input The input chat ID.
     * @return string The sanitized chat ID.
     */
    public static function sanitize_chat_id($input) {
        $current = get_option('lsec_chat_id');
        // If the input matches the masked format of the current chat ID, return the current chat ID.
        if ($current && $input === '****' . substr($current, -4) && !empty($current)) {
            return $current;
        }
        return sanitize_text_field($input);
    }

    /**
     * Sanitizes an array of WordPress roles.
     *
     * @param array $input_roles An array of roles from input.
     * @return array An array of sanitized and valid roles.
     */
    public static function sanitize_roles($input_roles) {
        $valid_roles = array_keys(get_editable_roles());
        $sanitized_roles = [];
        if (is_array($input_roles)) {
            foreach ($input_roles as $role) {
                $s_role = sanitize_text_field($role);
                if (in_array($s_role, $valid_roles, true)) {
                    $sanitized_roles[] = $s_role;
                }
            }
        }
        return $sanitized_roles;
    }

    /**
     * Sanitizes a textarea input containing a list of IP addresses.
     *
     * @param string $input The input string.
     * @return string A string of unique, valid IP addresses separated by newlines.
     */
    public static function sanitize_ip_list_textarea($input) {
        if (!is_string($input)) {
            return '';
        }
        $ips = explode("\n", $input);
        $sanitized_ips = [];
        foreach ($ips as $ip) {
            $trimmed_ip = trim($ip);
            if (!empty($trimmed_ip) && filter_var($trimmed_ip, FILTER_VALIDATE_IP)) {
                $sanitized_ips[] = $trimmed_ip;
            }
        }
        return implode("\n", array_unique($sanitized_ips));
    }

    /**
     * Sanitizes the custom admin URL slug.
     *
     * @param string $input The input URL slug.
     * @return string The sanitized URL slug.
     */
    public static function sanitize_custom_admin_url($input) {
        $input = trim($input);
        if (empty($input)) {
            return '';
        }
        // Remove leading/trailing slashes and sanitize as a URL slug
        $input = trim($input, '/');
        $input = sanitize_title_with_dashes($input);

        // Prevent using reserved WordPress slugs
        $reserved_slugs = ['wp-admin', 'wp-login', 'wp-login.php', 'admin', 'login', 'wp-content', 'wp-includes'];
        if (in_array($input, $reserved_slugs, true)) {
            add_settings_error(
                'lsec_custom_admin_url',
                'lsec_reserved_slug',
                __('The custom admin URL cannot use reserved WordPress slugs. Please choose a different URL.', 'login-security-with-telegram-alerts'),
                'error'
            );
            return get_option('lsec_custom_admin_url', '');
        }

        return $input;
    }

    /**
     * Adds the plugin's settings page to the WordPress admin menu.
     */
    public function add_admin_menu() {
        add_options_page(
            __('Login Security Settings', 'login-security-with-telegram-alerts'),
            __('Login Security', 'login-security-with-telegram-alerts'),
            'manage_options',
            'login-security-telegram-alerts-settings',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Renders the plugin's settings page.
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'login-security-with-telegram-alerts'));
        }

        global $wpdb;
        $table_failed_attempts = $wpdb->prefix . 'lsec_login_attempts';

        $max_attempts              = get_option('lsec_max_attempts', 5);
        $lockout_duration          = get_option('lsec_lockout_duration', 20);
        $bot_token                 = get_option('lsec_bot_token');
        $display_bot_token         = $bot_token ? preg_replace('/(.{4}).+(.{4})/', '$1****$2', $bot_token) : '';
        $chat_id                   = get_option('lsec_chat_id');
        $display_chat_id           = $chat_id ? '****' . substr($chat_id, -4) : '';
        $selected_roles            = get_option('lsec_notify_roles', ['administrator']);
        $enable_geolocation        = get_option('lsec_enable_geolocation', 1);
        $notify_on_failed          = get_option('lsec_notify_on_failed_attempt', 1);
        $notify_admin_on_lockout   = get_option('lsec_notify_admin_on_lockout', 1);
        $manual_blacklist_ips      = get_option('lsec_manual_blacklist_ips', '');
        $manual_whitelist_ips      = get_option('lsec_manual_whitelist_ips', '');
        $custom_admin_url          = get_option('lsec_custom_admin_url', '');
        $activity_log_items_per_page_display = 50;

        $blocked_ips_details = [];
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check is necessary and doesn't need caching.
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_failed_attempts)) == $table_failed_attempts) {
            $max_attempts_config    = (int) get_option('lsec_max_attempts', 5);
            $lockout_duration_config= (int) get_option('lsec_lockout_duration', 20);
            $table_failed_attempts_escaped = esc_sql($table_failed_attempts);
            // Query does not have placeholders, so direct execution is used. Table name is properly escaped with esc_sql() and only contains prefix + hardcoded string.
            $results = $wpdb->get_results( "SELECT ip_address, last_attempt, attempts FROM `{$table_failed_attempts_escaped}`" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            if ($results) {
                foreach ($results as $row) {
                    if (!lsec_is_ip_whitelisted($row->ip_address) && $row->attempts >= $max_attempts_config) {
                        $lockout_expires_at_timestamp = strtotime($row->last_attempt) + ($lockout_duration_config * 60);
                        if (time() < $lockout_expires_at_timestamp) {
                            $blocked_ips_details[$row->ip_address] = true;
                        }
                    }
                }
            }
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Filter is for display purposes only.
        $current_filter = isset($_GET['lsec_log_filter']) ? sanitize_key($_GET['lsec_log_filter']) : 'all';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Pagination is for display purposes only.
        $paged_activity = isset($_GET['paged_activity']) ? max(1, intval($_GET['paged_activity'])) : 1;

        $activity_log_data = $this->login_tracker->get_activity_log_entries($current_filter, $paged_activity, $activity_log_items_per_page_display);
        $activity_log_entries = $activity_log_data['entries'];
        $total_activity_count = $activity_log_data['total_count'];
        $total_pages_activity = $total_activity_count > 0 ? ceil($total_activity_count / $activity_log_items_per_page_display) : 0;

        // This is the initial value for the hidden input. JavaScript will update it.
        $initial_hidden_tab_value = 'tab-security-settings';

        ?>
        <div class="wrap lsec-settings-wrap">
            <h1><?php esc_html_e('Login Security Settings', 'login-security-with-telegram-alerts'); ?></h1>
            <?php
            // This will display the standard "Settings saved." message
            // if the form submitted successfully to options.php.
            settings_errors('lsec_options_group');
            ?>

            <h2 class="nav-tab-wrapper" id="lsec-tabs">
                <a href="#tab-security-settings"
                    class="nav-tab"><?php esc_html_e('Security Settings', 'login-security-with-telegram-alerts'); ?></a>
                <a href="#tab-telegram-settings"
                    class="nav-tab"><?php esc_html_e('Telegram Notifications', 'login-security-with-telegram-alerts'); ?></a>
                <a href="#tab-ip-access-control"
                    class="nav-tab"><?php esc_html_e('IP Access Control', 'login-security-with-telegram-alerts'); ?></a>
                <a href="#tab-file-integrity"
                    class="nav-tab"><?php esc_html_e('File Integrity & Permissions', 'login-security-with-telegram-alerts'); ?></a>
                <a href="#tab-activity-log-display"
                    class="nav-tab"><?php esc_html_e('Activity Log', 'login-security-with-telegram-alerts'); ?></a>
            </h2>

            <form method="post" action="options.php" id="lsec-main-settings-form">
                <?php settings_fields('lsec_options_group'); ?>
                <?php wp_nonce_field('lsec_clear_activity_log_action', 'lsec_clear_activity_log_nonce'); ?>
                <?php wp_nonce_field('lsec_clear_failed_log_action', 'lsec_clear_failed_log_nonce'); ?>

                <input type="hidden" name="lsec_current_tab" id="lsec_current_tab" value="<?php echo esc_attr($initial_hidden_tab_value); ?>">

                <div id="tab-security-settings" class="lsec-tab-content">
                    <h3><?php esc_html_e('Login Attempt Configuration', 'login-security-with-telegram-alerts'); ?></h3>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_max_attempts"><?php esc_html_e('Max Failed Attempts Allowed', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <input name="lsec_max_attempts" type="number" id="lsec_max_attempts"
                                    value="<?php echo esc_attr($max_attempts); ?>" class="small-text" min="1">
                                <?php esc_html_e('attempts', 'login-security-with-telegram-alerts'); ?>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_lockout_duration"><?php esc_html_e('Lockout Duration', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <input name="lsec_lockout_duration" type="number" id="lsec_lockout_duration"
                                    value="<?php echo esc_attr($lockout_duration); ?>" class="small-text" min="1">
                                <?php esc_html_e('minutes', 'login-security-with-telegram-alerts'); ?>
                            </td>
                        </tr>
                    </table>

                    <h3><?php esc_html_e('Advanced Security Settings', 'login-security-with-telegram-alerts'); ?></h3>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_custom_admin_url"><?php esc_html_e('Custom Admin URL', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <?php echo esc_url(home_url('/')); ?><input name="lsec_custom_admin_url" type="text" id="lsec_custom_admin_url"
                                    value="<?php echo esc_attr($custom_admin_url); ?>" class="regular-text" placeholder="my-secret-admin">
                                <p class="description">
                                    <?php esc_html_e('Set a custom URL slug to access wp-admin (e.g., "my-secret-admin"). Leave empty to use the default wp-admin URL. This adds an extra layer of security by hiding the standard admin login URL.', 'login-security-with-telegram-alerts'); ?>
                                    <?php if (!empty($custom_admin_url)) : ?>
                                        <br><strong><?php esc_html_e('Your admin URL:', 'login-security-with-telegram-alerts'); ?></strong>
                                        <a href="<?php echo esc_url(home_url($custom_admin_url)); ?>" target="_blank"><?php echo esc_url(home_url($custom_admin_url)); ?></a>
                                    <?php endif; ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div id="tab-telegram-settings" class="lsec-tab-content">
                    <h3><?php esc_html_e('Telegram API Configuration', 'login-security-with-telegram-alerts'); ?></h3>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_bot_token"><?php esc_html_e('Bot Token', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <input type="password" id="lsec_bot_token" name="lsec_bot_token"
                                    value="<?php echo esc_attr($display_bot_token); ?>" class="regular-text"
                                    autocomplete="new-password">
                                <p class="description">
                                    <?php esc_html_e('From @BotFather on Telegram.', 'login-security-with-telegram-alerts'); ?></p>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_chat_id"><?php esc_html_e('Chat ID', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <input type="password" id="lsec_chat_id" name="lsec_chat_id"
                                    value="<?php echo esc_attr($display_chat_id); ?>" class="regular-text"
                                    autocomplete="new-password">
                                <p class="description">
                                    <?php esc_html_e('Your Telegram user/group/channel ID.', 'login-security-with-telegram-alerts'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <p>
                        <button type="button" id="lsec-send-test-telegram" class="button button-secondary">
                            <?php esc_html_e('Send Test Telegram Message', 'login-security-with-telegram-alerts'); ?>
                        </button>
                        <span id="lsec-test-telegram-spinner" class="spinner"
                            style="float: none; vertical-align: middle; margin-left: 5px;"></span>
                    </p>
                    <div id="lsec-test-telegram-result" style="margin-top: 10px;"></div>

                    <h3><?php esc_html_e('Notification Preferences', 'login-security-with-telegram-alerts'); ?></h3>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row">
                                <?php esc_html_e('Notify for successful logins by roles', 'login-security-with-telegram-alerts'); ?>
                            </th>
                            <td>
                                <fieldset>
                                    <?php
                                        $roles = get_editable_roles();
                                        foreach ($roles as $role_slug => $details) : ?>
                                    <label>
                                        <input type="checkbox" name="lsec_notify_roles[]"
                                            value="<?php echo esc_attr($role_slug); ?>"
                                            <?php checked(in_array($role_slug, (array) $selected_roles)); ?>>
                                        <?php echo esc_html($details['name']); ?>
                                    </label><br>
                                    <?php endforeach; ?>
                                </fieldset>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_notify_on_failed_attempt"><?php esc_html_e('Notify on Failed Login', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <input type="hidden" name="lsec_notify_on_failed_attempt" value="0" />
                                <label>
                                    <input type="checkbox" id="lsec_notify_on_failed_attempt"
                                        name="lsec_notify_on_failed_attempt" value="1" <?php checked($notify_on_failed); ?>>
                                    <?php esc_html_e('Send Telegram message on any failed login attempt.', 'login-security-with-telegram-alerts'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_notify_admin_on_lockout"><?php esc_html_e('Notify on Account Lockout', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <input type="hidden" name="lsec_notify_admin_on_lockout" value="0" />
                                <label>
                                    <input type="checkbox" id="lsec_notify_admin_on_lockout" name="lsec_notify_admin_on_lockout"
                                        value="1" <?php checked($notify_admin_on_lockout); ?>>
                                    <?php esc_html_e('Send Telegram message when a user account/IP is locked out.', 'login-security-with-telegram-alerts'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_enable_geolocation"><?php esc_html_e('Enable Geolocation', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <input type="hidden" name="lsec_enable_geolocation" value="0" />
                                <label>
                                    <input type="checkbox" id="lsec_enable_geolocation" name="lsec_enable_geolocation" value="1"
                                        <?php checked($enable_geolocation); ?>>
                                    <?php esc_html_e('Include approximate location in Telegram messages (uses external IP lookup services).', 'login-security-with-telegram-alerts'); ?>
                                </label>
                            </td>
                        </tr>
                    </table>
                </div>

                <div id="tab-ip-access-control" class="lsec-tab-content">
                    <h3><?php esc_html_e('Manual IP Access Control', 'login-security-with-telegram-alerts'); ?></h3>
                    <p><?php esc_html_e('Enter one IP address per line. Whitelisted IPs will bypass all login attempt checks and will not be automatically blocked. Blacklisted IPs will be completely blocked from accessing the site.', 'login-security-with-telegram-alerts'); ?>
                    </p>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_manual_whitelist_ips"><?php esc_html_e('Whitelisted IP Addresses', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <textarea name="lsec_manual_whitelist_ips" id="lsec_manual_whitelist_ips" rows="5"
                                    class="large-text code"><?php echo esc_textarea($manual_whitelist_ips); ?></textarea>
                                <p class="description">
                                    <?php esc_html_e('One IP address per line. These IPs will bypass lockout rules.', 'login-security-with-telegram-alerts'); ?>
                                </p>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><label
                                    for="lsec_manual_blacklist_ips"><?php esc_html_e('Blacklisted IP Addresses', 'login-security-with-telegram-alerts'); ?></label>
                            </th>
                            <td>
                                <textarea name="lsec_manual_blacklist_ips" id="lsec_manual_blacklist_ips" rows="5"
                                    class="large-text code"><?php echo esc_textarea($manual_blacklist_ips); ?></textarea>
                                <p class="description">
                                    <?php esc_html_e('One IP address per line. These IPs will be completely blocked from accessing the site.', 'login-security-with-telegram-alerts'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div id="tab-file-integrity" class="lsec-tab-content">
                    <h3><?php esc_html_e('WordPress Core Files Integrity Check', 'login-security-with-telegram-alerts'); ?></h3>
                    <p><?php esc_html_e('Check if WordPress core files have been modified or are missing. This helps detect unauthorized changes to your installation.', 'login-security-with-telegram-alerts'); ?></p>

                    <?php
                    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only reading GET parameter for display logic
                    if (isset($_GET['lsec_check_integrity']) && $_GET['lsec_check_integrity'] === '1') {
                        $file_integrity = new LSEC_File_Integrity();
                        $integrity_results = $file_integrity->check_core_files();

                        if ($integrity_results['status'] === 'success') {
                            echo '<div class="notice notice-success"><p>' . esc_html($integrity_results['message']) . '</p></div>';
                        } else {
                            echo '<div class="notice notice-warning"><p>' . esc_html($integrity_results['message']) . '</p></div>';

                            if (!empty($integrity_results['modified_files'])) {
                                echo '<h4>' . esc_html__('Modified Files:', 'login-security-with-telegram-alerts') . '</h4>';
                                echo '<ul style="list-style: disc; margin-left: 20px;">';
                                foreach ($integrity_results['modified_files'] as $file) {
                                    echo '<li><code>' . esc_html($file) . '</code></li>';
                                }
                                echo '</ul>';
                            }

                            if (!empty($integrity_results['missing_files'])) {
                                echo '<h4>' . esc_html__('Missing Files:', 'login-security-with-telegram-alerts') . '</h4>';
                                echo '<ul style="list-style: disc; margin-left: 20px;">';
                                foreach ($integrity_results['missing_files'] as $file) {
                                    echo '<li><code>' . esc_html($file) . '</code></li>';
                                }
                                echo '</ul>';
                            }
                        }
                    }
                    ?>

                    <p>
                        <a href="<?php echo esc_url(add_query_arg(['page' => 'login-security-telegram-alerts-settings', 'lsec_check_integrity' => '1'], admin_url('options-general.php')) . '#tab-file-integrity'); ?>" class="button button-primary">
                            <?php esc_html_e('Check Core Files Integrity', 'login-security-with-telegram-alerts'); ?>
                        </a>
                    </p>

                    <hr style="margin: 30px 0;">

                    <h3><?php esc_html_e('File Permissions Check', 'login-security-with-telegram-alerts'); ?></h3>
                    <p><?php esc_html_e('Check and fix file permissions according to WordPress security standards.', 'login-security-with-telegram-alerts'); ?></p>

                    <?php
                    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only reading GET parameter for display logic
                    if (isset($_GET['lsec_check_permissions']) && $_GET['lsec_check_permissions'] === '1') {
                        $file_integrity = new LSEC_File_Integrity();
                        $permissions_results = $file_integrity->check_file_permissions();

                        if ($permissions_results['status'] === 'success') {
                            echo '<div class="notice notice-success"><p>' . esc_html($permissions_results['message']) . '</p></div>';
                        } else {
                            echo '<div class="notice notice-warning"><p>' . esc_html($permissions_results['message']) . '</p></div>';

                            if (!empty($permissions_results['issues'])) {
                                echo '<table class="widefat fixed striped" style="margin-top: 15px;">';
                                echo '<thead><tr>';
                                echo '<th>' . esc_html__('File/Directory', 'login-security-with-telegram-alerts') . '</th>';
                                echo '<th>' . esc_html__('Current', 'login-security-with-telegram-alerts') . '</th>';
                                echo '<th>' . esc_html__('Recommended', 'login-security-with-telegram-alerts') . '</th>';
                                echo '<th>' . esc_html__('Description', 'login-security-with-telegram-alerts') . '</th>';
                                echo '</tr></thead><tbody>';

                                foreach ($permissions_results['issues'] as $issue) {
                                    echo '<tr>';
                                    echo '<td><code>' . esc_html($issue['file']) . '</code></td>';
                                    echo '<td>' . esc_html($issue['current']) . '</td>';
                                    echo '<td>' . esc_html($issue['recommended']) . '</td>';
                                    echo '<td>' . esc_html($issue['description']) . '</td>';
                                    echo '</tr>';
                                }

                                echo '</tbody></table>';

                                // Add fix button
                                echo '<p style="margin-top: 15px;">';
                                echo '<a href="' . esc_url(wp_nonce_url(add_query_arg(['page' => 'login-security-telegram-alerts-settings', 'lsec_fix_permissions' => '1'], admin_url('options-general.php')) . '#tab-file-integrity', 'lsec_fix_permissions_action', 'lsec_fix_permissions_nonce')) . '" class="button button-secondary">';
                                echo esc_html__('Attempt to Fix Permissions', 'login-security-with-telegram-alerts');
                                echo '</a></p>';
                            }
                        }
                    }

                    // Handle fix permissions action
                    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified below
                    if (isset($_GET['lsec_fix_permissions'], $_GET['lsec_fix_permissions_nonce']) && current_user_can('manage_options')) {
                        $nonce = sanitize_text_field(wp_unslash($_GET['lsec_fix_permissions_nonce'])); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                        if (wp_verify_nonce($nonce, 'lsec_fix_permissions_action')) {
                            $file_integrity = new LSEC_File_Integrity();
                            $permissions_results = $file_integrity->check_file_permissions();

                            if (!empty($permissions_results['issues'])) {
                                $fix_results = $file_integrity->fix_file_permissions($permissions_results['issues']);

                                echo '<div class="notice notice-info"><p><strong>' . esc_html__('Fix Results:', 'login-security-with-telegram-alerts') . '</strong></p><ul style="list-style: disc; margin-left: 20px;">';
                                foreach ($fix_results as $file => $result) {
                                    $icon = $result['success'] ? '✓' : '✗';
                                    echo '<li>' . esc_html($icon . ' ' . $file . ': ' . $result['message']) . '</li>';
                                }
                                echo '</ul></div>';

                                echo '<p><a href="' . esc_url(add_query_arg(['page' => 'login-security-telegram-alerts-settings', 'lsec_check_permissions' => '1'], admin_url('options-general.php')) . '#tab-file-integrity') . '" class="button button-secondary">' . esc_html__('Re-check Permissions', 'login-security-with-telegram-alerts') . '</a></p>';
                            }
                        }
                    }
                    ?>

                    <p>
                        <a href="<?php echo esc_url(add_query_arg(['page' => 'login-security-telegram-alerts-settings', 'lsec_check_permissions' => '1'], admin_url('options-general.php')) . '#tab-file-integrity'); ?>" class="button button-primary">
                            <?php esc_html_e('Check File Permissions', 'login-security-with-telegram-alerts'); ?>
                        </a>
                    </p>
                </div>

                <div id="tab-activity-log-display" class="lsec-tab-content">
                    <h3><?php esc_html_e('Login Activity Log', 'login-security-with-telegram-alerts'); ?></h3>
                    <div style="margin-bottom: 15px;">
                        <label
                            for="lsec_log_filter_select"><?php esc_html_e('Filter:', 'login-security-with-telegram-alerts'); ?></label>
                        <select id="lsec_log_filter_select" onchange="window.location.href = '<?php
                                echo esc_url(add_query_arg([
                                    'page' => 'login-security-telegram-alerts-settings',
                                ], admin_url('options-general.php')));
                                ?>&lsec_log_filter=' + this.value + '#tab-activity-log-display';">
                            <option value="all" <?php selected($current_filter, 'all'); ?>>
                                <?php esc_html_e('All', 'login-security-with-telegram-alerts'); ?></option>
                            <option value="success" <?php selected($current_filter, 'success'); ?>>
                                <?php esc_html_e('Successful', 'login-security-with-telegram-alerts'); ?></option>
                            <option value="failed" <?php selected($current_filter, 'failed'); ?>>
                                <?php esc_html_e('Failed', 'login-security-with-telegram-alerts'); ?></option>
                        </select>
                    </div>

                    <p class="submit">
                        <input type="submit" name="lsec_clear_activity_log_submit" class="button button-secondary"
                            value="<?php esc_attr_e('Clear Activity Log', 'login-security-with-telegram-alerts'); ?>"
                            onclick="return confirm('<?php esc_attr_e('Are you sure you want to clear the entire activity log?', 'login-security-with-telegram-alerts'); ?>');">
                        <input type="submit" name="lsec_clear_failed_log_submit" class="button button-secondary"
                            value="<?php esc_attr_e('Clear Failed Login Log', 'login-security-with-telegram-alerts'); ?>"
                            onclick="return confirm('<?php esc_attr_e('Are you sure you want to clear the failed login attempts log? This will remove all temporary IP blocks.', 'login-security-with-telegram-alerts'); ?>');">
                    </p>

                    <table class="widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('ID', 'login-security-with-telegram-alerts'); ?></th>
                                <th><?php esc_html_e('Status', 'login-security-with-telegram-alerts'); ?></th>
                                <th><?php esc_html_e('Username', 'login-security-with-telegram-alerts'); ?></th>
                                <th><?php esc_html_e('Role', 'login-security-with-telegram-alerts'); ?></th>
                                <th><?php esc_html_e('IP Address', 'login-security-with-telegram-alerts'); ?></th>
                                <th><?php esc_html_e('Time', 'login-security-with-telegram-alerts'); ?></th>
                                <th style="width:20%;"><?php esc_html_e('User Agent', 'login-security-with-telegram-alerts'); ?></th>
                                <th><?php esc_html_e('Actions', 'login-security-with-telegram-alerts'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($activity_log_entries) : foreach ($activity_log_entries as $entry) : ?>
                            <tr>
                                <td><?php echo esc_html($entry->id); ?></td>
                                <td>
                                    <span
                                        class="lsec-status-<?php echo esc_attr(sanitize_title(str_replace(['(', ')'], '', $entry->status))); ?>">
                                        <?php echo esc_html(ucfirst($entry->status)); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html($entry->username); ?></td>
                                <td><?php echo esc_html($entry->role ?: esc_html__('N/A', 'login-security-with-telegram-alerts')); ?>
                                </td>
                                <td>
                                    <a href="https://ip-api.com/#<?php echo esc_attr($entry->ip_address); ?>" target="_blank"
                                        rel="noopener noreferrer"><?php echo esc_html($entry->ip_address); ?></a>
                                    <?php if (lsec_is_ip_whitelisted($entry->ip_address)) : ?>
                                    <span class="dashicons dashicons-yes-alt"
                                        title="<?php esc_attr_e('IP Whitelisted', 'login-security-with-telegram-alerts'); ?>"
                                        style="color:green; font-size:1.3em; vertical-align:middle;"></span>
                                    <?php elseif (lsec_is_ip_blacklisted($entry->ip_address)) : ?>
                                    <span class="dashicons dashicons-dismiss"
                                        title="<?php esc_attr_e('IP Manually Blacklisted', 'login-security-with-telegram-alerts'); ?>"
                                        style="color:red; font-size:1.3em; vertical-align:middle;"></span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo esc_html($entry->login_time); ?></td>
                                <td style="word-wrap:break-word;white-space:normal;font-size:0.9em;">
                                    <?php echo esc_html($entry->user_agent); ?></td>
                                <td>
                                    <?php
                                        $is_blacklisted = lsec_is_ip_blacklisted($entry->ip_address);
                                        $is_whitelisted = lsec_is_ip_whitelisted($entry->ip_address);

                                        // Check if IP is currently blocked based on failed attempts
                                        $attempt_data = $this->login_tracker->get_failed_attempts_data($entry->ip_address);
                                        $is_temporarily_blocked = false;
                                        if ($attempt_data && !$is_whitelisted) {
                                            $max_attempts_config    = (int) get_option('lsec_max_attempts', 5);
                                            $lockout_duration_config= (int) get_option('lsec_lockout_duration', 20);
                                            if ($attempt_data->attempts >= $max_attempts_config) {
                                                $lockout_expires_at_timestamp = strtotime($attempt_data->last_attempt) + ($lockout_duration_config * 60);
                                                if (time() < $lockout_expires_at_timestamp) {
                                                    $is_temporarily_blocked = true;
                                                }
                                            }
                                        }

                                        $url_params = [
                                            'page' => 'login-security-telegram-alerts-settings',
                                        ];
                                        if (isset($_GET['paged_activity'])) {
                                            $url_params['paged_activity'] = intval($_GET['paged_activity']);
                                        }
                                        if (isset($_GET['lsec_log_filter'])) {
                                            $url_params['lsec_log_filter'] = sanitize_key($_GET['lsec_log_filter']);
                                        }

                                        // Show unblock button if IP is blacklisted or temporarily blocked
                                        if ($is_blacklisted || $is_temporarily_blocked) {
                                            $url_params['lsec_unblock_ip'] = $entry->ip_address;
                                            $unblock_url = wp_nonce_url(
                                                add_query_arg($url_params, admin_url('options-general.php')),
                                                'lsec_unblock_ip_action_' . $entry->ip_address,
                                                'lsec_unblock_nonce'
                                            );
                                            $unblock_url .= '#tab-activity-log-display';
                                            echo '<a href="' . esc_url($unblock_url) . '" class="button button-small lsec-unblock-ip-btn">'
                                                . esc_html__('Unblock IP', 'login-security-with-telegram-alerts')
                                                . '</a>';
                                        }
                                        // Show block button if IP is not blacklisted and not whitelisted
                                        elseif (!$is_whitelisted) {
                                            $url_params['lsec_block_ip'] = $entry->ip_address;
                                            $block_url = wp_nonce_url(
                                                add_query_arg($url_params, admin_url('options-general.php')),
                                                'lsec_block_ip_action_' . $entry->ip_address,
                                                'lsec_block_nonce'
                                            );
                                            $block_url .= '#tab-activity-log-display';
                                            echo '<a href="' . esc_url($block_url) . '" class="button button-small lsec-block-ip-btn">'
                                                . esc_html__('Block IP', 'login-security-with-telegram-alerts')
                                                . '</a>';
                                        } else {
                                            echo esc_html__('N/A', 'login-security-with-telegram-alerts');
                                        }
                                        ?>
                                </td>
                            </tr>
                            <?php endforeach;
                                else : ?>
                            <tr>
                                <td colspan="8">
                                    <?php esc_html_e('No activity recorded or filter returned no results.', 'login-security-with-telegram-alerts'); ?>
                                </td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                    <?php
                        $pagination_links_args = [
                            'base'      => add_query_arg(['paged_activity' => '%#%', 'lsec_log_filter' => $current_filter], admin_url('options-general.php?page=login-security-telegram-alerts-settings#tab-activity-log-display')),
                            'format'    => '',
                            'current'   => $paged_activity,
                            'total'     => $total_pages_activity,
                            'prev_text' => esc_html__('&laquo; Previous', 'login-security-with-telegram-alerts'),
                            'next_text' => esc_html__('Next &raquo;', 'login-security-with-telegram-alerts'),
                        ];
                        $pagination_links = paginate_links($pagination_links_args);
                        if ($pagination_links) {
                            echo '<div class="tablenav"><div class="tablenav-pages">' . $pagination_links . '</div></div>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- paginate_links output is safe.
                        }
                        ?>
                </div>

                <?php submit_button(esc_html__('Save All Settings', 'login-security-with-telegram-alerts')); ?>
            </form>

            <p style="margin-top: 20px; font-size: 12px; color: #666;">
                <?php
                printf(
                    /* translators: %s: Link to support page */
                    esc_html__('Like this plugin? %s', 'login-security-with-telegram-alerts'),
                    '<a href="https://ko-fi.com/gabrielrosca" target="_blank" rel="noopener noreferrer">' . esc_html__('Support development', 'login-security-with-telegram-alerts') . '</a>'
                );
                ?>
            </p>
        </div>
        <?php
    }
}