jQuery(document).ready(function($) {
    // Tab functionality
    var $tabsContainer = $('#lsec-tabs');
    // Reference to the hidden input for storing the active tab
    var $currentTabInput = $('#lsec_current_tab'); 

    if ($tabsContainer.length) {
        var $tabs = $tabsContainer.find('a.nav-tab');
        var $tabContents = $('.lsec-tab-content');

        /**
         * Activates a specific tab and displays its content.
         * Updates the URL hash and the hidden input for the active tab.
         * @param {string} tabHref The href attribute of the tab link (e.g., '#tab-security-settings').
         * @param {boolean} updateUrlHash Whether to update the browser's URL hash. Default is true.
         */
        function activateTab(tabHref, updateUrlHash = true) {
            var contentId = tabHref.substring(1); // Remove # from href
            $tabs.removeClass('nav-tab-active');
            $tabContents.hide(); // Hide all content

            $tabs.filter('[href="' + tabHref + '"]').addClass('nav-tab-active');
            $('#' + contentId).show(); // Show target content

            // Crucially, update the hidden input with the current tab ID
            // This ensures that when the form is submitted, the server knows which tab was active.
            if ($currentTabInput.length) {
                $currentTabInput.val(contentId); 
            }

            // Update URL hash for direct linking and browser history
            if (updateUrlHash) {
                if (history.pushState) {
                    history.pushState(null, null, tabHref);
                } else {
                    window.location.hash = tabHref;
                }
            }
        }

        // Determine the initial tab to activate on page load
        var initialTabHref;

        // 1. Check for 'lsec_active_tab' query parameter (used after form submission redirect)
        var urlParams = new URLSearchParams(window.location.search);
        var activeTabFromQuery = urlParams.get('lsec_active_tab');

        if (activeTabFromQuery) {
            initialTabHref = '#' + activeTabFromQuery;
            // Clean up the URL: remove the query parameter after using it
            urlParams.delete('lsec_active_tab');
            // Remove 'settings-updated' as well for a cleaner URL, as WordPress automatically adds it
            urlParams.delete('settings-updated'); 
            
            // Reconstruct the URL without the parameters but with the hash
            var newUrl = window.location.pathname + window.location.search.replace(/(\?|&)(lsec_active_tab|settings-updated)=[^&]*/g, '').replace(/^&/, '?');
            
            if (history.replaceState) {
                // Ensure there's a '?' if there are other query parameters remaining
                if (newUrl.indexOf('?') === -1 && newUrl.indexOf('&') !== -1) {
                    newUrl = newUrl.replace('&', '?'); // Replace first '&' with '?' if no '?' exists
                }
                history.replaceState(null, null, newUrl + initialTabHref); // Append hash
            } else {
                window.location.hash = initialTabHref;
            }
            // Activate the tab without immediately pushing another state to history
            activateTab(initialTabHref, false);
        } else if (window.location.hash) {
            // 2. Check for existing URL hash (for direct navigation or browser history)
            initialTabHref = window.location.hash;
            activateTab(initialTabHref, false); // Activate without re-updating hash
        } else {
            // 3. Default to the first tab if no specific instruction
            initialTabHref = $tabs.first().attr('href') || '#tab-security-settings'; 
            activateTab(initialTabHref, true); // Activate and set hash
        }
        
        // Click event handler for tab navigation
        $tabs.click(function(e) {
            e.preventDefault(); // Prevent default anchor link behavior
            var clickedTabHref = $(this).attr('href');
            activateTab(clickedTabHref); // Activate the clicked tab (will update URL hash)
        });

        // Handle browser back/forward buttons (hashchange event)
        $(window).on('hashchange', function() {
            var hash = window.location.hash;
            if (hash && $tabs.filter('[href="' + hash + '"]').length > 0) {
                activateTab(hash, false); // Don't update hash again
            } else if (!hash && $tabs.length > 0) {
                // If hash is cleared or invalid, revert to the default (first) tab
                activateTab($tabs.first().attr('href') || '#tab-security-settings', false); // Don't update hash again
            }
        });
    }

    // --- AJAX for sending test Telegram message ---
    $('#lsec-send-test-telegram').on('click', function() {
        var $button = $(this);
        var $spinner = $('#lsec-test-telegram-spinner');
        var $resultDiv = $('#lsec-test-telegram-result');

        // Store original button text before changing it to 'Sending...'
        var originalButtonText = lsec_ajax_object.send_test_button_text || $button.text();

        // Disable button and show spinner to indicate activity
        $button.prop('disabled', true).text(lsec_ajax_object.sending_text);
        $spinner.addClass('is-active');
        // Clear previous results and remove notice classes
        $resultDiv.empty().removeClass('notice-success notice-error notice is-dismissible');

        // Send AJAX request
        $.post(lsec_ajax_object.ajax_url, {
            action: 'lsec_send_test_telegram', // WordPress AJAX action
            nonce: lsec_ajax_object.lsec_test_nonce // Security nonce
        }, function(response) {
            // Success callback for the AJAX request
            if (response.success) {
                $resultDiv.removeClass('notice-error')
                          .addClass('notice notice-success is-dismissible')
                          .html('<p>' + (response.data.message || lsec_ajax_object.success_message) + '</p>');
            } else {
                // Handle errors returned from the server (e.g., invalid token/chat ID)
                var errorMessage = (response.data && response.data.message) ? response.data.message : lsec_ajax_object.error_message;
                $resultDiv.removeClass('notice-success')
                          .addClass('notice notice-error is-dismissible').html('<p>' + errorMessage + '</p>');
            }
        }).fail(function() {
            // Handle AJAX request failures (e.g., network issues, server error)
            $resultDiv.removeClass('notice-success')
                      .addClass('notice notice-error is-dismissible').html('<p>' + lsec_ajax_object.request_failed_text + '</p>');
        }).always(function() {
            // This function runs regardless of success or failure
            // Re-enable button and hide spinner
            $button.prop('disabled', false).text(originalButtonText);
            $spinner.removeClass('is-active');
        });
    });
});