<?php
/**
 * Plugin Name:       LNH Lunar Calendar
 * Plugin URI:        https://wordpress.org/plugins/lnh-lunar-calendar/
 * Description:       Displays a beautiful, modern, and accurate Vietnamese Lunar Calendar via shortcode.
 * Version:           2.8.5
 * Author:            tobevn
 * Author URI:        https://profiles.wordpress.org/tobevn/
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       lnh-lunar-calendar
 * Domain Path:       /languages
 * Requires at least: 5.0
 * Requires PHP:      7.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// ===== Plugin constants =====
if ( ! defined( 'LNHLC_VERSION' ) ) {
    define( 'LNHLC_VERSION', '2.8.5' );
}
if ( ! defined( 'LNHLC_PLUGIN_DIR' ) ) {
    define( 'LNHLC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'LNHLC_PLUGIN_URL' ) ) {
    define( 'LNHLC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}
if ( ! defined( 'LNHLC_OPTIONS_NAME' ) ) {
    define( 'LNHLC_OPTIONS_NAME', 'lnhlc_options' );
}

// ===== DEV switch: set true to bypass caches while developing =====
if ( ! defined( 'LNHLC_DEV' ) ) {
    define( 'LNHLC_DEV', false );
}

/**
 * Get cache buster value stored in options (updated when settings are saved).
 * Falls back to plugin version if not set.
 *
 * @return string
 */
function lnhlc_get_cache_buster() {
    $buster = get_option( 'lnhlc_cache_buster', '' );
    if ( empty( $buster ) ) {
        $buster = LNHLC_VERSION;
    }
    return (string) $buster;
}

/**
 * Return a dynamic version for assets based on filemtime.
 * If file not found, fall back to cache buster / plugin version.
 *
 * @param string $relative_path Relative path under plugin root (e.g. 'assets/css/main.css')
 * @return string|int
 */
function lnhlc_asset_ver( $relative_path ) {
    // During development, always use current timestamp to defeat caches
    if ( defined('LNHLC_DEV') && LNHLC_DEV ) {
        return time();
    }
    $path = trailingslashit( LNHLC_PLUGIN_DIR ) . ltrim( $relative_path, '/\\' );
    if ( file_exists( $path ) ) {
        $mt = filemtime( $path );
        if ( $mt ) {
            return $mt;
        }
    }
    return lnhlc_get_cache_buster();
}

/**
 * The main plugin class
 */
final class LNH_Lunar_Calendar {

    private static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    private function __construct() {
        $this->includes();
        $this->init_hooks();
    }

    private function includes() {
        require_once LNHLC_PLUGIN_DIR . 'includes/functions.php';
        require_once LNHLC_PLUGIN_DIR . 'includes/class-lnh-lunar-calendar-settings.php';
        require_once LNHLC_PLUGIN_DIR . 'includes/class-lnh-lunar-calendar-shortcode.php';
    }

    private function init_hooks() {
        // Enqueue scripts and styles
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_scripts' ] );

        // Append cache-buster (cb) to our asset URLs as a last resort (helps bust CDN/browser caches)
        add_filter( 'style_loader_src', [ $this, 'append_cb_to_src' ], 10, 2 );
        add_filter( 'script_loader_src', [ $this, 'append_cb_to_src' ], 10, 2 );

        new LNH_Lunar_Calendar_Settings();
        new LNH_Lunar_Calendar_Shortcode();
    }

    public function enqueue_scripts() {
        // Frontend assets with filemtime-based versioning
        wp_register_style(
            'lnhlc-styles',
            LNHLC_PLUGIN_URL . 'assets/css/main.css',
            [],
            lnhlc_asset_ver( 'assets/css/main.css' )
        );

        wp_register_script(
            'lnhlc-script',
            LNHLC_PLUGIN_URL . 'assets/js/main.js',
            [],
            lnhlc_asset_ver( 'assets/js/main.js' ),
            true
        );
    }

    public function enqueue_admin_scripts( $hook ) {
        if ( 'settings_page_lnh-lunar-calendar' !== $hook ) {
            return;
        }
        wp_enqueue_style(
            'lnhlc-admin-styles',
            LNHLC_PLUGIN_URL . 'assets/css/admin.css',
            [],
            lnhlc_asset_ver( 'assets/css/admin.css' )
        );
    }

    /**
     * Append cache buster query param to our plugin asset URLs.
     * Helps when hosts/CDNs ignore ?ver param or when assets are behind aggressive caches.
     */
    public function append_cb_to_src( $src, $handle ) {
        // Only touch our plugin's assets
        $base = trailingslashit( LNHLC_PLUGIN_URL );
        if ( strpos( $src, $base ) === 0 ) {
            $join   = ( strpos( $src, '?' ) === false ) ? '?' : '&';
            $cb     = lnhlc_get_cache_buster();
            $src   .= $join . 'cb=' . rawurlencode( $cb );
        }
        return $src;
    }
}

/**
 * Begins execution of the plugin.
 */
function lnhlc_run_lunar_calendar() {
    return LNH_Lunar_Calendar::instance();
}

lnhlc_run_lunar_calendar();
