<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Get the list of strings that can be translated in the settings.
 */
function lnhlc_get_translatable_strings() {
    return [
        'lunar_date'   => 'Lunar Date',
        'solar_date'   => 'Solar Date',
        'day'          => 'Day',
        'month'        => 'Month',
        'year'         => 'Year',
        'zodiac_hours' => 'Zodiac Hours',
        'today'        => 'Today',
        'sunday'       => 'Sunday',
        'monday'       => 'Monday',
        'tuesday'      => 'Tuesday',
        'wednesday'    => 'Wednesday',
        'thursday'     => 'Thursday',
        'friday'       => 'Friday',
        'saturday'     => 'Saturday',
        'view'         => 'View',
        'lunar_day_short' => 'd',
        'lunar_month_short' => 'm',
    ];
}

/**
 * Get a translated string from options.
 */
function lnhlc_t( $key ) {
    $defaults = lnhlc_get_translatable_strings();
    if ( ! isset( $defaults[$key] ) ) {
        return '';
    }
    $options = get_option( LNHLC_OPTIONS_NAME );
    $option_key = 'translate_' . $key;
    if ( ! empty( $options[$option_key] ) ) {
        return $options[$option_key];
    }
    return $defaults[$key];
}

// --- RELIABLE ALGORITHM v2.4.3 ---

class LNH_Lunar_Calendar_Engine {
    private static $instance = null;
    private $time_zone = 7;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function jd_from_date($dd, $mm, $yy) {
        return gregoriantojd($mm, $dd, $yy);
    }

    private function get_new_moon_day($k) {
        $T = $k / 1236.85; $T2 = $T * $T; $T3 = $T2 * $T;
        $Jd1 = 2415020.75933 + 29.53058868 * $k + 0.0001178 * $T2 - 0.000000155 * $T3;
        $Jd1 += 0.00033 * sin(deg2rad(166.56 + 132.87 * $T - 0.009173 * $T2));
        $M = 359.2242 + 29.10535608 * $k - 0.0000333 * $T2 - 0.00000347 * $T3;
        $Mpr = 306.0253 + 385.81691806 * $k + 0.0107306 * $T2 + 0.00001236 * $T3;
        $F = 21.2964 + 390.67050646 * $k - 0.0016528 * $T2 - 0.00000239 * $T3;
        $C1 = (0.1734 - 0.000393 * $T) * sin(deg2rad($M)) + 0.0021 * sin(deg2rad(2 * $M));
        $C1 -= 0.4068 * sin(deg2rad($Mpr)) + 0.0161 * sin(deg2rad(2 * $Mpr));
        $C1 -= 0.0004 * sin(deg2rad(3 * $Mpr));
        $C1 += 0.0104 * sin(deg2rad(2 * $F)) - 0.0051 * sin(deg2rad($M + $Mpr));
        $C1 -= 0.0074 * sin(deg2rad($M - $Mpr)) + 0.0004 * sin(deg2rad(2 * $F + $M));
        $C1 -= 0.0004 * sin(deg2rad(2 * $F - $M)) - 0.0006 * sin(deg2rad(2 * $F + $Mpr));
        $C1 += 0.0010 * sin(deg2rad(2 * $F - $Mpr)) + 0.0005 * sin(deg2rad(2 * $Mpr + $M));
        $deltat = -0.000278 + 0.000265 * $T + 0.000262 * $T2;
        $JdNew = $Jd1 + $C1 - $deltat;
        return floor($JdNew + 0.5 + $this->time_zone / 24);
    }

    private function get_sun_longitude($jdn) {
        $T = ($jdn - 2451545.5 - $this->time_zone / 24) / 36525;
        $M = 357.52910 + 35999.05030 * $T;
        $L0 = 280.46645 + 36000.76983 * $T;
        $DL = (1.914600 - 0.004817 * $T) * sin(deg2rad($M));
        $DL += (0.019993 - 0.000101 * $T) * sin(deg2rad(2 * $M)) + 0.000290 * sin(deg2rad(3 * $M));
        $L = $L0 + $DL;
        $omega = 125.04 - 1934.136 * $T;
        $L = $L - 0.00569 - 0.00478 * sin(deg2rad($omega));
        $L = fmod($L, 360);
        return floor($L / 30);
    }
    
    private function get_lunar_month11($yy) {
        $off = $this->jd_from_date(31, 12, $yy) - 2415021;
        $k = floor($off / 29.530588853);
        $nm = $this->get_new_moon_day($k);
        if ($this->get_sun_longitude($nm) >= 9) {
            $nm = $this->get_new_moon_day($k - 1);
        }
        return $nm;
    }

    private function get_leap_month_offset($a11) {
        $k = floor(($a11 - 2415021.076998695) / 29.530588853 + 0.5);
        $last = 0; $i = 1;
        $arc = $this->get_sun_longitude($this->get_new_moon_day($k + $i));
        do {
            $last = $arc; $i++;
            $arc = $this->get_sun_longitude($this->get_new_moon_day($k + $i));
        } while ($arc != $last && $i < 14);
        return $i - 1;
    }

    public function convert_solar_to_lunar($dd, $mm, $yy) {
        $dayNumber = $this->jd_from_date($dd, $mm, $yy);
        $k = floor(($dayNumber - 2415021.076998695) / 29.530588853);
        $monthStart = $this->get_new_moon_day($k + 1);
        if ($monthStart > $dayNumber) {
            $monthStart = $this->get_new_moon_day($k);
        }
        $a11 = $this->get_lunar_month11($yy);
        $b11 = $a11;
        if ($a11 >= $monthStart) {
            $lunarYear = $yy;
            $a11 = $this->get_lunar_month11($yy - 1);
        } else {
            $lunarYear = $yy + 1;
            $b11 = $this->get_lunar_month11($yy + 1);
        }
        $lunarDay = $dayNumber - $monthStart + 1;
        $diff = floor(($monthStart - $a11) / 29);
        $is_leap = 0;
        $lunarMonth = $diff + 11;
        if ($b11 - $a11 > 365) {
            $leapMonthDiff = $this->get_leap_month_offset($a11);
            if ($diff >= $leapMonthDiff) {
                $lunarMonth = $diff + 10;
                if ($diff == $leapMonthDiff) {
                    $is_leap = 1;
                }
            }
        }
        if ($lunarMonth > 12) {
            $lunarMonth -= 12;
        }
        if ($lunarMonth >= 11 && $diff < 4) {
            $lunarYear -= 1;
        }
        return [$lunarDay, $lunarMonth, $lunarYear, $is_leap];
    }
}

/**
 * **NEW**: Calculates Zodiac Hours based on the day's Earthly Branch (Chi).
 */
function lnhlc_get_zodiac_hours($chi_day) {
    $zodiac_hours_map = [
        'Tý' => ['Tý', 'Sửu', 'Mão', 'Ngọ', 'Thân', 'Dậu'],
        'Sửu' => ['Dần', 'Mão', 'Tỵ', 'Thân', 'Tuất', 'Hợi'],
        'Dần' => ['Tý', 'Sửu', 'Thìn', 'Tỵ', 'Mùi', 'Tuất'],
        'Mão' => ['Tý', 'Dần', 'Mão', 'Ngọ', 'Mùi', 'Dậu'],
        'Thìn' => ['Dần', 'Thìn', 'Tỵ', 'Thân', 'Dậu', 'Hợi'],
        'Tỵ' => ['Sửu', 'Thìn', 'Ngọ', 'Mùi', 'Tuất', 'Hợi'],
        'Ngọ' => ['Tý', 'Sửu', 'Mão', 'Ngọ', 'Thân', 'Dậu'],
        'Mùi' => ['Dần', 'Mão', 'Tỵ', 'Thân', 'Tuất', 'Hợi'],
        'Thân' => ['Tý', 'Sửu', 'Thìn', 'Tỵ', 'Mùi', 'Tuất'],
        'Dậu' => ['Tý', 'Dần', 'Mão', 'Ngọ', 'Mùi', 'Dậu'],
        'Tuất' => ['Dần', 'Thìn', 'Tỵ', 'Thân', 'Dậu', 'Hợi'],
        'Hợi' => ['Sửu', 'Thìn', 'Ngọ', 'Mùi', 'Tuất', 'Hợi'],
    ];

    $chi_times = [
        'Tý' => '23-1', 'Sửu' => '1-3', 'Dần' => '3-5', 'Mão' => '5-7',
        'Thìn' => '7-9', 'Tỵ' => '9-11', 'Ngọ' => '11-13', 'Mùi' => '13-15',
        'Thân' => '15-17', 'Dậu' => '17-19', 'Tuất' => '19-21', 'Hợi' => '21-23'
    ];

    $day_chi_name = explode(' ', $chi_day)[1];
    if (!isset($zodiac_hours_map[$day_chi_name])) {
        return 'N/A';
    }

    $hours = $zodiac_hours_map[$day_chi_name];
    $result = [];
    foreach ($hours as $hour) {
        $result[] = $hour . ' (' . $chi_times[$hour] . ')';
    }

    return implode(', ', $result);
}



/**
 * Get Can Chi data for a specific SOLAR date.
 */
function lnhlc_get_can_chi($solar_day, $solar_month, $solar_year, $lunar_month, $lunar_year) {
    $can = ['Giáp', 'Ất', 'Bính', 'Đinh', 'Mậu', 'Kỷ', 'Canh', 'Tân', 'Nhâm', 'Quý'];
    $chi = ['Tý', 'Sửu', 'Dần', 'Mão', 'Thìn', 'Tỵ', 'Ngọ', 'Mùi', 'Thân', 'Dậu', 'Tuất', 'Hợi'];
    
    $jd = gregoriantojd($solar_month, $solar_day, $solar_year);
    
    $can_day = $can[($jd + 9) % 10];
    $chi_day = $chi[($jd + 1) % 12];
    
    $can_year = $can[($lunar_year - 4) % 10];
    $chi_year = $chi[($lunar_year - 4) % 12];

    // Calculate Can Chi for the month
    $can_month = $can[($lunar_year * 12 + $lunar_month + 3) % 10];
    $chi_month = $chi[($lunar_month + 1) % 12];

    return [
        'day'   => $can_day . ' ' . $chi_day,
        'month' => $can_month . ' ' . $chi_month,
        'year'  => $can_year . ' ' . $chi_year,
    ];
}

/**
 * Get all calendar data for a specific SOLAR date.
 */
function lnhlc_get_lunar_date_data($timestamp = null) {
    if (is_null($timestamp)) {
        $timestamp = time();
    }

    $solar_day = (int)wp_date('d', $timestamp);
    $solar_month = (int)wp_date('m', $timestamp);
    $solar_year = (int)wp_date('Y', $timestamp);
    
    $engine = LNH_Lunar_Calendar_Engine::get_instance();
    list($lunar_day, $lunar_month, $lunar_year, $is_leap) = $engine->convert_solar_to_lunar($solar_day, $solar_month, $solar_year);
    $can_chi = lnhlc_get_can_chi($solar_day, $solar_month, $solar_year, $lunar_month, $lunar_year);
    
    $weekday_num = (int)wp_date('w', $timestamp);
    $weekdays = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];

    return [
        'solar_day'   => $solar_day,
        'solar_month' => $solar_month,
        'solar_year'  => $solar_year,
        'weekday'     => lnhlc_t($weekdays[$weekday_num]),
        'lunar_day'   => $lunar_day,
        'lunar_month' => $lunar_month,
        'lunar_year'  => $lunar_year,
        'is_leap'     => $is_leap,
        'can_chi_day' => $can_chi['day'],
        'can_chi_month' => $can_chi['month'],
        'can_chi_year' => $can_chi['year'],
        // **UPDATE**: Call the new function to get dynamic zodiac hours
        'zodiac_hours_list' => lnhlc_get_zodiac_hours($can_chi['day'])
    ];
}
