<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Handles the plugin settings page.
 */
class LNH_Lunar_Calendar_Settings {

    private $options;

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'add_plugin_page' ] );
        add_action( 'admin_init', [ $this, 'page_init' ] );
    }

    public function add_plugin_page() {
        add_options_page(
            __( 'LNH Lunar Calendar Settings', 'lnh-lunar-calendar' ),
            __( 'LNH Lunar Calendar', 'lnh-lunar-calendar' ),
            'manage_options',
            'lnh-lunar-calendar',
            [ $this, 'create_admin_page' ]
        );
    }

    public function create_admin_page() {
        $this->options = get_option( LNHLC_OPTIONS_NAME );
        ?>
        <div class="wrap lnhlc-admin-wrap">
            <h1><?php esc_html_e( 'LNH Lunar Calendar Settings', 'lnh-lunar-calendar' ); ?></h1>
            
            <?php
            // **SECURITY FIX**: Sanitize and validate the tab input.
            $allowed_tabs = ['general', 'translate'];
            $active_tab = 'general'; // Default to a safe value.

            if ( isset( $_GET['tab'] ) ) {
                $selected_tab = sanitize_key( $_GET['tab'] );
                if ( in_array( $selected_tab, $allowed_tabs, true ) ) {
                    $active_tab = $selected_tab;
                }
            }
            ?>
            <h2 class="nav-tab-wrapper">
                <a href="?page=lnh-lunar-calendar&tab=general" class="nav-tab <?php echo $active_tab === 'general' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'General', 'lnh-lunar-calendar' ); ?></a>
                <a href="?page=lnh-lunar-calendar&tab=translate" class="nav-tab <?php echo $active_tab === 'translate' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Translate', 'lnh-lunar-calendar' ); ?></a>
            </h2>

            <form method="post" action="options.php">
                <?php
                settings_fields( 'lnh_lunar_calendar_option_group' );
                
                if ( $active_tab === 'general' ) {
                    do_settings_sections( 'lnh-lunar-calendar-admin-general' );
                } else { // 'translate'
                    do_settings_sections( 'lnh-lunar-calendar-admin-translate' );
                }
                
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public function page_init() {
        register_setting(
            'lnh_lunar_calendar_option_group',
            LNHLC_OPTIONS_NAME,
            [ $this, 'sanitize' ]
        );

        // General Settings Section
        add_settings_section(
            'lnhlc_setting_section_general',
            __( 'Default Shortcode Settings', 'lnh-lunar-calendar' ),
            null,
            'lnh-lunar-calendar-admin-general'
        );

        add_settings_field(
            'default_theme',
            __( 'Default Theme', 'lnh-lunar-calendar' ),
            [ $this, 'default_theme_callback' ],
            'lnh-lunar-calendar-admin-general',
            'lnhlc_setting_section_general'
        );

        // Translate Settings Section
        add_settings_section(
            'lnhlc_setting_section_translate',
            __( 'Frontend Text Translation', 'lnh-lunar-calendar' ),
            function() {
                echo '<p>' . esc_html__( 'Enter your custom text for the frontend display. Leave blank to use the English default.', 'lnh-lunar-calendar' ) . '</p>';
            },
            'lnh-lunar-calendar-admin-translate'
        );

        $translatable_strings = lnhlc_get_translatable_strings();
        foreach ($translatable_strings as $key => $default_value) {
             add_settings_field(
                'translate_' . $key,
                $default_value,
                [ $this, 'translate_field_callback' ],
                'lnh-lunar-calendar-admin-translate',
                'lnhlc_setting_section_translate',
                ['key' => $key]
            );
        }
    }

    public function sanitize( $input ) {
        $new_input = [];
        if ( isset( $input['default_theme'] ) ) {
            $new_input['default_theme'] = sanitize_text_field( $input['default_theme'] );
        }
        
        $translatable_strings = lnhlc_get_translatable_strings();
        foreach ($translatable_strings as $key => $default_value) {
            if ( isset( $input['translate_' . $key] ) ) {
                $new_input['translate_' . $key] = sanitize_text_field( $input['translate_' . $key] );
            }
        }
        return $new_input;
    }

    public function default_theme_callback() {
        $options = get_option(LNHLC_OPTIONS_NAME);
        $theme = isset( $options['default_theme'] ) ? $options['default_theme'] : 'light';
        ?>
        <select name="<?php echo esc_attr(LNHLC_OPTIONS_NAME); ?>[default_theme]">
            <option value="light" <?php selected( $theme, 'light' ); ?>><?php esc_html_e( 'Light', 'lnh-lunar-calendar' ); ?></option>
            <option value="dark" <?php selected( $theme, 'dark' ); ?>><?php esc_html_e( 'Dark', 'lnh-lunar-calendar' ); ?></option>
        </select>
        <?php
    }

    public function translate_field_callback($args) {
        $options = get_option(LNHLC_OPTIONS_NAME);
        $key = $args['key'];
        $value = isset( $options['translate_' . $key] ) ? $options['translate_' . $key] : '';
        printf(
            '<input type="text" class="regular-text" name="%s[translate_%s]" value="%s" />',
            esc_attr( LNHLC_OPTIONS_NAME ),
            esc_attr( $key ),
            esc_attr( $value )
        );
    }
}
