<?php
/**
 * Plugin Name:     LiveFot Matches
 * Plugin URI:      https://api.livefootballcenter.com
 * Description:     Display live football matches and scores from the LiveFot API.
 * Version:         1.0.1
 * Author:          Jeff Kamran
 * Text Domain:     livefot-matches
 * License:         GPLv2 or later
 * License URI:     https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin constants
define('LIVEFOT_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('LIVEFOT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LIVEFOT_VERSION', '1.0.1');

// Include required files
require_once LIVEFOT_PLUGIN_PATH . 'includes/class-livefot-api.php';
require_once LIVEFOT_PLUGIN_PATH . 'includes/class-livefot-ajax.php';

// Table creation during plugin activation
register_activation_hook(__FILE__, 'livefot_activate_plugin');

function livefot_activate_plugin() {
    livefot_create_tables(); 

	// Schedule the daily cron job if enabled
    $enable_cron = get_option('livefot_enable_cron_job', 1); // Default to enabled
    if ($enable_cron && !wp_next_scheduled('livefot_daily_fetch_fixtures')) {
        livefot_schedule_cron_job();
    }
	
	  // Schedule the short-range cron job if enabled
    $enable_short_cron = get_option('livefot_enable_short_cron_job', 1); // Default to enabled
    if ($enable_short_cron && !wp_next_scheduled('livefot_short_fetch_fixtures')) {
        livefot_schedule_short_cron_job();
    }
}

// Plugin deactivation: unschedule cron jobs and cleanup
register_deactivation_hook(__FILE__, 'livefot_deactivate_plugin');

function livefot_deactivate_plugin() {
    // Unschedule the daily cron job
    wp_clear_scheduled_hook('livefot_daily_fetch_fixtures');

    // Unschedule the short-range cron job
    wp_clear_scheduled_hook('livefot_short_fetch_fixtures');

    // Perform any additional cleanup if necessary
    // Note: Uninstall hook handles database cleanup
}

// Plugin uninstall: cleanup
/*register_uninstall_hook(__FILE__, 'livefot_cleanup_on_uninstall');

function livefot_cleanup_on_uninstall() {
    global $wpdb;
    $tables = [
        'livefot_fixtures',
        'livefot_leagues',
        'livefot_countries',
        'livefot_scores',
        'livefot_time',
        'livefot_aggregate',
        'livefot_referees',
        'livefot_venues',
        'livefot_lineups',
        'livefot_players',
        'livefot_events',
        'livefot_teams',
        'livefot_formations',
		'livefot_standings',
		'livefot_team_stats',
        'livefot_api_calls'
    ];
    foreach ($tables as $table) {
        $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}$table");
    }

    // Delete plugin options
    delete_option('livefot_api_key');
    delete_option('livefot_api_url'); 
    delete_option('livefot_enable_cron_job');
    delete_option('livefot_cron_hour');
    delete_option('livefot_cron_minute');
    delete_option('livefot_manual_date');
    delete_option('livefot_interval_matches');
    delete_option('livefot_interval_lineups');
    delete_option('livefot_interval_events');
    delete_option('livefot_interval_stats');
	delete_option('livefot_interval_standings');

    // Delete short-range cron job options
    delete_option('livefot_enable_short_cron_job');
  //  delete_option('livefot_short_cron_start_hour');
  //  delete_option('livefot_short_cron_start_minute');
    delete_option('livefot_cron_interval_short');
	delete_option('livefot_cron_last_run_status');
	delete_option('livefot_cron_last_run_start');
	delete_option('livefot_cron_last_run_end');
    delete_option('livefot_cron_last_run_status_short');
    delete_option('livefot_cron_last_run_start_short');
    delete_option('livefot_cron_last_run_end_short');
}

// Hook the scheduled event to a callback function
add_action('livefot_daily_fetch_fixtures', 'livefot_fetch_and_insert_fixtures');*/

register_uninstall_hook( __FILE__, 'livefot_cleanup_on_uninstall' );

function livefot_cleanup_on_uninstall() {
    global $wpdb;

    // list of your custom tables (without prefix)
    $tables = [
        'livefot_fixtures',
        'livefot_leagues',
        'livefot_countries',
        'livefot_scores',
        'livefot_time',
        'livefot_aggregate',
        'livefot_referees',
        'livefot_venues',
        'livefot_lineups',
        'livefot_players',
        'livefot_events',
        'livefot_teams',
        'livefot_formations',
        'livefot_standings',
        'livefot_team_stats',
        'livefot_api_calls',
    ];

    foreach ( $tables as $table ) {
        // safely construct the full table name
        $full_table = $wpdb->prefix . $table;
        // drop it (backticks added around the name, and esc_sql to be extra safe)
        $wpdb->query(
            'DROP TABLE IF EXISTS `'
            . esc_sql( $full_table )
            . '`'
        );
    }

    // options to delete
    $options = [
        'livefot_api_key',
        'livefot_api_url',
        'livefot_enable_cron_job',
        'livefot_cron_hour',
        'livefot_cron_minute',
        'livefot_manual_date',
        'livefot_interval_matches',
        'livefot_interval_lineups',
        'livefot_interval_events',
        'livefot_interval_stats',
        'livefot_interval_standings',
        'livefot_enable_short_cron_job',
        'livefot_cron_interval_short',
        'livefot_cron_last_run_status',
        'livefot_cron_last_run_start',
        'livefot_cron_last_run_end',
        'livefot_cron_last_run_status_short',
        'livefot_cron_last_run_start_short',
        'livefot_cron_last_run_end_short',
    ];
    foreach ( $options as $opt ) {
        delete_option( $opt );
    }
}

// (Your existing scheduled hook can stay as-is.)
add_action( 'livefot_daily_fetch_fixtures', 'livefot_fetch_and_insert_fixtures' );



function livefot_fetch_and_insert_fixtures() {
    // Get current UTC time for start
    $utc_start_time = gmdate('Y-m-d H:i:s'); // UTC time
    update_option('livefot_cron_last_run_start', $utc_start_time);

    // Initialize status as running
    update_option('livefot_cron_last_run_status', 'Running');

    $api = new LiveFot_API();
    $result = $api->livefot_fetch_and_insert_fixtures(); // Your custom logic

    // Get current UTC time for end
    $utc_end_time = gmdate('Y-m-d H:i:s'); // UTC time
    update_option('livefot_cron_last_run_end', $utc_end_time);

    // Update status based on result
    if ($result === true) {
        update_option('livefot_cron_last_run_status', 'Success');
    } else {
        update_option('livefot_cron_last_run_status', 'Failed: ' . $result);
    }

    // Log start and end times for debugging
  //  error_log("Daily cron job started at: $utc_start_time UTC");
  //  error_log("Daily cron job ended at: $utc_end_time UTC");
}


// Register short-range cron event
add_action('livefot_short_fetch_fixtures', 'livefot_handle_short_fetch_fixtures');

function livefot_handle_short_fetch_fixtures() {
    try {
        // Mark start time and status for SHORT cron only
        $utc_start_time_short = gmdate('Y-m-d H:i:s');
        update_option('livefot_cron_last_run_status_short', 'Running');
        update_option('livefot_cron_last_run_start_short', $utc_start_time_short);

        // Perform the fetch logic
        $api = new LiveFot_API();
        $result = $api->livefot_handle_short_fetch_fixtures();

        // Update status based on result
        if ($result === true) {
            update_option('livefot_cron_last_run_status_short', 'Success');
        } else {
            update_option('livefot_cron_last_run_status_short', 'Failed: ' . $result);
        }
    } catch (Exception $e) {
        // Catch any unexpected exceptions
        update_option('livefot_cron_last_run_status_short', 'Failed: ' . $e->getMessage());
    } finally {
        // Mark end time
        $utc_end_time_short = gmdate('Y-m-d H:i:s');
        update_option('livefot_cron_last_run_end_short', $utc_end_time_short);

        // Log execution times
      //  error_log("Short-range cron started at (UTC): {$utc_start_time_short}");
      //  error_log("Short-range cron ended at (UTC):   {$utc_end_time_short}");
    }
}


/*add_filter('cron_schedules', 'livefot_add_cron_intervals');
function livefot_add_cron_intervals($schedules) {
    // Retrieve the interval from options, default to 15 minutes
    $interval_minutes = intval(get_option('livefot_cron_interval_short', 15));
    if ($interval_minutes < 15) {
        $interval_minutes = 15; // Enforce minimum 15 minutes
        update_option('livefot_cron_interval_short', $interval_minutes);
        error_log("Short-range cron interval too low. Set to minimum 15 minutes.");
    }

    // Add the custom interval to the schedules array
    $schedules['livefot_short_interval'] = array(
        'interval' => $interval_minutes * MINUTE_IN_SECONDS,
        'display'  => sprintf(__('Every %d minutes', 'livefot'), $interval_minutes)
    );
    return $schedules;
}*/  //jk18042025

add_filter( 'cron_schedules', 'livefot_add_cron_intervals' );
function livefot_add_cron_intervals( $schedules ) {
    // Retrieve the interval from options, default to 15 minutes
    $interval_minutes = intval( get_option( 'livefot_cron_interval_short', 15 ) );
    if ( $interval_minutes < 15 ) {
        $interval_minutes = 15; // Enforce minimum 15 minutes
        update_option( 'livefot_cron_interval_short', $interval_minutes );
      //  error_log( 'Short‑range cron interval too low. Set to minimum 15 minutes.' );
    }

     
    $schedules['livefot_short_interval'] = array(
        'interval' => $interval_minutes * MINUTE_IN_SECONDS,
		/* translators: %d is the number of minutes between each cron run. */
        'display'  => sprintf( __( 'Every %d minutes', 'livefot-matches' ), $interval_minutes ),
    );

    return $schedules;
}


// Table creation function
function livefot_create_tables() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    // Initialize an array to hold SQL statements
    $sql = [];

    // Lineup table
    $lineup_table = $wpdb->prefix . 'livefot_lineups';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $lineup_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            match_id BIGINT(20) NOT NULL,
            team_id BIGINT(20) NOT NULL,
            player_id BIGINT(20) NOT NULL,
            number INT NOT NULL,
            position VARCHAR(10) NOT NULL,
            formation_position INT DEFAULT NULL,
            captain BOOLEAN NOT NULL DEFAULT FALSE,
            type VARCHAR(20) NOT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			UNIQUE KEY unique_match_team_player (match_id, team_id, player_id)
        ) $charset_collate;
    ";

    // Leagues table
    $leagues_table = $wpdb->prefix . 'livefot_leagues';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $leagues_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            league_id BIGINT(20) NOT NULL UNIQUE,
            name VARCHAR(255) NOT NULL,
            is_cup BOOLEAN NOT NULL DEFAULT FALSE,
            logo_path VARCHAR(2083) DEFAULT NULL,
            current_season_id BIGINT(20) DEFAULT NULL,
            stage_name VARCHAR(100) DEFAULT NULL,
            country_id BIGINT(20) NOT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Countries table
    $countries_table = $wpdb->prefix . 'livefot_countries';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $countries_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            country_id BIGINT(20) NOT NULL UNIQUE,
            name VARCHAR(255) NOT NULL,
            image_path VARCHAR(2083) DEFAULT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Fixtures table
    $fixtures_table = $wpdb->prefix . 'livefot_fixtures';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $fixtures_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            fixture_id BIGINT(20) NOT NULL UNIQUE,
            season_id BIGINT(20) NOT NULL,
            league_id BIGINT(20) NOT NULL,
            group_id BIGINT(20) DEFAULT NULL,
            aggregate_id BIGINT(20) DEFAULT NULL,
            localteam_id BIGINT(20) NOT NULL,
            localteam_name VARCHAR(255) NOT NULL,
            visitorteam_id BIGINT(20) NOT NULL,
            visitorteam_name VARCHAR(255) NOT NULL,
            stage_type VARCHAR(50) DEFAULT NULL,
            localteam_coach_id BIGINT(20) DEFAULT NULL,
            visitorteam_coach_id BIGINT(20) DEFAULT NULL,
            winner_team_id BIGINT(20) DEFAULT NULL,
            commentaries BOOLEAN DEFAULT FALSE,
            leg VARCHAR(10) DEFAULT NULL,
            is_placeholder BOOLEAN DEFAULT FALSE,
            referee_id BIGINT(20) DEFAULT NULL,
            venue_id BIGINT(20) DEFAULT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Referees table
    $referees_table = $wpdb->prefix . 'livefot_referees';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $referees_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            referee_id BIGINT(20) NOT NULL UNIQUE,
            common_name VARCHAR(255) NOT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Venues table
    $venues_table = $wpdb->prefix . 'livefot_venues';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $venues_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            venue_id BIGINT(20) NOT NULL UNIQUE,
            name VARCHAR(255) NOT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Scores table
    $scores_table = $wpdb->prefix . 'livefot_scores';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $scores_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            fixture_id BIGINT(20) NOT NULL UNIQUE,
            localteam_score INT DEFAULT NULL,
            visitorteam_score INT DEFAULT NULL,
            localteam_pen_score INT DEFAULT NULL,
            visitorteam_pen_score INT DEFAULT NULL,
            ht_score VARCHAR(10) DEFAULT NULL,
            ft_score VARCHAR(10) DEFAULT NULL,
            et_score VARCHAR(10) DEFAULT NULL,
            ps_score VARCHAR(10) DEFAULT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Time table
    $time_table = $wpdb->prefix . 'livefot_time';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $time_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            fixture_id BIGINT(20) NOT NULL UNIQUE,
            status VARCHAR(50) DEFAULT NULL,
            starting_at_datetime DATETIME DEFAULT NULL,
            starting_at_date DATE DEFAULT NULL,
            starting_at_time TIME DEFAULT NULL,
            starting_at_timestamp BIGINT(20) DEFAULT NULL,
            starting_at_timezone VARCHAR(50) DEFAULT NULL,
            minute INT DEFAULT NULL,
            second INT DEFAULT NULL,
            added_time INT DEFAULT NULL,
            extra_minute INT DEFAULT NULL,
            injury_time INT DEFAULT NULL,
            match_period VARCHAR(50) DEFAULT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Aggregate table
    $aggregate_table = $wpdb->prefix . 'livefot_aggregate';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $aggregate_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            aggregate_id BIGINT(20) NOT NULL UNIQUE,
            league_id BIGINT(20) NOT NULL,
            season_id BIGINT(20) NOT NULL,
            stage_id BIGINT(20) DEFAULT NULL,
            localteam VARCHAR(255) DEFAULT NULL,
            localteam_id BIGINT(20) DEFAULT NULL,
            visitorteam VARCHAR(255) DEFAULT NULL,
            visitorteam_id BIGINT(20) DEFAULT NULL,
            result VARCHAR(50) DEFAULT NULL,
            winner BIGINT(20) DEFAULT NULL,
            detail VARCHAR(255) DEFAULT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Players table
    $players_table = $wpdb->prefix . 'livefot_players';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $players_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            player_id BIGINT(20) NOT NULL UNIQUE,
            player_name VARCHAR(255) NOT NULL,
            logo_path VARCHAR(2083) DEFAULT NULL
        ) $charset_collate;
    ";

    // Events table
    $events_table = $wpdb->prefix . 'livefot_events';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $events_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            event_id BIGINT(20) NOT NULL UNIQUE,
            match_id BIGINT(20) NOT NULL,
            team_id BIGINT(20) NOT NULL,
            player_id BIGINT(20) DEFAULT NULL,
            related_player_id BIGINT(20) DEFAULT NULL,
            type VARCHAR(50) NOT NULL,
            minute INT DEFAULT NULL,
            extra_minute INT DEFAULT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    // Teams table
    $teams_table = $wpdb->prefix . 'livefot_teams';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $teams_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            team_id BIGINT(20) NOT NULL UNIQUE,
            name VARCHAR(255) NOT NULL,
            shortcode VARCHAR(50) DEFAULT NULL,
            twitter VARCHAR(255) DEFAULT NULL,
            country_id BIGINT(20) DEFAULT NULL,
            national_team BOOLEAN DEFAULT FALSE,
            founded INT DEFAULT NULL,
            logo_path VARCHAR(2083) DEFAULT NULL,
            venue_id BIGINT(20) DEFAULT NULL,
            current_season_id BIGINT(20) DEFAULT NULL,
            gender VARCHAR(10) DEFAULT NULL,
            team_type VARCHAR(50) DEFAULT NULL
        ) $charset_collate;
    ";

    // Formations table
    $formations_table = $wpdb->prefix . 'livefot_formations';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $formations_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            match_id BIGINT(20) NOT NULL UNIQUE,
            local_team_id BIGINT(20) NOT NULL,
            local_team_formation VARCHAR(50) DEFAULT NULL,
            visitor_team_id BIGINT(20) NOT NULL,
            visitor_team_formation VARCHAR(50) DEFAULT NULL,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";
	
	
	
	// New Standings Table
$standings_table = $wpdb->prefix . 'livefot_standings';
$sql[] = "
    CREATE TABLE IF NOT EXISTS $standings_table (
        id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        standing_id BIGINT(20) NOT NULL UNIQUE,
        league_id BIGINT(20) NOT NULL,
        group_id BIGINT(20) DEFAULT NULL,
        season_id BIGINT(20) NOT NULL,
        stage_id BIGINT(20) NOT NULL,
        round_id BIGINT(20) NOT NULL,
        position INT NOT NULL,
        team_id BIGINT(20) NOT NULL,
        team_name VARCHAR(255) NOT NULL,
        shortcode VARCHAR(50) DEFAULT NULL,
        team_logo VARCHAR(2083) DEFAULT NULL,
        goals VARCHAR(20) DEFAULT NULL,
        goal_diff INT DEFAULT NULL,
        wins INT DEFAULT NULL,
        lost INT DEFAULT NULL,
        draws INT DEFAULT NULL,
        played INT DEFAULT NULL,
        points INT DEFAULT NULL,
        description VARCHAR(255) DEFAULT NULL,
        recent_form VARCHAR(255) DEFAULT NULL,
        standing_rule_id BIGINT(20) DEFAULT NULL,
        result VARCHAR(100) DEFAULT NULL,
        fairplay_points_lose INT DEFAULT NULL,
        updated_at TIMESTAMP NULL DEFAULT NULL,
        inserted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) $charset_collate;
";
	
	$team_stats_table = $wpdb->prefix . 'livefot_team_stats';

$sql[] = "
    CREATE TABLE IF NOT EXISTS $team_stats_table (
        stats_id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        fixture_id BIGINT(20) UNSIGNED NOT NULL,
        team_id BIGINT(20) UNSIGNED NOT NULL,
        fouls INT DEFAULT NULL,
        corners INT DEFAULT NULL,
        offsides INT DEFAULT NULL,
        possession_time INT DEFAULT NULL,
        yellow_cards INT DEFAULT NULL,
        red_cards INT DEFAULT NULL,
        yellow_red_cards INT DEFAULT NULL,
        saves INT DEFAULT NULL,
        substitutions INT DEFAULT NULL,
        goal_kick INT DEFAULT NULL,
        goal_attempts INT DEFAULT NULL,
        free_kick INT DEFAULT NULL,
        throw_in INT DEFAULT NULL,
        ball_safe INT DEFAULT NULL,
        goals INT DEFAULT NULL,
        penalties INT DEFAULT NULL,
        injuries INT DEFAULT NULL,
        tackles INT DEFAULT NULL,
        attacks INT DEFAULT NULL,
        dangerous_attacks INT DEFAULT NULL,
        passes_total INT DEFAULT NULL,
        passes_accurate INT DEFAULT NULL,
        passes_percentage INT DEFAULT NULL,
        shots_total INT DEFAULT NULL,
        shots_ongoal INT DEFAULT NULL,
        shots_blocked INT DEFAULT NULL,
        shots_offgoal INT DEFAULT NULL,
        shots_insidebox INT DEFAULT NULL,
        shots_outsidebox INT DEFAULT NULL,
        last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY unique_fixture_team (fixture_id, team_id)
    ) $charset_collate;
";



    // API call tracking table
    $api_calls_table = $wpdb->prefix . 'livefot_api_calls';
    $sql[] = "
        CREATE TABLE IF NOT EXISTS $api_calls_table (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            endpoint_name VARCHAR(255) NOT NULL UNIQUE,
            remaining_calls INT DEFAULT NULL,
            status VARCHAR(50) NOT NULL,
            last_call_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;
    ";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    foreach ($sql as $query) {
        dbDelta($query);
    }
}

// Initialize the plugin
add_action('init', 'livefot_init');
function livefot_init() {
    // Register scripts and styles
    add_action('wp_enqueue_scripts', 'livefot_enqueue_scripts');
	
	 // Admin scripts and styles
  //  add_action('admin_enqueue_scripts', 'livefot_enqueue_admin_scripts');
    
    
    // Register shortcode
    add_shortcode('livefot_matches', 'livefot_matches_shortcode');
}


/*
// Enqueue scripts and styles
function livefot_enqueue_scripts() {

    // Enqueue Flatpickr CSS from CDN
    wp_enqueue_style(
        'flatpickr-css',
        plugins_url('assets/vendor/flatpickr/flatpickr.min.css', __FILE__),
        array(),
        '4.6.13'
    );

    // Enqueue Flatpickr JS from CDN

    wp_enqueue_script(
        'flatpickr-js',
        plugins_url('assets/js/flatpickr/flatpickr.js', __FILE__),
        array('jquery'),
        '4.6.13',
        true
    );
	
    wp_enqueue_style(
        'livefot-styles',
        LIVEFOT_PLUGIN_URL . 'assets/css/styles.css',
        array(),
        LIVEFOT_VERSION
    );
	
    wp_enqueue_script(
        'livefot-matches',
        LIVEFOT_PLUGIN_URL . 'assets/js/matches.js',
        array('jquery'),
        LIVEFOT_VERSION,
        true
    );

    // Retrieve user-defined intervals from options (default values if not set)
    $matches_interval_ms = (int) get_option('livefot_interval_matches', 30) * 1000;   // default 30s
    $lineups_interval_ms = (int) get_option('livefot_interval_lineups', 60) * 1000;   // default 60s
    $events_interval_ms = (int) get_option('livefot_interval_events', 60) * 1000;     // default 60s
    $stats_interval_ms = (int) get_option('livefot_interval_stats', 120) * 1000;      // default 120s
	$standings_interval_ms = (int) get_option('livefot_interval_standings', 120) * 1000;      // default 120s

    wp_localize_script('livefot-matches', 'livefotAjax', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('livefot_nonce'),
        'intervals' => array(
            'matches' => $matches_interval_ms,
            'lineups' => $lineups_interval_ms,
            'events' => $events_interval_ms,
			'standings' => $standings_interval_ms,
            'statistics' => $stats_interval_ms
        ),
		'icons_base_url' => plugins_url('assets/images/', __FILE__),
    ));
}*/
  //aded by jk20042025

/**
 * Enqueue scripts & styles (no external CDN calls)
 */
/*function livefot_enqueue_scripts() {

	/* ───────────────────────────────
	 * 1.  jQuery UI Datepicker  (JS)
	 *     ─ WP core ships it at /wp‑includes/js/jquery/ui/
	 * ─────────────────────────────── */
/*	wp_enqueue_script( 'jquery-ui-datepicker' );           // depends on core jQuery

	/* ───────────────────────────────
	 * 2.  jQuery UI Theme  (CSS)
	 *     ─ WP core ships a minified theme file
	 *       /wp‑includes/css/jquery-ui-theme.min.css
	 * ─────────────────────────────── */
/*	wp_enqueue_style(
		'jquery-ui-theme',
		includes_url( 'css/jquery-ui-theme.min.css' ),
		array(),
		false            // let WP append its current version query‑arg
	);

	/* ───────────────────────────────
	 * 3.  Your plugin’s own CSS / JS
	 * ─────────────────────────────── */
/*	wp_enqueue_style(
		'livefot-styles',
		LIVEFOT_PLUGIN_URL . 'assets/css/styles.css',
		array( 'jquery-ui-theme' ),
		LIVEFOT_VERSION
	);

	wp_enqueue_script(
		'livefot-matches',
		LIVEFOT_PLUGIN_URL . 'assets/js/matches.js',
		array( 'jquery', 'jquery-ui-datepicker' ),
		LIVEFOT_VERSION,
		true
	);

	/* ───────────────────────────────
	 * 4.  Localised data for JS
	 * ─────────────────────────────── */
/*	$ms = fn ( $opt, $def ) => (int) get_option( $opt, $def ) * 1000;

	wp_localize_script(
		'livefot-matches',
		'livefotAjax',
		array(
			'ajaxurl'   => admin_url( 'admin-ajax.php' ),
			'nonce'     => wp_create_nonce( 'livefot_nonce' ),
			'intervals' => array(
				'matches'    => $ms( 'livefot_interval_matches',   30 ),
				'lineups'    => $ms( 'livefot_interval_lineups',   60 ),
				'events'     => $ms( 'livefot_interval_events',    60 ),
				'standings'  => $ms( 'livefot_interval_standings', 120 ),
				'statistics' => $ms( 'livefot_interval_stats',     120 ),
			),
			'icons_base_url' => plugins_url( 'assets/images/', __FILE__ ),
		)
	);
}
add_action( 'wp_enqueue_scripts', 'livefot_enqueue_scripts' );*/



function livefot_enqueue_scripts() {
    global $wp_version;

    /* 1. jQuery UI Datepicker (JS) */
    wp_enqueue_script( 'jquery-ui-datepicker' ); // depends on core jQuery

    /* 2. jQuery UI Theme (CSS) */
    wp_enqueue_style(
        'jquery-ui-theme',
        includes_url( 'css/jquery-ui-theme.min.css' ),
        array(),
        $wp_version  // explicitly pass the WP version instead of `false`
    );

    /* 3. Your plugin’s own CSS / JS */
    wp_enqueue_style(
        'livefot-styles',
        LIVEFOT_PLUGIN_URL . 'assets/css/styles.css',
        array( 'jquery-ui-theme' ),
        LIVEFOT_VERSION
    );
    wp_enqueue_script(
        'livefot-matches',
        LIVEFOT_PLUGIN_URL . 'assets/js/matches.js',
        array( 'jquery', 'jquery-ui-datepicker' ),
        LIVEFOT_VERSION,
        true
    );

    /* 4. Localised data for JS */
    $ms = fn( $opt, $def ) => (int) get_option( $opt, $def ) * 1000;
    wp_localize_script(
        'livefot-matches',
        'livefotAjax',
        array(
            'ajaxurl'       => admin_url( 'admin-ajax.php' ),
            'nonce'         => wp_create_nonce( 'livefot_nonce' ),
            'intervals'     => array(
                'matches'    => $ms( 'livefot_interval_matches',   30 ),
                'lineups'    => $ms( 'livefot_interval_lineups',   60 ),
                'events'     => $ms( 'livefot_interval_events',    60 ),
                'standings'  => $ms( 'livefot_interval_standings', 120 ),
                'statistics' => $ms( 'livefot_interval_stats',     120 ),
            ),
            'icons_base_url' => plugins_url( 'assets/images/', __FILE__ ),
        )
    );
}
add_action( 'wp_enqueue_scripts', 'livefot_enqueue_scripts' );



// Shortcode callback
function livefot_matches_shortcode($atts) {
	
    ob_start();
    include LIVEFOT_PLUGIN_PATH . 'templates/matches-container.php';
    return ob_get_clean();
}

// Add admin menu and settings pages
add_action('admin_menu', 'livefot_add_admin_menu');

function livefot_add_admin_menu() {
    // Add main plugin menu (LiveFot)
    add_menu_page(
        'LiveFot',                // Page title
        'LiveFot',                // Menu title
        'manage_options',         // Capability
        'livefot-main',           // Menu slug
        'livefot_welcome_page',   // Callback function for the main/welcome page
        'dashicons-admin-generic',// Icon
        20                        // Position
    );

    // Add submenu for Settings (API Key) under LiveFot
    add_submenu_page(
        'livefot-main',           // Parent slug
        'Settings',               // Page title
        'Settings',               // Menu title
        'manage_options',         // Capability
        'livefot-settings',       // Menu slug
        'livefot_settings_page'   // Callback function for the settings page
    );

    // Add submenus for Data Tables
    add_submenu_page(
        'livefot-main',
        'Lineup Data',
        'Lineup Data',
        'manage_options',
        'livefot-lineup-data',
        'livefot_display_lineup_data'
    );

    add_submenu_page(
        'livefot-main',
        'Leagues Data',
        'Leagues Data',
        'manage_options',
        'livefot-leagues-data',
        'livefot_display_leagues_data'
    );

    add_submenu_page(
        'livefot-main',
        'Countries Data',
        'Countries Data',
        'manage_options',
        'livefot-countries-data',
        'livefot_display_countries_data'
    );

    add_submenu_page(
        'livefot-main',
        'Fixtures Data',
        'Fixtures Data',
        'manage_options',
        'livefot-fixtures-data',
        'livefot_display_fixtures_data'
    );

    add_submenu_page(
        'livefot-main',
        'Referees Data',
        'Referees Data',
        'manage_options',
        'livefot-referees-data',
        'livefot_display_referees_data'
    );

    add_submenu_page(
        'livefot-main',
        'Venues Data',
        'Venues Data',
        'manage_options',
        'livefot-venues-data',
        'livefot_display_venues_data'
    );

    add_submenu_page(
        'livefot-main',
        'Scores Data',
        'Scores Data',
        'manage_options',
        'livefot-scores-data',
        'livefot_display_scores_data'
    );

    add_submenu_page(
        'livefot-main',
        'Time Data',
        'Time Data',
        'manage_options',
        'livefot-time-data',
        'livefot_display_time_data'
    );

    add_submenu_page(
        'livefot-main',
        'Aggregate Data',
        'Aggregate Data',
        'manage_options',
        'livefot-aggregate-data',
        'livefot_display_aggregate_data'
    );

    add_submenu_page(
        'livefot-main',
        'Players Data',
        'Players Data',
        'manage_options',
        'livefot-players-data',
        'livefot_display_players_data'
    );

    add_submenu_page(
        'livefot-main',
        'Events Data',
        'Events Data',
        'manage_options',
        'livefot-events-data',
        'livefot_display_events_data'
    );

    add_submenu_page(
        'livefot-main',
        'Teams Data',
        'Teams Data',
        'manage_options',
        'livefot-teams-data',
        'livefot_display_teams_data'
    );

    add_submenu_page(
        'livefot-main',
        'Formations Data',
        'Formations Data',
        'manage_options',
        'livefot-formations-data',
        'livefot_display_formations_data'
    );
	
	// New Standings Data Submenu
	add_submenu_page(
    	'livefot-main',
    	'Standings Data',
    	'Standings Data',
    	'manage_options',
    	'livefot-standings-data',
    	'livefot_display_standings_data'
	);


	
	
	
	// New Standings Data Submenu
	add_submenu_page(
    	'livefot-main',
    	'Stats Data',
    	'Stats Data',
    	'manage_options',
    	'livefot-team-stats-data',
    	'livefot_display_team_stats_data'
	);
	
	
    add_submenu_page(
        'livefot-main',
        'API Calls',
        'API Calls',
        'manage_options',
        'livefot-api-calls',
        'livefot_display_api_calls'
    );
}

// Welcome page callback (main page)

function livefot_welcome_page() {

    wp_enqueue_style(
        'livefot-styles',
        LIVEFOT_PLUGIN_URL . 'assets/css/admin/styles.css',
        array(),
        LIVEFOT_VERSION
    );

    ?>
    <div class="wrap livefot-welcome-page">
        <div class="header-title">
          <h1>Welcome to <span>LiveFot</span></h1>
          <p class="about-description">
                    Your ultimate solution for automatic live football scores, match lineups, standings, stats, and live events.
                </p>
        </div>
        <div class="livefot-welcome-header">
            <div class="livefot-welcome-text">
<!--                 <p class="about-description">
                    Your ultimate solution for automatic live football scores, match lineups, standings, stats, and live events.
                </p> -->
            </div>
        </div>

        <div class="livefot-features-grid">
            <div class="feature-section">
                <h2 class="live-section-heading">🚀 Key Features</h2>
                <div class="feature-grid">
                    <div class="feature-item">
                        <h3>Automatic Updates</h3>
                        <p>Daily updates covering -5 to +7 days at a user-specified time (configured in the Settings page).</p>
                    </div>
                    
                    <div class="feature-item">
                        <h3>Secondary Cron Job</h3>
                        <p>Frequent updates for matches within -1 to +1 day at an interval (in minutes) set by the user.</p>
                    </div>
                    
                    <div class="feature-item">
                        <h3>Manual Fetch Option</h3>
                        <p>Fetch match data for any specific date whenever needed.</p>
                    </div>
                    
                    <div class="feature-item">
                        <h3>Timezone Conversion</h3>
                        <p>Matches automatically adjust to each viewer’s local time—no manual setup required.</p>
                    </div>
					<div class="feature-item">
                        <h3>Responsive Design</h3>
                        <p>Works seamlessly on mobile and desktop with a modern, easy-to-use interface.</p>
                    </div>
					<div class="feature-item">
                        <h3>Cross-Platform Availability</h3>
                        <p>Monitor live updates online or via our mobile apps on Google Play Store and Apple App Store.</p>
                    </div>
					<div class="feature-item">
                        <h3>Real-Time Data from LiveFot API</h3>
                        <p>Enjoy fast and reliable data for live scores, lineups, standings, stats, and match events.</p>
                    </div>
                </div>
            </div>
        </div>
        <div class="livefot-features-grid livefot-getting-started-wrapper">
            <div class="livefot-getting-started">
                <h2>🎯 Getting Started</h2>
<!--                 <p>To begin displaying live football data on your website:</p> -->
                <ol>
                    <li>Log in or create an account</li>
                    <li>Purchase a plan</li>
                    <li>Receive your codes via email after the purchase</li>
					<li>Use these codes to add leagues to your plan</li>
					<h4>For more information or assistance, feel free to contact our support team. Thank you for choosing Live Football Center!</h4>
                </ol>
            </div>
        </div>
        <div class="livefot-features-grid">

            <div class="livefot-versions">
                <h2 class="live-section-heading">📦 Available Versions</h2>
                <div class="version-grid">
                    <div class="version-item">
                        <h3>Basic Version</h3>
                        <p>Fully functional with essential features for live scores and updates.</p>
                        <ul>
                            <li>Live Scores</li>
                            <li>Match Updates</li>
                            <li>Basic Statistics</li>
                            <li>Team Lineups</li>
                        </ul>
                    </div>
                    
                    <div class="version-item">
                        <h3>Advanced Version (Coming Soon)</h3>
                        <p>Enhanced features for a superior experience.</p>
                        <ul>
                            <li>Everything in Basic Version</li>
                            <li>Advanced Analytics</li>
                            <li>Custom Widgets</li>
                            <li>Priority Support</li>
                        </ul>
                    </div>
                </div>
            </div>

            <div class="livefot-requirements">
                <h2 class="live-section-heading">⚙️ Technical Requirements</h2>
                <div class="requirements-grid">
                    <div class="requirement-item">
                        <h3>Platform</h3>
                        <p>Optimized for WordPress</p>
                    </div>
                    
                    <div class="requirement-item">
                        <h3>Hosting</h3>
                        <p>Works on both dedicated and shared WordPress hosting</p>
                    </div>
                    
                    <div class="requirement-item">
                        <h3>PHP Version</h3>
                        <p>PHP 7.4 or higher recommended</p>
                    </div>
                </div>
            </div>

            <div class="livefot-support">
                <h2 class="live-section-heading">📞 Support & Contact</h2>
                <p>For inquiries regarding leagues, subscriptions, or technical support, please reach out to us at:</p>
                <ul>
                    <li>Email: <a href="mailto:info@livefootballcenter.com">info@livefootballcenter.com</a></li>
                    <li>To access our API services and view a live demo before subscribing, please visit: <a href="http://api.livefootballcenter.com/" target="_blank">api.livefootballcenter.com</a></li>
                </ul>
            </div>

            
        </div>
    </div>
    <?php
}


// Register settings for API key
/*add_action('admin_init', 'livefot_register_settings');
function livefot_register_settings() {
    register_setting('livefot_options', 'livefot_api_key'); // Register API key setting
    
    // Register the API URL (New)
    register_setting('livefot_options', 'livefot_api_url');
}*/

add_action( 'admin_init', 'livefot_register_settings' );
function livefot_register_settings() {
    register_setting(
        'livefot_options',
        'livefot_api_key',
        array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => '',
        )
    );

    register_setting(
        'livefot_options',
        'livefot_api_url',
        array(
            'type'              => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default'           => '',
        )
    );
}



// Register cron job enable/disable setting
add_action('admin_init', 'livefot_register_cron_job_setting');

function livefot_register_cron_job_setting() {
    register_setting('livefot_options', 'livefot_enable_cron_job', array(
        'type'              => 'boolean',
        'description'       => 'Enable or disable the daily fixtures fetch cron job.',
        'default'           => true,
        'sanitize_callback' => 'absint', // Ensures the value is an integer (0 or 1)
    ));
}

// Register settings for short-range cron job
add_action('admin_init', 'livefot_register_short_cron_job_settings');

function livefot_register_short_cron_job_settings() {
    register_setting('livefot_options', 'livefot_enable_short_cron_job', array(
        'type'              => 'boolean',
        'description'       => 'Enable or disable the short-range fixtures fetch cron job.',
        'default'           => true,
        'sanitize_callback' => 'absint',
    ));
}

// Register cron job time settings
add_action('admin_init', 'livefot_register_cron_time_settings');

function livefot_register_cron_time_settings() {
    register_setting('livefot_options', 'livefot_cron_hour', array(
        'type'              => 'integer',
        'description'       => 'Hour of the day to run the daily cron job (0-23).',
        'default'           => 8,
        'sanitize_callback' => 'absint',
    ));

    register_setting('livefot_options', 'livefot_cron_minute', array(
        'type'              => 'integer',
        'description'       => 'Minute of the hour to run the daily cron job (0-59).',
        'default'           => 0,
        'sanitize_callback' => 'absint',
    ));
}

// Register manual run date setting
add_action('admin_init', 'livefot_register_manual_run_date_setting');

function livefot_register_manual_run_date_setting() {
    register_setting('livefot_options', 'livefot_manual_date', array(
        'type'              => 'string',
        'description'       => 'Date for manual fixtures fetch.',
        'sanitize_callback' => 'sanitize_text_field',
    ));
}

// Register settings for intervals
/*add_action('admin_init', 'livefot_register_update_interval_settings');
function livefot_register_update_interval_settings() {
    register_setting('livefot_options', 'livefot_interval_matches', array('default' => 30));
    register_setting('livefot_options', 'livefot_interval_lineups', array('default' => 60));
    register_setting('livefot_options', 'livefot_interval_events', array('default' => 60));
    register_setting('livefot_options', 'livefot_interval_stats', array('default' => 120));
    register_setting('livefot_options', 'livefot_interval_standings', array('default' => 120));
	
    // Register short-range cron interval setting
    register_setting('livefot_options', 'livefot_cron_interval_short', array(
        'type'              => 'integer',
        'description'       => 'Interval in minutes for short-range fixtures fetch cron job.',
        'default'           => 15,
        'sanitize_callback' => 'absint',
    ));
}*/





/**
 * Register the front‑end polling intervals.
 * Each value is stored as a positive integer (seconds).
 */
add_action( 'admin_init', 'livefot_register_update_interval_settings' );
function livefot_register_update_interval_settings() {

    $int_option = function ( $name, $default ) {
        register_setting(
            'livefot_options',
            $name,
            array(
                'type'              => 'integer',
                'default'           => $default,
                'sanitize_callback' => 'absint',   // <‑‑ key fix
            )
        );
    };

    $int_option( 'livefot_interval_matches',   30  );
    $int_option( 'livefot_interval_lineups',   60  );
    $int_option( 'livefot_interval_events',    60  );
    $int_option( 'livefot_interval_stats',     120 );
    $int_option( 'livefot_interval_standings', 120 );

    /* short‑range cron interval (already had a callback, keep it) */
    register_setting(
        'livefot_options',
        'livefot_cron_interval_short',
        array(
            'type'              => 'integer',
            'description'       => 'Interval in minutes for short‑range fixtures fetch cron job.',
            'default'           => 15,
            'sanitize_callback' => 'absint',
        )
    );
}



// Handle cron job scheduling based on settings
add_action('update_option_livefot_enable_cron_job', 'livefot_handle_cron_job_setting', 10, 2);
add_action('update_option_livefot_cron_hour', 'livefot_handle_cron_job_time_setting', 10, 2);
add_action('update_option_livefot_cron_minute', 'livefot_handle_cron_job_time_setting', 10, 2);

// Handle short-range cron job scheduling based on settings
add_action('update_option_livefot_enable_short_cron_job', 'livefot_handle_short_cron_job_setting', 10, 2);
add_action('update_option_livefot_short_cron_start_hour', 'livefot_handle_short_cron_job_setting', 10, 2);
add_action('update_option_livefot_short_cron_start_minute', 'livefot_handle_short_cron_job_setting', 10, 2);
add_action('update_option_livefot_cron_interval_short', 'livefot_handle_short_cron_interval_update', 10, 2);

function livefot_handle_cron_job_setting($old_value, $new_value) {
    if ($new_value && !$old_value) {
        // Enable cron job
        if (!wp_next_scheduled('livefot_daily_fetch_fixtures')) {
            livefot_schedule_cron_job();
        }
    } elseif (!$new_value && $old_value) {
        // Disable cron job
        wp_clear_scheduled_hook('livefot_daily_fetch_fixtures');
    }
}

function livefot_handle_cron_job_time_setting($old_value, $new_value) {
    // Reschedule the daily cron job based on the new time
    if (get_option('livefot_enable_cron_job')) {
        wp_clear_scheduled_hook('livefot_daily_fetch_fixtures');
        livefot_schedule_cron_job();
    }
}

function livefot_handle_short_cron_job_setting($old_value, $new_value) {
    // Reschedule the short-range cron job based on the new settings
    if ($new_value) {
        // Enabled
        livefot_schedule_short_cron_job();
    } else {
        // Disabled
        wp_clear_scheduled_hook('livefot_short_fetch_fixtures');
    }
}

function livefot_handle_short_cron_interval_update($old_value, $new_value) {
    // Reschedule the short-range cron job based on the new interval
    wp_clear_scheduled_hook('livefot_short_fetch_fixtures');
    livefot_schedule_short_cron_job();
}

// Updated livefot_schedule_cron_job function for daily fixtures
/*function livefot_schedule_cron_job() {
    // Retrieve user-defined cron time settings
    $hour = intval(get_option('livefot_cron_hour', 8));
    $minute = intval(get_option('livefot_cron_minute', 0));

    // Get WordPress local time
    $current_time_local = current_time('timestamp');

    // Calculate the scheduled time based on WordPress's local time
    $scheduled_time_local = mktime($hour, $minute, 0, date('n', $current_time_local), date('j', $current_time_local), date('Y', $current_time_local));

    // Convert scheduled_time_local to UTC
    $scheduled_time_utc = get_gmt_from_date(date('Y-m-d H:i:s', $scheduled_time_local), 'U');

    // Get current UTC time
    $current_time_utc = current_time('timestamp', true); // 'true' gets GMT timestamp

    // If the scheduled time has already passed today, schedule for tomorrow
    if ($scheduled_time_utc <= $current_time_utc) {
        $scheduled_time_local += DAY_IN_SECONDS; // Add 24 hours
        $scheduled_time_utc = get_gmt_from_date(date('Y-m-d H:i:s', $scheduled_time_local), 'U');
    }

    // Schedule the cron event if not already scheduled
    if (!wp_next_scheduled('livefot_daily_fetch_fixtures')) {
        wp_schedule_event($scheduled_time_utc, 'daily', 'livefot_daily_fetch_fixtures');
    }

    // Log the scheduled time for debugging
    error_log("Scheduled daily cron job for WordPress local time: " . date('Y-m-d H:i:s', $scheduled_time_local) . " (UTC: " . date('Y-m-d H:i:s', $scheduled_time_utc) . ")");
}*/



function livefot_schedule_cron_job() {
    // 1) Get user‐configured hour & minute (assumed in UTC)
    $hour   = intval( get_option( 'livefot_cron_hour',   8  ) );
    $minute = intval( get_option( 'livefot_cron_minute', 0  ) );

    // 2) Current GMT timestamp
    $now_gmt = current_time( 'timestamp', true );

    // 3) Break out year/month/day in GMT
    $year  = gmdate( 'Y', $now_gmt );
    $month = gmdate( 'n', $now_gmt );
    $day   = gmdate( 'j', $now_gmt );

    // 4) Build the scheduled‐for‐today GMT timestamp
    $scheduled_gmt = gmmktime( $hour, $minute, 0, $month, $day, $year );

    // 5) If that time has already passed, bump by 24h
    if ( $scheduled_gmt <= $now_gmt ) {
        $scheduled_gmt += DAY_IN_SECONDS;
    }

    // 6) Schedule the event if not already set
    if ( ! wp_next_scheduled( 'livefot_daily_fetch_fixtures' ) ) {
        wp_schedule_event( $scheduled_gmt, 'daily', 'livefot_daily_fetch_fixtures' );
    }

    // 7) (Optional) debug log in GMT
   /* error_log( sprintf(
        'Scheduled daily cron (UTC) at %s',
        gmdate( 'Y-m-d H:i:s', $scheduled_gmt )
    ) );*/
}






function livefot_schedule_short_cron_job() {
    $enable_short_cron = get_option('livefot_enable_short_cron_job', 1);
    if (!$enable_short_cron) {
      //  error_log("Short-range cron job is disabled.");
        return; // Short-range cron job is disabled
    }

    // Clear any existing schedules to avoid duplicates
    wp_clear_scheduled_hook('livefot_short_fetch_fixtures');

    // Get interval in minutes (default to 15 if not set)
    $interval_minutes = intval(get_option('livefot_cron_interval_short', 15));
    if ($interval_minutes < 15) { // Enforce minimum 15 minutes
        $interval_minutes = 15;
        update_option('livefot_cron_interval_short', $interval_minutes);
      //  error_log("Short-range cron interval too low. Set to minimum 15 minutes.");
    }

    // Schedule the recurring cron event if it's not already scheduled
    if (!wp_next_scheduled('livefot_short_fetch_fixtures')) {
        wp_schedule_event(time(), 'livefot_short_interval', 'livefot_short_fetch_fixtures');
     //   error_log("Scheduled short-range cron job to run every $interval_minutes minutes.");
    }
}



// Settings page callback
function livefot_settings_page() {
    // Check if manual fetch form was submitted
    if (isset($_POST['fetch_fixtures']) && check_admin_referer('livefot_manual_fetch', '_wpnonce_livefot_manual_fetch')) {
        // Sanitize the input date
      $manual_date = sanitize_text_field($_POST['livefot_manual_date']);
		
        
        if (empty($manual_date)) {
            add_settings_error(
                'livefot_messages',
                'livefot_manual_date_error',
                __('Please select a valid date.', 'livefot-matches'),
                'error'
            );
        } else {
            // Create a new instance of LiveFot_API
            $api = new LiveFot_API();
            $result = $api->livefot_fetch_and_store_fixtures_manual($manual_date);

            if ($result === true) {
				
                add_settings_error(
                    'livefot_messages',
                    'livefot_fetch_success',
					/* translators: %s is the date (YYYY‑MM‑DD) for which fixtures were fetched. */
                    sprintf(__('Fixtures successfully fetched for %s.', 'livefot-matches'), esc_html($manual_date)),
                    'success'
                );
            } else {
                add_settings_error(
                    'livefot_messages',
                    'livefot_fetch_error',
					/* translators: %s is the error message returned when fetching fixtures. */
                    sprintf(__('Error fetching fixtures: %s', 'livefot-matches'), esc_html($result)),
                    'error'
                );
            }
        }
    }

    // Display any error/success messages
    settings_errors('livefot_messages');
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('LiveFot Settings', 'livefot-matches'); ?></h1>
        
        <!-- Main Settings Form -->
        <form method="post" action="options.php">
            <?php
            // Output security fields for the registered setting "livefot_options"
            settings_fields('livefot_options');
            // Output setting sections and their fields
            do_settings_sections('livefot_options');
            ?>

            <!-- API Settings Section -->
            <h2><?php esc_html_e('API Settings', 'livefot-matches'); ?></h2>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="livefot_api_key"><?php esc_html_e('API Key', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="text" id="livefot_api_key" name="livefot_api_key" value="<?php echo esc_attr(get_option('livefot_api_key')); ?>" class="regular-text" required>
                        <p class="description"><?php esc_html_e('Enter your API key to access LiveFot services.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
                 <tr>
                    <th scope="row">
                        <label for="livefot_api_url"><?php esc_html_e('API URL', 'livefot-matches'); ?></label>
                    </th>
                    <td>
                        <input type="text" id="livefot_api_url" name="livefot_api_url" 
                               value="<?php echo esc_attr(get_option('livefot_api_url')); ?>" 
                               class="regular-text" required>
                        <p class="description"><?php esc_html_e('Enter your custom API base URL if different from default.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
            </table>

            <!-- Cron Job Settings Section -->
            <h2><?php esc_html_e('Cron Job Settings', 'livefot-matches'); ?></h2>
            <table class="form-table">
                <!-- Daily Cron Job Enable/Disable -->
                <tr>
                    <th scope="row"><label for="livefot_enable_cron_job"><?php esc_html_e('Enable Daily Fixtures Cron Job', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="checkbox" id="livefot_enable_cron_job" name="livefot_enable_cron_job" value="1" <?php checked(1, get_option('livefot_enable_cron_job', 1)); ?>>
                        <p class="description"><?php esc_html_e('Enable the daily cron job to fetch fixtures and update the database.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
                <!-- Daily Cron Job Time -->
                <tr>
                    <th scope="row"><?php esc_html_e('Daily Cron Job Time', 'livefot-matches'); ?></th>
                    <td>
                        <label for="livefot_cron_hour"><?php esc_html_e('Hour:', 'livefot-matches'); ?></label>
                        <select name="livefot_cron_hour" id="livefot_cron_hour">
                           <?php
							$selected_hour = get_option( 'livefot_cron_hour', 8 );

							for ( $h = 0; $h < 24; $h++ ) {
								printf(
									'<option value="%s" %s>%s</option>',
									esc_attr( $h ),
									selected( $h, $selected_hour, false ),
									esc_html( sprintf( '%02d:00', $h ) )
								);
							}
							?>
                        </select>
                        &nbsp;&nbsp;
                        <label for="livefot_cron_minute"><?php esc_html_e('Minute:', 'livefot-matches'); ?></label>
                        <select name="livefot_cron_minute" id="livefot_cron_minute">
                          <?php
							$selected_minute = get_option( 'livefot_cron_minute', 0 );

							for ( $m = 0; $m < 60; $m += 5 ) {
								printf(
									'<option value="%s" %s>%s</option>',
									esc_attr( $m ),
									selected( $m, $selected_minute, false ),
									esc_html( sprintf( '%02d', $m ) )
								);
							}
							?>

                        </select>
                        <p class="description"><?php esc_html_e('Select the time of day when the daily fixtures fetch should run.', 'livefot-matches'); ?></p>
                    </td>
                </tr>

                <!-- Short-Range Cron Job Enable/Disable -->
                <tr>
                    <th scope="row"><label for="livefot_enable_short_cron_job"><?php esc_html_e('Enable Short-Range Fixtures Cron Job', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="checkbox" id="livefot_enable_short_cron_job" name="livefot_enable_short_cron_job" value="1" <?php checked(1, get_option('livefot_enable_short_cron_job', 1)); ?>>
                        <p class="description"><?php esc_html_e('Enable the short-range cron job to fetch fixtures periodically.', 'livefot-matches'); ?></p>
                    </td>
                </tr>

                <!-- Short-Range Cron Job Interval -->
             
				<tr>
    <th scope="row"><label for="livefot_cron_interval_short"><?php esc_html_e('Short-Range Fixtures Cron Interval (Minutes)', 'livefot-matches'); ?></label></th>
    <td>
        <input type="number" id="livefot_cron_interval_short" name="livefot_cron_interval_short" value="<?php echo esc_attr(get_option('livefot_cron_interval_short', 60)); ?>" class="small-text" min="15" step="1" required>
        <p class="description"><?php esc_html_e('Set how often (in minutes) the short-range fixtures fetch should run.', 'livefot-matches'); ?></p>
    </td>
</tr>


            </table>

            <!-- Update Intervals Section -->
            <h2><?php esc_html_e('Update Intervals', 'livefot-matches'); ?></h2>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="livefot_interval_matches"><?php esc_html_e('Matches Update Interval (seconds)', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="number" id="livefot_interval_matches" name="livefot_interval_matches" value="<?php echo esc_attr(get_option('livefot_interval_matches', 30)); ?>" class="small-text" min="30" step="5" required>
                        <p class="description"><?php esc_html_e('Set the interval at which live matches are updated. Minimum 30 seconds.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="livefot_interval_lineups"><?php esc_html_e('Lineups Update Interval (seconds)', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="number" id="livefot_interval_lineups" name="livefot_interval_lineups" value="<?php echo esc_attr(get_option('livefot_interval_lineups', 60)); ?>" class="small-text" min="60" step="5" required>
                        <p class="description"><?php esc_html_e('Set the interval at which lineups are updated. Minimum 60 seconds.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="livefot_interval_events"><?php esc_html_e('Events Update Interval (seconds)', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="number" id="livefot_interval_events" name="livefot_interval_events" value="<?php echo esc_attr(get_option('livefot_interval_events', 60)); ?>" class="small-text" min="30" step="5" required>
                        <p class="description"><?php esc_html_e('Set the interval at which events are updated. Minimum 30 seconds.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
                <tr>
				  <th scope="row"><label for="livefot_interval_standings"><?php esc_html_e('Standings Update Interval (seconds)', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="number" id="livefot_interval_standings" name="livefot_interval_standings" value="<?php echo esc_attr(get_option('livefot_interval_standings', 120)); ?>" class="small-text" min="60" step="5" required>
                        <p class="description"><?php esc_html_e('Set the interval at which standings are updated. Minimum 60 seconds.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="livefot_interval_stats"><?php esc_html_e('Statistics Update Interval (seconds)', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="number" id="livefot_interval_stats" name="livefot_interval_stats" value="<?php echo esc_attr(get_option('livefot_interval_stats', 120)); ?>" class="small-text" min="60" step="5" required>
                        <p class="description"><?php esc_html_e('Set the interval at which statistics are updated. Minimum 60 seconds.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
            </table>

            <!-- Save Changes Button -->
            <?php submit_button(); ?>
        </form>

        <!-- Manual Fetch Form -->
        <h2><?php esc_html_e('Manual Fixtures Fetch', 'livefot-matches'); ?></h2>
        <form method="post" action="">
            <?php
            // Add nonce for security
            wp_nonce_field('livefot_manual_fetch', '_wpnonce_livefot_manual_fetch');
            ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="livefot_manual_date"><?php esc_html_e('Select Date to Fetch Fixtures', 'livefot-matches'); ?></label></th>
                    <td>
                        <input type="date" id="livefot_manual_date" name="livefot_manual_date" value="<?php echo esc_attr(get_option('livefot_manual_date')); ?>" required>
                        <?php
                        // Render the submit button without wrapping it in a paragraph or div
                        submit_button(__('Fetch Fixtures for Selected Date', 'livefot-matches'), 'secondary', 'fetch_fixtures', false);
                        ?>
                        <p class="description"><?php esc_html_e('Choose a date and click the button to manually fetch fixtures for that specific date.', 'livefot-matches'); ?></p>
                    </td>
                </tr>
            </table>
        </form>

        <!-- Cron Job Status Section -->
        <h2><?php esc_html_e('Cron Job Status', 'livefot-matches'); ?></h2>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Last Run Start Time (Daily)', 'livefot-matches'); ?></th>
                <td>
                    <?php 
                        $last_run_start = get_option('livefot_cron_last_run_start', 'Never');
                        if ($last_run_start !== 'Never') {
                            // Convert UTC to WordPress local time
                            $last_run_start_local = get_date_from_gmt($last_run_start, 'Y-m-d H:i:s');
                        } else {
                            $last_run_start_local = 'Never';
                        }
                        echo esc_html($last_run_start_local);
                    ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Last Run End Time (Daily)', 'livefot-matches'); ?></th>
                <td>
                    <?php 
                        $last_run_end = get_option('livefot_cron_last_run_end', 'Never');
                        if ($last_run_end !== 'Never') {
                            // Convert UTC to WordPress local time
                            $last_run_end_local = get_date_from_gmt($last_run_end, 'Y-m-d H:i:s');
                        } else {
                            $last_run_end_local = 'Never';
                        }
                        echo esc_html($last_run_end_local);
                    ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Last Run Status (Daily)', 'livefot-matches'); ?></th>
                <td>
                    <?php 
                        $last_run_status = get_option('livefot_cron_last_run_status', 'Never');
                        echo esc_html($last_run_status);
                    ?>
                </td>
            </tr>
        </table>

        <!-- Short-Range Cron Job Status Section -->
        <h2><?php esc_html_e('Short-Range Cron Job Status', 'livefot-matches'); ?></h2>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Last Run Start Time', 'livefot-matches'); ?></th>
                <td>
                    <?php 
                        $last_run_start_short = get_option('livefot_cron_last_run_start_short', 'Never');
                        if ($last_run_start_short !== 'Never') {
                            // Convert UTC to WordPress local time
                            $last_run_start_short_local = get_date_from_gmt($last_run_start_short, 'Y-m-d H:i:s');
                        } else {
                            $last_run_start_short_local = 'Never';
                        }
                        echo esc_html($last_run_start_short_local);
                    ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Last Run End Time', 'livefot-matches'); ?></th>
                <td>
                    <?php 
                        $last_run_end_short = get_option('livefot_cron_last_run_end_short', 'Never');
                        if ($last_run_end_short !== 'Never') {
                            // Convert UTC to WordPress local time
                            $last_run_end_short_local = get_date_from_gmt($last_run_end_short, 'Y-m-d H:i:s');
                        } else {
                            $last_run_end_short_local = 'Never';
                        }
                        echo esc_html($last_run_end_short_local);
                    ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Last Run Status', 'livefot-matches'); ?></th>
                <td>
                    <?php 
                        $last_run_status_short = get_option('livefot_cron_last_run_status_short', 'Never');
                        echo esc_html($last_run_status_short);
                    ?>
                </td>
            </tr>
        </table>

        <!-- Current Time Section -->
        <h2><?php esc_html_e('Current Time', 'livefot-matches'); ?></h2>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Current Time (Local)', 'livefot-matches'); ?></th>
                <td>
                    <?php 
                        // Get current WordPress local time
                        $current_time = current_time('Y-m-d H:i:s');
                        echo esc_html($current_time);
                    ?>
                </td>
            </tr>
        </table>

        <!-- Shortcode Usage Section -->
        <div class="card" style="background-color: #f9f9f9; padding: 20px; margin-top: 30px; border-radius: 5px;">
            <h2><?php esc_html_e('Shortcode Usage', 'livefot-matches'); ?></h2>
            <p><?php esc_html_e('Use this shortcode to display matches on any page or post:', 'livefot-matches'); ?></p>
            <code>[livefot_matches]</code>
        </div>
    </div>
    <?php
}

// Helper function to display pagination
function livefot_display_pagination($current_page, $total_pages, $base_url) {
    if ($total_pages <= 1) {
        return;
    }

    echo '<div class="livefot-pagination" style="margin-top: 20px; text-align: center;">';

    // Previous Page
    if ($current_page > 1) {
        $prev_page = $current_page - 1;
        echo '<a href="' . esc_url(add_query_arg('paged', $prev_page, $base_url)) . '" class="livefot-paginate-button" style="margin-right: 10px; text-decoration: none; color: #0073aa;">';
        echo '<span class="dashicons dashicons-arrow-left"></span> Previous';
        echo '</a>';
    }

    // Next Page
    if ($current_page < $total_pages) {
        $next_page = $current_page + 1;
        echo '<a href="' . esc_url(add_query_arg('paged', $next_page, $base_url)) . '" class="livefot-paginate-button" style="text-decoration: none; color: #0073aa;">';
        echo 'Next <span class="dashicons dashicons-arrow-right"></span>';
        echo '</a>';
    }

    echo '</div>';
}

// Fixtures Data page callback with pagination
function livefot_display_fixtures_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Verify nonce when the search form is submitted */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_fixtures_search_action', 'livefot_fixtures_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_fixtures';

    /* 4) Pagination settings */
    $items_per_page = 20;
    $current_page   = isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] )
                      ? max( 1, intval( $_GET['paged'] ) )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize search inputs */
    $search_fixture_id = isset( $_GET['search_fixture_id'] )
                        ? trim( sanitize_text_field( wp_unslash( $_GET['search_fixture_id'] ) ) )
                        : '';
    $search_team       = isset( $_GET['search_team'] )
                        ? trim( sanitize_text_field( wp_unslash( $_GET['search_team'] ) ) )
                        : '';
    $search_venue_id   = isset( $_GET['search_venue_id'] )
                        ? trim( sanitize_text_field( wp_unslash( $_GET['search_venue_id'] ) ) )
                        : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( $search_fixture_id !== '' && is_numeric( $search_fixture_id ) ) {
        $where_parts[] = 'fixture_id = %d';
        $params[]      = intval( $search_fixture_id );
    }

    if ( $search_team !== '' ) {
        $where_parts[] = '(localteam_name LIKE %s OR visitorteam_name LIKE %s)';
        $like_pattern  = '%' . $wpdb->esc_like( $search_team ) . '%';
        $params[]      = $like_pattern;
        $params[]      = $like_pattern;
    }

    if ( $search_venue_id !== '' && is_numeric( $search_venue_id ) ) {
        $where_parts[] = 'venue_id = %d';
        $params[]      = intval( $search_venue_id );
    }

    $where = implode( ' AND ', $where_parts );

    /* 7) Count total items (with caching for 1 minute) */
    $cache_group      = 'livefot_fixtures_data';
    $cache_key_total  = 'livefot_fixtures_total_' . md5( $where . serialize( $params ) );
    $total_items      = wp_cache_get( $cache_key_total, $cache_group );

    if ( false === $total_items ) {
        $total_items = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table_name} WHERE {$where}",
                ...$params
            )
        );
        wp_cache_set( $cache_key_total, $total_items, $cache_group, MINUTE_IN_SECONDS );
    }

    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 8) Fetch paginated results (with caching for 1 minute) */
    $cache_key_results = 'livefot_fixtures_results_'
                       . md5( $where . serialize( $params ) . "_{$items_per_page}_{$offset}" );
    $results = wp_cache_get( $cache_key_results, $cache_group );

    if ( false === $results ) {
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table_name}
                 WHERE {$where}
                 ORDER BY id DESC
                 LIMIT %d OFFSET %d",
                ...array_merge( $params, [ $items_per_page, $offset ] )
            ),
            ARRAY_A
        );
        wp_cache_set( $cache_key_results, $results, $cache_group, MINUTE_IN_SECONDS );
    }

    /* 9) Build base URL for pagination & clear link */
    $base_url   = admin_url( 'admin.php?page=livefot-fixtures-data' );
    $query_args = [];
    if ( $search_fixture_id !== '' ) {
        $query_args['search_fixture_id'] = $search_fixture_id;
    }
    if ( $search_team !== '' ) {
        $query_args['search_team'] = $search_team;
    }
    if ( $search_venue_id !== '' ) {
        $query_args['search_venue_id'] = $search_venue_id;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 10) Output */
    echo '<div class="wrap"><h1>' . esc_html__( 'LiveFot Fixtures Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="<?php echo esc_url( admin_url( 'admin.php' ) ); ?>">
        <?php wp_nonce_field( 'livefot_fixtures_search_action', 'livefot_fixtures_search_nonce', true, true ); ?>
        <input type="hidden" name="page" value="livefot-fixtures-data" />
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="search_fixture_id"><?php esc_html_e( 'Fixture ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input type="number" id="search_fixture_id" name="search_fixture_id" value="<?php echo esc_attr( $search_fixture_id ); ?>" />
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="search_team"><?php esc_html_e( 'Team Name', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input type="text" id="search_team" name="search_team" value="<?php echo esc_attr( $search_team ); ?>" />
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="search_venue_id"><?php esc_html_e( 'Venue ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input type="number" id="search_venue_id" name="search_venue_id" value="<?php echo esc_attr( $search_venue_id ); ?>" />
                </td>
            </tr>
        </table>
        <?php
            submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false );
        ?>
        <a href="<?php echo esc_url( $base_url ); ?>" class="button">
            <?php esc_html_e( 'Clear Search', 'livefot-matches' ); ?>
        </a>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>' .
             '<th>' . esc_html__( 'ID',              'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Fixture ID',      'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'League ID',       'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Local Team ID',   'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Local Team Name', 'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Visitor Team ID', 'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Visitor Team Name','livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Referee ID',      'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Venue ID',        'livefot-matches' ) . '</th>' .
             '<th>' . esc_html__( 'Last Updated',    'livefot-matches' ) . '</th>' .
             '</tr></thead><tbody>';
        foreach ( $results as $row ) {
            echo '<tr>' .
                 '<td>' . esc_html( $row['id'] )                . '</td>' .
                 '<td>' . esc_html( $row['fixture_id'] )        . '</td>' .
                 '<td>' . esc_html( $row['league_id'] )         . '</td>' .
                 '<td>' . esc_html( $row['localteam_id'] )      . '</td>' .
                 '<td>' . esc_html( $row['localteam_name'] )    . '</td>' .
                 '<td>' . esc_html( $row['visitorteam_id'] )    . '</td>' .
                 '<td>' . esc_html( $row['visitorteam_name'] )  . '</td>' .
                 '<td>' . ( ! empty( $row['referee_id'] ) ? esc_html( $row['referee_id'] ) : esc_html__( 'N/A', 'livefot-matches' ) ) . '</td>' .
                 '<td>' . ( ! empty( $row['venue_id'] )   ? esc_html( $row['venue_id'] )   : esc_html__( 'N/A', 'livefot-matches' ) ) . '</td>' .
                 '<td>' . ( ! empty( $row['last_updated'] ) ? esc_html( $row['last_updated'] ) : esc_html__( 'N/A', 'livefot-matches' ) ) . '</td>' .
                 '</tr>';
        }
        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_pages <= 1 && $total_items > 0 ) {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d result(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}




function livefot_display_leagues_data() {
	global $wpdb;

	/* Capability check */
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
	}

	/* Nonce check on search submission */
	if ( isset( $_GET['search_submit'] ) ) {
		check_admin_referer( 'livefot_leagues_search_action', 'livefot_leagues_search_nonce' );
	}

	/* Table name (prefix safely escaped when used in prepare) */
	$table_name = $wpdb->prefix . 'livefot_leagues';

	/* Pagination */
	$items_per_page = 20;
	$current_page   = isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] )
	                  ? max( 1, intval( $_GET['paged'] ) )
	                  : 1;
	$offset = ( $current_page - 1 ) * $items_per_page;

	/* Sanitise search inputs */
	$get                = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$search_league_id   = isset( $get['search_league_id'] )   ? trim( sanitize_text_field( $get['search_league_id'] ) )   : '';
	$search_league_name = isset( $get['search_league_name'] ) ? trim( sanitize_text_field( $get['search_league_name'] ) ) : '';

	/* Build WHERE clauses and parameters */
	$where_parts = [ '1=1' ];
	$params      = [];

	if ( '' !== $search_league_id && is_numeric( $search_league_id ) ) {
		$where_parts[] = 'league_id = %d';
		$params[]      = intval( $search_league_id );
	}

	if ( '' !== $search_league_name ) {
		$where_parts[] = 'name LIKE %s';
		$params[]      = '%' . $wpdb->esc_like( $search_league_name ) . '%';
	}

	$where = implode( ' AND ', $where_parts );

	/* 1) Count total items */
	$total_items = (int) $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(*) FROM {$table_name} WHERE {$where}",
			...$params
		)
	);
	$total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

	/* 2) Fetch paginated rows */
	$results = $wpdb->get_results(
		$wpdb->prepare(
			"SELECT * FROM {$table_name}
			 WHERE {$where}
			 ORDER BY last_updated DESC
			 LIMIT %d OFFSET %d",
			...array_merge( $params, [ $items_per_page, $offset ] )
		),
		ARRAY_A
	);

	/* Build base URL for pagination & clear link */
	$base_url = admin_url( 'admin.php?page=livefot-leagues-data' );
	$args     = [];
	if ( '' !== $search_league_id )   { $args['search_league_id']   = $search_league_id; }
	if ( '' !== $search_league_name ) { $args['search_league_name'] = $search_league_name; }
	if ( $args ) {
		$base_url = add_query_arg( $args, $base_url );
	}

	/* Output */
	echo '<div class="wrap">';
	echo '<h1>' . esc_html__( 'LiveFot Leagues Data', 'livefot-matches' ) . '</h1>';

	// Search form with nonce
	?>
	<form method="get" action="">
		<?php wp_nonce_field( 'livefot_leagues_search_action', 'livefot_leagues_search_nonce' ); ?>
		<input type="hidden" name="page" value="livefot-leagues-data" />
		<table class="form-table">
			<tr>
				<th><label for="search_league_id"><?php esc_html_e( 'League ID', 'livefot-matches' ); ?></label></th>
				<td><input type="text" id="search_league_id" name="search_league_id" value="<?php echo esc_attr( $search_league_id ); ?>" /></td>
			</tr>
			<tr>
				<th><label for="search_league_name"><?php esc_html_e( 'League Name', 'livefot-matches' ); ?></label></th>
				<td><input type="text" id="search_league_name" name="search_league_name" value="<?php echo esc_attr( $search_league_name ); ?>" /></td>
			</tr>
		</table>
		<?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
		<a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-leagues-data' ) ); ?>" class="button">
			<?php esc_html_e( 'Clear', 'livefot-matches' ); ?>
		</a>
	</form>
	<hr/>
	<?php

	if ( $results ) {
		echo '<table class="widefat fixed striped"><thead><tr>'
		   . '<th>' . esc_html__( 'ID',           'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'League ID',    'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'Name',         'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'Is Cup',       'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'Logo',         'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'Season ID',    'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'Stage',        'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'Country ID',   'livefot-matches' ) . '</th>'
		   . '<th>' . esc_html__( 'Last Updated', 'livefot-matches' ) . '</th>'
		   . '</tr></thead><tbody>';

		foreach ( $results as $row ) {
			echo '<tr>'
			   . '<td>' . esc_html( $row['id'] )                 . '</td>'
			   . '<td>' . esc_html( $row['league_id'] )          . '</td>'
			   . '<td>' . esc_html( $row['name'] )               . '</td>'
			   . '<td>' . esc_html( $row['is_cup'] )             . '</td>'
			   . '<td>' . esc_html( $row['logo_path'] )          . '</td>'
			   . '<td>' . esc_html( $row['current_season_id'] )  . '</td>'
			   . '<td>' . esc_html( $row['stage_name'] )         . '</td>'
			   . '<td>' . esc_html( $row['country_id'] )         . '</td>'
			   . '<td>' . esc_html( $row['last_updated'] )       . '</td>'
			   . '</tr>';
		}

		echo '</tbody></table>';

		/* Pagination */
		if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
			livefot_display_pagination( $current_page, $total_pages, $base_url );
		} elseif ( $total_pages <= 1 && $total_items > 0 ) {
			printf(
				'<p>%s</p>',
				sprintf(
					/* translators: %d = number of leagues. */
					esc_html__( 'Displaying %d league(s).', 'livefot-matches' ),
					esc_html( $total_items )
				)
			);
		}
	} else {
		echo '<p>' . esc_html__( 'No data found.', 'livefot-matches' ) . '</p>';
	}

	echo '</div>';
}


// Countries Data page callback with pagination
/**
 * Admin screen: “LiveFot Countries Data” with pagination.
 *
 * All SQL is prepared inline to satisfy PHPCS.
 *
 * @since 1.0.0
 */
function livefot_display_countries_data() {
    global $wpdb;

    /* Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_countries_search_action', 'livefot_countries_search_nonce' );
    }

    /* Table name */
    $table_name = $wpdb->prefix . 'livefot_countries';

    /* Pagination */
    $items_per_page = 20;
    $current_page   = isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] )
                      ? max( 1, intval( $_GET['paged'] ) )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* Sanitise search inputs */
    $get                 = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_country_id   = isset( $get['search_country_id'] )   ? trim( sanitize_text_field( $get['search_country_id'] ) )   : '';
    $search_country_name = isset( $get['search_country_name'] ) ? trim( sanitize_text_field( $get['search_country_name'] ) ) : '';

    /* Build WHERE clauses and parameters */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_country_id && is_numeric( $search_country_id ) ) {
        $where_parts[] = 'country_id = %d';
        $params[]      = intval( $search_country_id );
    }

    if ( '' !== $search_country_name ) {
        $where_parts[] = 'name LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_country_name ) . '%';
    }

    $where = implode( ' AND ', $where_parts );

    /* 1) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where}",
            ...$params
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 2) Fetch paginated rows */
    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where}
             ORDER BY last_updated DESC
             LIMIT %d OFFSET %d",
            ...array_merge( $params, [ $items_per_page, $offset ] )
        ),
        ARRAY_A
    );

    /* Build base URL for pagination & clear link */
    $base_url = admin_url( 'admin.php?page=livefot-countries-data' );
    $args     = [];
    if ( '' !== $search_country_id )   { $args['search_country_id']   = $search_country_id; }
    if ( '' !== $search_country_name ) { $args['search_country_name'] = $search_country_name; }
    if ( $args ) {
        $base_url = add_query_arg( $args, $base_url );
    }

    /* Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Countries Data', 'livefot-matches' ) . '</h1>'; ?>

    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_countries_search_action', 'livefot_countries_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-countries-data" />
        <table class="form-table">
            <tr>
                <th><label for="search_country_id"><?php esc_html_e( 'Country ID',   'livefot-matches' ); ?></label></th>
                <td><input type="text" id="search_country_id" name="search_country_id" value="<?php echo esc_attr( $search_country_id ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_country_name"><?php esc_html_e( 'Country Name', 'livefot-matches' ); ?></label></th>
                <td><input type="text" id="search_country_name" name="search_country_name" value="<?php echo esc_attr( $search_country_name ); ?>" /></td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-countries-data' ) ); ?>" class="button">
            <?php esc_html_e( 'Clear', 'livefot-matches' ); ?>
        </a>
    </form>
    <hr/>

    <?php if ( $results ) : ?>
        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th><?php esc_html_e( 'ID',            'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Country ID',    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Name',          'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Image Path',    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Last Updated',  'livefot-matches' ); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ( $results as $row ) : ?>
                <tr>
                    <td><?php echo esc_html( $row['id'] );         ?></td>
                    <td><?php echo esc_html( $row['country_id'] ); ?></td>
                    <td><?php echo esc_html( $row['name'] );       ?></td>
                    <td><?php echo esc_html( $row['image_path'] ); ?></td>
                    <td><?php echo esc_html( $row['last_updated'] ); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <?php if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) : ?>
            <?php livefot_display_pagination( $current_page, $total_pages, $base_url ); ?>
        <?php elseif ( $total_items > 0 ) : ?>
            <p><?php printf(
					/* translators: %d = number of countires. */
                esc_html__( 'Displaying %d country(ies).', 'livefot-matches' ),
                esc_html( $total_items )
            ); ?></p>
        <?php endif; ?>

    <?php else : ?>
        <p><?php esc_html_e( 'No data found in the LiveFot Countries table.', 'livefot-matches' ); ?></p>
    <?php endif;

    echo '</div>';
}




// Scores Data page callback with pagination
/*function livefot_display_scores_data() {
    global $wpdb;

    // Define the table name
    $table_name = $wpdb->prefix . 'livefot_scores';

    // Pagination settings
    $items_per_page = 20;
    $current_page = isset($_GET['paged']) && is_numeric($_GET['paged']) ? intval($_GET['paged']) : 1;
    $offset = ($current_page - 1) * $items_per_page;

    // Get total items
    $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
    $total_pages = ceil($total_items / $items_per_page);

    // Fetch data with pagination
    $results = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name ORDER BY last_updated DESC LIMIT %d OFFSET %d", $items_per_page, $offset), ARRAY_A);

    // Base URL for pagination
    $base_url = admin_url('admin.php?page=livefot-scores-data');

    echo '<div class="wrap">';
    echo '<h1>LiveFot Scores Data</h1>';

    if (!empty($results)) {
        echo '<table class="widefat fixed striped">';
        echo '<thead>';
        echo '<tr>';
        echo '<th>ID</th>';
        echo '<th>Match ID</th>';
        echo '<th>Local Team Score</th>';
        echo '<th>Visitor Team Score</th>';
        echo '<th>Local Team Penalty Score</th>';
        echo '<th>Visitor Team Penalty Score</th>';
        echo '<th>HT Score</th>';
        echo '<th>FT Score</th>';
        echo '<th>ET Score</th>';
        echo '<th>PS Score</th>';
        echo '<th>Last Updated</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ($results as $row) {
            echo '<tr>';
            echo '<td>' . esc_html($row['id']) . '</td>';
            echo '<td>' . esc_html($row['fixture_id']) . '</td>';
            echo '<td>' . esc_html($row['localteam_score']) . '</td>';
            echo '<td>' . esc_html($row['visitorteam_score']) . '</td>';
            echo '<td>' . esc_html($row['localteam_pen_score']) . '</td>';
            echo '<td>' . esc_html($row['visitorteam_pen_score']) . '</td>';
            echo '<td>' . esc_html($row['ht_score']) . '</td>';
            echo '<td>' . esc_html($row['ft_score']) . '</td>';
            echo '<td>' . esc_html($row['et_score']) . '</td>';
            echo '<td>' . esc_html($row['ps_score']) . '</td>';
            echo '<td>' . esc_html($row['last_updated']) . '</td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';

        // Display pagination
        livefot_display_pagination($current_page, $total_pages, $base_url);
    } else {
        echo '<p>No data found in the LiveFot Scores table.</p>';
    }

    echo '</div>';
}*/


function livefot_display_scores_data() {
    global $wpdb;

    /* Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_scores_search_action', 'livefot_scores_search_nonce' );
    }

    /* Table name */
    $table_name = $wpdb->prefix . 'livefot_scores';

    /* Pagination */
    $items_per_page = 20;
    $current_page   = isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] )
                      ? max( 1, intval( $_GET['paged'] ) )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* Sanitise search inputs */
    $get                 = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_fixture_id   = isset( $get['search_fixture_id'] )
                           ? trim( sanitize_text_field( $get['search_fixture_id'] ) )
                           : '';

    /* Build WHERE clauses and parameters */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_fixture_id && is_numeric( $search_fixture_id ) ) {
        $where_parts[] = 'fixture_id = %d';
        $params[]      = intval( $search_fixture_id );
    }

    $where = implode( ' AND ', $where_parts );

    /* 1) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where}",
            ...$params
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 2) Fetch paginated rows */
    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where}
             ORDER BY last_updated DESC
             LIMIT %d OFFSET %d",
            ...array_merge( $params, [ $items_per_page, $offset ] )
        ),
        ARRAY_A
    );

    /* Build base URL for pagination & clear link */
    $base_url = admin_url( 'admin.php?page=livefot-scores-data' );
    $args     = [];
    if ( '' !== $search_fixture_id ) {
        $args['search_fixture_id'] = $search_fixture_id;
    }
    if ( $args ) {
        $base_url = add_query_arg( $args, $base_url );
    }

    /* Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Scores Data', 'livefot-matches' ) . '</h1>'; ?>

    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_scores_search_action', 'livefot_scores_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-scores-data" />
        <table class="form-table">
            <tr>
                <th><label for="search_fixture_id"><?php esc_html_e( 'Fixture ID', 'livefot-matches' ); ?></label></th>
                <td>
                    <input
                        type="text"
                        id="search_fixture_id"
                        name="search_fixture_id"
                        value="<?php echo esc_attr( $search_fixture_id ); ?>"
                    />
                </td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-scores-data' ) ); ?>" class="button">
            <?php esc_html_e( 'Clear', 'livefot-matches' ); ?>
        </a>
    </form>
    <hr/>

    <?php if ( $results ) : ?>
        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th><?php esc_html_e( 'ID',                          'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Fixture ID',                  'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Local Team Score',            'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Visitor Team Score',          'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Local Team Penalty Score',    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Visitor Team Penalty Score',  'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'HT Score',                    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'FT Score',                    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'ET Score',                    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'PS Score',                    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Last Updated',                'livefot-matches' ); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ( $results as $row ) : ?>
                <tr>
                    <td><?php echo esc_html( $row['id'] ); ?></td>
                    <td><?php echo esc_html( $row['fixture_id'] ); ?></td>
                    <td><?php echo esc_html( $row['localteam_score'] ); ?></td>
                    <td><?php echo esc_html( $row['visitorteam_score'] ); ?></td>
                    <td><?php echo esc_html( $row['localteam_pen_score'] ); ?></td>
                    <td><?php echo esc_html( $row['visitorteam_pen_score'] ); ?></td>
                    <td><?php echo esc_html( $row['ht_score'] ); ?></td>
                    <td><?php echo esc_html( $row['ft_score'] ); ?></td>
                    <td><?php echo esc_html( $row['et_score'] ); ?></td>
                    <td><?php echo esc_html( $row['ps_score'] ); ?></td>
                    <td><?php echo esc_html( $row['last_updated'] ); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <?php if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) : ?>
            <?php livefot_display_pagination( $current_page, $total_pages, $base_url ); ?>
        <?php elseif ( $total_items > 0 ) : ?>
            <p>
                <?php printf(
		            /* translators: %d = number of scores. */
                    esc_html__( 'Displaying %d score(s).', 'livefot-matches' ),
                    esc_html( $total_items )
                ); ?>
            </p>
        <?php endif; ?>

    <?php else : ?>
        <p><?php esc_html_e( 'No data found in the LiveFot Scores table.', 'livefot-matches' ); ?></p>
    <?php endif;

    echo '</div>';
}




function livefot_display_time_data() {
    global $wpdb;

    /* Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_time_search_action', 'livefot_time_search_nonce' );
    }

    /* Table name */
    $table_name = $wpdb->prefix . 'livefot_time';

    /* Pagination */
    $items_per_page = 20;
    $current_page   = isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] )
                      ? max( 1, intval( $_GET['paged'] ) )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* Sanitise search inputs */
    $get                = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_fixture_id  = isset( $get['search_fixture_id'] )
                           ? trim( sanitize_text_field( $get['search_fixture_id'] ) )
                           : '';

    /* Build WHERE clauses and parameters */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_fixture_id && is_numeric( $search_fixture_id ) ) {
        $where_parts[] = 'fixture_id = %d';
        $params[]      = intval( $search_fixture_id );
    }

    $where = implode( ' AND ', $where_parts );

    /* 1) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where}",
            ...$params
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 2) Fetch paginated rows */
    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where}
             ORDER BY last_updated DESC
             LIMIT %d OFFSET %d",
            ...array_merge( $params, [ $items_per_page, $offset ] )
        ),
        ARRAY_A
    );

    /* Build base URL for pagination & clear link */
    $base_url = admin_url( 'admin.php?page=livefot-time-data' );
    $args     = [];
    if ( '' !== $search_fixture_id ) {
        $args['search_fixture_id'] = $search_fixture_id;
    }
    if ( $args ) {
        $base_url = add_query_arg( $args, $base_url );
    }

    /* Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Time Data', 'livefot-matches' ) . '</h1>'; ?>

    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_time_search_action', 'livefot_time_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-time-data" />
        <table class="form-table">
            <tr>
                <th><label for="search_fixture_id"><?php esc_html_e( 'Fixture ID', 'livefot-matches' ); ?></label></th>
                <td>
                    <input
                        type="text"
                        id="search_fixture_id"
                        name="search_fixture_id"
                        value="<?php echo esc_attr( $search_fixture_id ); ?>"
                    />
                </td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-time-data' ) ); ?>" class="button">
            <?php esc_html_e( 'Clear', 'livefot-matches' ); ?>
        </a>
    </form>
    <hr/>

    <?php if ( $results ) : ?>
        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th><?php esc_html_e( 'ID',                        'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Fixture ID',                'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Status',                    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Starting DateTime',         'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Starting Date',             'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Starting Time',             'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Timestamp',                 'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Timezone',                  'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Minute',                    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Second',                    'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Added Time',                'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Extra Minute',              'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Injury Time',               'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Match Period',              'livefot-matches' ); ?></th>
                    <th><?php esc_html_e( 'Last Updated',              'livefot-matches' ); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ( $results as $row ) : ?>
                <tr>
                    <td><?php echo esc_html( $row['id'] ); ?></td>
                    <td><?php echo esc_html( $row['fixture_id'] ); ?></td>
                    <td><?php echo esc_html( $row['status'] ); ?></td>
                    <td><?php echo esc_html( $row['starting_at_datetime'] ); ?></td>
                    <td><?php echo esc_html( $row['starting_at_date'] ); ?></td>
                    <td><?php echo esc_html( $row['starting_at_time'] ); ?></td>
                    <td><?php echo esc_html( $row['starting_at_timestamp'] ); ?></td>
                    <td><?php echo esc_html( $row['starting_at_timezone'] ); ?></td>
                    <td><?php echo esc_html( $row['minute'] ); ?></td>
                    <td><?php echo esc_html( $row['second'] ); ?></td>
                    <td><?php echo esc_html( $row['added_time'] ); ?></td>
                    <td><?php echo esc_html( $row['extra_minute'] ); ?></td>
                    <td><?php echo esc_html( $row['injury_time'] ); ?></td>
                    <td><?php echo esc_html( $row['match_period'] ); ?></td>
                    <td><?php echo esc_html( $row['last_updated'] ); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <?php if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) : ?>
            <?php livefot_display_pagination( $current_page, $total_pages, $base_url ); ?>
        <?php elseif ( $total_items > 0 ) : ?>
            <p>
                <?php printf(
		            /* translators: %d = number of records. */
                    esc_html__( 'Displaying %d record(s).', 'livefot-matches' ),
                    esc_html( $total_items )
                ); ?>
            </p>
        <?php endif; ?>

    <?php else : ?>
        <p><?php esc_html_e( 'No data found in the LiveFot Time table.', 'livefot-matches' ); ?></p>
    <?php endif;

    echo '</div>';
}



function livefot_display_aggregate_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_aggregate_search_action', 'livefot_aggregate_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_aggregate';

    /* 4) Pagination */
    $items_per_page = 20;
    $current_page   = ( isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) )
                      ? max( 1, intval( $_GET['paged'] ) )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize inputs */
    $get                 = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_aggregate_id = isset( $get['search_aggregate_id'] )
                           ? trim( sanitize_text_field( $get['search_aggregate_id'] ) )
                           : '';
    $search_league_id    = isset( $get['search_league_id'] )
                           ? trim( sanitize_text_field( $get['search_league_id'] ) )
                           : '';
    $search_visitorteam  = isset( $get['search_visitorteam'] )
                           ? trim( sanitize_text_field( $get['search_visitorteam'] ) )
                           : '';
    $search_localteam    = isset( $get['search_localteam'] )
                           ? trim( sanitize_text_field( $get['search_localteam'] ) )
                           : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_aggregate_id && is_numeric( $search_aggregate_id ) ) {
        $where_parts[] = 'aggregate_id = %d';
        $params[]      = intval( $search_aggregate_id );
    }

    if ( '' !== $search_league_id && is_numeric( $search_league_id ) ) {
        $where_parts[] = 'league_id = %d';
        $params[]      = intval( $search_league_id );
    }

    if ( '' !== $search_visitorteam ) {
        $where_parts[] = 'visitorteam LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_visitorteam ) . '%';
    }

    if ( '' !== $search_localteam ) {
        $where_parts[] = 'localteam LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_localteam ) . '%';
    }

    $where = implode( ' AND ', $where_parts );

    /* 7) Count total items */
    if ( $params ) {
        $total_items = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table_name} WHERE {$where}",
                ...$params
            )
        );
    } else {
        $total_items = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
    }
    $total_pages = $total_items > 0
        ? ceil( $total_items / $items_per_page )
        : 0;

    /* 8) Fetch rows */
    $fetch_params = array_merge( $params, [ $items_per_page, $offset ] );
    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where}
             ORDER BY last_updated DESC
             LIMIT %d OFFSET %d",
            ...$fetch_params
        ),
        ARRAY_A
    );

    /* 9) Build base URL & clear link */
    $base_url   = admin_url( 'admin.php?page=livefot-aggregate-data' );
    $query_args = [];
    if ( '' !== $search_aggregate_id ) {
        $query_args['search_aggregate_id'] = $search_aggregate_id;
    }
    if ( '' !== $search_league_id ) {
        $query_args['search_league_id'] = $search_league_id;
    }
    if ( '' !== $search_visitorteam ) {
        $query_args['search_visitorteam'] = $search_visitorteam;
    }
    if ( '' !== $search_localteam ) {
        $query_args['search_localteam'] = $search_localteam;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 10) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Aggregate Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_aggregate_search_action', 'livefot_aggregate_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-aggregate-data" />
        <table class="form-table">
            <tr>
                <th><label for="search_aggregate_id"><?php esc_html_e( 'Aggregate ID', 'livefot-matches' ); ?></label></th>
                <td><input type="number" id="search_aggregate_id" name="search_aggregate_id" value="<?php echo esc_attr( $search_aggregate_id ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_league_id"><?php esc_html_e( 'League ID', 'livefot-matches' ); ?></label></th>
                <td><input type="number" id="search_league_id" name="search_league_id" value="<?php echo esc_attr( $search_league_id ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_visitorteam"><?php esc_html_e( 'Visitor Team', 'livefot-matches' ); ?></label></th>
                <td><input type="text" id="search_visitorteam" name="search_visitorteam" value="<?php echo esc_attr( $search_visitorteam ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_localteam"><?php esc_html_e( 'Local Team', 'livefot-matches' ); ?></label></th>
                <td><input type="text" id="search_localteam" name="search_localteam" value="<?php echo esc_attr( $search_localteam ); ?>" /></td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <a href="<?php echo esc_url( $base_url ); ?>" class="button">
            <?php esc_html_e( 'Clear', 'livefot-matches' ); ?>
        </a>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>'
           . '<th>' . esc_html__( 'ID',             'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Aggregate ID',   'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'League ID',      'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Season ID',      'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Stage ID',       'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Local Team',     'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Local Team ID',  'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Visitor Team',   'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Visitor Team ID','livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Result',         'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Winner',         'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Detail',         'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Last Updated',   'livefot-matches' ) . '</th>'
           . '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>'
               . '<td>' . esc_html( $row['id'] )               . '</td>'
               . '<td>' . esc_html( $row['aggregate_id'] )     . '</td>'
               . '<td>' . esc_html( $row['league_id'] )        . '</td>'
               . '<td>' . esc_html( $row['season_id'] )        . '</td>'
               . '<td>' . esc_html( $row['stage_id'] )         . '</td>'
               . '<td>' . esc_html( $row['localteam'] )        . '</td>'
               . '<td>' . esc_html( $row['localteam_id'] )     . '</td>'
               . '<td>' . esc_html( $row['visitorteam'] )      . '</td>'
               . '<td>' . esc_html( $row['visitorteam_id'] )   . '</td>'
               . '<td>' . esc_html( $row['result'] )           . '</td>'
               . '<td>' . esc_html( $row['winner'] )           . '</td>'
               . '<td>' . esc_html( $row['detail'] )           . '</td>'
               . '<td>' . esc_html( $row['last_updated'] )     . '</td>'
               . '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}

function livefot_display_referees_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_referees_search_action', 'livefot_referees_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_referees';

    /* 4) Pagination */
    $items_per_page = 20;
    $current_page   = ( isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) )
                      ? max( 1, intval( $_GET['paged'] ) )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize inputs */
    $get                = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_referee_id  = isset( $get['search_referee_id'] )
                          ? trim( sanitize_text_field( $get['search_referee_id'] ) )
                          : '';
    $search_common_name = isset( $get['search_common_name'] )
                          ? trim( sanitize_text_field( $get['search_common_name'] ) )
                          : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_referee_id && is_numeric( $search_referee_id ) ) {
        $where_parts[] = 'referee_id = %d';
        $params[]      = intval( $search_referee_id );
    }

    if ( '' !== $search_common_name ) {
        $where_parts[] = 'common_name LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_common_name ) . '%';
    }

    $where = implode( ' AND ', $where_parts );

    /* 7) Try cache first */
    $cache_key = 'livefot_referees_' . md5( $where . '|' . serialize( $params ) . "|page{$current_page}" );
    if ( false !== ( $cache = get_transient( $cache_key ) ) ) {
        list( $results, $total_items ) = $cache;
    } else {
        /* 8) Count total items */
        if ( $params ) {
            $total_items = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table_name} WHERE {$where}",
                    ...$params
                )
            );
        } else {
            $total_items = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
        }

        /* 9) Fetch rows */
        $fetch_params = array_merge( $params, [ $items_per_page, $offset ] );
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table_name}
                 WHERE {$where}
                 ORDER BY last_updated DESC
                 LIMIT %d OFFSET %d",
                ...$fetch_params
            ),
            ARRAY_A
        );

        /* 10) Save to cache for 5 minutes */
        set_transient( $cache_key, [ $results, $total_items ], 5 * MINUTE_IN_SECONDS );
    }

    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 11) Build base URL & clear link */
    $base_url   = admin_url( 'admin.php?page=livefot-referees-data' );
    $query_args = [];
    if ( '' !== $search_referee_id ) {
        $query_args['search_referee_id'] = $search_referee_id;
    }
    if ( '' !== $search_common_name ) {
        $query_args['search_common_name'] = $search_common_name;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 12) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Referees Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_referees_search_action', 'livefot_referees_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-referees-data" />
        <table class="form-table">
            <tr>
                <th><label for="search_referee_id"><?php esc_html_e( 'Referee ID', 'livefot-matches' ); ?></label></th>
                <td><input type="number" id="search_referee_id" name="search_referee_id" value="<?php echo esc_attr( $search_referee_id ); ?>" placeholder="<?php esc_attr_e( 'Enter Referee ID', 'livefot-matches' ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_common_name"><?php esc_html_e( 'Common Name', 'livefot-matches' ); ?></label></th>
                <td><input type="text" id="search_common_name" name="search_common_name" value="<?php echo esc_attr( $search_common_name ); ?>" placeholder="<?php esc_attr_e( 'Enter Common Name', 'livefot-matches' ); ?>" /></td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <?php if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-referees-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>'
           . '<th>' . esc_html__( 'ID',            'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Referee ID',    'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Common Name',   'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Last Updated',  'livefot-matches' ) . '</th>'
           . '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>'
               . '<td>' . esc_html( $row['id'] )            . '</td>'
               . '<td>' . esc_html( $row['referee_id'] )    . '</td>'
               . '<td>' . esc_html( $row['common_name'] )   . '</td>'
               . '<td>' . esc_html( $row['last_updated'] )  . '</td>'
               . '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
                /* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}



function livefot_display_venues_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_venues_search_action', 'livefot_venues_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_venues';

    /* 4) Pagination */
    $items_per_page = 20;
    $current_page   = ( isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) )
                      ? absint( $_GET['paged'] )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize inputs */
    $get             = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_venue_id = isset( $get['search_venue_id'] )
                       ? trim( sanitize_text_field( $get['search_venue_id'] ) )
                       : '';
    $search_name     = isset( $get['search_name'] )
                       ? trim( sanitize_text_field( $get['search_name'] ) )
                       : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_venue_id && is_numeric( $search_venue_id ) ) {
        $where_parts[] = 'venue_id = %d';
        $params[]      = absint( $search_venue_id );
    }

    if ( '' !== $search_name ) {
        $where_parts[] = 'name LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_name ) . '%';
    }

    $where = implode( ' AND ', $where_parts );

    /* 7) Try cache */
    $cache_key = 'livefot_venues_' . md5( $where . '|' . serialize( $params ) . "|page{$current_page}" );
    if ( false !== ( $cache = get_transient( $cache_key ) ) ) {
        list( $results, $total_items ) = $cache;
    } else {
        /* 8) Count total items */
        if ( $params ) {
            $total_items = (int) $wpdb->get_var(
                $wpdb->prepare( "SELECT COUNT(*) FROM {$table_name} WHERE {$where}", ...$params )
            );
        } else {
            $total_items = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
        }

        /* 9) Fetch rows */
        $fetch_args = array_merge( $params, [ $items_per_page, $offset ] );
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table_name}
                 WHERE {$where}
                 ORDER BY last_updated DESC
                 LIMIT %d OFFSET %d",
                ...$fetch_args
            ),
            ARRAY_A
        );

        /* 10) Save to cache for 5 minutes */
        set_transient( $cache_key, [ $results, $total_items ], 5 * MINUTE_IN_SECONDS );
    }

    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 11) Build base URL & clear link */
    $base_url   = admin_url( 'admin.php?page=livefot-venues-data' );
    $query_args = [];
    if ( '' !== $search_venue_id ) {
        $query_args['search_venue_id'] = $search_venue_id;
    }
    if ( '' !== $search_name ) {
        $query_args['search_name'] = $search_name;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 12) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Venues Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_venues_search_action', 'livefot_venues_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-venues-data" />
        <table class="form-table">
            <tr>
                <th><label for="search_venue_id"><?php esc_html_e( 'Venue ID', 'livefot-matches' ); ?></label></th>
                <td>
                    <input
                        type="number"
                        id="search_venue_id"
                        name="search_venue_id"
                        value="<?php echo esc_attr( $search_venue_id ); ?>"
                        placeholder="<?php esc_attr_e( 'Enter Venue ID', 'livefot-matches' ); ?>"
                    />
                </td>
            </tr>
            <tr>
                <th><label for="search_name"><?php esc_html_e( 'Venue Name', 'livefot-matches' ); ?></label></th>
                <td>
                    <input
                        type="text"
                        id="search_name"
                        name="search_name"
                        value="<?php echo esc_attr( $search_name ); ?>"
                        placeholder="<?php esc_attr_e( 'Enter Venue Name', 'livefot-matches' ); ?>"
                    />
                </td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <?php if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-venues-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>'
           . '<th>' . esc_html__( 'ID',           'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Venue ID',     'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Name',         'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Last Updated', 'livefot-matches' ) . '</th>'
           . '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>'
               . '<td>' . esc_html( $row['id'] )           . '</td>'
               . '<td>' . esc_html( $row['venue_id'] )     . '</td>'
               . '<td>' . esc_html( $row['name'] )         . '</td>'
               . '<td>' . esc_html( $row['last_updated'] ) . '</td>'
               . '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
                /* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}


function livefot_display_lineup_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_lineup_search_action', 'livefot_lineup_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_lineups';

    /* 4) Pagination */
    $items_per_page = 20;
    $current_page   = ( isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) )
                      ? absint( $_GET['paged'] )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize inputs */
    $get                  = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_match_id      = isset( $get['search_match_id'] )        ? sanitize_text_field( $get['search_match_id'] )        : '';
    $search_team_id       = isset( $get['search_team_id'] )         ? sanitize_text_field( $get['search_team_id'] )         : '';
    $search_player_id     = isset( $get['search_player_id'] )       ? sanitize_text_field( $get['search_player_id'] )       : '';
    $search_position      = isset( $get['search_position'] )        ? sanitize_text_field( $get['search_position'] )        : '';
    $search_formation_pos = isset( $get['search_formation_position'] ) ? sanitize_text_field( $get['search_formation_position'] ) : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_match_id && is_numeric( $search_match_id ) ) {
        $where_parts[] = 'match_id = %d';
        $params[]      = absint( $search_match_id );
    }
    if ( '' !== $search_team_id && is_numeric( $search_team_id ) ) {
        $where_parts[] = 'team_id = %d';
        $params[]      = absint( $search_team_id );
    }
    if ( '' !== $search_player_id && is_numeric( $search_player_id ) ) {
        $where_parts[] = 'player_id = %d';
        $params[]      = absint( $search_player_id );
    }
    if ( '' !== $search_position ) {
        $where_parts[] = 'position LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_position ) . '%';
    }
    if ( '' !== $search_formation_pos ) {
        $where_parts[] = 'formation_position LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_formation_pos ) . '%';
    }

    $where = implode( ' AND ', $where_parts );

    /* 7) Try cache */
    $cache_key = 'livefot_lineup_' . md5( $where . '|' . serialize( $params ) . "|page{$current_page}" );
    if ( false !== ( $cache = get_transient( $cache_key ) ) ) {
        list( $results, $total_items ) = $cache;
    } else {
        /* 8) Count total items */
        if ( $params ) {
            $total_items = (int) $wpdb->get_var(
                $wpdb->prepare( "SELECT COUNT(*) FROM {$table_name} WHERE {$where}", ...$params )
            );
        } else {
            $total_items = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
        }

        /* 9) Fetch rows */
        $fetch_args = array_merge( $params, [ $items_per_page, $offset ] );
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table_name}
                 WHERE {$where}
                 ORDER BY last_updated DESC
                 LIMIT %d OFFSET %d",
                ...$fetch_args
            ),
            ARRAY_A
        );

        /* 10) Save to cache for 5 minutes */
        set_transient( $cache_key, [ $results, $total_items ], 5 * MINUTE_IN_SECONDS );
    }

    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 11) Build base URL & reset link */
    $base_url   = admin_url( 'admin.php?page=livefot-lineup-data' );
    $query_args = [];
    if ( '' !== $search_match_id ) {
        $query_args['search_match_id'] = $search_match_id;
    }
    if ( '' !== $search_team_id ) {
        $query_args['search_team_id'] = $search_team_id;
    }
    if ( '' !== $search_player_id ) {
        $query_args['search_player_id'] = $search_player_id;
    }
    if ( '' !== $search_position ) {
        $query_args['search_position'] = $search_position;
    }
    if ( '' !== $search_formation_pos ) {
        $query_args['search_formation_position'] = $search_formation_pos;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 12) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Lineup Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_lineup_search_action', 'livefot_lineup_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-lineup-data" />
        <table class="form-table">
            <tr>
                <th><label for="search_match_id"><?php esc_html_e( 'Match ID', 'livefot-matches' ); ?></label></th>
                <td><input type="number" id="search_match_id" name="search_match_id" value="<?php echo esc_attr( $search_match_id ); ?>" placeholder="<?php esc_attr_e( 'Enter Match ID', 'livefot-matches' ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_team_id"><?php esc_html_e( 'Team ID', 'livefot-matches' ); ?></label></th>
                <td><input type="number" id="search_team_id" name="search_team_id" value="<?php echo esc_attr( $search_team_id ); ?>" placeholder="<?php esc_attr_e( 'Enter Team ID', 'livefot-matches' ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_player_id"><?php esc_html_e( 'Player ID', 'livefot-matches' ); ?></label></th>
                <td><input type="number" id="search_player_id" name="search_player_id" value="<?php echo esc_attr( $search_player_id ); ?>" placeholder="<?php esc_attr_e( 'Enter Player ID', 'livefot-matches' ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_position"><?php esc_html_e( 'Position', 'livefot-matches' ); ?></label></th>
                <td><input type="text" id="search_position" name="search_position" value="<?php echo esc_attr( $search_position ); ?>" placeholder="<?php esc_attr_e( 'Enter Position', 'livefot-matches' ); ?>" /></td>
            </tr>
            <tr>
                <th><label for="search_formation_position"><?php esc_html_e( 'Formation Position', 'livefot-matches' ); ?></label></th>
                <td><input type="text" id="search_formation_position" name="search_formation_position" value="<?php echo esc_attr( $search_formation_pos ); ?>" placeholder="<?php esc_attr_e( 'Enter Formation Position', 'livefot-matches' ); ?>" /></td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <?php if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-lineup-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>'
           . '<th>' . esc_html__( 'ID',                 'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Match ID',           'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Team ID',            'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Player ID',          'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Number',             'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Position',           'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Formation Position', 'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Captain',            'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Type',               'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Last Updated',       'livefot-matches' ) . '</th>'
           . '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>'
               . '<td>' . esc_html( $row['id'] )                   . '</td>'
               . '<td>' . esc_html( $row['match_id'] )             . '</td>'
               . '<td>' . esc_html( $row['team_id'] )              . '</td>'
               . '<td>' . esc_html( $row['player_id'] )            . '</td>'
               . '<td>' . esc_html( $row['number'] )               . '</td>'
               . '<td>' . esc_html( $row['position'] )             . '</td>'
               . '<td>' . esc_html( $row['formation_position'] )   . '</td>'
               . '<td>' . esc_html( $row['captain'] ? __( 'Yes', 'livefot-matches' ) : __( 'No', 'livefot-matches' ) ) . '</td>'
               . '<td>' . esc_html( $row['type'] )                 . '</td>'
               . '<td>' . esc_html( $row['last_updated'] )         . '</td>'
               . '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
                /* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}



/*
function livefot_display_players_data() {
    global $wpdb;

    // Capability check to ensure only authorized users can access this page
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die(
            esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' )
        );
    }

    // Define the table name
    $table_name = $wpdb->prefix . 'livefot_players';

    // Pagination settings
    $items_per_page = 20;
    $current_page = isset($_GET['paged']) && is_numeric($_GET['paged']) ? intval($_GET['paged']) : 1;
    $offset = ($current_page - 1) * $items_per_page;

    // Initialize search variables
    $search_player_id = isset($_GET['search_player_id']) ? sanitize_text_field($_GET['search_player_id']) : '';
    $search_player_name = isset($_GET['search_player_name']) ? sanitize_text_field($_GET['search_player_name']) : '';

    // Cast numerical input to integer for validation
    if (!empty($search_player_id)) {
        $search_player_id = intval($search_player_id);
    }

    // Build the WHERE clause based on search inputs
    $where = '1=1'; // Default where clause
    $params = [];

    if (!empty($search_player_id)) {
        $where .= ' AND player_id = %d';
        $params[] = $search_player_id;
    }

    if (!empty($search_player_name)) {
        $where .= ' AND player_name LIKE %s';
        $params[] = '%' . $wpdb->esc_like($search_player_name) . '%';
    }

    // Prepare the SQL query with dynamic WHERE clause
    if (!empty($params)) {
        // If any search parameters are set, include them in the query
        $sql = $wpdb->prepare(
            "SELECT * FROM $table_name WHERE $where ORDER BY id DESC LIMIT %d OFFSET %d",
            array_merge($params, [$items_per_page, $offset])
        );

        // Prepare count query with the same WHERE clause
        $count_sql = $wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name WHERE $where",
            $params
        );
    } else {
        // No search parameters, fetch all
        $sql = $wpdb->prepare(
            "SELECT * FROM $table_name ORDER BY id DESC LIMIT %d OFFSET %d",
            $items_per_page,
            $offset
        );
        $count_sql = "SELECT COUNT(*) FROM $table_name";
    }

    // Get total items for pagination
    $total_items = $wpdb->get_var($count_sql);
    $total_pages = ceil($total_items / $items_per_page);

    // Fetch data with pagination and search filters
    $results = $wpdb->get_results($sql, ARRAY_A);

    // Base URL for pagination and maintaining search parameters
    $base_url = admin_url('admin.php?page=livefot-players-data');

    // Append search parameters to the base URL if they exist
    if (!empty($search_player_id)) {
        $base_url = add_query_arg('search_player_id', $search_player_id, $base_url);
    }
    if (!empty($search_player_name)) {
        $base_url = add_query_arg('search_player_name', $search_player_name, $base_url);
    }

    echo '<div class="wrap">';
    echo '<h1>LiveFot Players Data</h1>';

    // Search Form
    ?>
    <form method="get" action="">
        <input type="hidden" name="page" value="livefot-players-data" />
        <table class="form-table">
            <tr>
                <th scope="row"><label for="search_player_id">Player ID</label></th>
                <td>
                    <input type="text" id="search_player_id" name="search_player_id" value="<?php echo esc_attr($search_player_id); ?>" placeholder="Enter Player ID" />
                </td>
            </tr>
            <tr>
                <th scope="row"><label for="search_player_name">Player Name</label></th>
                <td>
                    <input type="text" id="search_player_name" name="search_player_name" value="<?php echo esc_attr($search_player_name); ?>" placeholder="Enter Player Name" />
                </td>
            </tr>
        </table>
				<?php 
			// Translatable & escaped “Search” button
			submit_button( esc_html__( 'Search', 'livefot-matches' ) ); 
		?>

		<?php if ( ! empty( $search_player_id ) || ! empty( $search_player_name ) ) : ?>
			<a 
				href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-players-data' ) ); ?>" 
				class="button-secondary"
			>
				<?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
			</a>
		<?php endif; ?>

    </form>
    <?php

    if (!empty($results)) {
        echo '<table class="widefat fixed striped">';
        echo '<thead>';
        echo '<tr>';
        echo '<th>ID</th>';
        echo '<th>Player ID</th>';
        echo '<th>Player Name</th>';
        echo '<th>Logo Path</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ($results as $row) {
            echo '<tr>';
            echo '<td>' . esc_html($row['id']) . '</td>';
            echo '<td>' . esc_html($row['player_id']) . '</td>';
            echo '<td>' . esc_html($row['player_name']) . '</td>';
            echo '<td>' . esc_html($row['logo_path']) . '</td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';

        // Display pagination
        livefot_display_pagination($current_page, $total_pages, $base_url);
    } else {
        echo '<p>No data found in the LiveFot Players table with the specified search criteria.</p>';
    }

    echo '</div>';
}*/



/**
 * Render the “LiveFot Players” admin screen.
 *
 * – All SQL is prepared.
 * – Inputs are sanitised/validated.
 * – Pagination & search preserved.
 */
function livefot_display_players_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_players_search_action', 'livefot_players_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_players';

    /* 4) Pagination */
    $items_per_page = 20;
    $current_page   = ( isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) )
                      ? absint( $_GET['paged'] )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize inputs */
    $get                = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_player_id   = isset( $get['search_player_id'] )   ? trim( sanitize_text_field( $get['search_player_id'] ) )   : '';
    $search_player_name = isset( $get['search_player_name'] ) ? trim( sanitize_text_field( $get['search_player_name'] ) ) : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [ '1=1' ];
    $params      = [];

    if ( '' !== $search_player_id && is_numeric( $search_player_id ) ) {
        $where_parts[] = 'player_id = %d';
        $params[]      = absint( $search_player_id );
    }

    if ( '' !== $search_player_name ) {
        $where_parts[] = 'player_name LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_player_name ) . '%';
    }

    $where = implode( ' AND ', $where_parts );

    /* 7) Count total items */
    if ( $params ) {
        $total_items = (int) $wpdb->get_var(
            $wpdb->prepare( "SELECT COUNT(*) FROM {$table_name} WHERE {$where}", ...$params )
        );
    } else {
        $total_items = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
    }
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 8) Fetch rows */
    $fetch_args = array_merge( $params, [ $items_per_page, $offset ] );
    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where}
             ORDER BY id DESC
             LIMIT %d OFFSET %d",
            ...$fetch_args
        ),
        ARRAY_A
    );

    /* 9) Build base URL & reset link */
    $base_url   = admin_url( 'admin.php?page=livefot-players-data' );
    $query_args = [];
    if ( '' !== $search_player_id ) {
        $query_args['search_player_id'] = $search_player_id;
    }
    if ( '' !== $search_player_name ) {
        $query_args['search_player_name'] = $search_player_name;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 10) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Players Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_players_search_action', 'livefot_players_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-players-data" />
        <table class="form-table" role="presentation">
            <tr>
                <th><label for="search_player_id"><?php esc_html_e( 'Player ID', 'livefot-matches' ); ?></label></th>
                <td>
                    <input
                        type="number"
                        id="search_player_id"
                        name="search_player_id"
                        value="<?php echo esc_attr( $search_player_id ); ?>"
                        placeholder="<?php esc_attr_e( 'Enter Player ID', 'livefot-matches' ); ?>"
                    />
                </td>
            </tr>
            <tr>
                <th><label for="search_player_name"><?php esc_html_e( 'Player Name', 'livefot-matches' ); ?></label></th>
                <td>
                    <input
                        type="text"
                        id="search_player_name"
                        name="search_player_name"
                        value="<?php echo esc_attr( $search_player_name ); ?>"
                        placeholder="<?php esc_attr_e( 'Enter Player Name', 'livefot-matches' ); ?>"
                    />
                </td>
            </tr>
        </table>
        <?php submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false ); ?>
        <?php if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-players-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>'
           . '<th>' . esc_html__( 'ID',           'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Player ID',    'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Player Name',  'livefot-matches' ) . '</th>'
           . '<th>' . esc_html__( 'Logo Path',    'livefot-matches' ) . '</th>'
           . '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>'
               . '<td>' . esc_html( $row['id'] )          . '</td>'
               . '<td>' . esc_html( $row['player_id'] )   . '</td>'
               . '<td>' . esc_html( $row['player_name'] ) . '</td>'
               . '<td>' . esc_html( $row['logo_path'] )   . '</td>'
               . '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}





/**
 * Admin-screen renderer: “LiveFot Events Data”.
 *
 * Fully compliant with PHPCS WordPress‐VIP rules:
 * – Every SQL statement is prepared inline (no NotPrepared warnings).
 * – GET params are validated & sanitised.
 * – Pagination, search & reset preserved.
 *
 * @since 1.0.0
 */
function livefot_display_events_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_events_search_action', 'livefot_events_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_events';

    /* 4) Pagination */
    $items_per_page = 20;
    $current_page   = ( isset( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) )
                      ? absint( $_GET['paged'] )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize inputs */
    $get                        = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_match_id            = isset( $get['search_match_id'] )          ? sanitize_text_field( $get['search_match_id'] )          : '';
    $search_event_id            = isset( $get['search_event_id'] )          ? sanitize_text_field( $get['search_event_id'] )          : '';
    $search_type                = isset( $get['search_type'] )              ? sanitize_text_field( $get['search_type'] )              : '';
    $search_team_id             = isset( $get['search_team_id'] )           ? sanitize_text_field( $get['search_team_id'] )           : '';
    $search_player_id           = isset( $get['search_player_id'] )         ? sanitize_text_field( $get['search_player_id'] )         : '';
    $search_related_player_id   = isset( $get['search_related_player_id'] ) ? sanitize_text_field( $get['search_related_player_id'] ) : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [];
    $params      = [];

    if ( '' !== $search_match_id && is_numeric( $search_match_id ) ) {
        $where_parts[] = 'match_id = %d';
        $params[]      = absint( $search_match_id );
    }
    if ( '' !== $search_event_id && is_numeric( $search_event_id ) ) {
        $where_parts[] = 'event_id = %d';
        $params[]      = absint( $search_event_id );
    }
    if ( '' !== $search_type ) {
        $where_parts[] = 'type LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_type ) . '%';
    }
    if ( '' !== $search_team_id && is_numeric( $search_team_id ) ) {
        $where_parts[] = 'team_id = %d';
        $params[]      = absint( $search_team_id );
    }
    if ( '' !== $search_player_id && is_numeric( $search_player_id ) ) {
        $where_parts[] = 'player_id = %d';
        $params[]      = absint( $search_player_id );
    }
    if ( '' !== $search_related_player_id && is_numeric( $search_related_player_id ) ) {
        $where_parts[] = 'related_player_id = %d';
        $params[]      = absint( $search_related_player_id );
    }

    if ( $where_parts ) {
        $where_sql  = implode( ' AND ', $where_parts );
        $where_args = $params;
    } else {
        $where_sql  = '1 = %d';
        $where_args = [ 1 ];
    }

    /* 7) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where_sql}",
            ...$where_args
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 8) Fetch rows */
    $fetch_args = array_merge( $where_args, [ $items_per_page, $offset ] );
    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where_sql}
             ORDER BY last_updated DESC
             LIMIT %d OFFSET %d",
            ...$fetch_args
        ),
        ARRAY_A
    );

    /* 9) Build base URL & reset link */
    $base_url   = add_query_arg( 'page', 'livefot-events-data', admin_url( 'admin.php' ) );
    $query_args = [];
    foreach ( [
        'search_match_id'          => $search_match_id,
        'search_event_id'          => $search_event_id,
        'search_type'              => $search_type,
        'search_team_id'           => $search_team_id,
        'search_player_id'         => $search_player_id,
        'search_related_player_id' => $search_related_player_id,
    ] as $key => $val ) {
        if ( '' !== $val ) {
            $query_args[ $key ] = $val;
        }
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 10) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Events Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_events_search_action', 'livefot_events_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-events-data" />
        <table class="form-table" role="presentation">
            <?php
            $fields = [
                'search_match_id'          => esc_html__( 'Match ID', 'livefot-matches' ),
                'search_event_id'          => esc_html__( 'Event ID', 'livefot-matches' ),
                'search_type'              => esc_html__( 'Type', 'livefot-matches' ),
                'search_team_id'           => esc_html__( 'Team ID', 'livefot-matches' ),
                'search_player_id'         => esc_html__( 'Player ID', 'livefot-matches' ),
                'search_related_player_id' => esc_html__( 'Related Player ID', 'livefot-matches' ),
            ];
            foreach ( $fields as $name => $label ) :
                $value = ${$name};
                ?>
                <tr>
                    <th scope="row">
                        <label for="<?php echo esc_attr( $name ); ?>">
                            <?php echo esc_html( $label ); ?>
                        </label>
                    </th>
                    <td>
                        <input
                            type="text"
                            id="<?php echo esc_attr( $name ); ?>"
                            name="<?php echo esc_attr( $name ); ?>"
                            value="<?php echo esc_attr( $value ); ?>"
                        />
                    </td>
                </tr>
            <?php endforeach; ?>
        </table>
        <?php
        submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false );
        if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-events-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>';
        $headers = [
            'id'                  => esc_html__( 'ID', 'livefot-matches' ),
            'event_id'            => esc_html__( 'Event ID', 'livefot-matches' ),
            'match_id'            => esc_html__( 'Match ID', 'livefot-matches' ),
            'team_id'             => esc_html__( 'Team ID', 'livefot-matches' ),
            'player_id'           => esc_html__( 'Player ID', 'livefot-matches' ),
            'related_player_id'   => esc_html__( 'Related Player ID', 'livefot-matches' ),
            'type'                => esc_html__( 'Type', 'livefot-matches' ),
            'minute'              => esc_html__( 'Minute', 'livefot-matches' ),
            'extra_minute'        => esc_html__( 'Extra Minute', 'livefot-matches' ),
            'last_updated'        => esc_html__( 'Last Updated', 'livefot-matches' ),
        ];
        foreach ( $headers as $key => $label ) {
            echo '<th>' . esc_html( $label ) . '</th>';
        }
        echo '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>';
            foreach ( $headers as $key => $label ) {
                echo '<td>' . esc_html( $row[ $key ] ) . '</td>';
            }
            echo '</tr>';
        }
        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}




function livefot_display_teams_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_teams_search_action', 'livefot_teams_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_teams';

    /* 4) Pagination */
    $get            = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $current_page   = ( isset( $get['paged'] ) && is_numeric( $get['paged'] ) )
                      ? absint( $get['paged'] )
                      : 1;
    $items_per_page = 20;
    $offset         = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize search inputs */
    $search_team_id = isset( $get['search_team_id'] ) ? sanitize_text_field( $get['search_team_id'] ) : '';
    $search_name    = isset( $get['search_name'] )    ? sanitize_text_field( $get['search_name'] )    : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [];
    $params      = [];

    if ( '' !== $search_team_id && is_numeric( $search_team_id ) ) {
        $where_parts[] = 'team_id = %d';
        $params[]      = absint( $search_team_id );
    }
    if ( '' !== $search_name ) {
        $where_parts[] = 'name LIKE %s';
        $params[]      = '%' . $wpdb->esc_like( $search_name ) . '%';
    }

    if ( $where_parts ) {
        $where_sql  = implode( ' AND ', $where_parts );
        $where_args = $params;
    } else {
        $where_sql  = '1 = %d';
        $where_args = [ 1 ];
    }

    /* 7) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where_sql}",
            ...$where_args
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 8) Fetch paginated rows */
    $fetch_args = array_merge( $where_args, [ $items_per_page, $offset ] );
    $results    = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where_sql}
             ORDER BY id DESC
             LIMIT %d OFFSET %d",
            ...$fetch_args
        ),
        ARRAY_A
    );

    /* 9) Build base URL & preserve search params */
    $base_url   = add_query_arg( 'page', 'livefot-teams-data', admin_url( 'admin.php' ) );
    $query_args = [];
    if ( '' !== $search_team_id ) {
        $query_args['search_team_id'] = $search_team_id;
    }
    if ( '' !== $search_name ) {
        $query_args['search_name'] = $search_name;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 10) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Teams Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_teams_search_action', 'livefot_teams_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-teams-data" />
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row">
                    <label for="search_team_id"><?php esc_html_e( 'Team ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input
                        type="text"
                        id="search_team_id"
                        name="search_team_id"
                        value="<?php echo esc_attr( $search_team_id ); ?>"
                    />
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="search_name"><?php esc_html_e( 'Team Name', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input
                        type="text"
                        id="search_name"
                        name="search_name"
                        value="<?php echo esc_attr( $search_name ); ?>"
                    />
                </td>
            </tr>
        </table>
        <?php
        submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false );
        if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-teams-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>';
        $headers = [
            'id'                => esc_html__( 'ID',                 'livefot-matches' ),
            'team_id'           => esc_html__( 'Team ID',            'livefot-matches' ),
            'name'              => esc_html__( 'Name',               'livefot-matches' ),
            'shortcode'         => esc_html__( 'Shortcode',          'livefot-matches' ),
            'twitter'           => esc_html__( 'Twitter',            'livefot-matches' ),
            'country_id'        => esc_html__( 'Country ID',         'livefot-matches' ),
            'national_team'     => esc_html__( 'National Team',      'livefot-matches' ),
            'founded'           => esc_html__( 'Founded',            'livefot-matches' ),
            'logo_path'         => esc_html__( 'Logo Path',          'livefot-matches' ),
            'venue_id'          => esc_html__( 'Venue ID',           'livefot-matches' ),
            'current_season_id' => esc_html__( 'Current Season ID',  'livefot-matches' ),
            'gender'            => esc_html__( 'Gender',             'livefot-matches' ),
            'team_type'         => esc_html__( 'Team Type',          'livefot-matches' ),
        ];
        foreach ( $headers as $key => $label ) {
            echo '<th>' . esc_html( $label ) . '</th>';
        }
        echo '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>';
            foreach ( $headers as $key => $label ) {
                if ( 'national_team' === $key ) {
                    echo '<td>' . ( $row['national_team']
                                   ? esc_html__( 'Yes', 'livefot-matches' )
                                   : esc_html__( 'No',  'livefot-matches' )
                                 ) . '</td>';
                } else {
                    echo '<td>' . esc_html( $row[ $key ] ) . '</td>';
                }
            }
            echo '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found with the specified search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}




// Formations Data page callback with pagination
function livefot_display_formations_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_formations_search_action', 'livefot_formations_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_formations';

    /* 4) Pagination */
    $get            = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $current_page   = ( isset( $get['paged'] ) && is_numeric( $get['paged'] ) )
                      ? absint( $get['paged'] )
                      : 1;
    $items_per_page = 20;
    $offset         = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize search inputs */
    $search_match_id         = isset( $get['search_match_id'] )         ? sanitize_text_field( $get['search_match_id'] )         : '';
    $search_local_team_id    = isset( $get['search_local_team_id'] )    ? sanitize_text_field( $get['search_local_team_id'] )    : '';
    $search_visitor_team_id  = isset( $get['search_visitor_team_id'] )  ? sanitize_text_field( $get['search_visitor_team_id'] )  : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [];
    $params      = [];

    if ( '' !== $search_match_id && is_numeric( $search_match_id ) ) {
        $where_parts[] = 'match_id = %d';
        $params[]      = absint( $search_match_id );
    }
    if ( '' !== $search_local_team_id && is_numeric( $search_local_team_id ) ) {
        $where_parts[] = 'local_team_id = %d';
        $params[]      = absint( $search_local_team_id );
    }
    if ( '' !== $search_visitor_team_id && is_numeric( $search_visitor_team_id ) ) {
        $where_parts[] = 'visitor_team_id = %d';
        $params[]      = absint( $search_visitor_team_id );
    }

    if ( $where_parts ) {
        $where_sql  = implode( ' AND ', $where_parts );
        $where_args = $params;
    } else {
        $where_sql  = '1 = %d';
        $where_args = [ 1 ]; // dummy
    }

    /* 7) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where_sql}",
            ...$where_args
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 8) Fetch paginated rows */
    $fetch_args = array_merge( $where_args, [ $items_per_page, $offset ] );
    $results    = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where_sql}
             ORDER BY id DESC
             LIMIT %d OFFSET %d",
            ...$fetch_args
        ),
        ARRAY_A
    );

    /* 9) Build base URL & preserve search params */
    $base_url   = add_query_arg( 'page', 'livefot-formations-data', admin_url( 'admin.php' ) );
    $query_args = [];
    if ( '' !== $search_match_id ) {
        $query_args['search_match_id'] = $search_match_id;
    }
    if ( '' !== $search_local_team_id ) {
        $query_args['search_local_team_id'] = $search_local_team_id;
    }
    if ( '' !== $search_visitor_team_id ) {
        $query_args['search_visitor_team_id'] = $search_visitor_team_id;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 10) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Formations Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_formations_search_action', 'livefot_formations_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-formations-data" />
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row">
                    <label for="search_match_id"><?php esc_html_e( 'Match ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input
                        type="text"
                        id="search_match_id"
                        name="search_match_id"
                        value="<?php echo esc_attr( $search_match_id ); ?>"
                    />
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="search_local_team_id"><?php esc_html_e( 'Local Team ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input
                        type="text"
                        id="search_local_team_id"
                        name="search_local_team_id"
                        value="<?php echo esc_attr( $search_local_team_id ); ?>"
                    />
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="search_visitor_team_id"><?php esc_html_e( 'Visitor Team ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input
                        type="text"
                        id="search_visitor_team_id"
                        name="search_visitor_team_id"
                        value="<?php echo esc_attr( $search_visitor_team_id ); ?>"
                    />
                </td>
            </tr>
        </table>
        <?php
        submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false );
        if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-formations-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>';
        $headers = [
            'id'                     => esc_html__( 'ID', 'livefot-matches' ),
            'match_id'               => esc_html__( 'Match ID', 'livefot-matches' ),
            'local_team_id'          => esc_html__( 'Local Team ID', 'livefot-matches' ),
            'local_team_formation'   => esc_html__( 'Local Team Formation', 'livefot-matches' ),
            'visitor_team_id'        => esc_html__( 'Visitor Team ID', 'livefot-matches' ),
            'visitor_team_formation' => esc_html__( 'Visitor Team Formation', 'livefot-matches' ),
        ];
        foreach ( $headers as $key => $label ) {
            echo '<th>' . esc_html( $label ) . '</th>';
        }
        echo '</tr></thead><tbody>';
        foreach ( $results as $row ) {
            echo '<tr>';
            foreach ( $headers as $key => $label ) {
                echo '<td>' . esc_html( $row[ $key ] ) . '</td>';
            }
            echo '</tr>';
        }
        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}


function livefot_display_standings_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_standings_search_action', 'livefot_standings_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_standings';

    /* 4) Pagination */
    $items_per_page = 20;
    $current_page   = ( ! empty( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) )
                      ? absint( $_GET['paged'] )
                      : 1;
    $offset = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize inputs */
    $get                = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $search_standing_id = isset( $get['search_standing_id'] ) ? sanitize_text_field( $get['search_standing_id'] ) : '';
    $search_team_id     = isset( $get['search_team_id'] )     ? sanitize_text_field( $get['search_team_id'] )     : '';
    $search_league_id   = isset( $get['search_league_id'] )   ? sanitize_text_field( $get['search_league_id'] )   : '';
    $search_round_id    = isset( $get['search_round_id'] )    ? sanitize_text_field( $get['search_round_id'] )    : '';
    $search_season_id   = isset( $get['search_season_id'] )   ? sanitize_text_field( $get['search_season_id'] )   : '';
    $search_stage_id    = isset( $get['search_stage_id'] )    ? sanitize_text_field( $get['search_stage_id'] )    : '';
    $search_group_id    = isset( $get['search_group_id'] )    ? sanitize_text_field( $get['search_group_id'] )    : '';

    /* 6) Build WHERE and params */
    $where_parts = [];
    $params      = [];

    foreach ( [
        'standing_id' => $search_standing_id,
        'team_id'     => $search_team_id,
        'league_id'   => $search_league_id,
        'round_id'    => $search_round_id,
        'season_id'   => $search_season_id,
        'stage_id'    => $search_stage_id,
        'group_id'    => $search_group_id,
    ] as $column => $value ) {
        if ( '' !== $value && is_numeric( $value ) ) {
            $where_parts[] = "{$column} = %d";
            $params[]      = absint( $value );
        }
    }

    if ( $where_parts ) {
        $where_sql  = implode( ' AND ', $where_parts );
        $where_args = $params;
    } else {
        $where_sql  = '1 = %d';
        $where_args = [ 1 ];
    }

    /* 7) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare( "SELECT COUNT(*) FROM {$table_name} WHERE {$where_sql}", ...$where_args )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 8) Fetch rows */
    $fetch_args = array_merge( $where_args, [ $items_per_page, $offset ] );
    $results    = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where_sql}
             ORDER BY position ASC
             LIMIT %d OFFSET %d",
            ...$fetch_args
        ),
        ARRAY_A
    );

    /* 9) Build base URL & reset link */
    $base_url = add_query_arg( 'page', 'livefot-standings-data', admin_url( 'admin.php' ) );
    foreach ( [
        'search_standing_id' => $search_standing_id,
        'search_team_id'     => $search_team_id,
        'search_league_id'   => $search_league_id,
        'search_round_id'    => $search_round_id,
        'search_season_id'   => $search_season_id,
        'search_stage_id'    => $search_stage_id,
        'search_group_id'    => $search_group_id,
    ] as $key => $val ) {
        if ( '' !== $val ) {
            $base_url = add_query_arg( $key, $val, $base_url );
        }
    }

    /* 10) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Standings Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_standings_search_action', 'livefot_standings_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-standings-data" />
        <table class="form-table" role="presentation">
            <?php
            $fields = [
                'search_standing_id' => esc_html__( 'Standing ID', 'livefot-matches' ),
                'search_team_id'     => esc_html__( 'Team ID',     'livefot-matches' ),
                'search_league_id'   => esc_html__( 'League ID',   'livefot-matches' ),
                'search_round_id'    => esc_html__( 'Round ID',    'livefot-matches' ),
                'search_season_id'   => esc_html__( 'Season ID',   'livefot-matches' ),
                'search_stage_id'    => esc_html__( 'Stage ID',    'livefot-matches' ),
                'search_group_id'    => esc_html__( 'Group ID',    'livefot-matches' ),
            ];
            foreach ( $fields as $name => $label ) :
                $value = ${$name};
                ?>
                <tr>
                    <th scope="row">
                        <label for="<?php echo esc_attr( $name ); ?>">
                            <?php echo esc_html( $label ); ?>
                        </label>
                    </th>
                    <td>
                        <input
                            type="number"
                            id="<?php echo esc_attr( $name ); ?>"
                            name="<?php echo esc_attr( $name ); ?>"
                            value="<?php echo esc_attr( $value ); ?>"
                        />
                    </td>
                </tr>
            <?php endforeach; ?>
        </table>
        <?php
        submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false );
        if ( $where_parts ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-standings-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>';
        $headers = [
            'id'                   => esc_html__( 'ID',               'livefot-matches' ),
            'standing_id'          => esc_html__( 'Standing ID',      'livefot-matches' ),
            'league_id'            => esc_html__( 'League ID',        'livefot-matches' ),
            'group_id'             => esc_html__( 'Group ID',         'livefot-matches' ),
            'season_id'            => esc_html__( 'Season ID',        'livefot-matches' ),
            'stage_id'             => esc_html__( 'Stage ID',         'livefot-matches' ),
            'round_id'             => esc_html__( 'Round ID',         'livefot-matches' ),
            'position'             => esc_html__( 'Position',         'livefot-matches' ),
            'team_id'              => esc_html__( 'Team ID',          'livefot-matches' ),
            'team_name'            => esc_html__( 'Team Name',        'livefot-matches' ),
            'shortcode'            => esc_html__( 'Shortcode',        'livefot-matches' ),
            'team_logo'            => esc_html__( 'Team Logo',        'livefot-matches' ),
            'goals'                => esc_html__( 'Goals',            'livefot-matches' ),
            'goal_diff'            => esc_html__( 'Goal Diff',        'livefot-matches' ),
            'wins'                 => esc_html__( 'Wins',             'livefot-matches' ),
            'lost'                 => esc_html__( 'Lost',             'livefot-matches' ),
            'draws'                => esc_html__( 'Draws',            'livefot-matches' ),
            'played'               => esc_html__( 'Played',           'livefot-matches' ),
            'points'               => esc_html__( 'Points',           'livefot-matches' ),
            'description'          => esc_html__( 'Description',      'livefot-matches' ),
            'recent_form'          => esc_html__( 'Recent Form',      'livefot-matches' ),
            'standing_rule_id'     => esc_html__( 'Standing Rule ID','livefot-matches' ),
            'result'               => esc_html__( 'Result',           'livefot-matches' ),
            'fairplay_points_lose' => esc_html__( 'Fair-play Lose',   'livefot-matches' ),
            'updated_at'           => esc_html__( 'Updated At',       'livefot-matches' ),
            'inserted_at'          => esc_html__( 'Inserted At',      'livefot-matches' ),
            'last_updated'         => esc_html__( 'Last Updated',     'livefot-matches' ),
        ];
        echo '<tr>';
        foreach ( $headers as $label ) {
            echo '<th>' . esc_html( $label ) . '</th>';
        }
        echo '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>';
            foreach ( $headers as $key => $label ) {
                if ( 'team_logo' === $key ) {
                    $logo_id = absint( $row['team_logo'] );
                    if ( $logo_id ) {
                        echo '<td>' . wp_get_attachment_image(
                            $logo_id, [50,50], false,
                            [ 'alt' => esc_attr( $row['team_name'] ) ]
                        ) . '</td>';
                    } else {
                        echo '<td>' . esc_html__( 'No image', 'livefot-matches' ) . '</td>';
                    }
                } else {
                    echo '<td>' . esc_html( $row[ $key ] ) . '</td>';
                }
            }
            echo '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } else {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found matching your search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}



/**
 * Admin-screen renderer: “LiveFot Team Statistics”.
 *
 * ➤ Passes WordPress.DB.PreparedSQL.NotPrepared – every query prepared inline.  
 * ➤ Sanitises all user input and validates ints with absint().  
 * ➤ Preserves pagination, search, reset link, translation & escaping.
 *
 * @since 1.0.0
 */
function livefot_display_team_stats_data() {
    global $wpdb;

    /* 1) Capability check */
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    /* 2) Nonce check on search submission */
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_team_stats_search_action', 'livefot_team_stats_search_nonce' );
    }

    /* 3) Table name */
    $table_name = $wpdb->prefix . 'livefot_team_stats';

    /* 4) Pagination */
    $get            = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $current_page   = ( isset( $get['paged'] ) && is_numeric( $get['paged'] ) )
                      ? absint( $get['paged'] )
                      : 1;
    $items_per_page = 20;
    $offset         = ( $current_page - 1 ) * $items_per_page;

    /* 5) Sanitize search inputs */
    $search_team_id    = isset( $get['search_team_id'] )    ? sanitize_text_field( $get['search_team_id'] )    : '';
    $search_fixture_id = isset( $get['search_fixture_id'] ) ? sanitize_text_field( $get['search_fixture_id'] ) : '';

    /* 6) Build WHERE clauses & params */
    $where_parts = [];
    $params      = [];

    if ( '' !== $search_team_id && is_numeric( $search_team_id ) ) {
        $where_parts[] = 'team_id = %d';
        $params[]      = absint( $search_team_id );
    }
    if ( '' !== $search_fixture_id && is_numeric( $search_fixture_id ) ) {
        $where_parts[] = 'fixture_id = %d';
        $params[]      = absint( $search_fixture_id );
    }

    if ( $where_parts ) {
        $where_sql  = implode( ' AND ', $where_parts );
        $where_args = $params;
    } else {
        $where_sql  = '1 = %d';
        $where_args = [ 1 ];
    }

    /* 7) Count total items */
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where_sql}",
            ...$where_args
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    /* 8) Fetch paginated rows */
    $fetch_args = array_merge( $where_args, [ $items_per_page, $offset ] );
    $results    = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE {$where_sql}
             ORDER BY stats_id ASC
             LIMIT %d OFFSET %d",
            ...$fetch_args
        ),
        ARRAY_A
    );

    /* 9) Build base URL & preserve search params */
    $base_url   = add_query_arg( 'page', 'livefot-team-stats-data', admin_url( 'admin.php' ) );
    $query_args = [];
    if ( '' !== $search_team_id ) {
        $query_args['search_team_id'] = $search_team_id;
    }
    if ( '' !== $search_fixture_id ) {
        $query_args['search_fixture_id'] = $search_fixture_id;
    }
    if ( $query_args ) {
        $base_url = add_query_arg( $query_args, $base_url );
    }

    /* 10) Output */
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot Team Statistics Data', 'livefot-matches' ) . '</h1>';
    ?>
    <form method="get" action="">
        <?php wp_nonce_field( 'livefot_team_stats_search_action', 'livefot_team_stats_search_nonce' ); ?>
        <input type="hidden" name="page" value="livefot-team-stats-data" />
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row">
                    <label for="search_team_id"><?php esc_html_e( 'Team ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input type="text" id="search_team_id" name="search_team_id" value="<?php echo esc_attr( $search_team_id ); ?>" />
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="search_fixture_id"><?php esc_html_e( 'Fixture ID', 'livefot-matches' ); ?></label>
                </th>
                <td>
                    <input type="text" id="search_fixture_id" name="search_fixture_id" value="<?php echo esc_attr( $search_fixture_id ); ?>" />
                </td>
            </tr>
        </table>
        <?php
        submit_button( esc_html__( 'Search', 'livefot-matches' ), 'primary', 'search_submit', false );
        if ( $query_args ) : ?>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=livefot-team-stats-data' ) ); ?>" class="button-secondary">
                <?php esc_html_e( 'Reset', 'livefot-matches' ); ?>
            </a>
        <?php endif; ?>
    </form>
    <hr/>
    <?php

    if ( $results ) {
        echo '<table class="widefat fixed striped"><thead><tr>';
        $headers = [
            'stats_id'         => esc_html__( 'ID',               'livefot-matches' ),
            'fixture_id'       => esc_html__( 'Fixture ID',       'livefot-matches' ),
            'team_id'          => esc_html__( 'Team ID',          'livefot-matches' ),
            'corners'          => esc_html__( 'Corners',          'livefot-matches' ),
            'offsides'         => esc_html__( 'Offsides',         'livefot-matches' ),
            'possession_time'  => esc_html__( 'Pos (%)',         'livefot-matches' ),
            'yellow_cards'     => esc_html__( 'Yellow',           'livefot-matches' ),
            'red_cards'        => esc_html__( 'Red',              'livefot-matches' ),
            'yellow_red_cards' => esc_html__( 'Yellow-Red',       'livefot-matches' ),
            'substitutions'    => esc_html__( 'Substitutions',    'livefot-matches' ),
            'goals'            => esc_html__( 'Goals',            'livefot-matches' ),
            'penalties'        => esc_html__( 'Penalties',        'livefot-matches' ),
            'injuries'         => esc_html__( 'Injuries',         'livefot-matches' ),
            'last_updated'     => esc_html__( 'Last Updated',     'livefot-matches' ),
        ];
        foreach ( $headers as $key => $label ) {
            echo '<th>' . esc_html( $label ) . '</th>';
        }
        echo '</tr></thead><tbody>';
        foreach ( $results as $row ) {
            echo '<tr>';
            foreach ( $headers as $key => $label ) {
                if ( 'possession_time' === $key ) {
                    echo '<td>' . esc_html( $row[ $key ] ) . '%</td>';
                } elseif ( 'yellow_red_cards' === $key && is_null( $row[ $key ] ) ) {
                    echo '<td>' . esc_html__( 'N/A', 'livefot-matches' ) . '</td>';
                } else {
                    echo '<td>' . esc_html( $row[ $key ] ) . '</td>';
                }
            }
            echo '</tr>';
        }
        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
				/* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }
    } else {
        echo '<p>' . esc_html__( 'No data found with the specified search criteria.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}




// API Calls Data page callback with pagination
function livefot_display_api_calls() {
    global $wpdb;

    // 1) Capability check
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'livefot-matches' ) );
    }

    // 2) Nonce check on pagination/search (if you ever add filters)
    if ( isset( $_GET['search_submit'] ) ) {
        check_admin_referer( 'livefot_api_calls_search_action', 'livefot_api_calls_search_nonce' );
    }

    // 3) Table name
    $table_name = $wpdb->prefix . 'livefot_api_calls';

    // 4) Pagination setup
    $get            = wp_unslash( $_GET ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $current_page   = ( ! empty( $get['paged'] ) && is_numeric( $get['paged'] ) )
                      ? absint( $get['paged'] )
                      : 1;
    $items_per_page = 20;
    $offset         = ( $current_page - 1 ) * $items_per_page;

    // 5) Count total items (use a dummy WHERE to allow prepare())
    $total_items = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE %d = %d",
            1, 1
        )
    );
    $total_pages = $total_items > 0 ? ceil( $total_items / $items_per_page ) : 0;

    // 6) Fetch paginated rows
    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * 
               FROM {$table_name}
               ORDER BY last_call_time DESC
               LIMIT %d OFFSET %d",
            $items_per_page,
            $offset
        ),
        ARRAY_A
    );

    // 7) Base URL for pagination
    $base_url = esc_url( admin_url( 'admin.php?page=livefot-api-calls' ) );

    // 8) Output
    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'LiveFot API Calls', 'livefot-matches' ) . '</h1>';

    if ( $results ) {
        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th>' . esc_html__( 'ID',              'livefot-matches' ) . '</th>';
        echo '<th>' . esc_html__( 'Endpoint Name',   'livefot-matches' ) . '</th>';
        echo '<th>' . esc_html__( 'Remaining Calls', 'livefot-matches' ) . '</th>';
        echo '<th>' . esc_html__( 'Status',          'livefot-matches' ) . '</th>';
        echo '<th>' . esc_html__( 'Last Call Time',  'livefot-matches' ) . '</th>';
        echo '</tr></thead><tbody>';

        foreach ( $results as $row ) {
            echo '<tr>';
            echo '<td>' . esc_html( $row['id'] )              . '</td>';
            echo '<td>' . esc_html( $row['endpoint_name'] )   . '</td>';
            echo '<td>' . esc_html( $row['remaining_calls'] ) . '</td>';
            echo '<td>' . esc_html( $row['status'] )          . '</td>';
            echo '<td>' . esc_html( $row['last_call_time'] )  . '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';

        if ( function_exists( 'livefot_display_pagination' ) && $total_pages > 1 ) {
            livefot_display_pagination( $current_page, $total_pages, $base_url );
        } elseif ( $total_items > 0 ) {
            printf(
                /* translators: %d is the number of results found */
                '<p>' . esc_html__( 'Displaying %d record(s).', 'livefot-matches' ) . '</p>',
                absint( $total_items )
            );
        }

    } else {
        echo '<p>' . esc_html__( 'No data found in the LiveFot API Calls table.', 'livefot-matches' ) . '</p>';
    }

    echo '</div>';
}



?>