<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// phpcs:set WordPress.WP.I18n text_domain livechatai

class LiveChatAI {

    // Production URL for LiveChatAI
    private $app_url = 'https://app.livechatai.com';

    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
        add_shortcode('livechatai', array($this, 'render_shortcode'));
        add_action('widgets_init', array($this, 'register_widget'));
        add_action('init', array($this, 'register_gutenberg_block'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_block_editor_assets'));
        add_action('wp_footer', array($this, 'add_messenger_script'));
        // Add settings link to plugins page (use dynamic plugin basename)
        $plugin_basename = plugin_basename(dirname(__DIR__) . '/livechatai.php');
        add_filter('plugin_action_links_' . $plugin_basename, array($this, 'add_settings_link'));
    }

    /**
     * Add a Settings link to the plugin action links on the plugins page
     */
    public function add_settings_link($links) {
        // phpcs:ignore WordPress.WP.I18n.TextDomainMismatch -- Dev folder slug differs; actual slug is 'livechatai' when packaged.
        $settings_link = '<a href="' . esc_url( admin_url('options-general.php?page=livechatai') ) . '">' . esc_html__('Settings', 'livechatai') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    public function enqueue_admin_styles($hook) {
        // Only load on our plugin page
        if ($hook !== 'settings_page_livechatai') {
            return;
        }
        
        wp_enqueue_style(
            'livechatai-admin-styles',
            LIVECHATAI_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            LIVECHATAI_VERSION
        );
        
        wp_enqueue_script(
            'livechatai-admin-script',
            LIVECHATAI_PLUGIN_URL . 'assets/js/admin.js',
            array(),
            LIVECHATAI_VERSION,
            true
        );
    }

    public function run() {
        // Additional initialization if needed
    }

    public function register_settings() {
        register_setting(
            'livechatai_options', 
            'livechatai_chatbot_id',
            array(
                'type'              => 'string',
                // Sanitize and validate on save to prevent storing quoted/escaped IDs
                'sanitize_callback' => function( $value ) {
                    $sanitized = LiveChatAI_Security::sanitize_chatbot_id( $value );
                    return LiveChatAI_Security::validate_chatbot_id( $sanitized ) ? $sanitized : '';
                },
                'default'           => ''
            )
        );
        
        add_settings_section(
            'livechatai_main_section',
            'LiveChatAI Configuration',
            array($this, 'settings_section_callback'),
            'livechatai'
        );

        add_settings_field(
            'livechatai_chatbot_id',
            'Chatbot ID',
            array($this, 'chatbot_id_field_callback'),
            'livechatai',
            'livechatai_main_section'
        );
    }

    public function settings_section_callback() {
        echo '<p>Enter your LiveChatAI chatbot ID to automatically add a messenger widget to all pages of your website.</p>';
        echo '<p>Don\'t have a chatbot yet? <a href="https://app.livechatai.com" target="_blank">Create one here</a> - it only takes a few minutes!</p>';
    }

    public function chatbot_id_field_callback() {
        $chatbot_id = get_option('livechatai_chatbot_id');
        echo '<input type="text" id="livechatai_chatbot_id" name="livechatai_chatbot_id" value="' . esc_attr($chatbot_id) . '" class="regular-text" placeholder="e.g., YOUR_DATA_ID" />';
        echo '<p class="description">Find your chatbot ID in <strong>LiveChatAI Dashboard → Your Chatbot → Embed & Integrate → Add as Messenger</strong>. Look for the <code>data-id</code> value in the script.</p>';
    }

    public function add_admin_menu() {
        add_options_page(
            'LiveChatAI Settings',
            'LiveChatAI',
            'manage_options',
            'livechatai',
            array($this, 'settings_page')
        );
    }

    public function settings_page() {
        $chatbot_id = get_option('livechatai_chatbot_id');
        $is_configured = !empty($chatbot_id);
        
        $updated_flag = filter_input( INPUT_GET, 'settings-updated', FILTER_UNSAFE_RAW );
        if ( current_user_can( 'manage_options' ) && $updated_flag && '1' === sanitize_text_field( $updated_flag ) ) {
            // phpcs:ignore WordPress.WP.I18n.TextDomainMismatch -- Dev folder slug differs; actual slug is 'livechatai' when packaged.
            add_settings_error('livechatai_messages', 'livechatai_message', esc_html__( 'Settings saved successfully!', 'livechatai' ), 'updated');
        }

        settings_errors('livechatai_messages');
        ?>
        
        <div class="livechatai-admin-wrap livechatai-fade-in">
            <!-- Header Section -->
            <div class="livechatai-header">
                <?php echo wp_get_attachment_image( 
                    attachment_url_to_postid( esc_url( LIVECHATAI_PLUGIN_URL . 'assets/images/logo-black.png' ) ), 
                    'medium', 
                    false, 
                    array( 'alt' => 'LiveChatAI' )
                ); ?>
                <?php if ( ! wp_get_attachment_image( attachment_url_to_postid( esc_url( LIVECHATAI_PLUGIN_URL . 'assets/images/logo-black.png' ) ) ) ): ?>
                    <img src="<?php echo esc_url( LIVECHATAI_PLUGIN_URL . 'assets/images/logo-black.png' ); ?>" 
                         alt="LiveChatAI" />
                <?php endif; ?>
                <h1>AI chatbot for WordPress</h1>
                <p>Resolve 70% of customer support queries instantly.<br>Save time for your team and customers with AI-powered answers.</p>
            </div>

            <!-- Status Alert -->
            <?php if ($is_configured): ?>
                <div class="livechatai-alert livechatai-alert-success">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <div>
                        <strong>Your LiveChatAI is Active!</strong>
                        <p>The messenger widget is now live on your website. Visitors can start chatting with your AI assistant.</p>
                    </div>
                </div>
            <?php else: ?>
                <div class="livechatai-alert livechatai-alert-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <div>
                        <strong>Setup Required</strong>
                        <p>Enter your chatbot ID below to activate the AI messenger on your website.</p>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Quick Setup Card -->
            <div style="max-width: 600px; margin: 0 auto 40px;">
                <div class="livechatai-card">
                    <div class="livechatai-card-header" style="text-align: center;">
                        <h2>
                            Quick Setup
                        </h2>
                        <p>Connect your LiveChatAI chatbot in just a few seconds</p>
                    </div>
                    <div class="livechatai-card-body">
                        <form action="options.php" method="post">
                            <?php settings_fields('livechatai_options'); ?>
                            
                            <div class="livechatai-form-group">
                                <label for="livechatai_chatbot_id">Chatbot ID</label>
                                <input type="text" 
                                       id="livechatai_chatbot_id" 
                                       name="livechatai_chatbot_id" 
                                       value="<?php echo esc_attr($chatbot_id); ?>" 
                                       placeholder="e.g., YOUR_DATA_ID" />
                                <div class="livechatai-form-help">
                                    Find your ID in <strong><a href="https://app.livechatai.com" target="_blank">LiveChatAI Dashboard</a> → Embed & Integrate → Add as Messenger</strong>. 
                                    Look for the <code>data-id</code> value in the script.
                                </div>
                            </div>

                            <div class="livechatai-form-group">
                                <button type="submit" class="livechatai-btn livechatai-btn-primary">
                                    <span class="dashicons dashicons-saved"></span>
                                    Save Settings
                                </button>
                            </div>
                        </form>

                        <!-- Status Display -->
                        <div style="margin-top: 24px; padding-top: 24px; border-top: 1px solid var(--livechatai-gray-200);">
                            <h4 style="margin: 0 0 12px 0; color: var(--livechatai-gray-700);">Status:</h4>
                            <?php if ($is_configured): ?>
                                <div class="livechatai-status livechatai-status-active">
                                    <span class="livechatai-status-dot"></span>
                                    Live & Active
                                </div>
                            <?php else: ?>
                                <div class="livechatai-status livechatai-status-inactive">
                                    <span class="livechatai-status-dot"></span>
                                    Not Configured
                                </div>
                            <?php endif; ?>
                                                </div>
                    </div>
                </div>
            </div>

            <!-- Implementation Guide -->
            <div class="livechatai-card" style="margin-top: 60px;">
                <div class="livechatai-card-header">
                    <h2>
                        How to Get Your Chatbot ID
                    </h2>
                    <p>Follow these simple steps to connect your AI assistant</p>
                </div>
                <div class="livechatai-card-body">
                    <div class="livechatai-steps">
                        <div class="livechatai-step">
                            <h3>Create or Access Your Chatbot</h3>
                            <p>Login to <a href="https://app.livechatai.com" target="_blank">LiveChatAI Dashboard</a>. If you don't have a chatbot yet, click <strong>"Create New Chatbot"</strong> and follow the setup wizard. Configure your AI assistant with your knowledge base and customize its responses.</p>
                        </div>

                        <div class="livechatai-step">
                            <h3>Navigate to Embed & Integrate</h3>
                            <p>Once your chatbot is ready, click on the <strong>"Embed & Integrate"</strong> tab in the top navigation menu. This section contains all the integration options for your AI assistant.</p>
                        </div>

                        <div class="livechatai-step">
                            <h3>Find the Messenger Script</h3>
                            <p>Look for the <strong>"Add as Messenger"</strong> section. You'll see a code snippet that looks like this:</p>
                            <div class="livechatai-code-block">
&lt;script
  src="https://app.livechatai.com/embed.js"
  data-id="YOUR_DATA_ID"
  async defer&gt;
&lt;/script&gt;
                            </div>
                        </div>

                        <div class="livechatai-step">
                            <h3>Copy Your Chatbot ID</h3>
                            <p>Copy the value from the <code>data-id</code> attribute (the text between the quotes). In the example above, the Chatbot ID is <code>YOUR_DATA_ID</code>. Paste this ID into the field above and save your settings.</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Footer -->
            <div class="livechatai-footer">
                <p>
                    Need help? Visit our <a href="https://help.livechatai.com" target="_blank">Help Center</a> or 
                    <a href="https://livechatai.com/contact" target="_blank">contact support</a>. 
                    Happy chatting! 🎉
                </p>
            </div>
        </div>
        
        <?php
    }

    public function render_shortcode($atts) {
        $atts = shortcode_atts(array(
            'chatbot_id' => get_option('livechatai_chatbot_id'),
        ), $atts);

        if (empty($atts['chatbot_id'])) {
            return '<div style="padding: 10px; background: #f0f0f0; border: 1px solid #ddd; border-radius: 4px;">
                        <strong>LiveChatAI:</strong> Missing chatbot ID. Please configure your chatbot in 
                        <a href="' . esc_url( admin_url('options-general.php?page=livechatai') ) . '">plugin settings</a>.
                    </div>';
        }

        return '<!-- LiveChatAI Messenger will load automatically -->';
    }

    public function add_messenger_script() {
        $chatbot_id = get_option('livechatai_chatbot_id');
        
        if (!empty($chatbot_id)) {
            // Enqueue the embed script without version query string
            wp_enqueue_script(
                'livechatai-messenger',
                'https://app.livechatai.com/embed.js',
                array(),
                null,
                true
            );

            // Ensure the required data-id attribute is present on the script tag at load time
            // so the embed can initialize correctly, and add async/defer attributes.
            // Sanitize the ID to remove any quotes or invalid characters.
            $chatbot_id_sanitized = LiveChatAI_Security::sanitize_chatbot_id( $chatbot_id );
            add_filter(
                'script_loader_tag',
                function( $tag, $handle, $src ) use ( $chatbot_id_sanitized ) {
                    if ( 'livechatai-messenger' !== $handle ) {
                        return $tag;
                    }

                    $data_id = esc_attr( $chatbot_id_sanitized );

                    // Inject attributes (data-id, async, defer, id) into the existing enqueued script tag
                    $tag = preg_replace(
                        '/<script(\s+[^>]*src=\"[^\"]+\")(.*)>/i',
                        '<script$1 data-id=\'' . $data_id . '\' async defer id=\'livechatai-messenger-js\'$2>',
                        $tag,
                        1
                    );

                    return $tag;
                },
                10,
                3
            );
        }
    }

    public function register_widget() {
        register_widget('LiveChatAI_Widget');
    }

    public function register_gutenberg_block() {
        // Register block only if Gutenberg is available
        if (function_exists('register_block_type')) {
            register_block_type('livechatai/embed', array(
                'editor_script' => 'livechatai-block-editor',
                'render_callback' => array($this, 'render_gutenberg_block'),
            ));
        }
    }

    public function enqueue_block_editor_assets() {
        wp_enqueue_script(
            'livechatai-block-editor',
            LIVECHATAI_PLUGIN_URL . 'blocks/livechatai-block.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'),
            LIVECHATAI_VERSION,
            true
        );
    }

    public function render_gutenberg_block($attributes) {
        $chatbot_id = !empty($attributes['chatbotId']) ? $attributes['chatbotId'] : get_option('livechatai_chatbot_id');
        
        return $this->render_shortcode(array(
            'chatbot_id' => $chatbot_id
        ));
    }
}

class LiveChatAI_Widget extends WP_Widget {

    public function __construct() {
        parent::__construct(
            'livechatai_widget',
            'LiveChatAI',
            array('description' => 'Embed LiveChatAI chatbot in your sidebar')
        );
    }

    public function widget($args, $instance) {
        echo wp_kses_post( $args['before_widget'] );
        
        if (!empty($instance['title'])) {
            echo wp_kses_post( $args['before_title'] ) . esc_html( apply_filters('widget_title', $instance['title']) ) . wp_kses_post( $args['after_title'] );
        }
        
        $chatbot_id = !empty($instance['chatbot_id']) ? $instance['chatbot_id'] : get_option('livechatai_chatbot_id');
        
        if (!empty($chatbot_id)) {
            echo '<p>LiveChatAI messenger widget is active on this site.</p>';
        } else {
            echo '<p>Please configure your LiveChatAI chatbot in <a href="' . esc_url( admin_url('options-general.php?page=livechatai') ) . '">plugin settings</a>.</p>';
        }
        
        echo wp_kses_post( $args['after_widget'] );
    }

    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $chatbot_id = !empty($instance['chatbot_id']) ? $instance['chatbot_id'] : get_option('livechatai_chatbot_id');
        ?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>">Widget Title:</label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>" 
                   name="<?php echo esc_attr($this->get_field_name('title')); ?>" 
                   type="text" value="<?php echo esc_attr($title); ?>" 
                   placeholder="Optional title">
        </p>
        
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('chatbot_id')); ?>">Chatbot ID:</label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('chatbot_id')); ?>" 
                   name="<?php echo esc_attr($this->get_field_name('chatbot_id')); ?>" 
                   type="text" value="<?php echo esc_attr($chatbot_id); ?>" 
                   placeholder="Leave empty to use default">
            <small>Leave empty to use the default chatbot ID from settings.</small>
        </p>
        
        <p><strong>Note:</strong> This widget shows messenger chat widget status. The actual chat appears automatically on all pages once configured.</p>
        <?php
    }

    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? sanitize_text_field($new_instance['title']) : '';
        $instance['chatbot_id'] = (!empty($new_instance['chatbot_id'])) ? sanitize_text_field($new_instance['chatbot_id']) : '';
        return $instance;
    }
} 