<?php
/**
 * LiveChatAI WordPress Plugin - Security Class
 * 
 * Handles security, validation, and sanitization for the plugin.
 * 
 * @package LiveChatAI
 * @version 1.0.0
 * @license GPL-2.0+
 */

/*
LiveChatAI WordPress Plugin
Copyright (C) 2024 LiveChatAI Team

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// phpcs:set WordPress.WP.I18n text_domain livechatai

/**
 * Security and validation class for LiveChatAI plugin
 */
class LiveChatAI_Security {

    /**
     * Validate chatbot ID format
     *
     * @param string $chatbot_id The chatbot ID to validate
     * @return bool True if valid, false otherwise
     */
    public static function validate_chatbot_id( $chatbot_id ) {
        // Remove whitespace
        $chatbot_id = trim( $chatbot_id );
        
        // Allow empty for removal
        if ( empty( $chatbot_id ) ) {
            return true;
        }
        
        // Check format: alphanumeric, hyphens, underscores, reasonable length
        if ( ! preg_match( '/^[a-zA-Z0-9_-]{10,50}$/', $chatbot_id ) ) {
            return false;
        }
        
        // Additional security checks
        if ( self::contains_suspicious_patterns( $chatbot_id ) ) {
            return false;
        }
        
        return true;
    }

    /**
     * Sanitize chatbot ID
     *
     * @param string $chatbot_id The chatbot ID to sanitize
     * @return string Sanitized chatbot ID
     */
    public static function sanitize_chatbot_id( $chatbot_id ) {
        // Decode HTML entities (e.g., &quot; to ") and then trim whitespace
        $chatbot_id = trim( html_entity_decode( $chatbot_id, ENT_QUOTES | ENT_HTML5, 'UTF-8' ) );

        // Remove backslashes that might precede quotes or entities
        $chatbot_id = str_replace('\\', '', $chatbot_id);

        // Trim surrounding quotes if present
        $chatbot_id = trim( $chatbot_id, "\"'" );
        
        // Allow only alphanumeric characters, hyphens, and underscores
        $chatbot_id = preg_replace( '/[^a-zA-Z0-9_-]/', '', $chatbot_id );
        
        // Limit length
        $chatbot_id = substr( $chatbot_id, 0, 50 );
        
        return $chatbot_id;
    }

    /**
     * Check for suspicious patterns in input
     *
     * @param string $input The input to check
     * @return bool True if suspicious patterns found
     */
    private static function contains_suspicious_patterns( $input ) {
        $suspicious_patterns = array(
            '/javascript:/i',
            '/data:/i',
            '/vbscript:/i',
            '/<script/i',
            '/<iframe/i',
            '/<object/i',
            '/<embed/i',
            '/on\w+\s*=/i', // Event handlers like onclick=
            '/expression\s*\(/i',
            '/url\s*\(/i',
        );
        
        foreach ( $suspicious_patterns as $pattern ) {
            if ( preg_match( $pattern, $input ) ) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Verify nonce for admin forms
     *
     * @param string $action The action to verify
     * @param string $name The nonce name (optional)
     * @return bool True if valid nonce
     */
    public static function verify_nonce( $action, $name = '_wpnonce' ) {
        return wp_verify_nonce( 
            isset( $_REQUEST[ $name ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $name ] ) ) : '', 
            $action 
        );
    }

    /**
     * Check if user has required capabilities
     *
     * @param string $capability The capability to check
     * @return bool True if user has capability
     */
    public static function user_can( $capability = 'manage_options' ) {
        return current_user_can( $capability );
    }

    /**
     * Sanitize and validate settings before saving
     *
     * @param array $settings The settings to validate
     * @return array Validated and sanitized settings
     */
    public static function validate_settings( $settings ) {
        $validated = array();
        
        if ( isset( $settings['livechatai_chatbot_id'] ) ) {
            $chatbot_id = self::sanitize_chatbot_id( $settings['livechatai_chatbot_id'] );
            
            if ( self::validate_chatbot_id( $chatbot_id ) ) {
                $validated['livechatai_chatbot_id'] = $chatbot_id;
            } else {
                // Log invalid attempt only in debug mode
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    /**
                     * Fires when an invalid chatbot ID is attempted to be saved.
                     *
                     * @param string $attempted_id The attempted chatbot ID value.
                     */
                    do_action( 'livechatai_security_log', 'invalid_chatbot_id', array( 'attempted_id' => $settings['livechatai_chatbot_id'] ) );
                }
                
                // Return empty to prevent saving invalid data
                $validated['livechatai_chatbot_id'] = '';
                
                // Add admin notice
                add_action( 'admin_notices', array( __CLASS__, 'show_validation_error' ) );
            }
        }
        
        return $validated;
    }

    /**
     * Show validation error notice
     */
    public static function show_validation_error() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p>
                <?php // phpcs:ignore WordPress.WP.I18n.TextDomainMismatch -- Dev folder slug differs; actual slug is 'livechatai' when packaged. ?>
                <strong><?php esc_html_e( 'LiveChatAI Error:', 'livechatai' ); ?></strong>
                <?php // phpcs:ignore WordPress.WP.I18n.TextDomainMismatch -- Dev folder slug differs; actual slug is 'livechatai' when packaged. ?>
                <?php esc_html_e( 'Invalid chatbot ID format. Please enter a valid ID from your LiveChatAI dashboard.', 'livechatai' ); ?>
            </p>
        </div>
        <?php
    }

    /**
     * Escape output for safe display
     *
     * @param string $output The output to escape
     * @param string $context The context (html, attr, js, etc.)
     * @return string Escaped output
     */
    public static function escape_output( $output, $context = 'html' ) {
        switch ( $context ) {
            case 'attr':
                return esc_attr( $output );
            case 'js':
                return esc_js( $output );
            case 'url':
                return esc_url( $output );
            case 'textarea':
                return esc_textarea( $output );
            case 'html':
            default:
                return esc_html( $output );
        }
    }

    /**
     * Rate limiting for API-like operations
     *
     * @param string $key The rate limit key
     * @param int $limit Number of attempts allowed
     * @param int $window Time window in seconds
     * @return bool True if within rate limit
     */
    public static function check_rate_limit( $key, $limit = 10, $window = 300 ) {
        $transient_key = 'livechatai_rate_limit_' . md5( $key );
        $attempts = get_transient( $transient_key );
        
        if ( false === $attempts ) {
            // First attempt
            set_transient( $transient_key, 1, $window );
            return true;
        }
        
        if ( $attempts >= $limit ) {
            return false;
        }
        
        // Increment attempts
        set_transient( $transient_key, $attempts + 1, $window );
        return true;
    }

    /**
     * Log security events
     *
     * @param string $event The security event
     * @param array $context Additional context
     */
    public static function log_security_event( $event, $context = array() ) {
        if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG ) {
            return;
        }
        
        $log_entry = array(
            'timestamp' => current_time( 'mysql' ),
            'event' => $event,
            'user_id' => get_current_user_id(),
            'ip' => self::get_client_ip(),
            'user_agent' => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
            'context' => $context,
        );
        
        /**
         * Fires when a LiveChatAI security event occurs.
         *
         * @param array $log_entry The structured log entry data.
         */
        do_action( 'livechatai_security_event', $log_entry );
    }

    /**
     * Get client IP address
     *
     * @return string Client IP address
     */
    private static function get_client_ip() {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );
        
        foreach ( $ip_headers as $header ) {
            if ( ! empty( $_SERVER[ $header ] ) ) {
                $ip = sanitize_text_field( wp_unslash( $_SERVER[ $header ] ) );
                
                // Handle comma-separated IPs
                if ( strpos( $ip, ',' ) !== false ) {
                    $ip = trim( explode( ',', $ip )[0] );
                }
                
                // Validate IP
                if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) ) {
                    return $ip;
                }
            }
        }
        
        return '0.0.0.0';
    }
} 