/**
 * LiveChatAI WordPress Plugin - Admin JavaScript
 * Simple and clean functionality
 * 
 * @package LiveChatAI
 * @version 1.0.0
 * @license GPL-2.0+
 */

(function($) {
    'use strict';

    /**
     * Initialize the LiveChatAI admin interface
     */
    function initLiveChatAI() {
        initChatbotIdValidation();
        initFormEnhancements();
        initCopyButtons();
        
        console.log('LiveChatAI Admin: Initialized');
    }

    /**
     * Validate chatbot ID format
     */
    function initChatbotIdValidation() {
        const $chatbotIdField = $('#livechatai_chatbot_id');
        
        if ($chatbotIdField.length === 0) return;

        let validationTimeout;

        $chatbotIdField.on('input', function() {
            const $field = $(this);
            const value = $field.val().trim();
            
            clearTimeout(validationTimeout);
            $field.removeClass('valid invalid');
            $('.livechatai-validation-message').remove();
            
            if (value === '') return;

            validationTimeout = setTimeout(function() {
                validateChatbotId($field, value);
            }, 500);
        });
    }

    /**
     * Validate chatbot ID format
     */
    function validateChatbotId($field, value) {
        const isValid = /^[a-zA-Z0-9_-]{10,50}$/.test(value);
        
        $field.removeClass('valid invalid');
        $('.livechatai-validation-message').remove();
        
        if (isValid) {
            $field.addClass('valid');
            $field.after('<div class="livechatai-validation-message" style="color: #00a32a;">✓ Format looks correct</div>');
        } else {
            $field.addClass('invalid');
            $field.after('<div class="livechatai-validation-message" style="color: #f56e28;">⚠ Please check your chatbot ID</div>');
        }
    }

    /**
     * Enhance form interactions
     */
    function initFormEnhancements() {
        // Add loading state to save button
        $('form').on('submit', function() {
            const $submitBtn = $(this).find('.livechatai-btn-primary');
            
            if ($submitBtn.length) {
                $submitBtn
                    .prop('disabled', true)
                    .html('Saving...');
            }
        });

        // Auto-trim whitespace
        $('#livechatai_chatbot_id').on('blur', function() {
            $(this).val($(this).val().trim());
        });
    }

    /**
     * Add copy buttons for code examples
     */
    function initCopyButtons() {
        $('.livechatai-code-block').each(function() {
            const $codeBlock = $(this);
            const code = $codeBlock.text();
            
            const $copyBtn = $('<button type="button" style="position: absolute; top: 4px; right: 4px; background: rgba(0,0,0,0.1); border: none; padding: 4px 6px; border-radius: 3px; font-size: 11px; cursor: pointer;">Copy</button>');
            
            $codeBlock.css('position', 'relative').append($copyBtn);
            
            $copyBtn.on('click', function() {
                copyToClipboard(code, $copyBtn);
            });
        });
    }

    /**
     * Copy text to clipboard
     */
    function copyToClipboard(text, $button) {
        if (navigator.clipboard) {
            navigator.clipboard.writeText(text).then(function() {
                showCopySuccess($button);
            }).catch(function() {
                fallbackCopyTextToClipboard(text, $button);
            });
        } else {
            fallbackCopyTextToClipboard(text, $button);
        }
    }

    /**
     * Fallback copy method
     */
    function fallbackCopyTextToClipboard(text, $button) {
        const textArea = document.createElement("textarea");
        textArea.value = text;
        textArea.style.position = "fixed";
        textArea.style.left = "-999999px";
        
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();
        
        try {
            const successful = document.execCommand('copy');
            if (successful) {
                showCopySuccess($button);
            }
        } catch (err) {
            console.error('Copy failed:', err);
        }
        
        document.body.removeChild(textArea);
    }

    /**
     * Show copy success feedback
     */
    function showCopySuccess($button) {
        const originalText = $button.text();
        $button.text('Copied!').css('background', 'rgba(0, 163, 42, 0.2)');
        
        setTimeout(function() {
            $button.text(originalText).css('background', 'rgba(0,0,0,0.1)');
        }, 1500);
    }

    /**
     * Initialize when document is ready
     */
    $(document).ready(function() {
        if ($('.livechatai-admin-wrap').length > 0) {
            initLiveChatAI();
        }
    });

})(jQuery); 