<?php
/**
 * Main WC Font Selector Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class Live_Font_Preview {
    
    public function __construct() {
        // Frontend hooks
        add_action('woocommerce_before_add_to_cart_button', array($this, 'display_font_selector'));
        add_filter('woocommerce_add_cart_item_data', array($this, 'add_cart_item_data'), 10, 3);
        add_filter('woocommerce_get_item_data', array($this, 'display_cart_item_data'), 10, 2);
        add_action('woocommerce_checkout_create_order_line_item', array($this, 'add_order_item_meta'), 10, 4);
        
        // Assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        
        // Admin
        if (is_admin()) {
            require_once LIVE_FONT_PREVIEW_PLUGIN_DIR . 'includes/class-admin.php';
            new Live_Font_Preview_Admin();
        }
    }
    
    /**
     * Check if should display
     */
    private function should_display() {
        $enable_setting = get_option('live_font_preview_enable_all', 'yes');
        
        if ($enable_setting === 'yes') {
            return true;
        }
        
        global $product;
        if (!$product || !is_a($product, 'WC_Product')) {
            return false;
        }
        
        // Check if explicitly enabled for this specific product
        $enabled_products = get_option('live_font_preview_products', array());
        if (!empty($enabled_products) && in_array($product->get_id(), array_map('intval', $enabled_products))) {
            return true;
        }

        // Check if enabled for product category
        $enabled_categories = get_option('live_font_preview_categories', array());
        if (!empty($enabled_categories)) {
            $product_categories = wp_get_post_terms($product->get_id(), 'product_cat', array('fields' => 'ids'));
            if (!empty(array_intersect($enabled_categories, $product_categories))) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Display font selector
     */
    public function display_font_selector() {
        if (!$this->should_display()) {
            return;
        }
        
        $fonts = get_option('live_font_preview_fonts', array(
            'Arial', 'Georgia', 'Times New Roman', 'Helvetica', 'Verdana',
            'Open Sans', 'Roboto', 'Montserrat', 'Playfair Display', 'Dancing Script'
        ));
        
        $title = get_option('live_font_preview_title', 'Customize Your Text');
        $text_label = get_option('live_font_preview_text_label', 'Your Text');
        $font_label = get_option('live_font_preview_font_label', 'Choose Font');
        $preview_label = get_option('live_font_preview_preview_label', 'Preview');
        $text_placeholder = get_option('live_font_preview_text_placeholder', 'Enter your custom text here...');
        $preview_text = get_option('live_font_preview_preview_text', 'This is how your text will look');
        $max_length = get_option('live_font_preview_max_text_length', 100);
        
        ?>

        <div class="live-font-preview-for-woocommerce-wrapper">
            <h4><?php echo esc_html($title); ?></h4>
            
            <div class="live-font-preview-for-woocommerce-fields">
                <?php wp_nonce_field('live_font_preview_add_to_cart', 'live_font_preview_nonce'); ?>
                <div class="live-font-preview-for-woocommerce-field">
                    <label for="live_font_preview_text"><?php echo esc_html($text_label); ?> <span class="required">*</span></label>
                    <input type="text" 
                           id="live_font_preview_text" 
                           name="live_font_preview_text" 
                           class="live-font-preview-for-woocommerce-text-input"
                           placeholder="<?php echo esc_attr($text_placeholder); ?>"
                           maxlength="<?php echo esc_attr($max_length); ?>"
                           required>
                    <small>Enter your custom text, maximum <?php echo esc_html($max_length); ?> characters</small>
                </div>
                
                <div class="live-font-preview-for-woocommerce-field">
                    <label for="live_font_preview_font"><?php echo esc_html($font_label); ?> <span class="required">*</span></label>
                    <select id="live_font_preview_font" name="live_font_preview_font" class="live-font-preview-for-woocommerce-font-select" required>
                        <option value="">Select a font...</option>
                        <?php foreach ($fonts as $font): ?>
                            <option value="<?php echo esc_attr($font); ?>"><?php echo esc_html($font); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <small>Choose the font style for your custom text</small>
                </div>
            </div>
            
            <div class="live-font-preview-for-woocommerce-preview-section">
                <label><?php echo esc_html($preview_label); ?></label>
                <div class="live-font-preview-for-woocommerce-preview" id="live_font_preview_preview">
                    <?php echo esc_html($preview_text); ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Add to cart
     */
    public function add_cart_item_data($cart_item_data, $product_id, $variation_id) {
        if (!empty($_POST['live_font_preview_text']) && !empty($_POST['live_font_preview_font'])) {
            // Verify Nonce
            if (!isset($_POST['live_font_preview_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['live_font_preview_nonce'])), 'live_font_preview_add_to_cart')) {
                return $cart_item_data; // Fail silently or handle error
            }
            
            $cart_item_data['live_font_preview_text'] = sanitize_text_field(wp_unslash($_POST['live_font_preview_text']));
            $cart_item_data['live_font_preview_font'] = sanitize_text_field(wp_unslash($_POST['live_font_preview_font']));
            $cart_item_data['unique_key'] = md5(microtime() . wp_rand());
        }
        return $cart_item_data;
    }
    
    /**
     * Display in cart
     */
    public function display_cart_item_data($item_data, $cart_item) {
        if (isset($cart_item['live_font_preview_text'])) {
            $text_label = get_option('live_font_preview_cart_text_label', 'Custom Text');
            $font_label = get_option('live_font_preview_cart_font_label', 'Font');
            
            $item_data[] = array(
                'key' => $text_label,
                'value' => $cart_item['live_font_preview_text'],
                'display' => ''
            );
            
            $item_data[] = array(
                'key' => $font_label,
                'value' => $cart_item['live_font_preview_font'],
                'display' => ''
            );
        }
        return $item_data;
    }
    
    /**
     * Add to order
     */
    public function add_order_item_meta($item, $cart_item_key, $values, $order) {
        if (isset($values['live_font_preview_text'])) {
            $text_label = get_option('live_font_preview_order_text_label', 'Custom Text');
            $font_label = get_option('live_font_preview_order_font_label', 'Font');
            
            $item->add_meta_data($text_label, $values['live_font_preview_text']);
            $item->add_meta_data($font_label, $values['live_font_preview_font']);
        }
    }
    
    /**
     * Enqueue assets
     */
    public function enqueue_frontend_assets() {
        if (!is_product()) {
            return;
        }
        
        wp_enqueue_style(
            'live-font-preview-for-woocommerce-frontend',
            LIVE_FONT_PREVIEW_ASSETS_URL . 'css/frontend.css',
            array(),
            LIVE_FONT_PREVIEW_VERSION
        );
        
        wp_enqueue_script(
            'live-font-preview-for-woocommerce-frontend',
            LIVE_FONT_PREVIEW_ASSETS_URL . 'js/frontend.js',
            array('jquery'),
            LIVE_FONT_PREVIEW_VERSION,
            true
        );
    }
}