<?php
/**
 * Admin functionality
 *
 * @package Live_Font_Preview
 * @since 1.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Live_Font_Preview_Admin Class
 */
class Live_Font_Preview_Admin {
    
    /**
     * Settings instance
     */
    private $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        // Include settings class first
        if (!class_exists('Live_Font_Preview_Settings')) {
            require_once LIVE_FONT_PREVIEW_PLUGIN_DIR . 'includes/class-settings.php';
        }
        
        $this->settings = new Live_Font_Preview_Settings();
        
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));

        add_action('admin_init', array($this, 'handle_settings_save'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('Live Font Preview Settings', 'live-font-preview-for-woocommerce'),
            __('Live Font Preview', 'live-font-preview-for-woocommerce'),
            'manage_options',
            'live-font-preview-for-woocommerce',
            array($this, 'display_settings_page')
        );
    }
    
    /**
     * Handle settings save
     */
    public function handle_settings_save() {
        // Check if this is our settings page and form was submitted
        if (!isset($_POST['live_font_preview_save']) || !isset($_POST['live_font_preview_nonce'])) {
            return;
        }
        
        // Verify nonce
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['live_font_preview_nonce'])), 'live_font_preview_save_settings')) {
            wp_die(esc_html__('Security check failed.', 'live-font-preview-for-woocommerce'));
        }
        
        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'live-font-preview-for-woocommerce'));
        }
        
        // Process and save settings
        $this->process_settings_form();
        
        // Redirect with success message
        $redirect_url = add_query_arg(
            array(
                'page' => 'live-font-preview-for-woocommerce',
                'settings-updated' => 'true'
            ),
            admin_url('admin.php')
        );
        
        wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function admin_enqueue_scripts($hook) {
        // Only load on our settings page
        if ('woocommerce_page_live-font-preview-for-woocommerce' !== $hook) {
            return;
        }
        
        // Enqueue WooCommerce Admin Styles
        wp_enqueue_style('woocommerce_admin_styles');

        // Enqueue WooCommerce Admin Scripts for Select2
        // Enqueue WooCommerce Admin Scripts for Select2
        wp_enqueue_script('wc-enhanced-select');

        // Enqueue our admin CSS
        wp_enqueue_style(
            'live-font-preview-for-woocommerce-admin',
            LIVE_FONT_PREVIEW_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            LIVE_FONT_PREVIEW_VERSION
        );
        
        // Enqueue our admin JavaScript
        wp_enqueue_script(
            'live-font-preview-for-woocommerce-admin',
            LIVE_FONT_PREVIEW_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery', 'wc-enhanced-select'),
            LIVE_FONT_PREVIEW_VERSION,
            true
        );
        
        // Localize script
        wp_localize_script('live-font-preview-for-woocommerce-admin', 'live_font_preview_admin_params', array(
            'nonce' => wp_create_nonce('live_font_preview_admin_nonce'),
        ));
    }
    
    /**
     * Display settings page
     */
    public function display_settings_page() {
        $this->settings->display();
    }
    
    /**
     * Process settings form
     */
    private function process_settings_form() {
        // Get POST data
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_settings_save caller
        $posted_data = $_POST;

        // Define all settings fields
        $settings_fields = array(
            'live_font_preview_enable_all' => 'sanitize_text_field',
            'live_font_preview_categories' => 'array_map_int',
            'live_font_preview_products' => 'array_map_int',
            'live_font_preview_fonts' => 'array_map_text',
            'live_font_preview_title' => 'sanitize_text_field',
            'live_font_preview_text_label' => 'sanitize_text_field',
            'live_font_preview_text_placeholder' => 'sanitize_text_field',
            'live_font_preview_font_label' => 'sanitize_text_field',
            'live_font_preview_preview_label' => 'sanitize_text_field',
            'live_font_preview_preview_text' => 'sanitize_text_field',
            'live_font_preview_error_text' => 'sanitize_text_field',
            'live_font_preview_error_font' => 'sanitize_text_field',
            'live_font_preview_cart_text_label' => 'sanitize_text_field',
            'live_font_preview_cart_font_label' => 'sanitize_text_field',
            'live_font_preview_order_text_label' => 'sanitize_text_field',
            'live_font_preview_order_font_label' => 'sanitize_text_field'
        );
        
        foreach ($settings_fields as $field => $sanitize_callback) {
            if (isset($posted_data[$field])) {
                $value = wp_unslash($posted_data[$field]);
                
                // Apply appropriate sanitization
                switch ($sanitize_callback) {
                    case 'array_map_int':
                        $value = is_array($value) ? array_map('intval', $value) : array();
                        break;
                    case 'array_map_text':
                        $value = is_array($value) ? array_map('sanitize_text_field', $value) : array();
                        // Remove empty values
                        $value = array_filter($value);
                        break;
                    default:
                        $value = sanitize_text_field($value);
                        break;
                }
                
                update_option($field, $value);
            } else {
                // Handle unchecked checkboxes or empty multi-selects
                if ($field === 'live_font_preview_enable_all') {
                    update_option($field, 'no');
                } elseif (in_array($sanitize_callback, array('array_map_int', 'array_map_text'))) {
                    // If an array field is missing from POST, it means it was cleared
                    update_option($field, array());
                }
            }
        }
        
        // Clear any cached data
        wp_cache_flush();
    }
    

}