/**
 * Admin JavaScript for WooCommerce Font Selector Plugin
 */

(function ($) {
    'use strict';

    // Tab functionality
    function initTabs() {
        console.log('WC Font Selector: Initializing tabs...');

        // Make sure we have the right elements
        if (!$('.live-font-preview-for-woocommerce-tabs').length) {
            console.log('WC Font Selector: Tab container not found');
            return;
        }

        // Initially hide all tab content except the first
        $('.tab-content').removeClass('active').hide();
        $('#general').addClass('active').show();

        // Handle tab clicks
        $(document).off('click.wcfstabs').on('click.wcfstabs', '.nav-tab', function (e) {
            e.preventDefault();
            e.stopPropagation();

            var $this = $(this);
            var target = $this.attr('href') || '#' + $this.data('tab');

            // Clean up target
            if (target.indexOf('#') !== 0) {
                target = '#' + target;
            }

            console.log('Tab clicked:', target);

            // Update nav tabs
            $('.nav-tab').removeClass('nav-tab-active');
            $this.addClass('nav-tab-active');

            // Hide all content and show target
            $('.tab-content').removeClass('active').hide();
            $(target).addClass('active').show();

            // Save active tab
            localStorage.setItem('live_font_preview_active_tab', target);

            console.log('Tab switched to:', target);
        });

        // Restore active tab from hash or localStorage
        var activeTab = window.location.hash || localStorage.getItem('live_font_preview_active_tab');
        if (activeTab && $(activeTab).length) {
            $('a[href="' + activeTab + '"], a[data-tab="' + activeTab.replace('#', '') + '"]').trigger('click');
        }

        console.log('WC Font Selector: Tabs initialized');
    }

    // Conditional fields functionality
    function initConditionalFields() {
        function toggleConditionalFields() {
            var enableAll = $('#live_font_preview_enable_all').is(':checked');
            if (enableAll) {
                $('.live-font-preview-for-woocommerce-conditional').fadeOut();
            } else {
                $('.live-font-preview-for-woocommerce-conditional').fadeIn();
            }
        }

        // Initial toggle
        toggleConditionalFields();

        // Handle change
        $(document).on('change', '#live_font_preview_enable_all', toggleConditionalFields);
    }

    // Font management functionality
    function initFontManager() {
        // Add font functionality
        $(document).on('click', '#add-font-btn', function (e) {
            e.preventDefault();

            var fontName = $('#new-font-name').val().trim();
            if (!fontName) {
                alert('Please enter a font name');
                return;
            }

            // Check if font already exists
            var exists = false;
            $('.font-name').each(function () {
                if ($(this).text().toLowerCase() === fontName.toLowerCase()) {
                    exists = true;
                    return false;
                }
            });

            if (exists) {
                alert('This font already exists');
                return;
            }

            // Add font to list
            var fontHtml = '<div class="live-font-preview-for-woocommerce-font-item">' +
                '<input type="hidden" name="live_font_preview_fonts[]" value="' + fontName + '" />' +
                '<span class="font-name" style="font-family: \'' + fontName + '\', sans-serif;">' + fontName + '</span>' +
                '<button type="button" class="button remove-font" title="Remove Font">' +
                '<span class="dashicons dashicons-trash"></span>' +
                '</button>' +
                '</div>';

            $('.live-font-preview-for-woocommerce-fonts-list').append(fontHtml);
            $('#new-font-name').val('');

            // Load the font
            loadGoogleFont(fontName);
        });

        // Remove font functionality
        $(document).on('click', '.remove-font', function (e) {
            e.preventDefault();

            if (confirm('Are you sure you want to remove this font?')) {
                $(this).closest('.live-font-preview-for-woocommerce-font-item').fadeOut(300, function () {
                    $(this).remove();
                });
            }
        });

        // Enter key on font input
        $(document).on('keypress', '#new-font-name', function (e) {
            if (e.which === 13) {
                e.preventDefault();
                $('#add-font-btn').trigger('click');
            }
        });
    }

    // Load Google Font
    function loadGoogleFont(fontName) {
        if (!fontName) return;

        var fontUrl = 'https://fonts.googleapis.com/css2?family=' +
            fontName.replace(/\s+/g, '+') + ':wght@400;700&display=swap';

        if ($('link[href*="' + fontName.replace(/\s+/g, '+') + '"]').length === 0) {
            $('<link>', {
                rel: 'stylesheet',
                type: 'text/css',
                href: fontUrl
            }).appendTo('head');
        }
    }



    // Initialize everything when document is ready
    $(document).ready(function () {
        console.log('WC Font Selector: Document ready');

        // Check if we're on the right page
        if (!$('.live-font-preview-for-woocommerce-settings').length) {
            console.log('WC Font Selector: Not on settings page');
            return;
        }

        console.log('WC Font Selector: Initializing...');

        // Initialize components
        initTabs();
        initConditionalFields();
        initFontManager();



        // Load existing fonts
        $('.font-name').each(function () {
            loadGoogleFont($(this).text());
        });

        console.log('WC Font Selector: Initialization complete');
    });

})(jQuery);