<?php
/*
Plugin Name: Live Chat Inserter
Plugin URI: 
Description: Insert a validated Live Chat script into the footer of all or selected pages.
Version: 1.0.0
Author: gongcj
Author URI: 
Text Domain: live-chat-inserter
Domain Path: /languages
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Register plugin settings with sanitization callbacks.
 */
function livechatins_register_settings()
{
    register_setting('livechatins_settings_group', 'livechatins_base_url', [
        'sanitize_callback' => 'esc_url_raw',
    ]);
    register_setting('livechatins_settings_group', 'livechatins_channel_number', [
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('livechatins_settings_group', 'livechatins_insert_mode', [
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('livechatins_settings_group', 'livechatins_selected_pages', [
        'sanitize_callback' => 'livechatins_sanitize_selected_pages',
    ]);
    register_setting('livechatins_settings_group', 'livechatins_enable_script', [
        'sanitize_callback' => 'sanitize_text_field',
    ]);
}
add_action('admin_init', 'livechatins_register_settings');

/**
 * Sanitize selected pages input, ensuring it is an array of positive integers.
 */
function livechatins_sanitize_selected_pages($input)
{
    if (!is_array($input)) {
        return [];
    }
    return array_map('absint', $input);
}

/**
 * Add plugin options page to the WordPress admin menu.
 */
function livechatins_add_admin_menu()
{
    add_options_page(
        esc_html__('Live Chat Inserter', 'live-chat-inserter'),
        esc_html__('Live Chat Inserter', 'live-chat-inserter'),
        'manage_options',
        'live-chat-inserter',
        'livechatins_settings_page'
    );
}
add_action('admin_menu', 'livechatins_add_admin_menu');

/**
 * Display the plugin settings page form.
 */
function livechatins_settings_page()
{
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Live Chat Inserter Settings', 'live-chat-inserter'); ?></h1>
        <?php settings_errors(); ?>
        <form method="post" action="options.php">
            <?php
            settings_fields('livechatins_settings_group');
            do_settings_sections('livechatins_settings_group');

            $base_url = get_option('livechatins_base_url', '');
            $channel_number = get_option('livechatins_channel_number', '');
            $insert_mode = get_option('livechatins_insert_mode', 'all');
            $selected_pages = (array) get_option('livechatins_selected_pages', []);
            $pages = get_pages(['post_status' => 'publish']);
            ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Base URL', 'live-chat-inserter'); ?></th>
                    <td>
                        <input type="text" name="livechatins_base_url" value="<?php echo esc_attr($base_url); ?>"
                            size="60" />
                        <br /><small><?php esc_html_e('Example: https://hello.test.smartpbx.cn', 'live-chat-inserter'); ?></small>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Channel Number', 'live-chat-inserter'); ?></th>
                    <td>
                        <input type="text" name="livechatins_channel_number"
                            value="<?php echo esc_attr($channel_number); ?>" size="30" />
                        <br /><small><?php esc_html_e('Example: LC00001', 'live-chat-inserter'); ?></small>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Insert Mode', 'live-chat-inserter'); ?></th>
                    <td>
                        <label>
                            <input type="radio" name="livechatins_insert_mode" value="all" <?php checked($insert_mode, 'all'); ?> id="livechatins_mode_all" />
                            <?php esc_html_e('All Pages', 'live-chat-inserter'); ?>
                        </label><br />
                        <label>
                            <input type="radio" name="livechatins_insert_mode" value="selected" <?php checked($insert_mode, 'selected'); ?> id="livechatins_mode_selected" />
                            <?php esc_html_e('Only Selected Pages', 'live-chat-inserter'); ?>
                        </label>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Select Pages', 'live-chat-inserter'); ?></th>
                    <td>
                        <div id="livechatins_pages_list"
                            style="max-height: 300px; overflow-y: auto; border: 1px solid #ccc; padding: 10px;">
                            <?php foreach ($pages as $page): ?>
                                <label style="display: block; margin-bottom: 5px;">
                                    <input type="checkbox" name="livechatins_selected_pages[]"
                                        value="<?php echo esc_attr($page->ID); ?>" <?php checked(in_array($page->ID, $selected_pages)); ?> />
                                    <?php echo esc_html($page->post_title); ?>
                                </label>
                            <?php endforeach; ?>
                        </div>
                        <small><?php esc_html_e('Only applies if "Only Selected Pages" is selected above.', 'live-chat-inserter'); ?></small>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Enable Script Insertion', 'live-chat-inserter'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="livechatins_enable_script" value="1" <?php checked(get_option('livechatins_enable_script'), '1'); ?> />
                            <?php esc_html_e('Enable script injection on the front end', 'live-chat-inserter'); ?>
                        </label>
                    </td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

/**
 * Enqueue the live chat script on the front end based on settings.
 */
function livechatins_enqueue_script()
{
    if (get_option('livechatins_enable_script') !== '1') {
        return;
    }

    $base_url = get_option('livechatins_base_url', '');
    $channel_number = get_option('livechatins_channel_number', '');
    $insert_mode = get_option('livechatins_insert_mode', 'all');
    $selected_pages = (array) get_option('livechatins_selected_pages', []);

    if (!$base_url || !$channel_number) {
        return;
    }

    $final_url = rtrim($base_url, '/') . '/live_chat.v1.0.0.js?channelNumber=' . urlencode($channel_number);

    if ($insert_mode === 'all' || (is_page() && in_array(get_the_ID(), $selected_pages, true))) {
        wp_enqueue_script('livechatins-custom-script', $final_url, [], '1.0.0', true);
    }
}
add_action('wp_enqueue_scripts', 'livechatins_enqueue_script');

/**
 * Cleanup plugin options upon uninstall.
 */
function livechatins_cleanup_settings()
{
    delete_option('livechatins_base_url');
    delete_option('livechatins_channel_number');
    delete_option('livechatins_insert_mode');
    delete_option('livechatins_selected_pages');
    delete_option('livechatins_enable_script');
}
register_uninstall_hook(__FILE__, 'livechatins_cleanup_settings');
