/**
 * LISA AI Chat Widget
 */

(function () {
    'use strict';

    // Widget state
    var state = {
        isOpen: false,
        conversationId: null,
        isLoading: false,
        conversationHistory: [],
        lastProducts: JSON.parse(localStorage.getItem('lisa_last_products') || '[]'),
        productOffset: parseInt(localStorage.getItem('lisa_product_offset') || '0', 10),
        messageTimestamp: 0,
        messageCount: 0,
        lastMessageTime: 0,
        hasDragged: false
    };

    // DOM elements
    var elements = {};

    // Initialize widget
    function init() {
        // Get DOM elements
        elements.widget = document.getElementById('lisa-ai-widget');
        elements.chatButton = document.getElementById('lisa-chat-button');
        elements.chatWindow = document.getElementById('lisa-chat-window');
        elements.chatClose = document.getElementById('lisa-chat-close');
        elements.chatMessages = document.getElementById('lisa-chat-messages');
        elements.chatInput = document.getElementById('lisa-chat-input');
        elements.chatSend = document.getElementById('lisa-chat-send');
        elements.iconChat = elements.chatButton.querySelector('.lisa-icon-chat');
        elements.iconClose = elements.chatButton.querySelector('.lisa-icon-close');
        elements.chatReset = document.getElementById('lisa-chat-reset');
        elements.chatInfo = document.getElementById('lisa-chat-info');
        elements.chatSize = document.getElementById('lisa-chat-size');
        elements.chatHeader = document.querySelector('.lisa-chat-header');

        if (!elements.widget) return;

        // Bind events
        elements.chatButton.addEventListener('click', function (e) {
            if (e.target.classList.contains('lisa-button-close')) {
                // Click on × hides pill for session
                elements.chatButton.style.display = 'none';
                sessionStorage.setItem('lisa_widget_hidden', 'true');
                return;
            }
            toggleChat();
        });
        elements.chatClose.addEventListener('click', closeChat);
        elements.chatReset.addEventListener('click', resetConversation);
        elements.chatInfo.addEventListener('click', showInfoPopup);
        elements.chatSize.addEventListener('click', toggleChatSize);
        elements.chatSend.addEventListener('click', sendMessage);
        elements.chatInput.addEventListener('keypress', function (e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });

        // Record timestamp when user starts typing (for bot timing check)
        elements.chatInput.addEventListener('focus', function () {
            state.messageTimestamp = Date.now();
        });

        // Draggable widget (matching Shopify pattern)
        var isDragging = false;
        var dragStartX, dragStartY, startRight, startBottom;

        elements.chatHeader.addEventListener('mousedown', function (e) {
            isDragging = true;
            state.hasDragged = false;
            dragStartX = e.clientX;
            dragStartY = e.clientY;

            var rect = elements.widget.getBoundingClientRect();
            startRight = window.innerWidth - rect.right;
            startBottom = window.innerHeight - rect.bottom;

            elements.widget.style.transition = 'none';
            e.preventDefault();
        });

        document.addEventListener('mousemove', function (e) {
            if (!isDragging) return;

            var deltaX = dragStartX - e.clientX;
            var deltaY = dragStartY - e.clientY;

            if (Math.abs(deltaX) > 5 || Math.abs(deltaY) > 5) {
                state.hasDragged = true;
            }

            var newRight = startRight + deltaX;
            var newBottom = startBottom + deltaY;

            // Keep within viewport
            newRight = Math.max(10, Math.min(newRight, window.innerWidth - 100));
            newBottom = Math.max(10, Math.min(newBottom, window.innerHeight - 100));

            elements.widget.style.position = 'fixed';
            elements.widget.style.right = newRight + 'px';
            elements.widget.style.bottom = newBottom + 'px';
            elements.widget.style.left = 'auto';
            elements.widget.style.top = 'auto';
        });

        document.addEventListener('mouseup', function () {
            if (isDragging) {
                isDragging = false;
                elements.widget.style.transition = '';
                if (state.hasDragged) {
                    sessionStorage.setItem('lisa_widget_position', JSON.stringify({
                        right: parseInt(elements.widget.style.right),
                        bottom: parseInt(elements.widget.style.bottom)
                    }));
                }
            }
        });

        // Restore saved position
        var savedPosition = sessionStorage.getItem('lisa_widget_position');
        if (savedPosition) {
            var pos = JSON.parse(savedPosition);
            elements.widget.style.position = 'fixed';
            elements.widget.style.right = pos.right + 'px';
            elements.widget.style.bottom = pos.bottom + 'px';
            elements.widget.style.left = 'auto';
            elements.widget.style.top = 'auto';
        }

        // Load conversation from local storage
        var savedConversation = localStorage.getItem('lisa_conversation_id');
        
        if (savedConversation) {
            state.conversationId = savedConversation;
        }

        // Load conversation history
        state.conversationHistory = loadConversation();

        // Auto-open chat on page load (unless user dismissed it)
        if (!sessionStorage.getItem('lisa_widget_hidden')) {
            openChat();
        }
    }

    // Toggle chat window
    function toggleChat() {
        if (state.hasDragged) {
            state.hasDragged = false;
            return;
        }
        if (state.isOpen) {
            closeChat();
        } else {
            openChat();
        }
    }

    // Open chat window
    function openChat() {
        state.isOpen = true;
        elements.chatWindow.style.display = 'flex';
        elements.chatButton.style.display = 'none';
        elements.chatInput.focus();

        // Show welcome message or restore previous conversation
        if (elements.chatMessages.children.length === 0) {
            if (state.conversationHistory.length > 0) {
                // Restore previous messages
                state.conversationHistory.forEach(function (msg) {
                    if (msg.role === 'user') {
                        addMessage(msg.content, 'user');
                    } else {
                        var displayText = escapeHtml(msg.content);
                        // Re-linkify product titles
                        if (msg.products && msg.products.length > 0) {
                            msg.products.forEach(function (product) {
                                if (product.title) {
                                    var productUrl = product.url || (product.handle ? window.location.origin + '/product/' + product.handle + '/' : null);
                                    if (productUrl) {
                                        var regex = new RegExp(escapeHtml(product.title).replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi');
                                        displayText = displayText.replace(regex, '<a href="' + escapeHtml(productUrl) + '" target="_blank" style="color: #0066cc; text-decoration: underline;">' + escapeHtml(product.title) + '</a>');
                                    }
                                }
                            });
                            addMessage(displayText, 'assistant', true);
                            addProductCards(msg.products);
                        } else {
                            addMessage(displayText, 'assistant', true);
                        }
                    }
                });
            } else {
                addMessage(lisaAiSettings.welcomeMessage, 'assistant');
            }
        }
    }

    // Reset conversation (trash button)
    function resetConversation() {
        state.conversationHistory = [];
        state.lastProducts = [];
        state.productOffset = 0;
        localStorage.removeItem('lisa_conversation');
        localStorage.removeItem('lisa_last_products');
        localStorage.removeItem('lisa_product_offset');

        // Clear all messages from DOM
        while (elements.chatMessages.firstChild) {
            elements.chatMessages.removeChild(elements.chatMessages.firstChild);
        }

        // Show fresh welcome message
        addMessage(lisaAiSettings.welcomeMessage, 'assistant');
        elements.chatInput.focus();
    }

    // Close chat window
    function closeChat() {
        state.isOpen = false;
        elements.chatWindow.style.display = 'none';
        elements.chatButton.style.display = 'flex';

        // Reset widget position to default
        elements.widget.style.right = '20px';
        elements.widget.style.bottom = '20px';
        elements.widget.style.left = 'auto';
        elements.widget.style.top = 'auto';
        sessionStorage.removeItem('lisa_widget_position');
    }

    // Show info popup
    function showInfoPopup() {
        var overlay = document.createElement('div');
        overlay.id = 'lisa-info-overlay';
        overlay.style.cssText = 'position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.5);z-index:9999999;display:flex;align-items:center;justify-content:center;';

        var popup = document.createElement('div');
        popup.style.cssText = 'background:white;padding:24px;border-radius:16px;max-width:320px;margin:20px;box-shadow:0 8px 32px rgba(0,0,0,0.3);text-align:center;';
        popup.innerHTML = '<div style="font-size:32px;margin-bottom:12px;">💬</div>' +
            '<div style="font-weight:600;font-size:16px;margin-bottom:12px;color:#333;">Conversation Memory</div>' +
            '<div style="font-size:14px;line-height:1.6;color:#666;margin-bottom:20px;">Your conversation stays open while you browse for a better shopping experience. It clears automatically when you close the browser or after 15 minutes of inactivity.</div>' +
            '<button id="lisa-info-close" style="background:#1a1a1a;color:white;border:none;padding:10px 24px;border-radius:20px;cursor:pointer;font-size:14px;">Got it</button>';

        overlay.appendChild(popup);
        document.body.appendChild(overlay);

        overlay.addEventListener('click', function (e) {
            if (e.target === overlay || e.target.id === 'lisa-info-close') {
                overlay.remove();
            }
        });
    }

    // Toggle chat size (expand/collapse)
    function toggleChatSize() {
        state.isExpanded = !state.isExpanded;
        if (state.isExpanded) {
            elements.chatWindow.style.height = '600px';
            elements.chatSize.textContent = '⬇️';
            elements.chatSize.title = 'Collapse';
        } else {
            elements.chatWindow.style.height = '500px';
            elements.chatSize.textContent = '⬆️';
            elements.chatSize.title = 'Expand';
        }
        scrollToBottom();
    }

    // Send message
    function sendMessage() {
        var message = elements.chatInput.value.trim();

        if (!message || state.isLoading) return;

        // Rate limiting - max 10 messages per minute
        var now = Date.now();
        if (now - state.lastMessageTime < 2000) {
            return; // Ignore messages sent within 2 seconds
        }
        state.lastMessageTime = now;

        // Message cap - max 5 user messages per conversation
        var MAX_USER_MESSAGES = 5;
        var userMessageCount = state.conversationHistory.filter(function (msg) {
            return msg.role === 'user';
        }).length;
        if (userMessageCount >= MAX_USER_MESSAGES) {
            addMessage("You've reached the message limit for this conversation. Click the 🗑️ button above to start a fresh chat!", 'assistant');
            return;
        }

        // Add user message to chat
        addMessage(message, 'user');
        elements.chatInput.value = '';

        // Record in conversation history
        state.conversationHistory.push({ role: 'user', content: message });
        saveConversation();

        // Reset offset for new searches, keep for "show me more"
        var showMorePhrases = ['show more', 'show me more', 'more products', 'more options', 'see more', 'next', 'what else', 'any more', 'anything else'];
        if (showMorePhrases.indexOf(message.toLowerCase()) === -1) {
            state.productOffset = 0;
            localStorage.setItem('lisa_product_offset', '0');
        }

        // Show typing indicator
        state.isLoading = true;
        showTypingIndicator();

        // Call LISA backend directly for streaming
        var requestBody = {
            shop: window.location.hostname,
            user_id: 'woo_' + lisaAiSettings.storeId,
            message: message,
            conversation_history: state.conversationHistory.map(function (msg) {
                return { role: msg.role, content: msg.content, products: msg.products || [] };
            }),
            product_offset: state.productOffset,
            context: { page: window.location.pathname, referrer: document.referrer },
            _hp: '',
            _ts: state.messageTimestamp
        };

        // Send to backend streaming endpoint
        fetch(lisaAiSettings.backendUrl + '/lisa/query/stream', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': 'Bearer ' + lisaAiSettings.accessToken
            },
            body: JSON.stringify(requestBody)
        })
            .then(function (response) {
                var contentType = response.headers.get('content-type');

                if (contentType && contentType.includes('text/event-stream')) {
                    // Handle streaming response - show progressively
                    var reader = response.body.getReader();
                    var decoder = new TextDecoder();
                    var fullText = '';
                    var products = [];
                    var streamMsgDiv = null;
                    var streamTextSpan = null;

                    function readStream() {
                        reader.read().then(function (result) {
                            if (result.done) {
                                if (fullText && !state.streamDone) {
                                    hideTypingIndicator();
                                    state.isLoading = false;
                                    if (!streamMsgDiv) {
                                        showStreamedResponse(fullText, products);
                                    }
                                    state.conversationHistory.push({ role: 'assistant', content: fullText, products: products });
                                    saveConversation();
                                }
                                return;
                            }

                            var chunk = decoder.decode(result.value, { stream: true });
                            var lines = chunk.split('\n');

                            for (var i = 0; i < lines.length; i++) {
                                var line = lines[i];
                                if (line.indexOf('data: ') === 0) {
                                    try {
                                        var data = JSON.parse(line.substring(6));

                                        if (data.type === 'products') {
                                            products = data.products || [];
                                            state.productOffset += products.length;
                                            localStorage.setItem('lisa_product_offset', state.productOffset.toString());
                                            if (products.length > 0) {
                                                state.lastProducts = products;
                                                localStorage.setItem('lisa_last_products', JSON.stringify(products));
                                                // Show products immediately
                                                hideTypingIndicator();
                                                addProductCards(products);
                                                // Create message div for streaming text
                                                streamMsgDiv = document.createElement('div');
                                                streamMsgDiv.className = 'lisa-message lisa-message-assistant';
                                                var nameLabel = document.createElement('div');
                                                nameLabel.className = 'lisa-assistant-name';
                                                nameLabel.textContent = lisaAiSettings.assistantName || 'LISA';
                                                streamMsgDiv.appendChild(nameLabel);
                                                streamTextSpan = document.createElement('span');
                                                streamMsgDiv.appendChild(streamTextSpan);
                                                elements.chatMessages.appendChild(streamMsgDiv);
                                                scrollToBottom();
                                            }
                                        } else if (data.type === 'text_chunk') {
                                            fullText += data.chunk;
                                            // Show text progressively
                                            if (streamTextSpan) {
                                                streamTextSpan.textContent = fullText;
                                                scrollToBottom();
                                            }
                                        } else if (data.type === 'done') {
                                            state.isLoading = false;
                                            state.streamDone = true;
                                            // Final update with product links
                                            if (streamMsgDiv && streamTextSpan) {
                                                var displayText = escapeHtml(fullText);
                                                var productsToLink = products.length > 0 ? products : state.lastProducts;
                                                if (productsToLink && productsToLink.length > 0) {
                                                    productsToLink.forEach(function (product) {
                                                        if (product.title) {
                                                            var productUrl = product.url || (product.handle ? window.location.origin + '/product/' + product.handle + '/' : null);
                                                            if (productUrl) {
                                                                var regex = new RegExp(escapeHtml(product.title).replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi');
                                                                displayText = displayText.replace(regex, '<a href="' + escapeHtml(productUrl) + '" target="_blank" style="color: #0066cc; text-decoration: underline;">' + escapeHtml(product.title) + '</a>');
                                                            }
                                                        }
                                                    });
                                                }
                                                streamTextSpan.innerHTML = displayText;
                                            } else {
                                                showStreamedResponse(fullText, products);
                                            }
                                            state.conversationHistory.push({ role: 'assistant', content: fullText, products: products });
                                            saveConversation();
                                        }
                                    } catch (e) {
                                        // Skip parse errors
                                    }
                                }
                            }

                            readStream();
                        }).catch(function (error) {
                            hideTypingIndicator();
                            state.isLoading = false;
                            console.error('LISA AI Stream Error:', error);
                        });
                    }

                    state.streamDone = false;
                    readStream();

                } else {
                    // Fallback to regular JSON response
                    response.json().then(function (data) {
                        hideTypingIndicator();
                        state.isLoading = false;

                        var reply = data.reply || data.response || '';
                        var dataProducts = data.products || [];

                        if (reply) {
                            showStreamedResponse(reply, dataProducts);
                        }

                        if (dataProducts.length > 0) {
                            addProductCards(dataProducts);
                        }
                    });
                }
            })
            .catch(function (error) {
                hideTypingIndicator();
                state.isLoading = false;
                addMessage('Sorry, something went wrong. Please try again.', 'assistant');
                console.error('LISA AI Error:', error);
            });
    }

    // Show streamed response with product title links
    function showStreamedResponse(text, products) {
        // Record assistant reply in conversation history
        state.conversationHistory.push({ role: 'assistant', content: text, products: products || [] });
        saveConversation();

        var displayText = escapeHtml(text);

        // Use current products if available, otherwise fall back to last products
        var productsToLink = (products && products.length > 0) ? products : state.lastProducts;

        if (productsToLink && productsToLink.length > 0) {
            productsToLink.forEach(function (product) {
                if (product.title) {
                    var productUrl = product.url || (product.handle ? window.location.origin + '/product/' + product.handle + '/' : null);
                    if (productUrl) {
                        var regex = new RegExp(escapeHtml(product.title).replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi');
                        displayText = displayText.replace(regex, '<a href="' + escapeHtml(productUrl) + '" target="_blank" style="color: #0066cc; text-decoration: underline;">' + escapeHtml(product.title) + '</a>');
                    }
                }
            });
        }

        addMessage(displayText, 'assistant', true);
    }

    // Add message to chat
    function addMessage(text, sender, useHtml) {
        var messageDiv = document.createElement('div');
        messageDiv.className = 'lisa-message lisa-message-' + sender;
        if (sender === 'assistant') {
            var nameLabel = document.createElement('div');
            nameLabel.className = 'lisa-assistant-name';
            nameLabel.textContent = lisaAiSettings.assistantName || 'LISA';
            messageDiv.appendChild(nameLabel);
        }
        if (sender === 'user' && lisaAiSettings.fontColor) {
            messageDiv.style.color = lisaAiSettings.fontColor;
        }
        if (useHtml) {
            var textSpan = document.createElement('span');
            textSpan.innerHTML = text;
            messageDiv.appendChild(textSpan);
        } else {
            var textSpan = document.createElement('span');
            textSpan.textContent = text;
            messageDiv.appendChild(textSpan);
        }
        elements.chatMessages.appendChild(messageDiv);
        scrollToBottom();
    }

    // Add product cards
    function addProductCards(products) {
        var gridDiv = document.createElement('div');
        gridDiv.className = 'lisa-message lisa-message-assistant';

        var gridHtml = '<div class="lisa-products-grid">';

        products.forEach(function (product) {
            var productUrl = product.url || (product.handle ? window.location.origin + '/product/' + product.handle + '/' : null);
            if (productUrl) {
                gridHtml += '<a href="' + escapeHtml(productUrl) + '" class="lisa-product-card" target="_blank" style="text-decoration: none; color: inherit; display: block;">';
            } else {
                gridHtml += '<div class="lisa-product-card">';
            }
            if (product.image) {
                gridHtml += '<img src="' + escapeHtml(product.image) + '" alt="' + escapeHtml(product.title) + '">';
            }
            gridHtml += '<div class="lisa-product-card-title">' + escapeHtml(product.title) + '</div>';
            gridHtml += '<div class="lisa-product-card-price">' + escapeHtml((lisaAiSettings.currencySymbol || '$') + product.price) + '</div>';
            if (productUrl) {
                gridHtml += '</a>';
            } else {
                gridHtml += '</div>';
            }
        });

        gridHtml += '</div>';
        gridDiv.innerHTML = gridHtml;
        elements.chatMessages.appendChild(gridDiv);
        scrollToBottom();
    }

    // Show typing indicator
    function showTypingIndicator() {
        var typingDiv = document.createElement('div');
        typingDiv.id = 'lisa-typing-indicator';
        typingDiv.className = 'lisa-message lisa-message-assistant lisa-message-typing';
        typingDiv.innerHTML = '<span></span><span></span><span></span>';
        elements.chatMessages.appendChild(typingDiv);
        scrollToBottom();
    }

    // Hide typing indicator
    function hideTypingIndicator() {
        var typingDiv = document.getElementById('lisa-typing-indicator');
        if (typingDiv) {
            typingDiv.remove();
        }
    }

    // Scroll to bottom of messages
    function scrollToBottom() {
        elements.chatMessages.scrollTop = elements.chatMessages.scrollHeight;
    }

    // Save conversation to session storage
    function saveConversation() {
        var data = {
            messages: state.conversationHistory,
            lastActivity: Date.now()
        };
        localStorage.setItem('lisa_conversation', JSON.stringify(data));
    }

    // Load conversation from session storage (15-min timeout)
    function loadConversation() {
        try {
            var stored = localStorage.getItem('lisa_conversation');
            if (!stored) return [];
            var data = JSON.parse(stored);
            var elapsed = Date.now() - data.lastActivity;
            if (elapsed > 15 * 60 * 1000) {
                localStorage.removeItem('lisa_conversation');
                return [];
            }
            return data.messages || [];
        } catch (e) {
            return [];
        }
    }

    // Escape HTML
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();