<?php
/**
 * LISA Chat Widget
 *
 * Handles the frontend chat widget display
 *
 * @package LISA_AI
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Lisa_Widget {

    /**
     * Plugin settings
     */
    private $settings;

    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('lisa_ai_settings');

        // Only load widget if enabled
        if ($this->is_widget_enabled()) {
            add_action('wp_footer', array($this, 'render_widget'));
            add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        }

        // Register REST API endpoint for chat
        add_action('rest_api_init', array($this, 'register_rest_routes'));

        // Register AJAX streaming endpoint (supports SSE unlike REST API)
        add_action('wp_ajax_lisa_stream', array($this, 'handle_stream_request'));
        add_action('wp_ajax_nopriv_lisa_stream', array($this, 'handle_stream_request'));
    }

    /**
     * Check if widget is enabled
     */
    private function is_widget_enabled() {
        return isset($this->settings['widget_enabled']) && $this->settings['widget_enabled'];
    }

    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_scripts() {
        wp_enqueue_style(
            'lisa-ai-inter-font',
            'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap',
            array(),
            LISA_AI_VERSION
        );

        wp_enqueue_style(
            'lisa-ai-widget',
            LISA_AI_PLUGIN_URL . 'public/css/lisa-widget.css',
            array('lisa-ai-inter-font'),
            LISA_AI_VERSION
        );

        wp_enqueue_script(
            'lisa-ai-widget',
            LISA_AI_PLUGIN_URL . 'public/js/lisa-widget.js',
            array('jquery'),
            LISA_AI_VERSION,
            true
        );

        // Pass settings to JavaScript
        wp_localize_script('lisa-ai-widget', 'lisaAiSettings', array(
            'ajaxUrl' => rest_url('lisa-ai/v1/chat'),
            'streamUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wp_rest'),
            'storeId' => $this->settings['store_id'] ?? '',
            'accessToken' => $this->settings['access_token'] ?? '',
            'backendUrl' => LISA_AI_API_URL,
            'position' => $this->settings['widget_position'] ?? 'bottom-right',
            'color' => $this->settings['widget_color'] ?? '#0066cc',
            'welcomeMessage' => $this->settings['welcome_message'] ?? "Hi! I'm LISA, your shopping assistant. How can I help you today?",
            'assistantName' => $this->settings['assistant_name'] ?? 'LISA',
            'fontColor' => $this->settings['widget_font_color'] ?? '#ffffff',
            'currencySymbol' => function_exists('get_woocommerce_currency_symbol') ? get_woocommerce_currency_symbol() : '$',
        ));
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('lisa-ai/v1', '/chat', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_chat_request'),
            'permission_callback' => function() {
                return $this->is_widget_enabled() && wp_verify_nonce(
                    isset($_SERVER['HTTP_X_WP_NONCE']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_X_WP_NONCE'])) : '',
                    'wp_rest'
                );
            },
        ));
    }

    /**
     * Handle chat API request
     */
    public function handle_chat_request($request) {
        $message = sanitize_text_field($request->get_param('message'));
        $conversation_id = sanitize_text_field($request->get_param('conversation_id'));

        if (empty($message)) {
            return new WP_Error('missing_message', __('Message is required', 'lisa-ai-shopping-assistant'), array('status' => 400));
        }

        $api_client = new Lisa_Api_Client();

        if (!$api_client->is_configured()) {
            return new WP_Error('not_configured', __('LISA AI is not configured', 'lisa-ai-shopping-assistant'), array('status' => 500));
        }

        $response = $api_client->send_message($message, $conversation_id);

        if (!$response['success']) {
            return new WP_Error('api_error', $response['error'], array('status' => 500));
        }

        return rest_ensure_response($response['data']);
    }

    /**
     * Handle streaming chat request via AJAX (SSE passthrough)
     */
    public function handle_stream_request() {
        // Verify nonce
        if (!check_ajax_referer('wp_rest', '_nonce', false)) {
            wp_send_json_error('Invalid nonce', 403);
        }

        $message = sanitize_text_field(wp_unslash($_POST['message'] ?? ''));
        $conversation_history_raw = json_decode(wp_unslash($_POST['conversation_history'] ?? '[]'), true);
        $conversation_history = is_array($conversation_history_raw) ? $this->sanitize_conversation_history($conversation_history_raw) : array();
        $product_offset = intval($_POST['product_offset'] ?? 0);
        $hp = sanitize_text_field(wp_unslash($_POST['_hp'] ?? ''));
        $ts = sanitize_text_field(wp_unslash($_POST['_ts'] ?? ''));

        // Honeypot check - if filled, it's a bot
        if (!empty($hp)) {
            wp_send_json(array('reply' => 'Thanks for your message!', 'products' => array()));
            return;
        }

        // Timing check - if sent too fast (<500ms), likely a bot
        if (!empty($ts) && strtolower($message) !== 'show me more') {
            $time_taken = round(microtime(true) * 1000) - intval($ts);
            if ($time_taken < 500) {
                wp_send_json(array('reply' => 'Thanks for your message!', 'products' => array()));
                return;
            }
        }

        if (empty($message)) {
            wp_send_json_error('Message is required', 400);
        }

        $api_client = new Lisa_Api_Client();

        if (!$api_client->is_configured()) {
            wp_send_json_error('LISA AI is not configured', 500);
        }

        $stream_url = $api_client->get_stream_url();
        $access_token = $api_client->get_access_token();

        $context_raw = json_decode(wp_unslash($_POST['context'] ?? '{}'), true);
        $context = is_array($context_raw) ? array_map('sanitize_text_field', $context_raw) : array();

        $body = wp_json_encode(array(
            'store_id' => $api_client->get_store_id(),
            'message' => $message,
            'conversation_history' => $conversation_history ?: array(),
            'product_offset' => $product_offset,
            'context' => $context ?: array(),
        ));

        // Set SSE headers
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no');

        // Use wp_remote_post to stream from backend
        $response = wp_remote_post($stream_url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $access_token,
            ),
            'body' => $body,
            'stream' => false,
        ));

        if (is_wp_error($response)) {
            echo "data: " . wp_json_encode(array('error' => $response->get_error_message())) . "\n\n";
        } else {
            $response_body = wp_remote_retrieve_body($response);
            // Validate and sanitize SSE response - only allow valid SSE lines
            $lines = explode("\n", $response_body);
            foreach ($lines as $line) {
                $trimmed = trim($line);
                if ($trimmed === '') {
                    echo "\n";
                } elseif (strpos($trimmed, 'data: ') === 0) {
                    $json_part = substr($trimmed, 6);
                    $decoded = json_decode($json_part, true);
                    if (null !== $decoded) {
                        echo "data: " . wp_json_encode($decoded) . "\n";
                    }
                }
            }
            echo "\n";
        }

        if (ob_get_level()) {
            ob_flush();
        }
        flush();

        exit;
    }

    /**
     * Render the chat widget HTML
     */
    public function render_widget() {
        $position = $this->settings['widget_position'] ?? 'bottom-right';
        $color = $this->settings['widget_color'] ?? '#0066cc';
        ?>
        <div id="lisa-ai-widget" class="lisa-widget lisa-widget-<?php echo esc_attr($position); ?>" style="--lisa-primary-color: <?php echo esc_attr($color); ?>">
            <!-- Chat Button (pill style) -->
            <button id="lisa-chat-button" class="lisa-chat-button-pill" aria-label="<?php esc_attr_e('Open chat', 'lisa-ai-shopping-assistant'); ?>">
                <span class="lisa-button-text"><?php echo esc_html($this->settings['assistant_name'] ?? 'LISA'); ?></span>
                <span class="lisa-button-close">&times;</span>
            </button>

            <!-- Chat Window -->
            <div id="lisa-chat-window" class="lisa-chat-window" style="display:none;">
                <?php $font_color = $this->settings['widget_font_color'] ?? '#ffffff'; ?>
                <div class="lisa-chat-header" style="color: <?php echo esc_attr($font_color); ?>;">
                    <div class="lisa-chat-header-info">
                        <span class="lisa-chat-title"><?php echo esc_html($this->settings['assistant_name'] ?? 'LISA'); ?></span>
                        <span class="lisa-chat-subtitle"><?php esc_html_e('Shopping Assistant', 'lisa-ai-shopping-assistant'); ?></span>
                    </div>
                    <button id="lisa-chat-info" class="lisa-chat-header-btn" aria-label="<?php esc_attr_e('Info', 'lisa-ai-shopping-assistant'); ?>" title="<?php esc_attr_e('Info', 'lisa-ai-shopping-assistant'); ?>">&#8505;&#65039;</button>
                    <button id="lisa-chat-reset" class="lisa-chat-header-btn" aria-label="<?php esc_attr_e('New conversation', 'lisa-ai-shopping-assistant'); ?>" title="<?php esc_attr_e('New conversation', 'lisa-ai-shopping-assistant'); ?>">&#128465;&#65039;</button>
                    <button id="lisa-chat-size" class="lisa-chat-header-btn" aria-label="<?php esc_attr_e('Resize', 'lisa-ai-shopping-assistant'); ?>" title="<?php esc_attr_e('Expand', 'lisa-ai-shopping-assistant'); ?>">&#11014;&#65039;</button>
                    <button id="lisa-chat-close" class="lisa-chat-header-btn" aria-label="<?php esc_attr_e('Close chat', 'lisa-ai-shopping-assistant'); ?>" title="<?php esc_attr_e('Close chat', 'lisa-ai-shopping-assistant'); ?>">&times;</button>
                </div>

                <div id="lisa-chat-messages" class="lisa-chat-messages">
                    <!-- Messages will be inserted here -->
                </div>

                <div class="lisa-chat-input-container">
                    <input type="text" id="lisa-chat-input" class="lisa-chat-input" placeholder="<?php esc_attr_e('Type your message...', 'lisa-ai-shopping-assistant'); ?>" />
                    <button id="lisa-chat-send" class="lisa-chat-send" aria-label="<?php esc_attr_e('Send message', 'lisa-ai-shopping-assistant'); ?>">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="22" y1="2" x2="11" y2="13"></line>
                            <polygon points="22 2 15 22 11 13 2 9 22 2"></polygon>
                        </svg>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Sanitize conversation history array
     */
    private function sanitize_conversation_history($history) {
        $clean = array();
        foreach ($history as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $clean[] = array(
                'role' => sanitize_text_field($entry['role'] ?? ''),
                'content' => sanitize_textarea_field($entry['content'] ?? ''),
            );
        }
        return $clean;
    }

}