<?php
/**
 * LISA Product Sync
 *
 * Handles syncing WooCommerce products with LISA AI backend
 *
 * @package LISA_AI
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Lisa_Product_Sync {

    /**
     * API Client
     */
    private $api_client;

    /**
     * Constructor
     */
    public function __construct() {
        // Hook into WooCommerce product events
        add_action('woocommerce_new_product', array($this, 'sync_product'), 10, 1);
        add_action('woocommerce_update_product', array($this, 'sync_product'), 10, 1);
        add_action('woocommerce_delete_product', array($this, 'delete_product'), 10, 1);
        add_action('woocommerce_trash_product', array($this, 'delete_product'), 10, 1);

        // AJAX handlers for manual sync
        add_action('wp_ajax_lisa_sync_all_products', array($this, 'ajax_sync_all_products'));
    }

    /**
     * Get API client instance
     */
    private function get_api_client() {
        if (!$this->api_client) {
            $this->api_client = new Lisa_Api_Client();
        }
        return $this->api_client;
    }

    /**
     * Sync all products (returns result array)
     */
    public function sync_all_products() {
        $api_client = $this->get_api_client();

        if (!$api_client->is_configured()) {
            return array('success' => false, 'error' => 'LISA AI is not configured. Please enter your API key and Store ID.');
        }

        // Get all published products
        $products = wc_get_products(array(
            'status' => 'publish',
            'limit' => -1,
        ));

        if (empty($products)) {
            return array('success' => false, 'error' => 'No products found to sync.');
        }

        $formatted_products = array();
        foreach ($products as $product) {
            $formatted = $this->format_product($product);
            if ($formatted) {
                $formatted_products[] = $formatted;
            }
        }

        // Sync in batches of 50
        $batch_size = 50;
        $batches = array_chunk($formatted_products, $batch_size);
        $total_synced = 0;

        foreach ($batches as $batch) {
            $response = $api_client->sync_products($batch);
            if ($response['success']) {
                $total_synced += count($batch);
            }
        }

        return array(
            'success' => true,
            'message' => sprintf('Successfully synced %d products.', $total_synced),
            'count' => $total_synced,
        );
    }

    /**
     * Format product for LISA API
     */
    private function format_product($product) {
        if (!$product || !is_a($product, 'WC_Product')) {
            return null;
        }

        // Get product images
        $images = array();
        $image_id = $product->get_image_id();
        if ($image_id) {
            $images[] = array(
                'src' => wp_get_attachment_url($image_id),
                'position' => 0,
            );
        }

        // Get gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        foreach ($gallery_ids as $index => $gallery_id) {
            $images[] = array(
                'src' => wp_get_attachment_url($gallery_id),
                'position' => $index + 1,
            );
        }

        // Get categories
        $categories = array();
        $terms = get_the_terms($product->get_id(), 'product_cat');
        if ($terms && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[] = $term->name;
            }
        }

        // Get tags
        $tags = array();
        $tag_terms = get_the_terms($product->get_id(), 'product_tag');
        if ($tag_terms && !is_wp_error($tag_terms)) {
            foreach ($tag_terms as $term) {
                $tags[] = $term->name;
            }
        }

        // Build variants array
        $variants = array();
        if ($product->is_type('variable')) {
            $variations = $product->get_available_variations();
            foreach ($variations as $variation) {
                $variants[] = array(
                    'id' => $variation['variation_id'],
                    'title' => implode(' / ', $variation['attributes']),
                    'price' => $variation['display_price'],
                    'sku' => $variation['sku'],
                    'available' => $variation['is_in_stock'],
                );
            }
        } else {
            $variants[] = array(
                'id' => $product->get_id(),
                'title' => 'Default',
                'price' => $product->get_price(),
                'sku' => $product->get_sku(),
                'available' => $product->is_in_stock(),
            );
        }

        return array(
            'shopify_product_id' => (string) $product->get_id(),
            'title' => $product->get_name(),
            'description' => $product->get_description(),
            'product_type' => !empty($categories) ? $categories[0] : $product->get_type(),
            'vendor' => get_bloginfo('name'),
            'tags' => $tags,
            'price' => (float) $product->get_price(),
            'compare_at_price' => $product->get_regular_price() ? (float) $product->get_regular_price() : null,
            'currency' => get_woocommerce_currency(),
            'available' => $product->is_in_stock(),
            'inventory_quantity' => $product->get_stock_quantity() ?: 0,
            'image_url' => wp_get_attachment_url($product->get_image_id()) ?: null,
            'handle' => $product->get_slug(),
            'variants' => $variants,
            'options' => array(),
            'metadata' => array(
                'url' => get_permalink($product->get_id()),
                'categories' => $categories,
                'short_description' => $product->get_short_description(),
            ),
        );
    }

    /**
     * Sync a single product
     */
    public function sync_product($product_id) {
        $api_client = $this->get_api_client();

        if (!$api_client->is_configured()) {
            return;
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            return;
        }

        $formatted = $this->format_product($product);
        if (!$formatted) {
            return;
        }

        $api_client->sync_products(array($formatted));
    }

    /**
     * Delete product from LISA
     */
    public function delete_product($product_id) {
        $api_client = $this->get_api_client();

        if (!$api_client->is_configured()) {
            return;
        }

        $api_client->delete_product($product_id);
    }

    /**
     * Sync all products via AJAX
     */
    public function ajax_sync_all_products() {
        check_ajax_referer('lisa_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'lisa-ai-shopping-assistant')));
        }

        $api_client = $this->get_api_client();

        if (!$api_client->is_configured()) {
            wp_send_json_error(array('message' => __('LISA AI is not configured. Please enter your API key.', 'lisa-ai-shopping-assistant')));
        }

        // Get all published products
        $products = wc_get_products(array(
            'status' => 'publish',
            'limit' => -1,
        ));

        if (empty($products)) {
            wp_send_json_error(array('message' => __('No products found to sync.', 'lisa-ai-shopping-assistant')));
        }

        $formatted_products = array();
        foreach ($products as $product) {
            $formatted = $this->format_product($product);
            if ($formatted) {
                $formatted_products[] = $formatted;
            }
        }

        // Sync in batches of 50
        $batch_size = 50;
        $batches = array_chunk($formatted_products, $batch_size);
        $total_synced = 0;

        foreach ($batches as $batch) {
            $response = $api_client->sync_products($batch);
            if ($response['success']) {
                $total_synced += count($batch);
            }
        }

        wp_send_json_success(array(
            /* translators: %d: number of products synced */
            'message' => sprintf(__('Successfully synced %d products.', 'lisa-ai-shopping-assistant'), $total_synced),
            'count' => $total_synced,
        ));
    }

    /**
     * Get sync status
     */
    public function get_sync_status() {
        $products_count = wp_count_posts('product');
        $published = $products_count->publish ?? 0;

        return array(
            'total_products' => $published,
            'last_sync' => get_option('lisa_ai_last_sync', null),
        );
    }
}