<?php
/**
 * LISA API Client
 *
 * Handles communication with the LISA AI backend
 *
 * @package LISA_AI
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Lisa_Api_Client {

    /**
     * API base URL
     */
    private $api_url;

    /**
     * API key
     */
    private $api_key;

    /**
     * Store ID
     */
    private $store_id;

    /**
     * Access token (JWT)
     */
    private $access_token;

    /**
     * Constructor
     */
    public function __construct() {
        $this->api_url = LISA_AI_API_URL;
        $options = get_option('lisa_ai_settings');
        $this->api_key = $options['api_key'] ?? '';
        $this->store_id = $options['store_id'] ?? '';
        $this->access_token = $options['access_token'] ?? '';
    }

    /**
     * Make API request
     */
    private function request($endpoint, $method = 'GET', $body = null, $use_bearer = false) {
        $url = trailingslashit($this->api_url) . ltrim($endpoint, '/');

        $headers = array(
            'Content-Type' => 'application/json',
        );

        if ($use_bearer && !empty($this->access_token)) {
            $headers['Authorization'] = 'Bearer ' . $this->access_token;
        } else {
            $headers['X-API-Key'] = $this->api_key;
            $headers['X-Store-ID'] = $this->store_id;
        }

        $args = array(
            'method' => $method,
            'timeout' => 30,
            'headers' => $headers,
        );

        if ($body !== null) {
            $args['body'] = json_encode($body);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message(),
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($status_code >= 200 && $status_code < 300) {
            return array(
                'success' => true,
                'data' => $body,
            );
        }

        return array(
            'success' => false,
            'error' => $body['detail'] ?? 'Unknown error',
            'status_code' => $status_code,
        );
    }

    /**
     * Register store with LISA backend
     */
    public function register_store($store_data) {
        return $this->request('api/v1/stores/register', 'POST', $store_data);
    }

    /**
     * Sync products to LISA backend
     */
    public function sync_products($products) {
        return $this->request('api/v1/products/sync', 'POST', array(
            'products' => $products,
        ), true);
    }

    /**
     * Send chat message
     */
    public function send_message($message, $conversation_id = null) {
        $body = array(
            'store_id' => $this->store_id,
            'message' => $message,
        );

        if ($conversation_id) {
            $body['conversation_id'] = $conversation_id;
        }

        return $this->request('lisa/query', 'POST', $body);
    }

    /**
     * Update store settings
     */
    public function update_settings($settings) {
        return $this->request("api/v1/stores/{$this->store_id}/settings", 'PATCH', $settings);
    }

    /**
     * Get store info
     */
    public function get_store_info() {
        return $this->request("api/v1/stores/{$this->store_id}", 'GET');
    }

    /**
     * Delete product from LISA
     */
    public function delete_product($product_id) {
        return $this->request("api/v1/products/{$product_id}", 'DELETE');
    }

    /**
     * Check API connection
     */
    public function test_connection() {
        $response = $this->request('health', 'GET');
        return $response['success'] ?? false;
    }

    /**
     * Get API key
     */
    public function get_api_key() {
        return $this->api_key;
    }

    /**
     * Get store ID
     */
    public function get_store_id() {
        return $this->store_id;
    }

    /**
     * Check if configured
     */
    public function is_configured() {
        return !empty($this->api_key) && !empty($this->store_id);
    }

    /**
     * Get the streaming endpoint URL
     */
    public function get_stream_url() {
        return trailingslashit($this->api_url) . 'lisa/query/stream';
    }

    /**
     * Get access token for streaming requests
     */
    public function get_access_token() {
        return $this->access_token;
    }
}