<?php
/**
 * LISA Admin Settings Page
 *
 * @package LISA_AI
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Lisa_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_lisa_sync_all_products', array($this, 'ajax_sync_products'));
        add_action('wp_ajax_lisa_check_billing', array($this, 'ajax_check_billing'));
        add_action('wp_ajax_lisa_create_checkout', array($this, 'ajax_create_checkout'));
        add_action('wp_ajax_lisa_manage_subscription', array($this, 'ajax_manage_subscription'));
        add_action('wp_ajax_lisa_register_store', array($this, 'ajax_register_store'));
    }

    /**
     * Add admin menu item
     */
    public function add_admin_menu() {
        add_menu_page(
            __('LISA AI', 'lisa-ai-shopping-assistant'),
            __('LISA AI', 'lisa-ai-shopping-assistant'),
            'manage_options',
            'lisa-ai-settings',
            array($this, 'render_settings_page'),
            'dashicons-format-chat',
            56
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('lisa_ai_settings_group', 'lisa_ai_settings', array($this, 'sanitize_settings'));

        // API Settings Section
        add_settings_section(
            'lisa_ai_api_section',
            __('API Settings', 'lisa-ai-shopping-assistant'),
            array($this, 'api_section_callback'),
            'lisa-ai-settings'
        );

        add_settings_field(
            'api_key',
            __('API Key', 'lisa-ai-shopping-assistant'),
            array($this, 'api_key_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_api_section'
        );

        add_settings_field(
            'access_token',
            __('Access Token', 'lisa-ai-shopping-assistant'),
            array($this, 'access_token_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_api_section'
        );

        add_settings_field(
            'store_id',
            __('Store ID', 'lisa-ai-shopping-assistant'),
            array($this, 'store_id_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_api_section'
        );

        // Widget Settings Section
        add_settings_section(
            'lisa_ai_widget_section',
            __('Widget Settings', 'lisa-ai-shopping-assistant'),
            array($this, 'widget_section_callback'),
            'lisa-ai-settings'
        );

        add_settings_field(
            'widget_enabled',
            __('Enable Widget', 'lisa-ai-shopping-assistant'),
            array($this, 'widget_enabled_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_widget_section'
        );

        add_settings_field(
            'widget_position',
            __('Widget Position', 'lisa-ai-shopping-assistant'),
            array($this, 'widget_position_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_widget_section'
        );

        add_settings_field(
            'widget_color',
            __('Widget Color', 'lisa-ai-shopping-assistant'),
            array($this, 'widget_color_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_widget_section'
        );

        add_settings_field(
            'widget_font_color',
            __('Font Color', 'lisa-ai-shopping-assistant'),
            array($this, 'widget_font_color_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_widget_section'
        );

        add_settings_field(
            'welcome_message',
            __('Welcome Message', 'lisa-ai-shopping-assistant'),
            array($this, 'welcome_message_field_callback'),
            'lisa-ai-settings',
            'lisa_ai_widget_section'
        );
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();

        $sanitized['api_key'] = sanitize_text_field($input['api_key'] ?? '');
        $sanitized['store_id'] = sanitize_text_field($input['store_id'] ?? '');
        $sanitized['access_token'] = sanitize_text_field($input['access_token'] ?? '');
        $sanitized['widget_enabled'] = isset($input['widget_enabled']) ? (bool) $input['widget_enabled'] : false;
        $sanitized['widget_position'] = sanitize_text_field($input['widget_position'] ?? 'bottom-right');
        $sanitized['widget_color'] = sanitize_hex_color($input['widget_color'] ?? '#0066cc');
        $sanitized['widget_font_color'] = sanitize_hex_color($input['widget_font_color'] ?? '#ffffff');
        $sanitized['welcome_message'] = sanitize_textarea_field($input['welcome_message'] ?? '');

        return $sanitized;
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        if ('toplevel_page_lisa-ai-settings' !== $hook) {
            return;
        }

        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        wp_enqueue_style(
            'lisa-ai-admin',
            LISA_AI_PLUGIN_URL . 'admin/css/lisa-admin.css',
            array(),
            LISA_AI_VERSION
        );

        wp_enqueue_script(
            'lisa-ai-admin',
            LISA_AI_PLUGIN_URL . 'admin/js/lisa-admin.js',
            array('jquery', 'wp-color-picker'),
            LISA_AI_VERSION,
            true
        );

        wp_localize_script('lisa-ai-admin', 'lisaAdminSettings', array(
            'nonce' => wp_create_nonce('lisa_admin_nonce'),
        ));
    }

    /**
     * API section callback
     */
    public function api_section_callback() {
        echo '<p>' . esc_html__('Connect your store to LISA AI backend.', 'lisa-ai-shopping-assistant') . '</p>';
    }

    /**
     * Widget section callback
     */
    public function widget_section_callback() {
        echo '<p>' . esc_html__('Customize the chat widget appearance.', 'lisa-ai-shopping-assistant') . '</p>';
    }

    /**
     * API key field callback
     */
    public function api_key_field_callback() {
        $options = get_option('lisa_ai_settings');
        $value = $options['api_key'] ?? '';
        echo '<input type="text" name="lisa_ai_settings[api_key]" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__('Enter your LISA AI API key.', 'lisa-ai-shopping-assistant') . '</p>';
    }

    /**
     * Store ID field callback
     */
    public function store_id_field_callback() {
        $options = get_option('lisa_ai_settings');
        $value = $options['store_id'] ?? '';
        echo '<input type="text" name="lisa_ai_settings[store_id]" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__('Enter your LISA AI Store ID.', 'lisa-ai-shopping-assistant') . '</p>';
    }

    /**
     * Access token field callback
     */
    public function access_token_field_callback() {
        $options = get_option('lisa_ai_settings');
        $value = $options['access_token'] ?? '';
        echo '<input type="text" name="lisa_ai_settings[access_token]" value="' . esc_attr($value) . '" class="large-text" />';
        echo '<p class="description">' . esc_html__('Enter your LISA AI Access Token (JWT).', 'lisa-ai-shopping-assistant') . '</p>';
    }

    /**
     * Widget enabled field callback
     */
    public function widget_enabled_field_callback() {
        $options = get_option('lisa_ai_settings');
        $checked = isset($options['widget_enabled']) && $options['widget_enabled'] ? 'checked' : '';
        echo '<input type="checkbox" name="lisa_ai_settings[widget_enabled]" value="1" ' . esc_attr($checked) . ' />';
        echo '<label>' . esc_html__('Show LISA chat widget on your store', 'lisa-ai-shopping-assistant') . '</label>';
    }

    /**
     * Widget position field callback
     */
    public function widget_position_field_callback() {
        $options = get_option('lisa_ai_settings');
        $value = $options['widget_position'] ?? 'bottom-right';
        ?>
        <select name="lisa_ai_settings[widget_position]">
            <option value="bottom-right" <?php selected($value, 'bottom-right'); ?>><?php esc_html_e('Bottom Right', 'lisa-ai-shopping-assistant'); ?></option>
            <option value="bottom-left" <?php selected($value, 'bottom-left'); ?>><?php esc_html_e('Bottom Left', 'lisa-ai-shopping-assistant'); ?></option>
        </select>
        <?php
    }

    /**
     * Widget color field callback
     */
    public function widget_color_field_callback() {
        $options = get_option('lisa_ai_settings');
        $value = $options['widget_color'] ?? '#0066cc';
        echo '<input type="text" name="lisa_ai_settings[widget_color]" value="' . esc_attr($value) . '" class="lisa-color-picker" />';
    }

    /**
     * Widget font color field callback
     */
    public function widget_font_color_field_callback() {
        $options = get_option('lisa_ai_settings');
        $value = $options['widget_font_color'] ?? '#ffffff';
        echo '<input type="text" name="lisa_ai_settings[widget_font_color]" value="' . esc_attr($value) . '" class="lisa-color-picker" />';
        echo '<p class="description">' . esc_html__('Text color for the widget header. Use dark color (#000000) for light widget colors.', 'lisa-ai-shopping-assistant') . '</p>';
    }

    /**
     * Welcome message field callback
     */
    public function welcome_message_field_callback() {
        $options = get_option('lisa_ai_settings');
        $value = $options['welcome_message'] ?? "Hi! I'm LISA, your shopping assistant. How can I help you today?";
        echo '<textarea name="lisa_ai_settings[welcome_message]" rows="3" class="large-text">' . esc_textarea($value) . '</textarea>';
    }

    /**
     * AJAX handler for product sync
     */
    public function ajax_sync_products() {
        check_ajax_referer('lisa_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        $sync = new Lisa_Product_Sync();
        $result = $sync->sync_all_products();

        if ($result['success']) {
            wp_send_json_success(array('message' => $result['message']));
        } else {
            wp_send_json_error(array('message' => $result['error']));
        }
    }

    /**
     * AJAX handler for checking billing status
     */
    public function ajax_check_billing() {
        check_ajax_referer('lisa_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        $options = get_option('lisa_ai_settings');
        $store_id = $options['store_id'] ?? '';

        if (empty($store_id)) {
            wp_send_json_error(array('message' => 'Store ID not configured'));
        }

        $response = wp_remote_get(
            'https://lisa-ai-backend-production.up.railway.app/api/v1/billing/status/' . urlencode($store_id),
            array('timeout' => 15)
        );

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        wp_send_json_success($body);
    }

    /**
     * AJAX handler for creating Stripe checkout session
     */
    public function ajax_create_checkout() {
        check_ajax_referer('lisa_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        $plan = sanitize_text_field(wp_unslash($_POST['plan'] ?? ''));
        $options = get_option('lisa_ai_settings');
        $store_id = $options['store_id'] ?? '';

        if (empty($store_id) || empty($plan)) {
            wp_send_json_error(array('message' => 'Missing store ID or plan'));
        }

        $response = wp_remote_post(
            'https://lisa-ai-backend-production.up.railway.app/api/v1/billing/create-checkout-session',
            array(
                'timeout' => 15,
                'headers' => array('Content-Type' => 'application/json'),
                'body' => wp_json_encode(array(
                    'store_id' => $store_id,
                    'plan' => $plan,
                )),
            )
        );

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['checkout_url'])) {
            wp_send_json_success(array('checkout_url' => $body['checkout_url']));
        } else {
            wp_send_json_error(array('message' => $body['detail'] ?? 'Failed to create checkout session'));
        }
    }

    /**
     * AJAX handler for managing subscription (Stripe portal)
     */
    public function ajax_manage_subscription() {
        check_ajax_referer('lisa_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        $options = get_option('lisa_ai_settings');
        $store_id = $options['store_id'] ?? '';

        if (empty($store_id)) {
            wp_send_json_error(array('message' => 'Store ID not configured'));
        }

        $response = wp_remote_post(
            'https://lisa-ai-backend-production.up.railway.app/api/v1/billing/create-portal-session',
            array(
                'timeout' => 15,
                'headers' => array('Content-Type' => 'application/json'),
                'body' => wp_json_encode(array('store_id' => $store_id)),
            )
        );

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['portal_url'])) {
            wp_send_json_success(array('portal_url' => $body['portal_url']));
        } else {
            wp_send_json_error(array('message' => $body['detail'] ?? 'Failed to create portal session'));
        }
    }

    /**
     * AJAX handler for store registration
     */
    public function ajax_register_store() {
        check_ajax_referer('lisa_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        $store_name = sanitize_text_field(wp_unslash($_POST['store_name'] ?? ''));
        $email = sanitize_email(wp_unslash($_POST['email'] ?? ''));
        $store_url = sanitize_text_field(wp_unslash($_POST['store_url'] ?? ''));

        if (empty($store_name) || empty($email)) {
            wp_send_json_error(array('message' => 'Store name and email are required'));
        }

        $response = wp_remote_post(
            'https://lisa-ai-backend-production.up.railway.app/api/v1/stores/register',
            array(
                'timeout' => 30,
                'headers' => array('Content-Type' => 'application/json'),
                'body' => wp_json_encode(array(
                    'shopify_domain' => $store_url ?: sanitize_title($store_name) . '.woocommerce',
                    'shopify_store_id' => 'woo_' . wp_generate_password(12, false),
                    'store_name' => $store_name,
                    'email' => $email,
                    'plan' => 'free',
                )),
            )
        );

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code >= 200 && $status_code < 300 && isset($body['store'])) {
            $options = get_option('lisa_ai_settings');
            $options['api_key'] = $body['api_key'] ?? '';
            $options['access_token'] = $body['access_token'] ?? '';
            $options['store_id'] = $body['store']['id'] ?? '';
            update_option('lisa_ai_settings', $options);

            wp_send_json_success(array(
                'message' => 'Store registered successfully!',
                'api_key' => $options['api_key'],
                'access_token' => $options['access_token'],
                'store_id' => $options['store_id'],
            ));
        } else {
            wp_send_json_error(array('message' => $body['detail'] ?? 'Registration failed'));
        }
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="lisa-ai-admin-container">
                <div class="lisa-ai-settings-main">
                    <?php
                    $options = get_option('lisa_ai_settings');
                    $has_credentials = !empty($options['api_key']) && !empty($options['store_id']);
                    if (!$has_credentials) :
                    ?>
                    <div id="lisa-registration-section" class="lisa-registration-box">
                        <h2><?php esc_html_e('Get Started with LISA AI', 'lisa-ai-shopping-assistant'); ?></h2>
                        <p><?php esc_html_e('Register your store to get your API credentials automatically.', 'lisa-ai-shopping-assistant'); ?></p>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><label for="lisa-reg-store-name"><?php esc_html_e('Store Name', 'lisa-ai-shopping-assistant'); ?></label></th>
                                <td><input type="text" id="lisa-reg-store-name" class="regular-text" value="<?php echo esc_attr(get_bloginfo('name')); ?>" /></td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="lisa-reg-email"><?php esc_html_e('Email', 'lisa-ai-shopping-assistant'); ?></label></th>
                                <td><input type="email" id="lisa-reg-email" class="regular-text" value="<?php echo esc_attr(get_option('admin_email')); ?>" /></td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="lisa-reg-url"><?php esc_html_e('Store URL', 'lisa-ai-shopping-assistant'); ?></label></th>
                                <td><input type="text" id="lisa-reg-url" class="regular-text" value="<?php echo esc_attr(home_url()); ?>" readonly /></td>
                            </tr>
                        </table>
                        <p>
                            <button type="button" id="lisa-register-btn" class="button button-primary button-hero"><?php esc_html_e('Register & Get Credentials', 'lisa-ai-shopping-assistant'); ?></button>
                        </p>
                        <div id="lisa-reg-status"></div>
                    </div>
                    <hr />
                    <?php endif; ?>

                    <form action="options.php" method="post">
                        <?php
                        settings_fields('lisa_ai_settings_group');
                        do_settings_sections('lisa-ai-settings');
                        submit_button(esc_html__('Save Settings', 'lisa-ai-shopping-assistant'));
                        ?>
                    </form>

                    <hr />

                    <h2><?php esc_html_e('Plan & Billing', 'lisa-ai-shopping-assistant'); ?></h2>
                    <div id="lisa-billing-section">
                        <p><?php esc_html_e('Loading billing status...', 'lisa-ai-shopping-assistant'); ?></p>
                    </div>

                    <hr />

                    <h2><?php esc_html_e('Product Sync', 'lisa-ai-shopping-assistant'); ?></h2>
                    <p><?php esc_html_e('Sync your WooCommerce products with LISA AI for intelligent product recommendations.', 'lisa-ai-shopping-assistant'); ?></p>
                    <button type="button" id="lisa-sync-products" class="button button-secondary">
                        <?php esc_html_e('Sync Products Now', 'lisa-ai-shopping-assistant'); ?>
                    </button>
                    <span id="lisa-sync-status"></span>
                </div>

                <div class="lisa-ai-settings-sidebar">
                    <div class="lisa-ai-info-box">
                        <h3><?php esc_html_e('About LISA AI', 'lisa-ai-shopping-assistant'); ?></h3>
                        <p><?php esc_html_e('LISA is an AI-powered shopping assistant that helps convert browsers into buyers through natural conversation.', 'lisa-ai-shopping-assistant'); ?></p>
                        <p><a href="https://codenamelisa.com" target="_blank"><?php esc_html_e('Learn more', 'lisa-ai-shopping-assistant'); ?> &rarr;</a></p>
                    </div>

                    <div class="lisa-ai-info-box">
                        <h3><?php esc_html_e('Need Help?', 'lisa-ai-shopping-assistant'); ?></h3>
                        <p><a href="https://codenamelisa.com/docs" target="_blank"><?php esc_html_e('Documentation', 'lisa-ai-shopping-assistant'); ?></a></p>
                        <p><a href="https://codenamelisa.com/support" target="_blank"><?php esc_html_e('Contact Support', 'lisa-ai-shopping-assistant'); ?></a></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}