<?php
/**
 * Plugin Name: Link to Headline - Anchor Link Generator
 * Plugin URI: https://wordpress.org/plugins/link-to-headline-anchor-link-generator
 * Description: Automatically adds anchor links to headings for easy section sharing
 * Version: 1.0.5
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.2
 * Author: Vinod
 * Author URI: https://profiles.wordpress.org/vinodkudavoor/
 * License:     GPL-2.0+
 * Text Domain: link-to-headline-anchor-link-generator
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LTHALG_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LTHALG_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('LTHALG_PLUGIN_VERSION', '1.2.1');

// Include necessary files
require_once LTHALG_PLUGIN_PATH . 'includes/class-anchor-links.php';
require_once LTHALG_PLUGIN_PATH . 'includes/class-settings.php';

// Initialize the plugin
function lthalg_init_plugin() {
    $anchor_links = new LTHALG_Anchor_Links();
    $settings = new LTHALG_Settings();
}
add_action('plugins_loaded', 'lthalg_init_plugin');

// Register activation hook
register_activation_hook(__FILE__, 'lthalg_activate_plugin');
function lthalg_activate_plugin() {
    // Set default options on activation
    if (false === get_option('lthalg_settings')) {
        $defaults = array(
            'heading_levels' => array('h2', 'h3', 'h4'),
            'icon_position' => 'right',
            'icon_type' => 'default',
            'custom_icon' => '',
            'excluded_ids' => '',
            'sticky_header_offset' => 0,
        );
        add_option('lthalg_settings', $defaults);
    }
}

// Add settings link to plugin listing
function lthalg_add_settings_link($links) {
    $settings_link = '<a href="' . admin_url('options-general.php?page=link-to-headline-anchor-link-generator') . '">' . __('Settings', 'link-to-headline-anchor-link-generator') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'lthalg_add_settings_link');

/**
 * Display an admin notice for donation.
 */
function lthalg_admin_donate_notice() {
    $screen = get_current_screen();
    // Only display on the plugins page.
    if ( isset($screen->id) && 'plugins' === $screen->id ) {
        ?>
        <div class="notice notice-info is-dismissible">
            <p>
                <?php
                printf(
                    /* translators: %s: Link to Buy Me a Coffee page. */
                    esc_html__( 'If you find the Link to Headline - Anchor Link Generator plugin useful, please consider supporting its development via %s.', 'link-to-headline-anchor-link-generator' ),
                    '<a href="https://buymeacoffee.com/wemakeitlive" target="_blank">' . esc_html__( 'Buy Me a Coffee', 'link-to-headline-anchor-link-generator' ) . '</a>'
                );
                ?>
            </p>
        </div>
        <?php
    }
}

/**
 * Add "Show Icon" field under Settings -> Reading for quick access.
 */
add_action('admin_init', function() {
    // Register standalone option so Reading settings can save it separately
    register_setting('reading', 'lthalg_show_icon', array(
        'type' => 'string',
        'sanitize_callback' => function($val) {
            $val = sanitize_text_field($val);
            return in_array($val, array('always','hover')) ? $val : 'always';
        },
        'default' => 'always',
    ));

    add_settings_field('lthalg_show_icon_field', __('Link to Headline: Show Icon', 'link-to-headline-anchor-link-generator'), function() {
        $val = get_option('lthalg_show_icon', 'always');
        ?>
        <select name="lthalg_show_icon">
            <option value="always" <?php selected($val, 'always'); ?>><?php esc_html_e('Always', 'link-to-headline-anchor-link-generator'); ?></option>
            <option value="hover" <?php selected($val, 'hover'); ?>><?php esc_html_e('On Hover', 'link-to-headline-anchor-link-generator'); ?></option>
        </select>
        <p class="description"><?php esc_html_e('Control whether the anchor icon is always visible or only shown on heading hover.', 'link-to-headline-anchor-link-generator'); ?></p>
        <?php
    }, 'reading', 'default');
});    
add_action( 'admin_notices', 'lthalg_admin_donate_notice' );