<?php
class LTHALG_Anchor_Links {
    private static $used_ids = [];
    private $is_excluded = false;

    public function __construct() {
        // Delay exclusion check until WordPress has loaded the queried object
        add_action('wp', [$this, 'determine_exclusion']);
        
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_filter('the_content', [$this, 'add_anchor_links']);
        add_filter('body_class', [$this, 'add_body_class']);
    }

    public function determine_exclusion() {
        if (!is_singular()) return;

        $options = get_option('lthalg_settings');
        $excluded_ids = isset($options['excluded_ids']) ? array_map('trim', explode(',', $options['excluded_ids'])) : [];

        if (in_array(get_the_ID(), $excluded_ids)) {
            $this->is_excluded = true;
        }
    }

    public function enqueue_scripts() {
        if ($this->is_excluded || !is_singular()) return;

        $options = get_option('lthalg_settings');
        wp_enqueue_script('lthalg-frontend', LTHALG_PLUGIN_URL . 'assets/js/frontend.js', [], LTHALG_PLUGIN_VERSION, true);
        wp_enqueue_style('lthalg-frontend', LTHALG_PLUGIN_URL . 'assets/css/frontend.css', [], LTHALG_PLUGIN_VERSION);

        wp_localize_script('lthalg-frontend', 'lthalg_settings', [
            'sticky_header_offset' => isset($options['sticky_header_offset']) ? absint($options['sticky_header_offset']) : 0,
        ]);

        wp_localize_script('lthalg-frontend', 'lthalg_i18n', [
            'link_copied' => __('Link Copied!', 'link-to-headline-anchor-link-generator'),
            'copy_error'  => __('Failed to copy link', 'link-to-headline-anchor-link-generator')
        ]);
    }

    public function add_anchor_links($content) {
        if ($this->is_excluded || !is_singular() || !in_the_loop() || !is_main_query()) return $content;

        self::$used_ids = [];
        $options = get_option('lthalg_settings');
        $heading_levels_option = isset($options['heading_levels']) ? $options['heading_levels'] : ['h2', 'h3', 'h4'];
        if (empty($heading_levels_option)) return $content;

        $heading_levels = array_map(fn($h) => substr($h, 1), $heading_levels_option);
        $pattern = '/<(?<tag>h[' . implode('', $heading_levels) . '])(?<attr>[^>]*)>(?<content>.*?)<\/\1>/is';
        return preg_replace_callback($pattern, [$this, 'add_anchor_link_callback'], $content);
    }
    
    private function add_anchor_link_callback($matches) {
        $tag = $matches['tag'];
        $attr = $matches['attr'];
        $content = $matches['content'];

        if (preg_match('/id=(["\'])(.*?)\1/', $attr, $id_matches)) {
            $id = $id_matches[2];
        } else {
            $id = sanitize_title(wp_strip_all_tags($content));
            $original_id = $id;
            $count = 1;
            while (in_array($id, self::$used_ids)) {
                $id = $original_id . '-' . $count++;
            }
            $attr .= ' id="' . esc_attr($id) . '"';
        }
        self::$used_ids[] = $id;

        $options = get_option('lthalg_settings');
        $icon_type = $options['icon_type'] ?? 'default';
        $custom_icon = $options['custom_icon'] ?? '';
        $icon_content = ($icon_type === 'custom' && $custom_icon)
            ? '<img src="' . esc_url($custom_icon) . '" alt="" class="lthalg-custom-icon">'
            : '🔗';

        $anchor_url = esc_attr(esc_url(get_permalink() . '#' . $id));
        $aria_label = esc_attr__('Copy link to this section', 'link-to-headline-anchor-link-generator');
        $anchor_link = '<a href="#' . esc_attr($id) . '" class="lthalg-anchor-link" data-url="' . $anchor_url . '" aria-label="' . $aria_label . '">' . $icon_content . '</a>';

        $position = $options['icon_position'] ?? 'right';
        if ($position === 'left') {
            $content = $anchor_link . $content;
        } else {
            $content .= $anchor_link;
        }

        return '<' . $tag . $attr . '>' . $content . '</' . $tag . '>';
    }

    public function add_body_class($classes) {
        $options = get_option('lthalg_settings', []);
        $show_icon = $options['show_icon'] ?? 'always';
        $position = $options['icon_position'] ?? 'right';

        if ($show_icon === 'hover') $classes[] = 'lthalg-icon-hover';
        if ($position === 'left') $classes[] = 'lthalg-icon-left';
        elseif ($position === 'right') $classes[] = 'lthalg-icon-right';

        return $classes;
    }
}
