(function() {
    'use strict';

    /**
     * Show tooltip/checkmark feedback for successful copy
     */
    function lthalgShowSuccess(link) {
        const iconWrapper = link.querySelector('.lthalg-icon-wrapper');
        const originalIconHTML = iconWrapper ? iconWrapper.innerHTML : '';

        // Remove existing tooltip if any
        const existingTooltip = link.querySelector('.lthalg-tooltip');
        if (existingTooltip) existingTooltip.remove();

        // Create tooltip
        const tooltip = document.createElement('span');
        tooltip.className = 'lthalg-tooltip';
        tooltip.textContent = lthalg_i18n.link_copied;
        link.appendChild(tooltip);

        setTimeout(() => {
            if (link.contains(tooltip)) link.removeChild(tooltip);
        }, 2000);

        // Temporary checkmark effect
        link.classList.add('lthalg-copied');
        if (iconWrapper) iconWrapper.innerHTML = '✓';

        setTimeout(() => {
            link.classList.remove('lthalg-copied');
            if (iconWrapper) iconWrapper.innerHTML = originalIconHTML;
        }, 2000);
    }

    /**
     * Fallback for older browsers (execCommand)
     */
    function lthalgFallbackCopy(url, resolve, reject) {
        const textArea = document.createElement("textarea");
        textArea.value = url;
        textArea.style.position = "fixed";
        textArea.style.opacity = "0";
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();

        try {
            if (document.execCommand('copy')) {
                resolve();
            } else {
                reject();
            }
        } catch (err) {
            reject(err);
        }

        document.body.removeChild(textArea);
    }

    /**
     * Copy-to-clipboard click handler
     */
    function handleAnchorClick(e) {
        const anchorLink = e.target.closest('.lthalg-anchor-link');
        if (!anchorLink) return;

        e.preventDefault(); // Prevent default anchor behavior

        const url = anchorLink.getAttribute('data-url');

        const copyToClipboard = new Promise((resolve, reject) => {
            if (navigator.clipboard && window.isSecureContext) {
                navigator.clipboard.writeText(url).then(resolve).catch(reject);
            } else {
                lthalgFallbackCopy(url, resolve, reject);
            }
        });

        copyToClipboard.then(() => {
            lthalgShowSuccess(anchorLink);
            // Do NOT change window.location.hash to prevent scroll
        }).catch(err => {
            console.error('Link to Headline: Could not copy text.', err);
        });
    }

    // Initialize
    document.addEventListener('DOMContentLoaded', function() {
        document.body.addEventListener('click', handleAnchorClick);
    });

})();
