<?php
if (!defined('ABSPATH')) {
    exit;
}

class Limbo_API {
    /**
     * Make a request to the Limbo API.
     *
     * @param string $endpoint API endpoint.
     * @param string $method HTTP method (GET, POST, etc.).
     * @param array $body Request body (for POST/PUT requests).
     * @param array $headers Additional request headers.
     * @return array|WP_Error API response or WP_Error on failure.
     */
    public static function request($endpoint, $method = 'GET', $body = [], $headers = []) {
        $args = [
            'method'  => $method,
            'headers' => array_merge(['Accept' => 'application/json'], $headers),
        ];

        if ($body) {
            $args['body'] = wp_json_encode($body);
            $args['headers']['Content-Type'] = 'application/json';
        }

        $response = wp_remote_request($endpoint, $args);
        if (is_wp_error($response)) {
            self::log_error('API request failed', ['endpoint' => $endpoint, 'error' => $response->get_error_message()]);
            return new WP_Error('api_error', __('API request failed.', 'limbo'));
        }

        $status = wp_remote_retrieve_response_code($response);
        $data = json_decode(wp_remote_retrieve_body($response), true);

        if ($status >= 200 && $status < 300) {
            return $data;
        }

        self::log_error('API error response', ['endpoint' => $endpoint, 'status' => $status, 'response' => $data]);
        return new WP_Error('api_error', $data['message'] ?? __('Unknown API error.', 'limbo'));
    }

    /**
     * Log API errors.
     *
     * @param string $message Log message.
     * @param array $context Additional context for debugging.
     */
    public static function log_error($message, $context = []) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $logger = wc_get_logger();
            $logger->error($message, ['source' => 'limbo-plugin', 'context' => $context]);
        }
    }
}
