document.addEventListener('DOMContentLoaded', async function() {
    const ITEMS_PER_PAGE = 10;
    let wcProducts = [];
    let limboProducts = [];
    let wcFilteredProducts = [];
    let limboFilteredProducts = [];
    let selectedWcProduct = null;
    let selectedLimboProduct = null;

    // Add search input listeners
    const wcSearchInput = document.getElementById('wc-product-search');
    const limboSearchInput = document.getElementById('limbo-product-search');

    wcSearchInput.addEventListener('input', debounce(() => {
        filterProducts('wc', wcSearchInput.value.toLowerCase());
    }, 300));

    limboSearchInput.addEventListener('input', debounce(() => {
        filterProducts('limbo', limboSearchInput.value.toLowerCase());
    }, 300));

    // Filter function
    function filterProducts(type, searchTerm) {
        if (type === 'wc') {
            wcFilteredProducts = wcProducts.filter(product => 
                product.name.toLowerCase().includes(searchTerm)
            );
            displayWooCommerceProducts();
        } else {
            limboFilteredProducts = limboProducts.filter(product => 
                product.productName.toLowerCase().includes(searchTerm)
            );
            displayLimboProducts();
        }
    }

    // Debounce helper function
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    // Fetch WooCommerce products
    async function fetchWooCommerceProducts() {
        try {
            const response = await fetch(limboNewConnection.ajaxurl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'limbo_new_connection_get_wc_products',
                    security: limboNewConnection.nonce
                })
            });

            if (!response.ok) {
                throw new Error(limboNewConnection.strings.networkError);
            }

            const data = await response.json();
            if (data.success) {
                wcProducts = data.data;
                wcFilteredProducts = wcProducts;
                displayWooCommerceProducts();
            }
        } catch (error) {
            console.error('Error fetching WooCommerce products:', error);
            document.getElementById('wc-products-list').innerHTML = 
                `<p style="color: red;">${limboNewConnection.strings.loadError}</p>`;
        }
    }

    // Fetch Limbo products
    async function fetchLimboProducts() {
        try {
            const userData = JSON.parse(localStorage.getItem('limboUserData'));
            if (!userData || !userData.id) {
                throw new Error('No user data found');
            }

            const response = await fetch(`https://www.serverlimbo.com/seller/productsBySellerId/${userData.id}`);
            if (!response.ok) {
                throw new Error(limboNewConnection.strings.networkError);
            }

            const data = await response.json();
            limboProducts = data.products;
            limboFilteredProducts = limboProducts;
            displayLimboProducts();
        } catch (error) {
            console.error('Error fetching Limbo products:', error);
            document.getElementById('limbo-products-list').innerHTML = 
                `<p style="color: red;">${limboNewConnection.strings.loadError}</p>`;
        }
    }

    // Display functions
    function displayWooCommerceProducts() {
        const productsList = document.getElementById('wc-products-list');
        productsList.innerHTML = '';

        if (!wcFilteredProducts || wcFilteredProducts.length === 0) {
            productsList.innerHTML = `<p>${limboNewConnection.strings.noProductsFound}</p>`;
            return;
        }

        wcFilteredProducts.forEach(product => {
            const productElement = createProductElement(product, 'wc');
            if (selectedWcProduct && selectedWcProduct.id === product.id) {
                productElement.classList.add('selected');
            }
            productsList.appendChild(productElement);
        });
    }

    function displayLimboProducts() {
        const productsList = document.getElementById('limbo-products-list');
        productsList.innerHTML = '';

        if (!limboFilteredProducts || limboFilteredProducts.length === 0) {
            productsList.innerHTML = `<p>${limboNewConnection.strings.noProductsFound}</p>`;
            return;
        }

        limboFilteredProducts.forEach(product => {
            const productElement = createProductElement(product, 'limbo');
            if (selectedLimboProduct && selectedLimboProduct === product) {
                productElement.classList.add('selected');
            }
            productsList.appendChild(productElement);
        });
    }

    function createProductElement(product, type) {
        const div = document.createElement('div');
        div.className = 'product-item';
        div.dataset.id = type === 'wc' ? product.id : product.id;
        div.dataset.type = type;
        
        const mainImage = type === 'wc' 
            ? (product.images && product.images.length > 0 ? product.images[0].thumbnail : '')
            : (product.images && product.images.length > 0 ? product.images[0] : '');
        
        const defaultImage = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="50" height="50" viewBox="0 0 50 50"%3E%3Crect width="50" height="50" fill="%23f0f0f0"/%3E%3Ctext x="50%25" y="50%25" text-anchor="middle" dy=".3em" fill="%23999" font-size="12"%3ENo Image%3C/text%3E%3C/svg%3E';

        div.innerHTML = `
            <img src="${mainImage || defaultImage}" 
                 alt="${type === 'wc' ? product.name : product.productName}" 
                 class="product-thumbnail"
                 onerror="this.src='${defaultImage}'"
            >
            <div class="product-info">
                <div class="product-name">${type === 'wc' ? product.name : product.productName}</div>
                <div class="product-price">${
                    type === 'wc' 
                        ? product.price 
                        : new Intl.NumberFormat('pt-BR', { 
                            style: 'currency', 
                            currency: 'BRL' 
                          }).format(product.price)
                }</div>
            </div>
        `;

        div.addEventListener('click', () => handleProductSelection(div, product, type));

        return div;
    }

    function handleProductSelection(element, product, type) {
        const container = type === 'wc' ? 'wc-products-list' : 'limbo-products-list';
        document.querySelectorAll(`#${container} .product-item.selected`)
            .forEach(item => item.classList.remove('selected'));

        element.classList.add('selected');

        if (type === 'wc') {
            selectedWcProduct = product;
        } else {
            selectedLimboProduct = product;
        }

        updateConnectionButton();
    }

    function updateConnectionButton() {
        let existingButton = document.getElementById('create-connection-btn');
        
        if (!existingButton) {
            existingButton = document.createElement('button');
            existingButton.id = 'create-connection-btn';
            existingButton.className = 'button button-primary';
            existingButton.textContent = limboNewConnection.strings.createConnection;
            existingButton.addEventListener('click', handleCreateConnection);
            document.querySelector('.connection-container').appendChild(existingButton);
        }

        existingButton.disabled = !(selectedWcProduct && selectedLimboProduct);
    }

    async function handleCreateConnection() {
        try {
            // Get user data from localStorage
            const userData = JSON.parse(localStorage.getItem('limboUserData'));
            if (!userData || !userData.id) {
                throw new Error('No user data found');
            }

            // Prepare the connection data with exactly what the backend needs
            const connectionData = {
                woocommerceProductId: selectedWcProduct.id.toString(), // Convert to string if needed
                limboProductId: selectedLimboProduct._id, // Use _id for Limbo products
                sellerId: userData.id,
                shopDomain: window.location.hostname
            };

            // Log the data being sent
            console.log('Creating connection with data:', connectionData);

            // Show loading state
            const button = document.getElementById('create-connection-btn');
            button.disabled = true;
            button.textContent = limboNewConnection.strings.creating;

            // Send request to create connection
            const response = await fetch('https://www.serverlimbo.com/seller/woocommerce/create-connection', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(connectionData)
            });

            const data = await response.json();
            console.log('Server response:', data);

            if (!response.ok) {
                throw new Error(data.message || limboNewConnection.strings.connectionError);
            }

            // Show success message
            showNotification(limboNewConnection.strings.connectionSuccess, 'success');
            
            // Reset selections
            selectedWcProduct = null;
            selectedLimboProduct = null;
            document.querySelectorAll('.product-item.selected')
                .forEach(item => item.classList.remove('selected'));
            updateConnectionButton();

        } catch (error) {
            console.error('Error creating connection:', error);
            showNotification(error.message || limboNewConnection.strings.connectionError, 'error');
        } finally {
            // Reset button state
            const button = document.getElementById('create-connection-btn');
            button.disabled = false;
            button.textContent = limboNewConnection.strings.createConnection;
        }
    }

    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notice notice-${type} is-dismissible`;
        notification.innerHTML = `<p>${message}</p>`;
        
        // Insert notification at the top of the page
        const container = document.querySelector('.wrap');
        container.insertBefore(notification, container.firstChild);

        // Auto-dismiss after 5 seconds
        setTimeout(() => {
            notification.remove();
        }, 5000);
    }

    // Initial load
    fetchWooCommerceProducts();
    fetchLimboProducts();
}); 