<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Add AJAX handler for WooCommerce product fetching
add_action('wp_ajax_get_woo_product', 'limbo_get_woo_product');
function limbo_get_woo_product() {
    // Verify nonce first
    if (!check_ajax_referer('limbo-connections-nonce', 'security', false)) {
        wp_send_json_error('Invalid security token');
        return;
    }
    
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error('Permission denied');
        return;
    }

    // Validate and sanitize product_id
    if (!isset($_GET['product_id'])) {
        wp_send_json_error('Product ID is required');
        return;
    }

    $product_id = absint(wp_unslash($_GET['product_id']));
    if (!$product_id) {
        wp_send_json_error('Invalid product ID');
        return;
    }

    $product = wc_get_product($product_id);
    if (!$product) {
        wp_send_json_error('Product not found');
        return;
    }

    $product_data = array(
        'id' => $product->get_id(),
        'name' => $product->get_name(),
        'image' => wp_get_attachment_image_url($product->get_image_id(), 'thumbnail'),
        'price' => $product->get_price(),
        'status' => $product->get_status()
    );

    wp_send_json_success($product_data);
}

// Enqueue scripts and styles
function limbo_connections_enqueue_scripts($hook) {
    // Only load on our plugin's page
    if ('limbo_page_limbo-connections' !== $hook) {
        return;
    }

    // Enqueue styles
    wp_enqueue_style(
        'limbo-connections-styles', 
        plugins_url('css/limbo-connections.css', __FILE__),
        array(),
        LIMBO_VERSION
    );

    // Enqueue jQuery
    wp_enqueue_script('jquery');

    // Enqueue our custom script
    wp_enqueue_script(
        'limbo-connections-script',
        plugins_url('js/limbo-connections.js', __FILE__),
        array('jquery'),
        LIMBO_VERSION,
        true
    );

    // Create nonce and localize script
    wp_localize_script(
        'limbo-connections-script',
        'limboAjaxData',
        array(
            'nonce' => wp_create_nonce('limbo-connections-nonce'),
            'ajaxurl' => admin_url('admin-ajax.php'),
            'strings' => array(
                'noConnections' => __('No connections found matching your search.', 'limbo'),
                'createNew' => __('Create a new connection!', 'limbo'),
                'pleaseLogin' => __('Please log in to view your connections', 'limbo'),
                'loading' => __('Loading connections...', 'limbo'),
                'loadingProduct' => __('Loading product details...', 'limbo'),
                'deleteConfirm' => __('Are you sure you want to delete the selected connections?', 'limbo'),
                'deleteSuccess' => __('Connections deleted successfully', 'limbo'),
                'deleteFailed' => __('Failed to delete connections', 'limbo'),
                'connectionSelected' => __('connection selected', 'limbo'),
                'connectionsSelected' => __('connections selected', 'limbo'),
            )
        )
    );
}
add_action('admin_enqueue_scripts', 'limbo_connections_enqueue_scripts');

// Manage Connections Page Callback Function
function limbo_display_connections() {
    // Check user capabilities
    if (!current_user_can('manage_woocommerce')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'limbo'));
    }

    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Manage Connections', 'limbo'); ?></h1>
        
        <div id="limbo-connections-content">
            <p class="description">
                <?php esc_html_e('View and manage your product connections between WooCommerce and Limbo.', 'limbo'); ?>
            </p>
            
            <!-- Search bar -->
            <div class="search-container">
                <input 
                    type="text" 
                    id="connection-search" 
                    class="search-input" 
                    placeholder="<?php esc_attr_e('Search connections by ID, product name, or product ID...', 'limbo'); ?>"
                >
                <div class="search-stats">
                    <span id="search-count"></span>
                </div>
            </div>
            
            <!-- Connections list container -->
            <div id="connections-list" class="connections-container">
                <div class="loading">
                    <?php esc_html_e('Loading connections...', 'limbo'); ?>
                </div>
            </div>
        </div>
    </div>
    <?php
}
