(function($) {
    // Global variables will be populated by wp_localize_script
    const limboAjax = window.limboAjaxData;
    
    // Global function declarations
    window.limbo_handleConnectionSelection = null;
    window.limbo_changePage = null;
    window.limbo_deleteSelectedConnections = null;
    window.limbo_filterConnections = null;

    document.addEventListener('DOMContentLoaded', async function() {
        const connectionsContainer = document.getElementById('connections-list');
        const searchInput = document.getElementById('connection-search');
        const searchCount = document.getElementById('search-count');
        let allConnections = []; 
        let parsedUserData;
        let selectedConnections = new Set();
        const ITEMS_PER_PAGE = 10;
        let currentPage = 1;

        async function limbo_fetchAndUpdateProductDetails(connection) {
            // Fetch Limbo product
            try {
                const productUrl = `https://www.serverlimbo.com/seller/product/${connection.limboProductId}/${parsedUserData.id}`;
                console.log('Fetching Limbo product:', productUrl);
                
                const productResponse = await fetch(productUrl, {
                    headers: {
                        'Authorization': `Bearer ${parsedUserData.token}`,
                        'Accept': 'application/json'
                    }
                });

                if (!productResponse.ok) {
                    throw new Error(`Failed to fetch Limbo product: ${productResponse.status}`);
                }

                const productData = await productResponse.json();
                console.log('Limbo product data:', productData);
                
                connection.limboProductData = productData.product;

                const productContainer = document.querySelector(`#connection-${connection._id} .limbo-product-info`);
                if (productContainer) {
                    const htmlContent = `
                        <div class="product-content">
                            ${productData.product.images && productData.product.images.length > 0 ? `
                                <img src="${productData.product.images[0]}" alt="${productData.product.productName}" class="product-image">
                            ` : ''}
                            <div class="product-text">
                                <p class="product-id">ID: ${connection.limboProductId}</p>
                                <p class="product-name">${productData.product.productName}</p>
                            </div>
                        </div>
                    `;
                    productContainer.innerHTML = htmlContent;
                }
            } catch (error) {
                console.error('Error fetching Limbo product details:', error);
                const productContainer = document.querySelector(`#connection-${connection._id} .limbo-product-info`);
                if (productContainer) {
                    productContainer.innerHTML = `
                        <p class="product-id">ID: ${connection.limboProductId}</p>
                        <p class="error-message">Failed to load product details: ${error.message}</p>
                    `;
                }
            }

            // Fetch WooCommerce product
            try {
                console.log('Fetching WooCommerce product:', connection.woocommerceProductId);
                
                const wooResponse = await fetch(
                    `${limboAjax.ajaxurl}?action=get_woo_product&product_id=${connection.woocommerceProductId}&security=${limboAjax.nonce}`,
                    {
                        credentials: 'same-origin',
                        headers: {
                            'Accept': 'application/json'
                        }
                    }
                );

                if (!wooResponse.ok) {
                    throw new Error(`Failed to fetch WooCommerce product: ${wooResponse.status}`);
                }

                const wooData = await wooResponse.json();
                console.log('WooCommerce product data:', wooData);
                
                if (!wooData.success) {
                    throw new Error(wooData.data || 'Failed to fetch WooCommerce product');
                }
                
                connection.wooProductData = wooData.data;

                const wooContainer = document.querySelector(`#connection-${connection._id} .woo-product-info`);
                if (wooContainer) {
                    wooContainer.innerHTML = `
                        <div class="product-content">
                            ${wooData.data.image ? `
                                <img src="${wooData.data.image}" alt="${wooData.data.name}" class="product-image">
                            ` : ''}
                            <div class="product-text">
                                <p class="product-id">ID: ${wooData.data.id}</p>
                                <p class="product-name">${wooData.data.name}</p>
                                <p class="product-price">Price: ${wooData.data.price}</p>
                                <p class="product-status">Status: ${wooData.data.status}</p>
                            </div>
                        </div>
                    `;
                }
            } catch (error) {
                console.error('Error fetching WooCommerce product:', error);
                const wooContainer = document.querySelector(`#connection-${connection._id} .woo-product-info`);
                if (wooContainer) {
                    wooContainer.innerHTML = `
                        <p class="product-id">ID: ${connection.woocommerceProductId}</p>
                        <p class="error-message">Failed to load product details: ${error.message}</p>
                    `;
                }
            }
        }

        function limbo_filter_connections(searchTerm) {
            searchTerm = searchTerm.toLowerCase();
            const filteredConnections = allConnections.filter(connection => {
                const wooProduct = connection.wooProductData || {};
                const limboProduct = connection.limboProductData || {};
                
                return (
                    connection._id.toLowerCase().includes(searchTerm) ||
                    connection.woocommerceProductId.toString().includes(searchTerm) ||
                    connection.limboProductId.toString().includes(searchTerm) ||
                    (wooProduct.name || '').toLowerCase().includes(searchTerm) ||
                    (limboProduct.productName || '').toLowerCase().includes(searchTerm)
                );
            });

            limbo_displayConnections(filteredConnections);
            limbo_update_search_count(filteredConnections.length, allConnections.length);
            return filteredConnections;
        }
        window.limbo_filterConnections = limbo_filter_connections;

        function limbo_update_search_count(filteredCount, totalCount) {
            if (searchCount) {
                searchCount.textContent = `Showing ${filteredCount} of ${totalCount} connections`;
            }
        }

        function limbo_handle_connection_selection(connectionId, checkbox) {
            if (checkbox.checked) {
                selectedConnections.add(connectionId);
            } else {
                selectedConnections.delete(connectionId);
            }
            limbo_updateSelectionCount();
            limbo_updateDeleteButton();
        }
        window.limbo_handleConnectionSelection = limbo_handle_connection_selection;

        function limbo_updateSelectionCount() {
            const selectionCount = document.getElementById('selection-count');
            if (selectionCount) {
                selectionCount.textContent = `${selectedConnections.size} connection${selectedConnections.size !== 1 ? 's' : ''} selected`;
            }
        }

        function limbo_updateDeleteButton() {
            const deleteButton = document.getElementById('delete-connections');
            if (deleteButton) {
                deleteButton.disabled = selectedConnections.size === 0;
            }
        }

        function limbo_changePage(newPage) {
            currentPage = newPage;
            const filteredConnections = limbo_filter_connections(searchInput.value || '');
            limbo_displayConnections(filteredConnections);
        }
        window.limbo_changePage = limbo_changePage;

        async function limbo_deleteSelectedConnections() {
            if (!selectedConnections.size) return;

            const confirmDelete = confirm(`Are you sure you want to delete ${selectedConnections.size} connection${selectedConnections.size !== 1 ? 's' : ''}?`);
            if (!confirmDelete) return;

            const deleteButton = document.getElementById('delete-connections');
            const originalText = deleteButton.textContent;
            deleteButton.textContent = 'Deleting...';
            deleteButton.disabled = true;

            try {
                const shopDomain = window.location.origin;
                const connectionsToDelete = Array.from(selectedConnections).map(connectionId => ({
                    shopDomain,
                    connectionId
                }));

                const response = await fetch('https://www.serverlimbo.com/seller/woocommerce/delete-connections', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${parsedUserData.token}`,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify(connectionsToDelete)
                });

                if (!response.ok) {
                    throw new Error('Failed to delete connections');
                }

                const data = await response.json();
                
                if (!data.success) {
                    throw new Error(data.message || 'Failed to delete connections');
                }

                allConnections = allConnections.filter(conn => !selectedConnections.has(conn._id));
                selectedConnections.clear();
                
                currentPage = 1;
                limbo_displayConnections(allConnections);
                limbo_update_search_count(allConnections.length, allConnections.length);

                limbo_showNotification('Connections deleted successfully', 'success');

            } catch (error) {
                console.error('Delete error:', error);
                limbo_showNotification(error.message || 'Failed to delete connections', 'error');
            } finally {
                deleteButton.textContent = originalText;
                deleteButton.disabled = false;
            }
        }
        window.limbo_deleteSelectedConnections = limbo_deleteSelectedConnections;

        function limbo_displayConnections(connections) {
            if (connections.length === 0) {
                connectionsContainer.innerHTML = `
                    <div class="no-connections">
                        <p>No connections found matching your search. <a href="admin.php?page=limbo-new-connection">Create a new connection!</a></p>
                    </div>
                `;
                return;
            }

            const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
            const endIndex = startIndex + ITEMS_PER_PAGE;
            const paginatedConnections = connections.slice(startIndex, endIndex);

            const selectionInfoHtml = `
                <div class="selection-info">
                    <div class="selection-left">
                        <span id="selection-count">0 connections selected</span>
                    </div>
                    <div class="selection-actions">
                        <button 
                            id="delete-connections" 
                            class="button button-link-delete" 
                            ${selectedConnections.size === 0 ? 'disabled' : ''}
                            onclick="window.limbo_deleteSelectedConnections()"
                        >
                            Delete Selected
                        </button>
                    </div>
                </div>
            `;

            const connectionsHtml = paginatedConnections.map(connection => `
                <div class="connection-item" id="connection-${connection._id}">
                    <div class="connection-details">
                        <div class="connection-header">
                            <div class="connection-header-left">
                                <input type="checkbox" 
                                    id="checkbox-${connection._id}" 
                                    class="connection-checkbox"
                                    ${selectedConnections.has(connection._id) ? 'checked' : ''}
                                    onchange="window.limbo_handleConnectionSelection('${connection._id}', this)">
                                <h3>Connection ID: ${connection._id}</h3>
                            </div>
                            <span class="connection-date">Created: ${new Date(connection.createdAt).toLocaleDateString()}</span>
                        </div>
                        <div class="products-info">
                            <div class="product-detail woo">
                                <h4>WooCommerce Product</h4>
                                <div class="woo-product-info">
                                    <p>ID: ${connection.woocommerceProductId}</p>
                                    <div class="loading-product">Loading product details...</div>
                                </div>
                            </div>
                            <div class="product-detail limbo">
                                <h4>Limbo Product</h4>
                                <div class="limbo-product-info">
                                    <p>ID: ${connection.limboProductId}</p>
                                    <div class="loading-product">Loading product details...</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `).join('');

            const paginationControls = limbo_createPaginationControls(connections.length);

            connectionsContainer.innerHTML = selectionInfoHtml + connectionsHtml + paginationControls;

            paginatedConnections.forEach(connection => {
                limbo_fetchAndUpdateProductDetails(connection);
            });

            limbo_updateSelectionCount();
            limbo_updateDeleteButton();
        }

        function limbo_createPaginationControls(totalItems) {
            const totalPages = Math.ceil(totalItems / ITEMS_PER_PAGE);
            if (totalPages <= 1) return '';

            return `
                <div class="pagination-controls">
                    <button 
                        class="button pagination-button" 
                        onclick="window.limbo_changePage(${currentPage - 1})"
                        ${currentPage === 1 ? 'disabled' : ''}
                    >Previous</button>
                    <span class="pagination-info">Page ${currentPage} of ${totalPages}</span>
                    <button 
                        class="button pagination-button" 
                        onclick="window.limbo_changePage(${currentPage + 1})"
                        ${currentPage === totalPages ? 'disabled' : ''}
                    >Next</button>
                </div>
            `;
        }

        function limbo_showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `notice notice-${type} is-dismissible`;
            notification.innerHTML = `<p>${message}</p>`;
            
            const wrap = document.querySelector('.wrap');
            wrap.insertBefore(notification, wrap.firstChild);

            setTimeout(() => {
                notification.remove();
            }, 5000);
        }

        function limbo_debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }

        searchInput.addEventListener('input', limbo_debounce((e) => {
            currentPage = 1;
            limbo_filter_connections(e.target.value);
        }, 300));

        // Initial data fetch
        try {
            console.log('Starting initial data fetch...');
            
            const userData = localStorage.getItem('limboUserData');
            if (!userData) {
                connectionsContainer.innerHTML = `
                    <div class="error-message">
                        <p>Please log in to view your connections</p>
                    </div>
                `;
                return;
            }

            parsedUserData = JSON.parse(userData);
            console.log('User data found:', parsedUserData.id);
            
            const shopDomain = window.location.origin;
            console.log('Shop domain:', shopDomain);

            const response = await fetch('https://www.serverlimbo.com/seller/woocommerce/get-connections', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${parsedUserData.token}`,
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    shopDomain: shopDomain
                })
            });

            console.log('Response status:', response.status);
            
            if (!response.ok) {
                const errorText = await response.text();
                console.error('Response not OK:', errorText);
                
                try {
                    const errorData = JSON.parse(errorText);
                    if (errorData.message === "No connections found for this shop domain") {
                        connectionsContainer.innerHTML = `
                            <div class="no-connections">
                                <p>No connection was found. <a href="admin.php?page=limbo-new-connection">Create your first connection!</a></p>
                            </div>
                        `;
                        return;
                    }
                } catch (parseError) {
                    throw new Error(`Failed to fetch connections: ${response.status} ${response.statusText}`);
                }
                
                throw new Error(`Failed to fetch connections: ${response.status} ${response.statusText}`);
            }

            const data = await response.json();
            console.log('Data received:', data);
            
            if (!data.success) {
                throw new Error(data.message || 'Failed to load connections');
            }

            if (!data.connections) {
                connectionsContainer.innerHTML = `
                    <div class="no-connections">
                        <p>No connection was found. <a href="admin.php?page=limbo-new-connection">Create your first connection!</a></p>
                    </div>
                `;
                return;
            }

            allConnections = data.connections;
            console.log(`Found ${allConnections.length} connections`);
            
            limbo_displayConnections(allConnections);
            limbo_update_search_count(allConnections.length, allConnections.length);

        } catch (error) {
            console.error('Main error:', error);
            connectionsContainer.innerHTML = `
                <div class="error-message">
                    <p>Error: ${error.message || 'An error occurred while loading connections.'}</p>
                    ${error.stack ? `<pre>${error.stack}</pre>` : ''}
                </div>
            `;
        }
    });
})(jQuery); 