<?php
/**
 * Limbo Import Functions
 *
 * @package Limbo
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Add the AJAX handlers
add_action('wp_ajax_limbo_import_products', 'limbo_handle_product_import');

/**
 * Handle product import AJAX request
 */
function limbo_handle_product_import() {
    // Increase memory limit for large imports using WordPress function
    wp_raise_memory_limit('admin');
    
    try {
        // Verify nonce first
        if (!check_ajax_referer('limbo-import-nonce', 'security', false)) {
            wp_send_json_error([
                'message' => 'Security check failed',
                'code' => 'security_error'
            ]);
            return;
        }

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error([
                'message' => 'Permission denied',
                'code' => 'permission_error'
            ]);
            return;
        }

        // Validate input
        if (!isset($_POST['products'])) {
            wp_send_json_error([
                'message' => 'No products data received',
                'code' => 'no_data'
            ]);
            return;
        }

        // Get and decode products data with proper sanitization
        $products_raw = wp_unslash($_POST['products']);
        $products = json_decode($products_raw, true);

        // JSON decode error check
        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error([
                'message' => 'Invalid JSON data: ' . json_last_error_msg(),
                'code' => 'json_error'
            ]);
            return;
        }

        if (!is_array($products)) {
            wp_send_json_error([
                'message' => 'Products data must be an array',
                'code' => 'invalid_format'
            ]);
            return;
        }

        limbo_log_error("Starting import of " . count($products) . " products");

        $imported_count = 0;
        $errors = [];

        foreach ($products as $product) {
            try {
                limbo_log_error("Processing product: " . $product['productName']);
                
                // Sanitize product data
                $product_data = limbo_sanitize_product_data($product);
                
                // Create the product
                $product_id = limbo_create_product($product_data);

                if ($product_id) {
                    $imported_count++;
                    limbo_log_error("Successfully created product ID: " . $product_id);
                } else {
                    throw new Exception("Failed to create product");
                }

            } catch (Exception $e) {
                $error_msg = sprintf(
                    'Error importing %s: %s',
                    isset($product['productName']) ? $product['productName'] : 'unknown product',
                    $e->getMessage()
                );
                $errors[] = $error_msg;
                limbo_log_error($error_msg);
            }
        }

        if ($imported_count > 0) {
            wp_send_json_success([
                'imported' => $imported_count,
                'errors' => $errors,
                'message' => "Successfully imported {$imported_count} products"
            ]);
        } else {
            wp_send_json_error([
                'message' => 'No products were imported',
                'errors' => $errors,
                'code' => 'import_failed'
            ]);
        }

    } catch (Exception $e) {
        limbo_log_error("Critical import error: " . $e->getMessage());
        wp_send_json_error([
            'message' => 'Import failed: ' . $e->getMessage(),
            'code' => 'critical_error',
            'errors' => isset($errors) ? $errors : []
        ]);
    }
}

/**
 * Sanitize product data
 *
 * @param array $product Raw product data
 * @return array Sanitized product data
 */
function limbo_sanitize_product_data($product) {
    return array(
        'productName' => isset($product['productName']) ? sanitize_text_field($product['productName']) : '',
        'description' => isset($product['description']) ? wp_kses_post($product['description']) : '',
        'price' => isset($product['price']) ? floatval($product['price']) : 0,
        'category' => isset($product['category']) ? sanitize_text_field($product['category']) : '',
        'options' => isset($product['options']) ? limbo_sanitize_options($product['options']) : [],
        '_id' => isset($product['_id']) ? sanitize_text_field($product['_id']) : '',
        'sellerId' => isset($product['sellerId']) ? sanitize_text_field($product['sellerId']) : '',
        'sales' => isset($product['sales']) ? absint($product['sales']) : 0,
        'images' => isset($product['images']) ? array_map('esc_url_raw', $product['images']) : []
    );
}

/**
 * Sanitize product options
 *
 * @param array $options Raw options data
 * @return array Sanitized options
 */
function limbo_sanitize_options($options) {
    if (!is_array($options)) {
        return [];
    }

    return array_map(function($option) {
        $field_values = [];
        
        if (isset($option['fieldValues']) && is_array($option['fieldValues'])) {
            // Handle new format with value and additionalPrice
            foreach ($option['fieldValues'] as $field_value) {
                if (is_array($field_value) && isset($field_value['value'])) {
                    // New format
                    $field_values[] = [
                        'value' => sanitize_text_field($field_value['value']),
                        'additionalPrice' => isset($field_value['additionalPrice']) ? floatval($field_value['additionalPrice']) : 0
                    ];
                } else {
                    // Legacy format (just strings)
                    $field_values[] = [
                        'value' => sanitize_text_field($field_value),
                        'additionalPrice' => 0
                    ];
                }
            }
        }
        
        return [
            'fieldName' => isset($option['fieldName']) ? sanitize_text_field($option['fieldName']) : '',
            'fieldValues' => $field_values
        ];
    }, $options);
}

/**
 * Create a WooCommerce product
 *
 * @param array $product_data Sanitized product data
 * @return int|false Product ID on success, false on failure
 */
function limbo_create_product($product_data) {
    // Create appropriate product type based on options
    $wc_product = !empty($product_data['options']) 
        ? new WC_Product_Variable()
        : new WC_Product_Simple();
    
    // Set basic product data
    $wc_product->set_name($product_data['productName']);
    $wc_product->set_description($product_data['description']);
    $wc_product->set_regular_price($product_data['price']);
    $wc_product->set_manage_stock(false);
    $wc_product->set_stock_status('instock');
    $wc_product->set_status('publish');

    // Handle product category
    if (!empty($product_data['category'])) {
        limbo_handle_product_category($wc_product, $product_data['category']);
    }

    // Handle product attributes and variations
    if (!empty($product_data['options'])) {
        limbo_handle_product_attributes($wc_product, $product_data['options']);
    }

    // Save the product first to get ID
    $product_id = $wc_product->save();

    if ($product_id) {
        // Handle product images
        if (!empty($product_data['images'])) {
            limbo_handle_product_images($product_id, $wc_product, $product_data['images']);
        }

        // Store Limbo metadata
        update_post_meta($product_id, '_limbo_product_id', $product_data['_id']);
        update_post_meta($product_id, '_limbo_seller_id', $product_data['sellerId']);
        update_post_meta($product_id, '_limbo_sales', $product_data['sales']);

        // Create variations if needed
        if (!empty($product_data['options'])) {
            limbo_create_product_variations($product_id, $product_data['options'], $product_data['price']);
        }

        // Create connection with Limbo server
        $connection_result = limbo_create_product_connection($product_id, $product_data['_id'], $product_data['sellerId']);
        if (!$connection_result['success']) {
            limbo_log_error("Warning: Failed to create Limbo connection: " . $connection_result['message']);
        }

        return $product_id;
    }

    return false;
}

/**
 * Handle product category creation/assignment
 */
function limbo_handle_product_category($wc_product, $category_name) {
    $term = get_term_by('name', $category_name, 'product_cat');
    if (!$term) {
        $term = wp_insert_term($category_name, 'product_cat');
    }
    if (!is_wp_error($term)) {
        $term_id = is_object($term) ? $term->term_id : $term['term_id'];
        wp_set_object_terms($wc_product->get_id(), $term_id, 'product_cat');
    }
}

/**
 * Handle product attributes
 */
function limbo_handle_product_attributes($wc_product, $options) {
    $attributes = array();
    
    foreach ($options as $option) {
        $attribute_name = wc_sanitize_taxonomy_name($option['fieldName']);
        // Remove the automatic 'pa_' prefix for custom attributes
        $attribute_slug = $attribute_name;
        
        // Create a non-taxonomy attribute
        $attribute = new WC_Product_Attribute();
        $attribute->set_name($attribute_name);
        
        // Extract values from the new structure
        $attribute_values = array();
        foreach ($option['fieldValues'] as $field_value) {
            $attribute_values[] = $field_value['value'];
        }
        
        $attribute->set_options($attribute_values);
        $attribute->set_visible(true);
        $attribute->set_variation(true);
        $attribute->set_position(count($attributes));
        
        $attributes[] = $attribute;
    }
    
    if (!empty($attributes)) {
        $wc_product->set_attributes($attributes);
    }
}

/**
 * Register a new product attribute
 */
function limbo_register_product_attribute($slug, $label) {
    register_taxonomy(
        'pa_' . $slug,
        'product',
        array(
            'label' => $label,
            'rewrite' => array('slug' => $slug),
            'hierarchical' => false
        )
    );
    
    wc_create_attribute(array(
        'name' => $label,
        'slug' => $slug,
        'type' => 'select',
        'order_by' => 'menu_order',
        'has_archives' => false
    ));
}

/**
 * Handle product images
 */
function limbo_handle_product_images($product_id, $wc_product, $images) {
    limbo_log_error("=== Starting product image import for product ID: " . $product_id . " ===");
    limbo_log_error("Number of images to process: " . count($images));
    
    $image_ids = array();
    $failed_images = array();
    
    foreach ($images as $index => $image_url) {
        limbo_log_error("Processing image " . ($index + 1) . " of " . count($images));
        
        if (empty($image_url)) {
            limbo_log_error("❌ Empty image URL at index " . $index);
            continue;
        }

        // Add small delay between image processing to prevent overload
        if ($index > 0) {
            sleep(1);
        }

        $image_id = limbo_import_image($image_url);
        
        if ($image_id) {
            $image_ids[] = $image_id;
            limbo_log_error("✓ Image " . ($index + 1) . " successfully processed");
        } else {
            $failed_images[] = $image_url;
            limbo_log_error("❌ Failed to process image " . ($index + 1));
        }
    }
    
    if (!empty($image_ids)) {
        // Set featured image
        $featured_result = set_post_thumbnail($product_id, $image_ids[0]);
        limbo_log_error($featured_result ? "✓ Featured image set" : "❌ Failed to set featured image");
        
        // Set gallery images
        $gallery_ids = array_slice($image_ids, 1);
        if (!empty($gallery_ids)) {
            $wc_product->set_gallery_image_ids($gallery_ids);
            limbo_log_error("✓ Gallery images set: " . implode(', ', $gallery_ids));
        }
        
        $save_result = $wc_product->save();
        limbo_log_error($save_result ? "✓ Product saved with images" : "❌ Failed to save product with images");
    }

    limbo_log_error("=== End product image import ===");
    return count($image_ids);
}

/**
 * Import an image from URL using WordPress Filesystem
 */
function limbo_import_image($url) {
    require_once(ABSPATH . 'wp-admin/includes/media.php');
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    require_once(ABSPATH . 'wp-admin/includes/image.php');

    // Detailed URL logging
    limbo_log_error("=== Starting image import ===");
    limbo_log_error("Original URL: " . $url);
    
    // URL cleanup and validation
    $url = str_replace(' ', '%20', $url);
    $decoded_url = urldecode($url);
    limbo_log_error("Decoded URL: " . $decoded_url);

    if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
        limbo_log_error("❌ Invalid URL format: " . $url);
        return false;
    }

    // Test URL accessibility
    $response = wp_remote_head($url);
    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
        limbo_log_error("❌ URL not accessible: " . $url);
        return false;
    }

    // Download file
    $tmp = download_url($url);
    if (is_wp_error($tmp)) {
        limbo_log_error("❌ Download failed: " . $tmp->get_error_message());
        return false;
    }

    // Prepare file array
    $file_array = array(
        'name' => wp_basename($url),
        'tmp_name' => $tmp
    );

    // Handle sideload
    $id = media_handle_sideload($file_array, 0);

    if (is_wp_error($id)) {
        limbo_log_error("❌ Sideload failed: " . $id->get_error_message());
        wp_delete_file($tmp);
        return false;
    }

    limbo_log_error("✓ Successfully imported image. ID: " . $id);
    limbo_log_error("=== End image import ===");
    
    return $id;
}

/**
 * Create product variations
 */
function limbo_create_product_variations($product_id, $options, $base_price) {
    $variations = limbo_generate_variation_combinations($options);
    
    foreach ($variations as $variation) {
        try {
            $variation_product = new WC_Product_Variation();
            $variation_product->set_parent_id($product_id);
            
            $variation_attributes = array();
            $additional_price_total = 0;
            
            foreach ($variation as $attr_name => $attr_value) {
                // Use the attribute name directly without 'pa_' prefix
                $attribute_key = wc_sanitize_taxonomy_name($attr_name);
                $variation_attributes[$attribute_key] = $attr_value['value'];
                $additional_price_total += floatval($attr_value['additionalPrice']);
            }
            
            $variation_product->set_attributes($variation_attributes);
            
            $variation_product->set_status('publish');
            // Set price as base price plus any additional price from options
            $variation_price = $base_price + $additional_price_total;
            $variation_product->set_regular_price($variation_price);
            $variation_product->set_manage_stock(false);
            $variation_product->set_stock_status('instock');
            
            $variation_product->save();
        } catch (Exception $e) {
            limbo_log_error("Error creating variation: " . $e->getMessage());
            continue;
        }
    }
}

/**
 * Generate all possible variation combinations
 */
function limbo_generate_variation_combinations($options) {
    $arrays = array();
    foreach ($options as $option) {
        $arrays[$option['fieldName']] = $option['fieldValues'];
    }
    
    $result = array(array());
    foreach ($arrays as $key => $values) {
        $temp = array();
        foreach ($result as $product) {
            foreach ($values as $item) {
                $product[$key] = $item;
                $temp[] = $product;
            }
        }
        $result = $temp;
    }
    
    return $result;
}

/**
 * Log error messages with proper debug checking
 */
function limbo_log_error($message) {
    if (defined('WP_DEBUG') && WP_DEBUG) {
        if (function_exists('wc_get_logger')) {
            $logger = wc_get_logger();
            $logger->error($message, ['source' => 'limbo-import']);
        }
    }
}

/**
 * Check filesystem permissions using WordPress Filesystem
 */
function limbo_check_filesystem_permissions() {
    global $wp_filesystem;
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();
    
    $upload_dir = wp_upload_dir();
    $base_dir = $upload_dir['basedir'];
    
    limbo_log_error("Checking upload directory permissions:");
    limbo_log_error("Upload base dir: " . $base_dir);
    limbo_log_error("Is writable: " . ($wp_filesystem->is_writable($base_dir) ? 'Yes' : 'No'));
    limbo_log_error("Directory permissions: " . substr(sprintf('%o', $wp_filesystem->getchmod($base_dir)), -4));
    
    // Get process user info safely
    $process_user = function_exists('posix_getpwuid') ? posix_getpwuid(posix_geteuid())['name'] : 'Unknown';
    limbo_log_error("PHP process user: " . $process_user);
}

/**
 * Create connection between WooCommerce and Limbo products
 */
function limbo_create_product_connection($wc_product_id, $limbo_product_id, $seller_id) {
    // Verify nonce
    if (!check_ajax_referer('limbo-import-nonce', 'security', false)) {
        return array(
            'success' => false,
            'message' => 'Security check failed'
        );
    }

    // Validate and sanitize shop domain
    $shop_domain = '';
    if (isset($_POST['shopDomain'])) {
        $shop_domain = sanitize_text_field(wp_unslash($_POST['shopDomain']));
    }
    
    if (empty($shop_domain)) {
        return array(
            'success' => false,
            'message' => 'Invalid shop domain'
        );
    }
    
    $body = array(
        'woocommerceProductId' => (string)$wc_product_id,
        'limboProductId' => $limbo_product_id,
        'sellerId' => $seller_id,
        'shopDomain' => $shop_domain
    );

    $response = wp_remote_post('https://www.serverlimbo.com/seller/woocommerce/create-connection', array(
        'method' => 'POST',
        'timeout' => 45,
        'redirection' => 5,
        'httpversion' => '1.0',
        'blocking' => true,
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode($body),
        'cookies' => array()
    ));

    if (is_wp_error($response)) {
        limbo_log_error("Connection API error: " . $response->get_error_message());
        return array(
            'success' => false,
            'message' => $response->get_error_message()
        );
    }

    $response_body = json_decode(wp_remote_retrieve_body($response), true);
    $status_code = wp_remote_retrieve_response_code($response);

    if ($status_code !== 201) {
        limbo_log_error("Connection API failed with status {$status_code}");
        return array(
            'success' => false,
            'message' => isset($response_body['message']) ? $response_body['message'] : 'Unknown error'
        );
    }

    limbo_log_error("Successfully created connection with Limbo server");
    return array(
        'success' => true,
        'message' => 'Connection created successfully',
        'data' => $response_body
    );
}
