document.addEventListener('DOMContentLoaded', async function() {
    const userData = localStorage.getItem('limboUserData');
    let allProducts = []; // Store all products for filtering
    const selectedProducts = new Set(); // Track selected product IDs
    const ITEMS_PER_PAGE = 15;
    let currentPage = 1;
    let filteredProducts = [];
    let allSelected = false;

    if (userData) {
        const userDataObj = JSON.parse(userData);

        try {
            const response = await fetch(`https://www.serverlimbo.com/seller/productsBySellerId/${userDataObj.id}`);
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            const data = await response.json();
            allProducts = data.products;
            filteredProducts = allProducts;
            
            displayProducts();
            updatePaginationControls();
            
            // Add search functionality
            const searchInput = document.getElementById('product-search');
            searchInput.addEventListener('input', (e) => {
                const searchTerm = e.target.value.toLowerCase();
                filteredProducts = allProducts.filter(product => 
                    product.productName.toLowerCase().includes(searchTerm)
                );
                currentPage = 1; // Reset to first page on search
                
                // If all products were selected, maintain selection for filtered products
                if (allSelected) {
                    filteredProducts.forEach(product => {
                        selectedProducts.add(product._id);
                    });
                }
                
                displayProducts();
                updatePaginationControls();
                updateSelectedCount();
            });

            // Pagination event listeners
            document.getElementById('prev-page').addEventListener('click', () => {
                if (currentPage > 1) {
                    currentPage--;
                    displayProducts();
                    updatePaginationControls();
                }
            });

            document.getElementById('next-page').addEventListener('click', () => {
                const maxPages = Math.ceil(filteredProducts.length / ITEMS_PER_PAGE);
                if (currentPage < maxPages) {
                    currentPage++;
                    displayProducts();
                    updatePaginationControls();
                }
            });

        } catch (error) {
            console.error('Error fetching products:', error);
            document.getElementById('products-list').innerHTML = 
                '<p style="color: red;">Error loading products. Please try again later.</p>';
        }
    } else {
        document.getElementById('products-list').innerHTML = 
            '<p>No user data found in localStorage</p>';
    }

    // Function to display products for current page
    function displayProducts() {
        const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
        const endIndex = startIndex + ITEMS_PER_PAGE;
        const productsToShow = filteredProducts.slice(startIndex, endIndex);
        
        const productsList = document.getElementById('products-list');
        productsList.innerHTML = '';

        if (!productsToShow || productsToShow.length === 0) {
            productsList.innerHTML = '<p>No products found.</p>';
            return;
        }

        productsToShow.forEach(product => {
            const productItem = document.createElement('div');
            productItem.className = 'product-item';
            
            const mainImage = product.images && product.images.length > 0 
                ? product.images[0] 
                : limboImportData.pluginUrl + '/assets/images/placeholder.png';

            productItem.innerHTML = `
                <input type="checkbox" class="product-checkbox" value="${product._id}" 
                    ${selectedProducts.has(product._id) ? 'checked' : ''}>
                <img src="${mainImage}" alt="${product.productName}" class="product-thumbnail">
                <div class="product-info">
                    <div class="product-name">${product.productName}</div>
                    <div class="product-details">
                        <span class="detail-item">
                            <span class="detail-label">Stock:</span>
                            <span class="detail-value">${product.quantity}</span>
                        </span>
                        <span class="detail-item">
                            <span class="detail-label">Sales:</span>
                            <span class="detail-value">${product.sales}</span>
                        </span>
                        ${product.options && product.options.length > 0 ? `
                            <span class="detail-item">
                                <span class="detail-label">Options:</span>
                                <span class="detail-value">${product.options.length}</span>
                            </span>
                        ` : ''}
                    </div>
                </div>
                <div class="product-price-container">
                    <div class="price-label">Price</div>
                    <div class="product-price">$${product.price.toFixed(2)}</div>
                </div>
            `;
            
            // Add checkbox event listener
            const checkbox = productItem.querySelector('.product-checkbox');
            checkbox.addEventListener('change', (e) => {
                if (e.target.checked) {
                    selectedProducts.add(product._id);
                } else {
                    selectedProducts.delete(product._id);
                }
                updateSelectedCount();
            });
            
            productsList.appendChild(productItem);
        });
    }

    // Function to update pagination controls
    function updatePaginationControls() {
        const totalPages = Math.ceil(filteredProducts.length / ITEMS_PER_PAGE);
        const prevButton = document.getElementById('prev-page');
        const nextButton = document.getElementById('next-page');
        const pageInfo = document.getElementById('page-info');

        prevButton.disabled = currentPage === 1;
        nextButton.disabled = currentPage === totalPages;

        pageInfo.textContent = `Page ${currentPage} of ${totalPages}`;
    }

    // Update the selected count whenever checkboxes change
    function updateSelectedCount() {
        const count = selectedProducts.size;
        const countText = `${count} product${count !== 1 ? 's' : ''} selected`;
        document.getElementById('selected-count').textContent = countText;
        document.getElementById('import-selected').disabled = count === 0;
    }

    // Add import button event listener
    document.getElementById('import-selected').addEventListener('click', async () => {
        const importButton = document.getElementById('import-selected');
        importButton.disabled = true;
        importButton.textContent = 'Importing...';

        try {
            const selectedProductsData = allProducts.filter(product => 
                selectedProducts.has(product._id)
            );

            // Log the data being sent to verify images are included
            console.log('Selected products data:', selectedProductsData);
            
            // Make sure each product has its images array
            selectedProductsData.forEach(product => {
                if (!product.images) {
                    product.images = [];
                }
            });

            const response = await fetch(limboImportData.ajaxurl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'limbo_import_products',
                    security: limboImportData.nonce,
                    products: JSON.stringify(selectedProductsData),
                    shopDomain: window.location.origin
                })
            });

            const responseText = await response.text();
            console.log('Raw server response:', responseText);

            let result;
            try {
                result = JSON.parse(responseText);
            } catch (e) {
                console.error('Failed to parse server response:', responseText);
                throw new Error('Server returned invalid JSON response');
            }

            console.log('Parsed response:', result);

            if (result.success) {
                alert(`Successfully imported ${result.data.imported} products!`);
                if (result.data.errors && result.data.errors.length > 0) {
                    console.warn('Import warnings:', result.data.errors);
                }
                selectedProducts.clear();
                updateSelectedCount();
                displayProducts();
            } else {
                throw new Error(result.data?.message || 'Import failed');
            }
        } catch (error) {
            console.error('Import error:', error);
            alert(`Error importing products: ${error.message}`);
        } finally {
            importButton.disabled = false;
            importButton.textContent = 'Import Selected Products';
        }
    });

    // Add select all button handler
    document.getElementById('select-all').addEventListener('click', function() {
        allSelected = !allSelected;
        
        // Update button text
        this.textContent = allSelected ? 
            'Unselect All Products' : 
            'Select All Products';
        
        // Select/unselect all filtered products
        if (allSelected) {
            filteredProducts.forEach(product => {
                selectedProducts.add(product._id);
            });
        } else {
            selectedProducts.clear();
        }
        
        // Update checkboxes on current page
        displayProducts();
        
        // Update the selected count
        updateSelectedCount();
    });
}); 