<?php
// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Add the AJAX handlers
add_action('wp_ajax_limbo_get_wc_products', 'limbo_get_wc_products');
add_action('wp_ajax_limbo_export_products', 'limbo_handle_product_export');

function limbo_get_wc_products() {
    check_ajax_referer('limbo-export-nonce', 'security');

    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(['message' => __('Permission denied', 'limbo')]);
        return;
    }

    $products = wc_get_products(array(
        'limit' => -1,
        'status' => 'publish'
    ));

    $formatted_products = array();
    foreach ($products as $product) {
        $formatted_products[] = limbo_format_product($product);
    }

    wp_send_json_success($formatted_products);
}

function limbo_format_product($product) {
    // Get product price based on product type
    $price = limbo_get_product_price($product);

    // Get product attributes
    $attributes = limbo_get_product_attributes($product);

    // Get all product images
    $images = limbo_get_product_images($product);

    // Get variations if it's a variable product
    $variations = limbo_get_product_variations($product);

    return array(
        'id' => $product->get_id(),
        'name' => $product->get_name(),
        'type' => $product->get_type(),
        'status' => $product->get_status(),
        'sku' => $product->get_sku(),
        'price' => $price,
        'sale_price' => $product->get_sale_price(),
        'description' => $product->get_description(),
        'short_description' => $product->get_short_description(),
        'images' => $images,
        'attributes' => $attributes,
        'categories' => wp_list_pluck(get_the_terms($product->get_id(), 'product_cat'), 'name'),
        'tags' => wp_list_pluck(get_the_terms($product->get_id(), 'product_tag'), 'name'),
        'variations' => $variations,
        'stock_status' => $product->get_stock_status(),
        'stock_quantity' => $product->get_stock_quantity(),
        'weight' => $product->get_weight(),
        'dimensions' => array(
            'length' => $product->get_length(),
            'width' => $product->get_width(),
            'height' => $product->get_height()
        )
    );
}

function limbo_get_product_price($product) {
    $price = '';
    if ($product->is_type('variable')) {
        $price = $product->get_variation_price('min');
        if (empty($price)) {
            $variations = $product->get_available_variations();
            if (!empty($variations)) {
                $prices = array_map(function($variation) {
                    return $variation['display_price'];
                }, $variations);
                $price = min($prices);
            }
        }
    } else {
        $price = $product->get_price();
        if (empty($price)) {
            $price = $product->get_regular_price();
        }
    }
    return $price;
}

function limbo_get_product_attributes($product) {
    $attributes = array();
    foreach ($product->get_attributes() as $attribute) {
        $attribute_values = array();
        if ($attribute->is_taxonomy()) {
            $attribute_values = wc_get_product_terms($product->get_id(), $attribute->get_name(), array('fields' => 'names'));
        } else {
            $attribute_values = $attribute->get_options();
        }

        $attributes[] = array(
            'name' => wc_attribute_label($attribute->get_name()),
            'values' => $attribute_values
        );
    }
    return $attributes;
}

function limbo_get_product_images($product) {
    $images = array();
    $image_ids = array_filter(array_merge(
        array($product->get_image_id()),
        $product->get_gallery_image_ids()
    ));

    foreach ($image_ids as $image_id) {
        if (!empty($image_id)) {
            $file_path = get_attached_file($image_id);
            
            if ($file_path && file_exists($file_path)) {
                // Get image content and convert to base64
                $image_data = file_get_contents($file_path);
                $mime_type = mime_content_type($file_path);
                $base64 = base64_encode($image_data);
                $base64_url = "data:{$mime_type};base64,{$base64}";
                
                $images[] = array(
                    'id' => $image_id,
                    'full' => $base64_url,
                    'thumbnail' => $base64_url,
                    'alt' => get_post_meta($image_id, '_wp_attachment_image_alt', true)
                );
            }
        }
    }

    // If no images found, return array with placeholder
    if (empty($images)) {
        $images[] = array(
            'id' => 0,
            'full' => LIMBO_PLUGIN_URL . 'assets/images/placeholder.png',
            'thumbnail' => LIMBO_PLUGIN_URL . 'assets/images/placeholder.png',
            'alt' => __('Product image placeholder', 'limbo')
        );
    }

    return $images;
}

function limbo_get_product_variations($product) {
    $variations = array();
    if ($product->is_type('variable')) {
        foreach ($product->get_available_variations() as $variation) {
            $variation_obj = wc_get_product($variation['variation_id']);
            $variations[] = array(
                'id' => $variation['variation_id'],
                'attributes' => $variation['attributes'],
                'price' => $variation_obj->get_regular_price(),
                'sale_price' => $variation_obj->get_sale_price(),
                'sku' => $variation_obj->get_sku(),
                'stock_status' => $variation_obj->get_stock_status()
            );
        }
    }
    return $variations;
}

function limbo_handle_product_export() {
    check_ajax_referer('limbo-export-nonce', 'security');

    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(['message' => __('Permission denied', 'limbo')]);
        return;
    }

    // Validate and sanitize input data
    $product_ids_raw = isset($_POST['productIds']) ? sanitize_text_field(wp_unslash($_POST['productIds'])) : '';
    $seller_id_raw = isset($_POST['sellerId']) ? sanitize_text_field(wp_unslash($_POST['sellerId'])) : '';
    $shop_domain = isset($_POST['shopDomain']) ? esc_url_raw(wp_unslash($_POST['shopDomain'])) : '';

    // Validate required data
    if (empty($product_ids_raw) || empty($seller_id_raw) || empty($shop_domain)) {
        wp_send_json_error(['message' => __('Missing required data', 'limbo')]);
        return;
    }

    // Process product IDs
    $product_ids = is_array($product_ids_raw) 
        ? array_map('absint', $product_ids_raw)
        : array_map('absint', explode(',', $product_ids_raw));

    $seller_id = $seller_id_raw;

    $formatted_products = array();
    foreach ($product_ids as $product_id) {
        $product = wc_get_product($product_id);
        if (!$product) {
            continue;
        }

        try {
            // Get product data
            $images = limbo_get_product_images($product);
            $image_urls = array_map(function($img) {
                return $img['full'];
            }, $images);

            // Get categories and ensure we have a default value
            $categories = wp_get_post_terms($product_id, 'product_cat', ['fields' => 'names']);
            $category = !empty($categories) ? $categories[0] : 'Uncategorized';

            $attributes = limbo_get_product_attributes($product);
            
            // Format options according to the new structure
            $options = array_map(function($attr) {
                $field_values = array_map(function($value) {
                    return [
                        'value' => $value,
                        'additionalPrice' => 0 // Default additional price is 0
                    ];
                }, (array)$attr['values']);
                
                return [
                    'fieldName' => $attr['name'],
                    'fieldValues' => $field_values
                ];
            }, $attributes);

            $price = limbo_get_product_price($product);
            if (empty($price) || floatval($price) <= 0) {
                continue;
            }

            $formatted_products[] = [
                'id' => $product_id,
                'productName' => sanitize_text_field($product->get_name()),
                'description' => wp_kses_post($product->get_description() ?: $product->get_short_description()),
                'price' => floatval($price),
                'sellerId' => $seller_id,
                'quantity' => absint(50),
                'images' => $image_urls,
                'category' => sanitize_text_field($category),
                'options' => $options,
                'sales' => 0
            ];

        } catch (Exception $e) {
            continue;
        }
    }

    if (empty($formatted_products)) {
        wp_send_json_error(['message' => __('No valid products found to export', 'limbo')]);
        return;
    }

    wp_send_json_success($formatted_products);
}
