document.addEventListener('DOMContentLoaded', async function() {
    const selectedProducts = new Set();
    const ITEMS_PER_PAGE = limboExportData.itemsPerPage;
    let currentPage = 1;
    let allProducts = [];
    let filteredProducts = [];
    let allSelected = false;

    function formatPrice(price) {
        if (!price || price === '' || isNaN(price)) {
            return 'N/A';
        }
        const numericPrice = parseFloat(price.toString().replace(/[^0-9.-]+/g, ''));
        return isNaN(numericPrice) ? 'N/A' : `$${numericPrice.toFixed(2)}`;
    }

    function createProductElement(product) {
        console.log('Product:', product.name);
        console.log('Image URLs:', product.images.map(img => img.full));
        
        let mainImage = limboExportData.placeholderImage;
        if (product.images && Array.isArray(product.images) && product.images.length > 0) {
            mainImage = product.images[0].thumbnail || product.images[0].full || limboExportData.placeholderImage;
        }

        const attributesHtml = product.attributes.map(attr => `
            <div class="attribute-item">
                <span class="attribute-label">${attr.name}:</span>
                <span class="attribute-value">${attr.values.join(', ')}</span>
            </div>
        `).join('');

        const categoriesHtml = product.categories.length
            ? `<div class="categories">${product.categories.join(', ')}</div>`
            : '';

        return `
            <div class="product-item">
                <input type="checkbox"
                       class="product-checkbox"
                       value="${product.id}"
                       ${selectedProducts.has(product.id) ? 'checked' : ''}
                >
                <img src="${mainImage}"
                     alt="${product.name}"
                     class="product-thumbnail"
                     onerror="this.src='${limboExportData.placeholderImage}'"
                >
                <div class="product-info">
                    <div class="product-name">${product.name}</div>
                    <div class="product-meta">
                        <div class="meta-item">
                            <span class="meta-label">Type:</span>
                            <span class="meta-value">${product.type}</span>
                        </div>
                        <div class="meta-item">
                            <span class="meta-label">SKU:</span>
                            <span class="meta-value">${product.sku || 'N/A'}</span>
                        </div>
                        <div class="meta-item">
                            <span class="meta-label">Stock:</span>
                            <span class="meta-value">${product.stock_status}</span>
                        </div>
                    </div>
                    ${attributesHtml}
                    ${categoriesHtml}
                    <div class="product-description">${product.short_description || ''}</div>
                </div>
                <div class="product-price">
                    <div class="price-label">Price</div>
                    <div class="price-amount">${formatPrice(product.price)}</div>
                    ${product.sale_price ? `<div class="sale-price">Sale: ${formatPrice(product.sale_price)}</div>` : ''}
                </div>
            </div>
        `;
    }

    async function fetchProducts() {
        try {
            const response = await fetch(limboExportData.ajaxurl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'limbo_get_wc_products',
                    security: limboExportData.nonce
                })
            });

            if (!response.ok) {
                throw new Error(limboExportData.strings.networkError);
            }

            const data = await response.json();
            if (data.success) {
                allProducts = data.data;
                filteredProducts = allProducts;
                displayProducts();
                updatePaginationControls();
            } else {
                throw new Error(data.data.message);
            }
        } catch (error) {
            console.error('Error fetching products:', error);
            document.getElementById('products-list').innerHTML =
                `<p style="color: red;">${limboExportData.strings.loadError}</p>`;
        }
    }

    function displayProducts() {
        const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
        const endIndex = startIndex + ITEMS_PER_PAGE;
        const productsToShow = filteredProducts.slice(startIndex, endIndex);

        const productsList = document.getElementById('products-list');
        productsList.innerHTML = '';

        if (!productsToShow || productsToShow.length === 0) {
            productsList.innerHTML = `<p>${limboExportData.strings.noProducts}</p>`;
            return;
        }

        productsToShow.forEach(product => {
            const productElement = document.createElement('div');
            productElement.innerHTML = createProductElement(product);

            const checkbox = productElement.querySelector('.product-checkbox');
            checkbox.checked = selectedProducts.has(product.id);

            checkbox.addEventListener('change', (e) => {
                if (e.target.checked) {
                    selectedProducts.add(product.id);
                } else {
                    selectedProducts.delete(product.id);
                    if (allSelected) {
                        allSelected = false;
                        document.getElementById('select-all').textContent = limboExportData.strings.selectAll;
                    }
                }
                updateSelectedCount();
            });

            productsList.appendChild(productElement);
        });
    }

    function updatePaginationControls() {
        const totalPages = Math.ceil(filteredProducts.length / ITEMS_PER_PAGE);
        const prevButton = document.getElementById('prev-page');
        const nextButton = document.getElementById('next-page');
        const pageInfo = document.getElementById('page-info');

        prevButton.disabled = currentPage === 1;
        nextButton.disabled = currentPage === totalPages;

        pageInfo.textContent = `Page ${currentPage} of ${totalPages}`;
    }

    function updateSelectedCount() {
        const count = selectedProducts.size;
        const total = filteredProducts.length;
        const countText = `${count} of ${total} ${limboExportData.strings.productsSelected}`;
        document.getElementById('selected-count').textContent = countText;
        document.getElementById('export-selected').disabled = count === 0;
    }

    // Event Listeners
    const searchInput = document.getElementById('product-search');
    searchInput.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();
        filteredProducts = allProducts.filter(product =>
            product.name.toLowerCase().includes(searchTerm)
        );
        currentPage = 1;

        if (allSelected) {
            filteredProducts.forEach(product => {
                selectedProducts.add(product.id);
            });
        }

        displayProducts();
        updatePaginationControls();
        updateSelectedCount();
    });

    document.getElementById('prev-page').addEventListener('click', () => {
        if (currentPage > 1) {
            currentPage--;
            displayProducts();
            updatePaginationControls();
        }
    });

    document.getElementById('next-page').addEventListener('click', () => {
        const maxPages = Math.ceil(filteredProducts.length / ITEMS_PER_PAGE);
        if (currentPage < maxPages) {
            currentPage++;
            displayProducts();
            updatePaginationControls();
        }
    });

    document.getElementById('select-all').addEventListener('click', function() {
        allSelected = !allSelected;
        this.textContent = allSelected ? 
            limboExportData.strings.unselectAll : 
            limboExportData.strings.selectAll;

        if (allSelected) {
            filteredProducts.forEach(product => {
                selectedProducts.add(product.id);
            });
        } else {
            selectedProducts.clear();
        }

        displayProducts();
        updateSelectedCount();
    });

    document.getElementById('export-selected').addEventListener('click', async function() {
        if (selectedProducts.size === 0) {
            alert(limboExportData.strings.selectAtLeastOne);
            return;
        }

        try {
            const exportButton = this;
            exportButton.disabled = true;
            exportButton.textContent = limboExportData.strings.exporting;

            // Get the site URL
            const shopDomain = window.location.origin;

            const response = await fetch(limboExportData.ajaxurl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'limbo_export_products',
                    security: limboExportData.nonce,
                    productIds: Array.from(selectedProducts),
                    sellerId: JSON.parse(localStorage.getItem('limboUserData'))?.id,
                    shopDomain: shopDomain
                })
            });

            const data = await response.json();

            if (!data.success) {
                throw new Error(data.data?.message || limboExportData.strings.formatError);
            }

            if (!data.data || !data.data.length) {
                throw new Error(limboExportData.strings.noProductsFormatted);
            }

            console.log('Formatted products for export:', data.data);
            data.data.forEach(product => {
                console.log(`Image URLs for product "${product.productName}":`, product.images);
            });

            const backendResponse = await fetch(limboExportData.apiEndpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${JSON.parse(localStorage.getItem('limboUserData'))?.token}`
                },
                body: JSON.stringify({
                    woocommerceProducts: data.data,
                    sellerId: JSON.parse(localStorage.getItem('limboUserData'))?.id,
                    shopDomain: shopDomain
                })
            });

            const backendData = await backendResponse.json();

            if (backendData.success) {
                alert(limboExportData.strings.exportSuccess);
                selectedProducts.clear();
                displayProducts();
                updateSelectedCount();
            } else {
                throw new Error(backendData.message || limboExportData.strings.exportFailedLimbo);
            }
        } catch (error) {
            console.error('Export error:', error);
            alert(`${limboExportData.strings.exportFailed} ${error.message}`);
        } finally {
            const exportButton = document.getElementById('export-selected');
            exportButton.disabled = false;
            exportButton.textContent = limboExportData.strings.exportSelected;
        }
    });

    // Initial load
    fetchProducts();
}); 