<?php

namespace Limb_Chatbot\Includes\Widgets\Callbacks;

use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;

/**
 * Widget Callback Factory
 *
 * Creates appropriate callback handlers based on widget type.
 *
 * @since 1.0.11
 */
class Widget_Callback_Factory {

	/**
	 * Create a callback handler for the given widget type.
	 *
	 * @param  string  $widget_type  Widget type.
	 *
	 * @return Widget_Callback_Interface|null Callback handler or null if not found.
	 * @since 1.0.11
	 */
	public function make( string $widget_type ): ?Widget_Callback_Interface {
		$callback_class = $this->get_callback_class( $widget_type );

		if ( ! $callback_class || ! class_exists( $callback_class ) ) {
			throw new Exception( Error_Codes::NOT_SUPPORTED, __( 'Widget callback not supported', 'limb-chatbot' ) );
		}

		return new $callback_class();
	}

	/**
	 * Get the callback class for a widget type.
	 *
	 * @param  string  $widget_type  Widget type.
	 *
	 * @return string|null Fully qualified class name or null.
	 * @since 1.0.11
	 */
	private function get_callback_class( string $widget_type ): ?string {
		$callbacks = [
			'lead_capture' => Lead_Capture_Callback::class,
			'prompt'       => null, // No callback for prompt
			'message'      => null, // No callback for message
			'link'         => null, // No callback for link
			'text'         => null, // No callback for text
			'chat_history' => null, // No callback for chat history
		];

		return $callbacks[ $widget_type ] ?? null;
	}
}
