<?php

namespace Limb_Chatbot\Includes\Vector_Dbs\Pinecone\Endpoints;

use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Services\Collection;
use Limb_Chatbot\Includes\Vector_Dbs\Pinecone\Response_Handler;

/**
 * Class Index_Endpoint
 *
 * Handles creation and deletion of Pinecone vector indexes via HTTP API.
 *
 * @package Limb_Chatbot\Includes\Vector_Dbs\Pinecone\Endpoints
 *
 * @since 1.0.0
 */
class Index_Endpoint extends Pinecone_Endpoint {

	/**
	 * Create a new Pinecone index with specified parameters.
	 *
	 * Constructs the request body based on vector index settings and
	 * serverless or pod spec, then sends a POST request to create the index.
	 *
	 * @return Collection|null Array of metadata returned by the Pinecone API on success.
	 *
	 * @throws \Exception If the HTTP request fails or API returns an error.
	 *
	 * @since 1.0.0
	 */
	public function create() {
		$utility      = $this->utility;
		$vector_index = $utility->get_global_utility()->get_vector_index();
		$body         = [
			'name'      => $vector_index->get_name(),
			'dimension' => $utility->get_dimension(),
			'metric'    => $utility->get_metric(),
		];
		if ( $utility->is_serverless_spec() ) {
			$body['spec']['serverless'] = [
				'cloud'  => $utility->get_cloud(),
				'region' => $utility->get_region(),
			];
		} elseif ( $utility->is_pod_spec() ) {
			$body['spec']['pod'] = [
				'environment' => $utility->get_environment(),
				'pod_type'    => $utility->get_pod_type(),
				'pods'        => $utility->get_pods(),
			];
		}
		$http_client = $this->http_client_factory();
		$response    = $http_client->post( self::API_BASE_URL . '/indexes', array(
			'headers' => $this->get_header(),
			'body'    => json_encode( $body ),
			'timeout' => $utility->get_timeout()
		) );

		return ( new Index\Response_Handler( $response, $http_client, $this ) )->get_metas();
	}

	/**
	 * Delete an existing Pinecone index by name.
	 *
	 * Sends a DELETE request to the Pinecone API to remove the index.
	 * If the index is not found, returns true gracefully.
	 *
	 * @return bool True on successful deletion or if index not found.
	 *
	 * @throws Exception If deletion fails for reasons other than "not found".
	 *
	 * @since 1.0.0
	 */
	public function delete() {
		$utility      = $this->utility;
		$vector_index = $utility->get_global_utility()->get_vector_index();
		$http_client  = $this->http_client_factory();
		try {
			$response = $http_client->post( self::API_BASE_URL . '/indexes/' . $vector_index->get_name(), array(
				'method'  => 'DELETE',
				'headers' => $this->get_header(),
				'timeout' => $utility->get_timeout()
			) );
			$result   = ( new Response_Handler( $response, $http_client, $this ) );

			return $result->status_starts_with_2();

		} catch ( Exception $e ) {
			if ( ( $e->get_error_data() instanceof Response_Handler ) && $e->get_error_data()->status_code_is_not_found() ) {
				return true;
			}
			throw $e;
		}
	}
}